/*
 * xmms-itouch - iTouch keyboard control plugin for xmms
 * Copyright (C) 2000-2002 Ville Syrjl <syrjala@sci.fi>
 *                         Bryn Davies <curious@ihug.com.au>
 *                         Jonathan A. Davis <davis@jdhouse.org>
 *                         Jeremy Tan <nsx@nsx.homeip.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <X11/Xlib.h>

#include "i18n.h"
#include <gtk/gtk.h>
#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>
#include <xmms/configfile.h>

typedef struct {
	gint vol_increment;
	gint vol_decrement;

	gint suspend;
	gint my_home;
	gint email;
	gint search;
	gint go;
	gint mute;
	gint vol_down;
	gint vol_up;
	gint play;
	gint stop;
	gint prev_track;
	gint next_track;
	gint logitech;
	gint calculator;

	gchar *suspend_command;
	gchar *my_home_command;
	gchar *email_command;
	gchar *search_command;
	gchar *go_command;
	gchar *logitech_command;
	gchar *calculator_command;
} iTouchConfig;

iTouchConfig itouch_cfg;

static Display *display;
static Window root;
static gint timeout_tag = 0;
static gint init_done = 0;
static gint grabbed = 0;

static GtkWidget *suspend_entry, *my_home_entry, *email_entry,
    *search_entry, *go_entry, *logitech_entry, *calculator_entry;
static GtkWidget *vol_increment_spin, *vol_decrement_spin, *suspend_spin,
    *my_home_spin, *email_spin, *search_spin, *go_spin, *mute_spin,
    *vol_down_spin, *vol_up_spin, *play_spin, *stop_spin, *prev_track_spin,
    *next_track_spin, *logitech_spin, *calculator_spin;
static GtkWidget *config_win, *ktable;

void init(void);
void cleanup(void);
static gint timeout_func(gpointer);
void about(void);
void config(void);
static void exec_command(gchar *);
void config_ok_cb(GtkWidget *, gpointer);

void load_kbdcfgs(void);
void load_cfg(void);
void save_cfg(void);

typedef struct {
	gchar *name;
	gint suspend;
	gint my_home;
	gint email;
	gint search;
	gint go;
	gint mute;
	gint vol_down;
	gint vol_up;
	gint play;
	gint stop;
	gint prev_track;
	gint next_track;
	gint logitech;
	gint calculator;
} KeyboardConfig;

KeyboardConfig *kbdcfgs = NULL;
gint kbdcount = 0;

GeneralPlugin gp = {
	NULL,			/* handle */
	NULL,			/* filename */
	-1,			/* xmms_session */
	"iTouch Multimedia Keyboard Control",	/* Description */
	init,
	about,
	config,
	cleanup
};

GeneralPlugin *get_gplugin_info(void)
{
	gp.description = g_strdup_printf(_("iTouch Keyboard Control %s"),
					 VERSION);
	return &gp;
}


void grab_keys(Display * display, Window root)
{
	if (grabbed)
		return;
	if (itouch_cfg.suspend)
		XGrabKey(display, itouch_cfg.suspend, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.my_home)
		XGrabKey(display, itouch_cfg.my_home, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.email)
		XGrabKey(display, itouch_cfg.email, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.search)
		XGrabKey(display, itouch_cfg.search, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.go)
		XGrabKey(display, itouch_cfg.go, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.mute)
		XGrabKey(display, itouch_cfg.mute, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.vol_down)
		XGrabKey(display, itouch_cfg.vol_down, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.vol_up)
		XGrabKey(display, itouch_cfg.vol_up, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.play)
		XGrabKey(display, itouch_cfg.play, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.stop)
		XGrabKey(display, itouch_cfg.stop, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.prev_track)
		XGrabKey(display, itouch_cfg.prev_track, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.next_track)
		XGrabKey(display, itouch_cfg.next_track, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.logitech)
		XGrabKey(display, itouch_cfg.logitech, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	if (itouch_cfg.calculator)
		XGrabKey(display, itouch_cfg.calculator, AnyModifier, root,
			 False, GrabModeAsync, GrabModeAsync);
	grabbed = 1;
}

void ungrab_keys(Display * display, Window root)
{
	if (!grabbed)
		return;
	if (itouch_cfg.suspend)
		XUngrabKey(display, itouch_cfg.suspend, AnyModifier, root);
	if (itouch_cfg.my_home)
		XUngrabKey(display, itouch_cfg.my_home, AnyModifier, root);
	if (itouch_cfg.email)
		XUngrabKey(display, itouch_cfg.email, AnyModifier, root);
	if (itouch_cfg.search)
		XUngrabKey(display, itouch_cfg.search, AnyModifier, root);
	if (itouch_cfg.go)
		XUngrabKey(display, itouch_cfg.go, AnyModifier, root);
	if (itouch_cfg.mute)
		XUngrabKey(display, itouch_cfg.mute, AnyModifier, root);
	if (itouch_cfg.vol_down)
		XUngrabKey(display, itouch_cfg.vol_down, AnyModifier,
			   root);
	if (itouch_cfg.vol_up)
		XUngrabKey(display, itouch_cfg.vol_up, AnyModifier, root);
	if (itouch_cfg.play)
		XUngrabKey(display, itouch_cfg.play, AnyModifier, root);
	if (itouch_cfg.stop)
		XUngrabKey(display, itouch_cfg.stop, AnyModifier, root);
	if (itouch_cfg.prev_track)
		XUngrabKey(display, itouch_cfg.prev_track, AnyModifier,
			   root);
	if (itouch_cfg.next_track)
		XUngrabKey(display, itouch_cfg.next_track, AnyModifier,
			   root);
	if (itouch_cfg.logitech)
		XUngrabKey(display, itouch_cfg.logitech, AnyModifier,
			   root);
	if (itouch_cfg.calculator)
		XUngrabKey(display, itouch_cfg.calculator, AnyModifier,
			   root);
	grabbed = 0;
}

static void handle_keyevent(int keycode)
{
	static gint volume, old_volume;
	gboolean playing, muted;

	playing = xmms_remote_is_playing(gp.xmms_session);
	volume = xmms_remote_get_main_volume(gp.xmms_session);
	if (volume)
		muted = FALSE;
	else
		muted = TRUE;

	if (keycode == itouch_cfg.suspend) {
		exec_command(itouch_cfg.suspend_command);
		return;
	}
	if (keycode == itouch_cfg.my_home) {
		exec_command(itouch_cfg.my_home_command);
		return;
	}
	if (keycode == itouch_cfg.email) {
		exec_command(itouch_cfg.email_command);
		return;
	}
	if (keycode == itouch_cfg.search) {
		exec_command(itouch_cfg.search_command);
		return;
	}
	if (keycode == itouch_cfg.go) {
		exec_command(itouch_cfg.go_command);
		return;
	}
	if (keycode == itouch_cfg.mute) {
		if (!muted) {
			old_volume = volume;
			volume = 0;
			muted = TRUE;
		} else {
			volume = old_volume;
			muted = FALSE;
		}

		xmms_remote_set_main_volume(gp.xmms_session, volume);
		return;
	}
	if (keycode == itouch_cfg.vol_down) {
		if (muted) {
			volume = old_volume;
			old_volume = 0;
			muted = FALSE;
		}

		if ((volume -= itouch_cfg.vol_decrement) < 0)
			volume = 0;
		if (volume != old_volume)
			xmms_remote_set_main_volume(gp.xmms_session,
						    volume);
		old_volume = volume;
		return;
	}
	if (keycode == itouch_cfg.vol_up) {
		if (muted) {
			volume = old_volume;
			old_volume = 0;
			muted = FALSE;
		}

		if ((volume += itouch_cfg.vol_increment) > 100)
			volume = 100;
		if (volume != old_volume)
			xmms_remote_set_main_volume(gp.xmms_session,
						    volume);
		old_volume = volume;
		return;
	}
	if (keycode == itouch_cfg.play) {
		if (!playing)
			xmms_remote_play(gp.xmms_session);
		else
			xmms_remote_pause(gp.xmms_session);
		return;
	}
	if (keycode == itouch_cfg.stop) {
		if (playing)
			xmms_remote_stop(gp.xmms_session);
		else
			xmms_remote_eject(gp.xmms_session);
		return;
	}
	if (keycode == itouch_cfg.prev_track) {
		xmms_remote_playlist_prev(gp.xmms_session);
		return;
	}
	if (keycode == itouch_cfg.next_track) {
		xmms_remote_playlist_next(gp.xmms_session);
		return;
	}
	if (keycode == itouch_cfg.logitech) {
		exec_command(itouch_cfg.logitech_command);
		return;
	}
	if (keycode == itouch_cfg.calculator) {
		exec_command(itouch_cfg.calculator_command);
		return;
	}
}

void cleanup(void)
{
	ungrab_keys(display, root);
	if (timeout_tag)
		gtk_timeout_remove(timeout_tag);
	timeout_tag = 0;
	XCloseDisplay(display);

	free(kbdcfgs);
}

void x_init(void)
{
	char *display_name;

	display_name = getenv("DISPLAY");
	if (!display_name) {
		perror(_("No display\n"));
		return;
	}
	display = XOpenDisplay(display_name);
	if (!display) {
		perror(_("Can't open display\n"));
		return;
	}
	root = DefaultRootWindow(display);
	if (!root) {
		perror(_("No root window\n"));
		return;
	}
	init_done = 1;
}

void init(void)
{
	x_init();
	timeout_tag = gtk_timeout_add(100, timeout_func, NULL);

	load_cfg();

	grab_keys(display, root);
}

static gint timeout_func(gpointer data)
{
	long mask = KeyPressMask;
	XEvent event;

	while (XCheckMaskEvent(display, mask, &event))
		handle_keyevent(event.xkey.keycode);

	return TRUE;
}

void about(void)
{
	static GtkWidget *dialog;

	if (dialog != NULL) {
		gdk_window_raise(dialog->window);
		return;
	}

	dialog = xmms_show_message(_("About iTouch Plugin"),
				   _
				   ("XMMS iTouch Keyboard Control Plugin\n"
				    "Copyright (C) 2000-2002 Ville Syrjl <syrjala@sci.fi>\n"
				    "                        Bryn Davies <curious@ihug.com.au>\n"
				    "                        Jonathan A. Davis <davis@jdhouse.org>\n"
				    "                        Jeremy Tan <nsx@nsx.homeip.net>\n\n"
				    "This program is free software; you can redistribute it and/or modify\n"
				    "it under the terms of the GNU General Public License as published by\n"
				    "the Free Software Foundation; either version 2 of the License, or\n"
				    "(at your option) any later version.\n\n"
				    "This program is distributed in the hope that it will be useful,\n"
				    "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
				    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
				    "GNU General Public License for more details.\n\n"
				    "You should have received a copy of the GNU General Public License\n"
				    "along with this program; if not, write to the Free Software\n"
				    "Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA."),
				   _("Ok"), FALSE, NULL, NULL);

	gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
			   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &dialog);
}

void kbd_change(GtkWidget * w, gpointer p)
{
	gint i = (gint) p;
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(suspend_spin),
				  kbdcfgs[i].suspend);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(my_home_spin),
				  kbdcfgs[i].my_home);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(email_spin),
				  kbdcfgs[i].email);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(search_spin),
				  kbdcfgs[i].search);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(go_spin), kbdcfgs[i].go);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(mute_spin),
				  kbdcfgs[i].mute);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(vol_down_spin),
				  kbdcfgs[i].vol_down);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(vol_up_spin),
				  kbdcfgs[i].vol_up);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(play_spin),
				  kbdcfgs[i].play);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(stop_spin),
				  kbdcfgs[i].stop);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(prev_track_spin),
				  kbdcfgs[i].prev_track);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(next_track_spin),
				  kbdcfgs[i].next_track);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(logitech_spin),
				  kbdcfgs[i].logitech);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(calculator_spin),
				  kbdcfgs[i].calculator);
}

void grab_keyboard(GtkWidget * w)
{
	GtkWidget *message;
	long mask = KeyPressMask;
	XEvent event;

	if (!init_done)
		x_init();

	XGrabKeyboard(display, root, False, GrabModeAsync, GrabModeAsync,
		      CurrentTime);

	XMaskEvent(display, mask, &event);

	XUngrabKeyboard(display, CurrentTime);

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), event.xkey.keycode);
}

void set_sensitive(GtkWidget * w)
{
	gtk_widget_set_sensitive(w, TRUE);
}

void set_insensitive(GtkWidget * w)
{
	gtk_widget_set_sensitive(w, FALSE);
}

void config(void)
{
	GtkWidget *vbox, *frame, *table, *label, *kbox, *kmenu, *button,
	    *menu, *item, *bbox, *cancel_button, *ok_button;

	gint current_kbdcfg, i;

	if (config_win) {
		gdk_window_raise(config_win->window);
		return;
	}

	if (!kbdcfgs)
		load_kbdcfgs();

	load_cfg();

	/* Safety measure */
	ungrab_keys(display, root);

	current_kbdcfg = kbdcount;
	for (i = 0; i < kbdcount; i++) {
		if ((itouch_cfg.suspend == kbdcfgs[i].suspend) &&
		    (itouch_cfg.my_home == kbdcfgs[i].my_home) &&
		    (itouch_cfg.email == kbdcfgs[i].email) &&
		    (itouch_cfg.search == kbdcfgs[i].search) &&
		    (itouch_cfg.go == kbdcfgs[i].go) &&
		    (itouch_cfg.mute == kbdcfgs[i].mute) &&
		    (itouch_cfg.vol_down == kbdcfgs[i].vol_down) &&
		    (itouch_cfg.vol_up == kbdcfgs[i].vol_up) &&
		    (itouch_cfg.play == kbdcfgs[i].play) &&
		    (itouch_cfg.stop == kbdcfgs[i].stop) &&
		    (itouch_cfg.prev_track == kbdcfgs[i].prev_track) &&
		    (itouch_cfg.next_track == kbdcfgs[i].next_track) &&
		    (itouch_cfg.logitech == kbdcfgs[i].logitech) &&
		    (itouch_cfg.calculator == kbdcfgs[i].calculator))
			current_kbdcfg = i;
	}

	config_win = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_container_border_width(GTK_CONTAINER(config_win), 10);
	gtk_window_set_title(GTK_WINDOW(config_win),
			     "iTouch Configuration");
	gtk_signal_connect(GTK_OBJECT(config_win), "destroy",
			   GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			   &config_win);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(config_win), vbox);



	frame = gtk_frame_new("Keyboard Model");
	gtk_container_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	kbox = gtk_vbox_new(FALSE, 10);
	gtk_container_border_width(GTK_CONTAINER(kbox), 5);
	gtk_container_add(GTK_CONTAINER(frame), kbox);

	kmenu = gtk_option_menu_new();
	gtk_box_pack_start(GTK_BOX(kbox), kmenu, TRUE, TRUE, 0);

	ktable = gtk_table_new(7, 6, FALSE);
	gtk_box_pack_start(GTK_BOX(kbox), ktable, TRUE, TRUE, 0);

	label = gtk_label_new("Suspend");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 0, 1, 0, 1);
	suspend_spin = gtk_spin_button_new((GtkAdjustment *)
					   gtk_adjustment_new(0, 0, 255, 1,
							      5, 0), 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(suspend_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(suspend_spin),
				  itouch_cfg.suspend);
	gtk_table_attach_defaults(GTK_TABLE(ktable), suspend_spin, 1, 2, 0,
				  1);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(suspend_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 2, 3, 0, 1);

	label = gtk_label_new("My Home");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 0, 1, 1, 2);
	my_home_spin = gtk_spin_button_new((GtkAdjustment *)
					   gtk_adjustment_new(0, 0, 255, 1,
							      5, 0), 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(my_home_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(my_home_spin),
				  itouch_cfg.my_home);
	gtk_table_attach_defaults(GTK_TABLE(ktable), my_home_spin, 1, 2, 1,
				  2);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(my_home_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 2, 3, 1, 2);

	label = gtk_label_new("Email");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 0, 1, 2, 3);
	email_spin = gtk_spin_button_new((GtkAdjustment *)
					 gtk_adjustment_new(0, 0, 255, 1,
							    5, 0), 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(email_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(email_spin),
				  itouch_cfg.email);
	gtk_table_attach_defaults(GTK_TABLE(ktable), email_spin, 1, 2, 2,
				  3);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(email_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 2, 3, 2, 3);

	label = gtk_label_new("Search");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 0, 1, 3, 4);
	search_spin = gtk_spin_button_new((GtkAdjustment *)
					  gtk_adjustment_new(0, 0, 255, 1,
							     5, 0), 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(search_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(search_spin),
				  itouch_cfg.search);
	gtk_table_attach_defaults(GTK_TABLE(ktable), search_spin, 1, 2, 3,
				  4);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(search_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 2, 3, 3, 4);

	label = gtk_label_new("Go");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 0, 1, 4, 5);
	go_spin = gtk_spin_button_new((GtkAdjustment *)
				      gtk_adjustment_new(0, 0, 255, 1, 5,
							 0), 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(go_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(go_spin), itouch_cfg.go);
	gtk_table_attach_defaults(GTK_TABLE(ktable), go_spin, 1, 2, 4, 5);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(go_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 2, 3, 4, 5);

	label = gtk_label_new("Mute");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 0, 1, 5, 6);
	mute_spin = gtk_spin_button_new((GtkAdjustment *)
					gtk_adjustment_new(0, 0, 255, 1, 5,
							   0), 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(mute_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(mute_spin),
				  itouch_cfg.mute);
	gtk_table_attach_defaults(GTK_TABLE(ktable), mute_spin, 1, 2, 5,
				  6);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(mute_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 2, 3, 5, 6);

	label = gtk_label_new("Volume Down");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 0, 1, 6, 7);
	vol_down_spin = gtk_spin_button_new((GtkAdjustment *)
					    gtk_adjustment_new(0, 0, 255,
							       1, 5, 0), 0,
					    0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(vol_down_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(vol_down_spin),
				  itouch_cfg.vol_down);
	gtk_table_attach_defaults(GTK_TABLE(ktable), vol_down_spin, 1, 2,
				  6, 7);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(vol_down_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 2, 3, 6, 7);

	label = gtk_label_new("Volume Up");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 3, 4, 0, 1);
	vol_up_spin = gtk_spin_button_new((GtkAdjustment *)
					  gtk_adjustment_new(0, 0, 255, 1,
							     5, 0), 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(vol_up_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(vol_up_spin),
				  itouch_cfg.vol_up);
	gtk_table_attach_defaults(GTK_TABLE(ktable), vol_up_spin, 4, 5, 0,
				  1);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(vol_up_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 5, 6, 0, 1);

	label = gtk_label_new("Play");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 3, 4, 1, 2);
	play_spin = gtk_spin_button_new((GtkAdjustment *)
					gtk_adjustment_new(0, 0, 255, 1, 5,
							   0), 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(play_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(play_spin),
				  itouch_cfg.play);
	gtk_table_attach_defaults(GTK_TABLE(ktable), play_spin, 4, 5, 1,
				  2);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(play_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 5, 6, 1, 2);

	label = gtk_label_new("Stop");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 3, 4, 2, 3);
	stop_spin = gtk_spin_button_new((GtkAdjustment *)
					gtk_adjustment_new(0, 0, 255, 1, 5,
							   0), 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(stop_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(stop_spin),
				  itouch_cfg.stop);
	gtk_table_attach_defaults(GTK_TABLE(ktable), stop_spin, 4, 5, 2,
				  3);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(stop_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 5, 6, 2, 3);

	label = gtk_label_new("Previous Track");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 3, 4, 3, 4);
	prev_track_spin = gtk_spin_button_new((GtkAdjustment *)
					      gtk_adjustment_new(0, 0, 255,
								 1, 5, 0),
					      0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(prev_track_spin),
				    TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(prev_track_spin),
				  itouch_cfg.prev_track);
	gtk_table_attach_defaults(GTK_TABLE(ktable), prev_track_spin, 4, 5,
				  3, 4);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(prev_track_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 5, 6, 3, 4);

	label = gtk_label_new("Next Track");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 3, 4, 4, 5);
	next_track_spin = gtk_spin_button_new((GtkAdjustment *)
					      gtk_adjustment_new(0, 0, 255,
								 1, 5, 0),
					      0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(next_track_spin),
				    TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(next_track_spin),
				  itouch_cfg.next_track);
	gtk_table_attach_defaults(GTK_TABLE(ktable), next_track_spin, 4, 5,
				  4, 5);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(next_track_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 5, 6, 4, 5);

	label = gtk_label_new("Logitech");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 3, 4, 5, 6);
	logitech_spin = gtk_spin_button_new((GtkAdjustment *)
					    gtk_adjustment_new(0, 0, 255,
							       1, 5, 0), 0,
					    0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(logitech_spin), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(logitech_spin),
				  itouch_cfg.logitech);
	gtk_table_attach_defaults(GTK_TABLE(ktable), logitech_spin, 4, 5,
				  5, 6);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(logitech_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 5, 6, 5, 6);

	label = gtk_label_new("Calculator");
	gtk_table_attach_defaults(GTK_TABLE(ktable), label, 3, 4, 6, 7);
	calculator_spin = gtk_spin_button_new((GtkAdjustment *)
					      gtk_adjustment_new(0, 0, 255,
								 1, 5, 0),
					      0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(calculator_spin),
				    TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(calculator_spin),
				  itouch_cfg.calculator);
	gtk_table_attach_defaults(GTK_TABLE(ktable), calculator_spin, 4, 5,
				  6, 7);
	button = gtk_button_new_with_label("Grab");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  GTK_SIGNAL_FUNC(grab_keyboard),
				  GTK_OBJECT(calculator_spin));
	gtk_table_attach_defaults(GTK_TABLE(ktable), button, 5, 6, 6, 7);

	if (current_kbdcfg != kbdcount)
		set_insensitive(GTK_WIDGET(ktable));

	menu = gtk_menu_new();
	for (i = 0; i < kbdcount; i++) {
		item = gtk_menu_item_new_with_label(kbdcfgs[i].name);
		gtk_signal_connect(GTK_OBJECT(item),
				   "activate",
				   GTK_SIGNAL_FUNC(kbd_change),
				   GINT_TO_POINTER(i));
		gtk_signal_connect_object(GTK_OBJECT(item),
					  "activate",
					  GTK_SIGNAL_FUNC(set_insensitive),
					  GTK_OBJECT(ktable));
		gtk_menu_append(GTK_MENU(menu), item);
	}
	item = gtk_menu_item_new_with_label(_("Custom"));
	gtk_signal_connect_object(GTK_OBJECT(item),
				  "activate",
				  GTK_SIGNAL_FUNC(set_sensitive),
				  GTK_OBJECT(ktable));
	gtk_menu_append(GTK_MENU(menu), item);

	gtk_option_menu_remove_menu(GTK_OPTION_MENU(kmenu));
	gtk_option_menu_set_menu(GTK_OPTION_MENU(kmenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(kmenu),
				    current_kbdcfg);


	frame = gtk_frame_new("Keybindings");
	gtk_container_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	table = gtk_table_new(6, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), 5);
	gtk_container_add(GTK_CONTAINER(frame), table);

	label = gtk_label_new("Suspend:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);

	label = gtk_label_new("My Home:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);

	label = gtk_label_new("Email:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);

	label = gtk_label_new("Search:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);

	label = gtk_label_new("Go:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);

	label = gtk_label_new("Logitech:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 5, 6);

	label = gtk_label_new("Calculator:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 6, 7);

	suspend_entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(suspend_entry),
			   itouch_cfg.suspend_command);
	gtk_table_attach(GTK_TABLE(table), suspend_entry, 1, 2, 0, 1,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions) (0), 0, 0);

	my_home_entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(my_home_entry),
			   itouch_cfg.my_home_command);
	gtk_table_attach(GTK_TABLE(table), my_home_entry, 1, 2, 1, 2,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions) (0), 0, 0);

	email_entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(email_entry),
			   itouch_cfg.email_command);
	gtk_table_attach(GTK_TABLE(table), email_entry, 1, 2, 2, 3,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions) (0), 0, 0);

	search_entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(search_entry),
			   itouch_cfg.search_command);
	gtk_table_attach(GTK_TABLE(table), search_entry, 1, 2, 3, 4,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions) (0), 0, 0);

	go_entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(go_entry), itouch_cfg.go_command);
	gtk_table_attach(GTK_TABLE(table), go_entry, 1, 2, 4, 5,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions) (0), 0, 0);

	logitech_entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(logitech_entry),
			   itouch_cfg.logitech_command);
	gtk_table_attach(GTK_TABLE(table), logitech_entry, 1, 2, 5, 6,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions) (0), 0, 0);

	calculator_entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(calculator_entry),
			   itouch_cfg.calculator_command);
	gtk_table_attach(GTK_TABLE(table), calculator_entry, 1, 2, 6, 7,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions) (0), 0, 0);


	frame = gtk_frame_new("Volume Adjustment");
	gtk_container_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	table = gtk_table_new(7, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), 5);
	gtk_container_add(GTK_CONTAINER(frame), table);

	label = gtk_label_new("Volume Increment (%)");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
	vol_increment_spin = gtk_spin_button_new((GtkAdjustment *)
						 gtk_adjustment_new(8, 1,
								    50, 1,
								    5, 0),
						 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(vol_increment_spin),
				    TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(vol_increment_spin),
				  itouch_cfg.vol_increment);
	gtk_table_attach_defaults(GTK_TABLE(table), vol_increment_spin, 1,
				  2, 0, 1);

	label = gtk_label_new("Volume Decrement (%)");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
	vol_decrement_spin = gtk_spin_button_new((GtkAdjustment *)
						 gtk_adjustment_new(8, 1,
								    50, 1,
								    5, 0),
						 0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(vol_decrement_spin),
				    TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(vol_decrement_spin),
				  itouch_cfg.vol_decrement);
	gtk_table_attach_defaults(GTK_TABLE(table), vol_decrement_spin, 1,
				  2, 1, 2);


	bbox = gtk_hbutton_box_new();
	gtk_box_pack_start(GTK_BOX(vbox), bbox, TRUE, TRUE, 0);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);

	ok_button = gtk_button_new_with_label("Ok");
	gtk_container_add(GTK_CONTAINER(bbox), ok_button);
	gtk_signal_connect(GTK_OBJECT(ok_button), "clicked",
			   GTK_SIGNAL_FUNC(config_ok_cb),
			   GTK_OBJECT(ok_button));
	GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(ok_button);

	cancel_button = gtk_button_new_with_label("Cancel");
	gtk_container_add(GTK_CONTAINER(bbox), cancel_button);
	gtk_signal_connect_object(GTK_OBJECT(cancel_button), "clicked",
				  GTK_SIGNAL_FUNC(gtk_widget_destroy),
				  GTK_OBJECT(config_win));
	GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);

	gtk_widget_show_all(config_win);
}

void config_ok_cb(GtkWidget * w, gpointer data)
{
	itouch_cfg.vol_increment =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (vol_increment_spin));
	itouch_cfg.vol_decrement =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (vol_decrement_spin));

	itouch_cfg.suspend =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (suspend_spin));
	itouch_cfg.suspend =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (suspend_spin));
	itouch_cfg.my_home =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (my_home_spin));
	itouch_cfg.email =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(email_spin));
	itouch_cfg.search =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(search_spin));
	itouch_cfg.go =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(go_spin));
	itouch_cfg.mute =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(mute_spin));
	itouch_cfg.vol_down =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (vol_down_spin));
	itouch_cfg.vol_up =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(vol_up_spin));
	itouch_cfg.play =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(play_spin));
	itouch_cfg.stop =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(stop_spin));
	itouch_cfg.prev_track =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (prev_track_spin));
	itouch_cfg.next_track =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (next_track_spin));
	itouch_cfg.logitech =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (logitech_spin));
	itouch_cfg.calculator =
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (calculator_spin));

	g_free(itouch_cfg.suspend_command);
	g_free(itouch_cfg.my_home_command);
	g_free(itouch_cfg.email_command);
	g_free(itouch_cfg.search_command);
	g_free(itouch_cfg.go_command);
	g_free(itouch_cfg.logitech_command);
	g_free(itouch_cfg.calculator_command);

	itouch_cfg.suspend_command =
	    g_strdup_printf(gtk_entry_get_text(GTK_ENTRY(suspend_entry)));
	itouch_cfg.my_home_command =
	    g_strdup_printf(gtk_entry_get_text(GTK_ENTRY(my_home_entry)));
	itouch_cfg.email_command =
	    g_strdup_printf(gtk_entry_get_text(GTK_ENTRY(email_entry)));
	itouch_cfg.search_command =
	    g_strdup_printf(gtk_entry_get_text(GTK_ENTRY(search_entry)));
	itouch_cfg.go_command =
	    g_strdup_printf(gtk_entry_get_text(GTK_ENTRY(go_entry)));
	itouch_cfg.logitech_command =
	    g_strdup_printf(gtk_entry_get_text(GTK_ENTRY(logitech_entry)));
	itouch_cfg.calculator_command =
	    g_strdup_printf(gtk_entry_get_text
			    (GTK_ENTRY(calculator_entry)));

	save_cfg();

	grab_keys(display, root);

	gtk_widget_destroy(config_win);
}

void load_kbdcfgs(void)
{
	ConfigFile *cfgfile;
	gint i;

	if (cfgfile = xmms_cfg_open_file(DATADIR "/xmms-itouch.config")) {
		xmms_cfg_read_int(cfgfile, "Global", "config_count",
				  &kbdcount);

		kbdcfgs = calloc(kbdcount, sizeof(KeyboardConfig));
		for (i = 0; i < kbdcount; i++) {
			gchar *cfgname = g_strdup_printf("config_%d", i);
			xmms_cfg_read_string(cfgfile, cfgname, "name",
					     &kbdcfgs[i].name);
			xmms_cfg_read_int(cfgfile, cfgname, "suspend",
					  &kbdcfgs[i].suspend);
			xmms_cfg_read_int(cfgfile, cfgname, "my_home",
					  &kbdcfgs[i].my_home);
			xmms_cfg_read_int(cfgfile, cfgname, "email",
					  &kbdcfgs[i].email);
			xmms_cfg_read_int(cfgfile, cfgname, "search",
					  &kbdcfgs[i].search);
			xmms_cfg_read_int(cfgfile, cfgname, "go",
					  &kbdcfgs[i].go);
			xmms_cfg_read_int(cfgfile, cfgname, "mute",
					  &kbdcfgs[i].mute);
			xmms_cfg_read_int(cfgfile, cfgname, "vol_down",
					  &kbdcfgs[i].vol_down);
			xmms_cfg_read_int(cfgfile, cfgname, "vol_up",
					  &kbdcfgs[i].vol_up);
			xmms_cfg_read_int(cfgfile, cfgname, "play",
					  &kbdcfgs[i].play);
			xmms_cfg_read_int(cfgfile, cfgname, "stop",
					  &kbdcfgs[i].stop);
			xmms_cfg_read_int(cfgfile, cfgname, "prev_track",
					  &kbdcfgs[i].prev_track);
			xmms_cfg_read_int(cfgfile, cfgname, "next_track",
					  &kbdcfgs[i].next_track);
			xmms_cfg_read_int(cfgfile, cfgname, "logitech",
					  &kbdcfgs[i].logitech);
			xmms_cfg_read_int(cfgfile, cfgname, "calculator",
					  &kbdcfgs[i].calculator);
			g_free(cfgname);
		}
		xmms_cfg_free(cfgfile);
	}
}

void load_cfg(void)
{
	ConfigFile *cfgfile;

	/* 8% default */
	itouch_cfg.vol_increment = 8;
	itouch_cfg.vol_decrement = 8;

	itouch_cfg.suspend = 0;
	itouch_cfg.my_home = 0;
	itouch_cfg.email = 0;
	itouch_cfg.search = 0;
	itouch_cfg.go = 0;
	itouch_cfg.mute = 0;
	itouch_cfg.vol_down = 0;
	itouch_cfg.vol_up = 0;
	itouch_cfg.play = 0;
	itouch_cfg.stop = 0;
	itouch_cfg.prev_track = 0;
	itouch_cfg.next_track = 0;
	itouch_cfg.logitech = 0;
	itouch_cfg.calculator = 0;

	if (itouch_cfg.suspend_command) {
		g_free(itouch_cfg.suspend_command);
		itouch_cfg.suspend_command = NULL;
	}
	if (itouch_cfg.my_home_command) {
		g_free(itouch_cfg.my_home_command);
		itouch_cfg.my_home_command = NULL;
	}
	if (itouch_cfg.email_command) {
		g_free(itouch_cfg.email_command);
		itouch_cfg.email_command = NULL;
	}
	if (itouch_cfg.search_command) {
		g_free(itouch_cfg.search_command);
		itouch_cfg.search_command = NULL;
	}
	if (itouch_cfg.go_command) {
		g_free(itouch_cfg.go_command);
		itouch_cfg.go_command = NULL;
	}
	if (itouch_cfg.logitech_command) {
		g_free(itouch_cfg.logitech_command);
		itouch_cfg.logitech_command = NULL;
	}
	if (itouch_cfg.calculator_command) {
		g_free(itouch_cfg.calculator_command);
		itouch_cfg.calculator_command = NULL;
	}

	if (cfgfile = xmms_cfg_open_default_file()) {
		xmms_cfg_read_int(cfgfile, "iTouch", "vol_increment",
				  &itouch_cfg.vol_increment);
		xmms_cfg_read_int(cfgfile, "iTouch", "vol_decrement",
				  &itouch_cfg.vol_decrement);

		xmms_cfg_read_int(cfgfile, "iTouch", "suspend",
				  &itouch_cfg.suspend);
		xmms_cfg_read_int(cfgfile, "iTouch", "my_home",
				  &itouch_cfg.my_home);
		xmms_cfg_read_int(cfgfile, "iTouch", "email",
				  &itouch_cfg.email);
		xmms_cfg_read_int(cfgfile, "iTouch", "search",
				  &itouch_cfg.search);
		xmms_cfg_read_int(cfgfile, "iTouch", "go", &itouch_cfg.go);
		xmms_cfg_read_int(cfgfile, "iTouch", "mute",
				  &itouch_cfg.mute);
		xmms_cfg_read_int(cfgfile, "iTouch", "vol_down",
				  &itouch_cfg.vol_down);
		xmms_cfg_read_int(cfgfile, "iTouch", "vol_up",
				  &itouch_cfg.vol_up);
		xmms_cfg_read_int(cfgfile, "iTouch", "play",
				  &itouch_cfg.play);
		xmms_cfg_read_int(cfgfile, "iTouch", "stop",
				  &itouch_cfg.stop);
		xmms_cfg_read_int(cfgfile, "iTouch", "prev_track",
				  &itouch_cfg.prev_track);
		xmms_cfg_read_int(cfgfile, "iTouch", "next_track",
				  &itouch_cfg.next_track);
		xmms_cfg_read_int(cfgfile, "iTouch", "logitech",
				  &itouch_cfg.logitech);
		xmms_cfg_read_int(cfgfile, "iTouch", "calculator",
				  &itouch_cfg.calculator);

		xmms_cfg_read_string(cfgfile, "iTouch", "suspend_command",
				     &itouch_cfg.suspend_command);
		xmms_cfg_read_string(cfgfile, "iTouch", "my_home_command",
				     &itouch_cfg.my_home_command);
		xmms_cfg_read_string(cfgfile, "iTouch", "email_command",
				     &itouch_cfg.email_command);
		xmms_cfg_read_string(cfgfile, "iTouch", "search_command",
				     &itouch_cfg.search_command);
		xmms_cfg_read_string(cfgfile, "iTouch", "go_command",
				     &itouch_cfg.go_command);
		xmms_cfg_read_string(cfgfile, "iTouch", "logitech_command",
				     &itouch_cfg.logitech_command);
		xmms_cfg_read_string(cfgfile, "iTouch",
				     "calculator_command",
				     &itouch_cfg.calculator_command);

		xmms_cfg_free(cfgfile);
	}

	if (!itouch_cfg.suspend_command)
		itouch_cfg.suspend_command =
		    g_strdup_printf("xscreensaver-command -lock");
	if (!itouch_cfg.my_home_command)
		itouch_cfg.my_home_command =
		    g_strdup_printf("mozilla $WWW_HOME");
	if (!itouch_cfg.email_command)
		itouch_cfg.email_command = g_strdup_printf("balsa");
	if (!itouch_cfg.search_command)
		itouch_cfg.search_command =
		    g_strdup_printf("mozilla http://www.google.com/linux");
	if (!itouch_cfg.go_command)
		itouch_cfg.go_command = g_strdup_printf("gnome-terminal");
	if (!itouch_cfg.logitech_command)
		itouch_cfg.logitech_command =
		    g_strdup_printf
		    ("mozilla http://www.saunalahti.fi/~syrjala/xmms-itouch/");
	if (!itouch_cfg.calculator_command)
		itouch_cfg.calculator_command = g_strdup_printf("gcalc");
}

void save_cfg(void)
{
	ConfigFile *cfgfile;

	cfgfile = xmms_cfg_open_default_file();

	xmms_cfg_write_int(cfgfile, "iTouch", "vol_increment",
			   itouch_cfg.vol_increment);
	xmms_cfg_write_int(cfgfile, "iTouch", "vol_decrement",
			   itouch_cfg.vol_decrement);

	xmms_cfg_write_int(cfgfile, "iTouch", "suspend",
			   itouch_cfg.suspend);
	xmms_cfg_write_int(cfgfile, "iTouch", "my_home",
			   itouch_cfg.my_home);
	xmms_cfg_write_int(cfgfile, "iTouch", "email", itouch_cfg.email);
	xmms_cfg_write_int(cfgfile, "iTouch", "search", itouch_cfg.search);
	xmms_cfg_write_int(cfgfile, "iTouch", "go", itouch_cfg.go);
	xmms_cfg_write_int(cfgfile, "iTouch", "mute", itouch_cfg.mute);
	xmms_cfg_write_int(cfgfile, "iTouch", "vol_down",
			   itouch_cfg.vol_down);
	xmms_cfg_write_int(cfgfile, "iTouch", "vol_up", itouch_cfg.vol_up);
	xmms_cfg_write_int(cfgfile, "iTouch", "play", itouch_cfg.play);
	xmms_cfg_write_int(cfgfile, "iTouch", "stop", itouch_cfg.stop);
	xmms_cfg_write_int(cfgfile, "iTouch", "prev_track",
			   itouch_cfg.prev_track);
	xmms_cfg_write_int(cfgfile, "iTouch", "next_track",
			   itouch_cfg.next_track);
	xmms_cfg_write_int(cfgfile, "iTouch", "logitech",
			   itouch_cfg.logitech);
	xmms_cfg_write_int(cfgfile, "iTouch", "calculator",
			   itouch_cfg.calculator);

	xmms_cfg_write_string(cfgfile, "iTouch", "suspend_command",
			      itouch_cfg.suspend_command);
	xmms_cfg_write_string(cfgfile, "iTouch", "my_home_command",
			      itouch_cfg.my_home_command);
	xmms_cfg_write_string(cfgfile, "iTouch", "email_command",
			      itouch_cfg.email_command);
	xmms_cfg_write_string(cfgfile, "iTouch", "search_command",
			      itouch_cfg.search_command);
	xmms_cfg_write_string(cfgfile, "iTouch", "go_command",
			      itouch_cfg.go_command);
	xmms_cfg_write_string(cfgfile, "iTouch", "logitech_command",
			      itouch_cfg.logitech_command);
	xmms_cfg_write_string(cfgfile, "iTouch", "calculator_command",
			      itouch_cfg.calculator_command);

	xmms_cfg_write_default_file(cfgfile);
	xmms_cfg_free(cfgfile);
}

static void exec_command(gchar * arg)
{
	int err;
	gchar *command;

	command = g_strconcat(arg, " &", NULL);
	err = system(command);
	if (err == 127)
		g_error("Execution of /bin/sh failed!\n");
	else if (err == -1)
		g_error("Unknown execution error!\n");
	g_free(command);
}
