/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <string.h>
#include <gtk/gtk.h>

#include "singit_debug.h"
#include "singit_macros.h"

#include "dlg_input_time.h"


enum {
	NEW_TIME,
	LAST_SIGNAL
};

static guint input_time_dialog_signals[LAST_SIGNAL] = { 0 };
static GtkWidgetClass *parent_class = NULL;

static void input_time_dialog_destroy (GtkObject *object);
void input_time_ok_button_event(GtkButton *button, gpointer user_data);
static void input_time_cancel_button_event(GtkButton *button, gpointer data);
gint input_time_dialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data);

void gtk_millisecond_adjustment_value_changed (GtkAdjustment *adjustment, gpointer user_data);
void input_time_change_time_event (GtkButton *button, gpointer *user_data);
void input_timeline_zero_clicked_event (GtkButton *button, gpointer *user_data);

void input_time_dialog_set_visible_widgets_and_adj(InputTimeDialog *itd);

static void input_time_dialog_class_init (InputTimeDialogClass *class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass*) class;

	parent_class = gtk_type_class (gtk_widget_get_type ());

	input_time_dialog_signals[NEW_TIME] =
		gtk_signal_new ("new_time",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET (InputTimeDialogClass, new_time),
		gtk_marshal_NONE__INT,
		GTK_TYPE_NONE,
		1,
		GTK_TYPE_INT);

	gtk_object_class_add_signals (object_class, input_time_dialog_signals, LAST_SIGNAL);
  	class->new_time = NULL;

	object_class->destroy = input_time_dialog_destroy;
}

static void input_time_dialog_init (InputTimeDialog *dialog)
{
	GtkObject *input_time_millisecond_adj;

	GtkWidget
		*input_time_vbox,
			*input_time_label_hbox,
				*input_time_arrow_label,
			*input_time_plus_button_hbox,
				*input_time_plus_label,
				*hour_plus_pushbutton,
				*minute_plus_pushbutton,
				*second_plus_pushbutton,
			*input_timeline_hbox,
				*input_timeline_label,
			*input_time_minus_button_hbox,
				*input_time_minus_label,
				*hour_minus_pushbutton,
				*minute_minus_pushbutton,
				*second_minus_pushbutton,
			*input_time_hsep,
			*input_time_button_hbox,
				*input_time_ok_button,
				*input_time_cancel_button;

	gchar *trans_str;

	dialog->max_time = 1000 * 60 * 99 + 1000 * 59 + 999;
	dialog->min_time = 0;
	dialog->time_offset = 0;

	gtk_window_set_policy(GTK_WINDOW(dialog), TRUE, TRUE, TRUE);
	gtk_container_border_width(GTK_CONTAINER(dialog), 2);
	gtk_widget_realize(GTK_WIDGET(dialog));
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

	input_time_vbox = gtk_vbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(dialog), input_time_vbox);
	gtk_widget_show(input_time_vbox);

	input_time_hsep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(input_time_vbox), input_time_hsep, TRUE, TRUE, 5);
	gtk_widget_show(input_time_hsep);

	input_time_label_hbox = gtk_hbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(input_time_vbox), input_time_label_hbox);
	gtk_widget_show(input_time_label_hbox);

	dialog->time_offset_label = gtk_label_new("[00:00:000]");
	gtk_box_pack_start(GTK_BOX(input_time_label_hbox),
		dialog->time_offset_label, TRUE, FALSE, 2);
	gtk_widget_show(dialog->time_offset_label);

	input_time_arrow_label = gtk_label_new("-->");
	gtk_box_pack_start(GTK_BOX(input_time_label_hbox), input_time_arrow_label, FALSE, FALSE, 2);
	gtk_widget_show(input_time_arrow_label);

	dialog->input_time_timestamp_label = gtk_label_new("[00:00:000]");
	gtk_box_pack_start(GTK_BOX(input_time_label_hbox),
		dialog->input_time_timestamp_label, TRUE, FALSE, 2);
	gtk_widget_show(dialog->input_time_timestamp_label);

	input_time_hsep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(input_time_vbox), input_time_hsep, TRUE, TRUE, 5);
	gtk_widget_show(input_time_hsep);

// Plus button line
	input_time_plus_button_hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(input_time_vbox), input_time_plus_button_hbox, FALSE, FALSE, 0);
	gtk_widget_show(input_time_plus_button_hbox);

	input_time_plus_label = gtk_label_new (" ++ ");
	gtk_box_pack_start(GTK_BOX(input_time_plus_button_hbox), input_time_plus_label, FALSE, FALSE, 2);
	gtk_widget_show(input_time_plus_label);

	hour_plus_pushbutton = gtk_button_new_with_label(_("hour"));
	gtk_signal_connect(GTK_OBJECT(hour_plus_pushbutton), "clicked",
		GTK_SIGNAL_FUNC(input_time_change_time_event), (gpointer) (60 * 60 * 1000));
	gtk_box_pack_start(GTK_BOX(input_time_plus_button_hbox), hour_plus_pushbutton, FALSE, FALSE, 10);
	gtk_widget_show(hour_plus_pushbutton);

	minute_plus_pushbutton = gtk_button_new_with_label(_("min"));
	gtk_signal_connect(GTK_OBJECT(minute_plus_pushbutton), "clicked",
		GTK_SIGNAL_FUNC(input_time_change_time_event), (gpointer) (60 * 1000));
	gtk_box_pack_start(GTK_BOX(input_time_plus_button_hbox), minute_plus_pushbutton, FALSE, FALSE, 10);
	gtk_widget_show(minute_plus_pushbutton);

	second_plus_pushbutton = gtk_button_new_with_label(_("sec"));
	gtk_signal_connect(GTK_OBJECT(second_plus_pushbutton), "clicked",
		GTK_SIGNAL_FUNC(input_time_change_time_event), (gpointer) 1000);
	gtk_box_pack_start(GTK_BOX(input_time_plus_button_hbox), second_plus_pushbutton, FALSE, FALSE, 10);
	gtk_widget_show(second_plus_pushbutton);

// Time spin button
	input_time_hsep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(input_time_vbox), input_time_hsep, TRUE, TRUE, 2);
	gtk_widget_show(input_time_hsep);

	input_timeline_hbox = gtk_hbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(input_time_vbox), input_timeline_hbox);
	gtk_container_border_width(GTK_CONTAINER(input_timeline_hbox), 2);
	gtk_widget_show(input_timeline_hbox);

	dialog->input_timeline_zero_pushbutton = gtk_button_new_with_label("0");
	gtk_signal_connect(GTK_OBJECT(dialog->input_timeline_zero_pushbutton), "clicked",
		GTK_SIGNAL_FUNC(input_timeline_zero_clicked_event), NULL);
	gtk_box_pack_start(GTK_BOX(input_timeline_hbox), dialog->input_timeline_zero_pushbutton, FALSE, FALSE, 10);
	gtk_widget_show(dialog->input_timeline_zero_pushbutton);

	trans_str = g_strconcat(_("Time [ms]"), ":", NULL);
	input_timeline_label = gtk_label_new (trans_str);
	g_free(trans_str);
	gtk_box_pack_start(GTK_BOX(input_timeline_hbox), input_timeline_label, FALSE, FALSE, 2);
	gtk_widget_show(input_timeline_label);

	input_time_millisecond_adj = gtk_adjustment_new(0, dialog->min_time, dialog->max_time, 1, 50, 50);
	gtk_signal_connect (GTK_OBJECT(input_time_millisecond_adj), "value_changed",
		GTK_SIGNAL_FUNC (gtk_millisecond_adjustment_value_changed), (gpointer) dialog);

	dialog->time_spinbutton =
		GTK_SPIN_BUTTON(gtk_spin_button_new (GTK_ADJUSTMENT (input_time_millisecond_adj), 1, 0));
	gtk_box_pack_start(GTK_BOX(input_timeline_hbox),
		GTK_WIDGET(dialog->time_spinbutton), TRUE, TRUE, 2);
	gtk_widget_show (GTK_WIDGET(dialog->time_spinbutton));

	input_time_hsep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(input_time_vbox), input_time_hsep, TRUE, TRUE, 2);
	gtk_widget_show(input_time_hsep);

// Minus button line
	input_time_minus_button_hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(input_time_vbox), input_time_minus_button_hbox, FALSE, FALSE, 0);
	gtk_widget_show(input_time_minus_button_hbox);

	input_time_minus_label = gtk_label_new (" -- ");
	gtk_box_pack_start(GTK_BOX(input_time_minus_button_hbox), input_time_minus_label, FALSE, FALSE, 2);
	gtk_widget_show(input_time_minus_label);

	hour_minus_pushbutton = gtk_button_new_with_label(_("hour"));
	gtk_signal_connect(GTK_OBJECT(hour_minus_pushbutton), "clicked",
		GTK_SIGNAL_FUNC(input_time_change_time_event), (gpointer) (-1 * 60 * 60 * 1000));
	gtk_box_pack_start(GTK_BOX(input_time_minus_button_hbox), hour_minus_pushbutton, FALSE, FALSE, 10);
	gtk_widget_show(hour_minus_pushbutton);

	minute_minus_pushbutton = gtk_button_new_with_label(_("min"));
	gtk_signal_connect(GTK_OBJECT(minute_minus_pushbutton), "clicked",
		GTK_SIGNAL_FUNC(input_time_change_time_event), (gpointer) (-1 * 60 * 1000));
	gtk_box_pack_start(GTK_BOX(input_time_minus_button_hbox), minute_minus_pushbutton, FALSE, FALSE, 10);
	gtk_widget_show(minute_minus_pushbutton);

	second_minus_pushbutton = gtk_button_new_with_label(_("sec"));
	gtk_signal_connect(GTK_OBJECT(second_minus_pushbutton), "clicked",
		GTK_SIGNAL_FUNC(input_time_change_time_event), (gpointer) (-1 * 1000));
	gtk_box_pack_start(GTK_BOX(input_time_minus_button_hbox), second_minus_pushbutton, FALSE, FALSE, 10);
	gtk_widget_show(second_minus_pushbutton);

	input_time_hsep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(input_time_vbox), input_time_hsep, TRUE, TRUE, 5);
	gtk_widget_show(input_time_hsep);

// Ok / cancel button bar
	input_time_button_hbox = gtk_hbutton_box_new();
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(input_time_button_hbox), 5);
	gtk_box_pack_start(GTK_BOX(input_time_vbox), input_time_button_hbox, FALSE, FALSE, 0);
	gtk_widget_show(input_time_button_hbox);

	input_time_ok_button = gtk_button_new_with_label(_("Ok"));
	GTK_WIDGET_SET_FLAGS(input_time_ok_button, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(input_time_button_hbox), input_time_ok_button, TRUE, TRUE, 0);
	gtk_widget_show(input_time_ok_button);

	input_time_cancel_button = gtk_button_new_with_label(_("Cancel"));
	GTK_WIDGET_SET_FLAGS(input_time_cancel_button, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(input_time_button_hbox), input_time_cancel_button, TRUE, TRUE, 0);
	gtk_widget_grab_default(input_time_cancel_button);
	gtk_widget_show(input_time_cancel_button);

	gtk_signal_connect_object(GTK_OBJECT(input_time_ok_button), "clicked",
		GTK_SIGNAL_FUNC(input_time_ok_button_event), (gpointer) dialog);
	gtk_signal_connect_object(GTK_OBJECT(input_time_cancel_button), "clicked",
		GTK_SIGNAL_FUNC(input_time_cancel_button_event), (gpointer) dialog);
	gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
		GTK_SIGNAL_FUNC(input_time_dialog_delete_event), NULL);
}

GtkType input_time_dialog_get_type (void)
{
	static GtkType input_time_dialog_type = 0;

	if (!input_time_dialog_type) {

		static const GtkTypeInfo input_time_dialog_info =
		{
			(gchar*) "InputTimeDialog",
			sizeof (InputTimeDialog),
			sizeof (InputTimeDialogClass),
			(GtkClassInitFunc) input_time_dialog_class_init,
			(GtkObjectInitFunc) input_time_dialog_init,
			NULL,
			NULL,
			(GtkClassInitFunc) NULL,
		};

		input_time_dialog_type = gtk_type_unique (gtk_window_get_type(), &input_time_dialog_info);
	}

	return input_time_dialog_type;
}

void input_time_dialog_set_time (InputTimeDialog *itd, gint time)
{
	g_return_if_fail(itd != NULL);
	g_return_if_fail(IS_INPUT_TIME_DIALOG(itd));

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_input_time.c [input_time_dialog_set_time]\n"));
#endif

	gtk_spin_button_set_value(itd->time_spinbutton, time);
}

gint input_time_dialog_get_time (InputTimeDialog *itd)
{
	g_return_val_if_fail(itd != NULL, 0);
	g_return_val_if_fail(IS_INPUT_TIME_DIALOG(itd), 0);

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_input_time.c [input_time_dialog_get_time]\n"));
#endif

	return gtk_spin_button_get_value_as_int(itd->time_spinbutton);
}

void input_time_ok_button_event(GtkButton *button, gpointer user_data)
{
	InputTimeDialog* itd;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_input_time.c [input_time_ok_button_event]\n"));
#endif

	g_return_if_fail (IS_INPUT_TIME_DIALOG(gtk_widget_get_toplevel(GTK_WIDGET(button))));
	itd = INPUT_TIME_DIALOG(gtk_widget_get_toplevel(GTK_WIDGET(button)));

	gtk_signal_emit(GTK_OBJECT(itd), input_time_dialog_signals[NEW_TIME],
		gtk_spin_button_get_value_as_int(itd->time_spinbutton));
}

void input_time_change_time_event(GtkButton *button, gpointer *user_data)
{
	InputTimeDialog* itd = INPUT_TIME_DIALOG(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	gint value = gtk_spin_button_get_value_as_int(itd->time_spinbutton);
	gint change = value + (gint) user_data;
//	GtkAdjustment *adj = gtk_spin_button_get_adjustment(itd->time_spinbutton);

	if ((change + itd->time_offset) > itd->max_time) { change = itd->max_time - itd->time_offset; }
	else if ((change + itd->time_offset) < itd->min_time) { change = itd->min_time  - itd->time_offset; }

	gtk_spin_button_set_value(itd->time_spinbutton, change);
}

void input_timeline_zero_clicked_event (GtkButton *button, gpointer *user_data)
{
	InputTimeDialog* itd = INPUT_TIME_DIALOG(gtk_widget_get_toplevel(GTK_WIDGET(button)));

	if ((0 <= itd->max_time) && (0 >= itd->min_time))
		{ gtk_spin_button_set_value(itd->time_spinbutton, 0); }
}

static void input_time_cancel_button_event(GtkButton *button, gpointer data)
{
	InputTimeDialog* itd;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_input_time.c [input_time_cancel_button_event]\n"));
#endif

	g_return_if_fail(IS_INPUT_TIME_DIALOG(gtk_widget_get_toplevel(GTK_WIDGET(button))));

	itd = INPUT_TIME_DIALOG(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	
	input_time_dialog_hide(itd);
}

gint input_time_dialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_input_time.c [input_time_dialog_delete_event]\n"));
#endif

	gtk_widget_hide(widget);
	return (TRUE);
}

GtkWidget *input_time_dialog_new  (gchar  *caption,
				    gint    width,
				    gint    height)
{
	GtkWidget *dialog;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_input_time.c [input_time_dialog_new]\n"));
#endif

	dialog = GTK_WIDGET(gtk_type_new(TYPE_INPUT_TIME_DIALOG));

	if (caption) {
		gtk_window_set_title(GTK_WINDOW(dialog), caption);
	}

	return dialog;
}

static void input_time_dialog_destroy (GtkObject *object)
{
	InputTimeDialog *itd;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_input_time.c [input_time_dialog_destroy]\n"));
#endif

	g_return_if_fail (itd = INPUT_TIME_DIALOG (object));

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

void input_time_dialog_show (InputTimeDialog* dialog, GtkWindow *parent, 
	gboolean at_cursor, gboolean modal)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_input_time.c [input_time_dialog_show]\n"));
#endif

	g_return_if_fail(dialog != NULL);
	g_return_if_fail(IS_INPUT_TIME_DIALOG(dialog));

	if (parent != NULL)
		gtk_window_set_transient_for(GTK_WINDOW(dialog), GTK_WINDOW(parent));
	gtk_window_set_modal(GTK_WINDOW(dialog), modal);
	
	gtk_widget_show(GTK_WIDGET(dialog));

	if (at_cursor == FALSE)
		gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
	else
		gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);

	if (modal == TRUE)
		gtk_main();
}

void input_time_dialog_hide (InputTimeDialog* dialog)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_input_time.c [input_time_dialog_hide]\n"));
#endif

	g_return_if_fail(dialog != NULL);
	g_return_if_fail(IS_INPUT_TIME_DIALOG(dialog));

	gtk_widget_hide (GTK_WIDGET(dialog));
	
	if (GTK_WINDOW(dialog)->modal == TRUE)
		gtk_main_quit();
}

void gtk_millisecond_adjustment_value_changed (GtkAdjustment *adjustment, gpointer user_data)
{
	InputTimeDialog *itd = INPUT_TIME_DIALOG(user_data);
	gint value = itd->time_offset + (gint) adjustment->value;
	gchar *text = g_strdup_printf("[%.2i:%.2i:%.3i]",
		value / (60 * 1000), (value / 1000) % 60, value % 1000);
	gtk_label_set(GTK_LABEL(itd->input_time_timestamp_label), text);
	g_free(text);
}

static void input_time_dialog_set_values (InputTimeDialog *itd)
{
	GtkAdjustment *adj;
	gchar *text;
	gint value;

	adj = gtk_spin_button_get_adjustment(itd->time_spinbutton);
	adj->upper = itd->max_time - itd->time_offset;
	adj->lower = itd->min_time - itd->time_offset;

	text = g_strdup_printf("[%.2i:%.2i:%.3i]", itd->time_offset / (60 * 1000),
		(itd->time_offset / 1000) % 60, itd->time_offset % 1000);
	gtk_label_set(GTK_LABEL(itd->time_offset_label), text);
	g_free(text);

	value = gtk_spin_button_get_value_as_int(itd->time_spinbutton) + itd->time_offset;
	text = g_strdup_printf("[%.2i:%.2i:%.3i]",
		value / (60 * 1000), (value / 1000) % 60, value % 1000);
	gtk_label_set(GTK_LABEL(itd->input_time_timestamp_label), text);
	g_free(text);

	gtk_widget_set_sensitive(itd->input_timeline_zero_pushbutton,
		((itd->min_time <= 0) && (itd->max_time >= 0)));
}

void input_time_dialog_set_max_time (InputTimeDialog *itd, gint max)
{
	g_return_if_fail(itd != NULL);
	g_return_if_fail(IS_INPUT_TIME_DIALOG(itd));
	g_return_if_fail(max > itd->min_time);

	itd->max_time = max;

	input_time_dialog_set_values(itd);
}

void input_time_dialog_set_min_time (InputTimeDialog *itd, gint min)
{
	g_return_if_fail(itd != NULL);
	g_return_if_fail(IS_INPUT_TIME_DIALOG(itd));
	g_return_if_fail(min < itd->max_time);

	itd->min_time = min;

	input_time_dialog_set_values(itd);
}

void input_time_dialog_set_time_offset (InputTimeDialog *itd, gint offset)
{
	g_return_if_fail(itd != NULL);
	g_return_if_fail(IS_INPUT_TIME_DIALOG(itd));

	if (offset > itd->max_time) { itd->time_offset = itd->max_time; }
	else if (offset < itd->min_time) { itd->time_offset = itd->min_time; }
	else { itd->time_offset = offset; }

	input_time_dialog_set_values(itd);
}

gint input_time_dialog_get_time_offset (InputTimeDialog *itd)
{
	g_return_val_if_fail(itd != NULL, 0);
	g_return_val_if_fail(IS_INPUT_TIME_DIALOG(itd), 0);

	return itd->time_offset;
}
