/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifndef __MATHS_H__
#define __MATHS_H__

#include <stdlib.h>
#include <math.h>

#include <GL/gl.h>

#define PI 3.14159265358979f
#define PI2 (2.0f * PI)

typedef long fix16_16;

enum {
	ANGLE_2PI = 1 << 16,
	ANGLE_MASK = ANGLE_2PI - 1,
};

inline float RandFloat(void)      { return (float)(1.0f / ((float)RAND_MAX + 1.0f)) * (float)rand(); }
inline int RandTrueFalse(void)    { return (rand() > RAND_MAX/2) ? 0 : 1; }
inline int Rand(void)             { return rand(); }
inline int Rand(int max)          { return rand() / ((RAND_MAX / (max+1)) + 1); }
inline int Rand(int min,int max)  { return min + rand() / ((RAND_MAX / (max-min+1)) + 1); }
inline int RandPlusMinus(int plus_minus) { return Rand(-plus_minus,plus_minus); }

inline float Sqrt(float n)     { return sqrtf(n); }

inline float Sin(float radian) { return sinf(radian); }
inline float Cos(float radian) { return cosf(radian); }

inline float Sin(fix16_16 radian) { return sinf((radian & ANGLE_MASK) * 2*M_PI / ANGLE_2PI); }
inline float Cos(fix16_16 radian) { return cosf((radian & ANGLE_MASK) * 2*M_PI / ANGLE_2PI); }

inline void InitVector3(float *vec_p)
{
	*(vec_p + 0) = 0.0f;
	*(vec_p + 1) = 0.0f;
	*(vec_p + 2) = 0.0f;
}

inline void InitVector4(float *vec_p)
{
	*(vec_p + 0) = 0.0f;
	*(vec_p + 1) = 0.0f;
	*(vec_p + 2) = 0.0f;
	*(vec_p + 3) = 1.0f;
}

inline float GetDistance(const float *src_p,const float *dst_p)
{
	return Sqrt((*(src_p + 0) - *(dst_p + 0)) * (*(src_p + 0) - *(dst_p + 0)) +
		(*(src_p + 1) - *(dst_p + 1)) * (*(src_p + 1) - *(dst_p + 1)) +
		(*(src_p + 2) - *(dst_p + 2)) * (*(src_p + 2) - *(dst_p + 2)));
}

inline float GetInnerProduct(const float *vec0_p,const float *vec1_p)
{
	return (*(vec0_p + 0) * *(vec1_p + 0) +
		*(vec0_p + 1) * *(vec1_p + 1) +
		*(vec0_p + 2) * *(vec1_p + 2));
}


inline void GetOuterProduct(const float *vec_p,float *dst_p)
{
	enum {
		Z = 0,
		F = 1,
		S = 2,
	};

	*(dst_p + 0) = (*(vec_p + F*4 + 1) - *(vec_p + S*4 + 1)) * (*(vec_p + Z*4 + 2) - *(vec_p + F*4 + 2)) -
		(*(vec_p + F*4 + 2) - *(vec_p + S*4 + 2)) * (*(vec_p + Z*4 + 1) - *(vec_p + F*4 + 1));

	*(dst_p + 1) = (*(vec_p + F*4 + 2) - *(vec_p + S*4 + 2)) * (*(vec_p + Z*4 + 0) - *(vec_p + F*4 + 0)) -
		(*(vec_p + F*4 + 0) - *(vec_p + S*4 + 0)) * (*(vec_p + Z*4 + 2) - *(vec_p + F*4 + 2));

	*(dst_p + 2) = (*(vec_p + F*4 + 0) - *(vec_p + S*4 + 0)) * (*(vec_p + Z*4 + 1) - *(vec_p + F*4 + 1)) -
		(*(vec_p + F*4 + 1) - *(vec_p + S*4 + 1)) * (*(vec_p + Z*4 + 0) - *(vec_p + F*4 + 0));
}

inline void GetOuterProductXY(const float *vec_p,float *dst_p)
{
	enum {
		Z = 0,
		F = 1,
		S = 2,
	};

	*(dst_p + 0) = (*(vec_p + F*4 + 1) - *(vec_p + S*4 + 1)) * (*(vec_p + Z*4 + 2) - *(vec_p + F*4 + 2)) -
		(*(vec_p + F*4 + 2) - *(vec_p + S*4 + 2)) * (*(vec_p + Z*4 + 1) - *(vec_p + F*4 + 1));

	*(dst_p + 1) = (*(vec_p + F*4 + 2) - *(vec_p + S*4 + 2)) * (*(vec_p + Z*4 + 0) - *(vec_p + F*4 + 0)) -
		(*(vec_p + F*4 + 0) - *(vec_p + S*4 + 0)) * (*(vec_p + Z*4 + 2) - *(vec_p + F*4 + 2));
}

inline float GetOuterProductZ(const float *vec_p)
{
	enum {
		Z = 0,
		F = 1,
		S = 2,
	};

	return (*(vec_p + F*4 + 0) - *(vec_p + S*4 + 0)) * (*(vec_p + Z*4 + 1) - *(vec_p + F*4 + 1)) -
		(*(vec_p + F*4 + 1) - *(vec_p + S*4 + 1)) * (*(vec_p + Z*4 + 0) - *(vec_p + F*4 + 0));
}


inline void UnitVector(float *vec_p)
{
	float scalar;

	scalar = Sqrt(*(vec_p + 0) * *(vec_p + 0) +
		*(vec_p + 1) * *(vec_p + 1) +
		*(vec_p + 2) * *(vec_p + 2));

	if (scalar != 0.0f) {
		*(vec_p + 0) = *(vec_p + 0) / scalar;
		*(vec_p + 1) = *(vec_p + 1) / scalar;
		*(vec_p + 2) = *(vec_p + 2) / scalar;
	}
}

inline void UnitVector(const float *vec_p,float *dst_p)
{
	float scalar;

	scalar = Sqrt(*(vec_p + 0) * *(vec_p + 0) +
		*(vec_p + 1) * *(vec_p + 1) +
		*(vec_p + 2) * *(vec_p + 2));

	if (scalar != 0.0f) {
		*(dst_p + 0) = *(vec_p + 0) / scalar;
		*(dst_p + 1) = *(vec_p + 1) / scalar;
		*(dst_p + 2) = *(vec_p + 2) / scalar;
	}
}


inline void MulVector3(float *vec_p,float scalar)
{
	*(vec_p + 0) = *(vec_p + 0) * scalar;
	*(vec_p + 1) = *(vec_p + 1) * scalar;
	*(vec_p + 2) = *(vec_p + 2) * scalar;
}

inline void MulVector3(float *vec_p,float *dst_p,float scalar)
{
	*(dst_p + 0) = *(vec_p + 0) * scalar;
	*(dst_p + 1) = *(vec_p + 1) * scalar;
	*(dst_p + 2) = *(vec_p + 2) * scalar;
}

inline void DivVector3(float *vec_p,float scalar)
{
	*(vec_p + 0) = *(vec_p + 0) / scalar;
	*(vec_p + 1) = *(vec_p + 1) / scalar;
	*(vec_p + 2) = *(vec_p + 2) / scalar;
}

inline void DivVector3(float *vec_p,float *dst_p,float scalar)
{
	*(dst_p + 0) = *(vec_p + 0) / scalar;
	*(dst_p + 1) = *(vec_p + 1) / scalar;
	*(dst_p + 2) = *(vec_p + 2) / scalar;
}

inline void AddVector3(float *s0_p, float *s1_p, float *dst_p)
{
	*(dst_p + 0) = *(s0_p + 0) + *(s1_p + 0);
	*(dst_p + 1) = *(s0_p + 1) + *(s1_p + 1);
	*(dst_p + 2) = *(s0_p + 2) + *(s1_p + 2);
}

inline void SubVector3(float *s0_p, float *s1_p, float *dst_p)
{
	*(dst_p + 0) = *(s0_p + 0) - *(s1_p + 0);
	*(dst_p + 1) = *(s0_p + 1) - *(s1_p + 1);
	*(dst_p + 2) = *(s0_p + 2) - *(s1_p + 2);
}

class CMatrix {
public:
	friend inline CMatrix operator*(const CMatrix &m0,const CMatrix &m1);
	friend CMatrix operator+(const CMatrix &m0,const CMatrix &m1);
	friend CMatrix operator-(const CMatrix &m0,const CMatrix &m1);

	enum {
		I_ELEMENTS = 4,
		J_ELEMENTS = 4,
	};

	CMatrix();
	virtual ~CMatrix();

	static int GetIndex(int j,int i);
	void RotUnit(void);
	void TransZero(void);
	void Unit(void);
	void Transpose(void);
	void GetModelView(void);
	void MulRotX(float sine,float cosine);
	void MulRotX(float radian);
	void MulRotX(fix16_16 radian);
	void MulRotY(float sine,float cosine);
	void MulRotY(float radian);
	void MulRotY(fix16_16 radian);
	void MulRotZ(float sine,float cosine);
	void MulRotZ(float radian);
	void MulRotZ(fix16_16 radian);
	void SetTrans(float tx,float ty,float tz);
	void AddTrans(float tx,float ty,float tz);
	void MulScale(float sx,float sy,float sz);
	void MulVector(const float src[],float dst[]);
	void MulVector(const float src[],float dst[],int n);
	float MulVectorZ(const float src[]);
	void MulVectorZ(const float src[],float dst[],int n);
	void SetMatrix(const float src[]);

private:
	float m[I_ELEMENTS * J_ELEMENTS];
};

inline CMatrix operator*(const CMatrix &m0,const CMatrix &m1);
CMatrix operator+(const CMatrix &m0,const CMatrix &m1);
CMatrix operator-(const CMatrix &m0,const CMatrix &m1);

#endif // __MATHS_H__
