/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>

#include <stdlib.h>
#include <stdio.h>

#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include <SDL/SDL_types.h>
#include <SDL/SDL_syswm.h>
#include <SDL/SDL_ttf.h>

#include "singit_displayer_plugin.h"
#include "displayer_test_sdl_status.h"

#include "singit_config.h"

#include "singit_dialogs.h"

#include "sdl_test_helpers.h"

#define DEBUG(x)  g_print x

static void displayer_test_sdl_init(void);
static void displayer_test_sdl_finish(void);

static void displayer_test_sdl_show(SingitSong *cur_song);
static void displayer_test_sdl_hide(SingitSong *cur_song);

static void displayer_test_sdl_set_time(guint time, SingitSong *cur_song, GList *real_next);

static void displayer_test_sdl_playback_start(void);
static void displayer_test_sdl_playback_stop(void);

static void displayer_test_sdl_config_update(const SingitConfigData *singit_config);

DisplayerPlugin test_sdl_dp =
{
	NULL,   /* Filled in by main plugin */
	NULL,   /* Filled in by main plugin */
	0,      /* Filled in by main plugin */

	"Test SDL displayer",     /* The description that is shown in the preferences box */

	displayer_test_sdl_init,    /* Called when the displayer enabled */
	displayer_test_sdl_show,    /* Called when the displayer is shown */
	displayer_test_sdl_hide,    /* Called when the displayer is hidden */
	NULL,  /* Called when the displayer visual state should change */
	displayer_test_sdl_finish,  /* Called before the plugin is disabled */

	/* Call this with a pointer to your plugin struct to disable the plugin */
	NULL,

	 /* Called when the global plugin configuration has been updated */
	displayer_test_sdl_config_update,

	/* Called by the main thread with the current song and time */
	displayer_test_sdl_set_time,

	NULL,           /* Show the plugin about box */
	NULL,       /* Show the plugin configure box */
	displayer_test_sdl_playback_start,  /* Called when playback starts */
	displayer_test_sdl_playback_stop,   /* Called when playback stops */

	/* Render the PCM data, don't do anything time consuming in here - directly from xmms */
	NULL,
	/* Render the freq data, don't do anything time consuming in here - directly from xmms */
	NULL
};

DIPSPLAYER_SYMBOL(libdisplayer_test_sdl, test_sdl_dp)

static gint displayer_test_sdl_disable_func(gpointer data)
{
	GDK_THREADS_ENTER();
	test_sdl_dp.disable(&test_sdl_dp);
	GDK_THREADS_LEAVE();
	return (FALSE);
}

static void sdl_test_handler_expose_video(void);
static void sdl_test_handler_key_down(void *pEvent);
static void sdl_test_handler_resize(void *pEvent);

static void displayer_test_sdl_draw_text(gboolean lock)
{
	SDL_Rect rect;
	gboolean lock_it;

	if ((displayer_test_sdl_status.current_text != NULL) || (displayer_test_sdl_status.next_text != NULL)) {
		rect.x = 10;
		rect.y = 10;

		lock_it = (lock && SDL_MUSTLOCK(displayer_test_sdl_status.screen));

		if (lock_it) { SDL_LockSurface(displayer_test_sdl_status.screen); }

		if (displayer_test_sdl_status.current_text != NULL) {
			rect.w = displayer_test_sdl_status.current_text->w;
			rect.h = displayer_test_sdl_status.current_text->h;
			SDL_UpperBlit(displayer_test_sdl_status.current_text, NULL,
				displayer_test_sdl_status.screen, &rect);
		}

		rect.y += TTF_FontLineSkip(displayer_test_sdl_status.font);

		if (displayer_test_sdl_status.next_text != NULL) {
			rect.w = displayer_test_sdl_status.next_text->w;
			rect.h = displayer_test_sdl_status.next_text->h;
			SDL_UpperBlit(displayer_test_sdl_status.next_text, NULL,
				displayer_test_sdl_status.screen, &rect);
		}

		if (lock_it) { SDL_UnlockSurface(displayer_test_sdl_status.screen); }
	}
}

static gboolean CreateScreen(gint nWidth, gint nHeight, gboolean bFullscreen)
{
	static gint save_full[2] = { -1 };
	static gboolean was_fullscreen = FALSE;
	gint nFlags, w, h;

	if (bFullscreen == TRUE) {
		sdl_helper_blanking(BLANK_DISABLE);
		nFlags = SDL_SWSURFACE | SDL_FULLSCREEN;
//		nFlags = SDL_FULLSCREEN;
		save_full[0] = displayer_test_sdl_status.screen->w;
		save_full[1] = displayer_test_sdl_status.screen->h;
		w = 640;
		h = 480;
		was_fullscreen = TRUE;
	}
	else {
		if (was_fullscreen == TRUE) {
			sdl_helper_blanking(BLANK_ENABLE);
		}
		nFlags = SDL_SWSURFACE | SDL_RESIZABLE;
//		nFlags = SDL_RESIZABLE;
		if ((save_full[0] > -1) && (save_full[1] > -1)) {
			w = save_full[0];
			h = save_full[1];
			save_full[0] = -1;
		}
		else {
			w = nWidth;
			h = nHeight;
		}
	}

	displayer_test_sdl_status.screen = SDL_SetVideoMode(w, h, bFullscreen ? 16 : 0, nFlags);
	displayer_test_sdl_status.is_fullscreen = bFullscreen;

	return (displayer_test_sdl_status.screen != NULL);
}

/*
  Little additional SDL event prozessor inside an g_timeout
  Needed to process events while xmms is stopped
  FIXME: Find a better solution
*/
static gboolean displayer_test_sdl_event_thread_func(gpointer data)
{
	SDL_Event sdl_event;
	gboolean result = TRUE;

	if (displayer_test_sdl_status.sdl_inuse == TRUE)
		{ return FALSE; }

	if (!displayer_test_sdl_status.kill) {
		displayer_test_sdl_draw_text(TRUE);
		if (displayer_test_sdl_status.screen == SDL_GetVideoSurface())
			SDL_UpdateRect(displayer_test_sdl_status.screen, 0, 0, 0, 0);

		while (!displayer_test_sdl_status.kill && (SDL_PollEvent(&sdl_event) > 0)) {
			switch (sdl_event.type) {
			case SDL_KEYDOWN:
				sdl_test_handler_key_down(&sdl_event);
				break;
			case SDL_QUIT:
				// Prevent double shutdown
				sdl_helper_blanking(BLANK_RESET);
				if (displayer_test_sdl_status.kill == FALSE) {
					gtk_idle_add(displayer_test_sdl_disable_func, NULL);
				}
				break;
			case SDL_VIDEORESIZE:
				sdl_test_handler_resize(&sdl_event);
				break;
			case SDL_ACTIVEEVENT: // Application visibility event structure
				break;
			case SDL_VIDEOEXPOSE:
				sdl_test_handler_expose_video();
				break;
			}
		}
	}

	if (displayer_test_sdl_status.kill) {
		displayer_test_sdl_status.timeout = 0;
		displayer_test_sdl_status.kill = FALSE;
//		pthread_mutex_unlock(&displayer_test_sdl_status.sdl_event_mutex);
#	ifdef CODEDEBUG
		DEBUG(("  Quit render thread\n"));
#	endif
		result = FALSE;
	}

	return result;
}


static inline gboolean shutdown_gtk_render_idle(void)
{
	if (displayer_test_sdl_status.timeout != 0) {
#ifdef CODEDEBUG
		DEBUG(("  Kill render thread\n"));
#endif
		displayer_test_sdl_status.kill = TRUE;

//		GDK_THREADS_ENTER();
#ifdef USE_GTK_TIMEOUT
		gtk_timeout_remove(displayer_test_sdl_status.timeout);
#else
		gtk_idle_remove(displayer_test_sdl_status.timeout);
#endif
//		GDK_THREADS_LEAVE();
		displayer_test_sdl_status.timeout = 0;
		return TRUE;
	}
	return FALSE;
}

static inline gboolean set_gtk_render_idle(const gboolean start)
{
	if ((start == TRUE) && (displayer_test_sdl_status.timeout == 0) &&
		((displayer_test_sdl_status.xmms_playing == FALSE) ||
		(displayer_test_sdl_status.singit_playing == TRUE)))
	{
#ifdef CODEDEBUG
		DEBUG(("  Start render thread\n"));
#endif
		displayer_test_sdl_status.kill = FALSE;
		displayer_test_sdl_status.timeout =
#ifdef USE_GTK_TIMEOUT
			gtk_timeout_add(50, displayer_test_sdl_event_thread_func, NULL);
#else
			gtk_idle_add(displayer_test_sdl_event_thread_func, NULL);
#endif
		return TRUE;
	}
	else {
		if ((displayer_test_sdl_status.xmms_playing == TRUE) &&
			!xmms_remote_is_paused(test_sdl_dp.xmms_session))
		{
			return shutdown_gtk_render_idle();
		}
		return FALSE;
	}
}

static void displayer_test_sdl_init(void)
{
	gchar *title;

#	ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_test_sdl_init]\n"));
#	endif

	displayer_test_sdl_status_init();

	/* Initialise the SDL library */
	if (SDL_WasInit(SDL_INIT_VIDEO) > 0) {
		g_print("SDL Video was initialized before : no two SDL windows allowed : exit\n");
		displayer_test_sdl_status.sdl_inuse = TRUE;
		gtk_idle_add (displayer_test_sdl_disable_func, NULL);
		return;
	}
	else {
		if (SDL_Init(SDL_INIT_VIDEO) != 0) {
//			fprintf (stderr, "Unable to init SDL: %s", SDL_GetError());
			g_print("Unable to init SDL Video: %s\n", SDL_GetError());
			gtk_idle_add (displayer_test_sdl_disable_func, NULL);
			return;
		}
	}

	displayer_test_sdl_status.song = NULL;

	if (CreateScreen(320, 200, FALSE) == FALSE) {
//		fprintf (stderr, "Graphic mode is not available: %s\n", SDL_GetError());
		g_print("Graphic mode is not available: %s\n", SDL_GetError());
		gtk_idle_add (displayer_test_sdl_disable_func, NULL);
		return;
	}

	title = g_strconcat(SINGIT_VERSION_STRING, " - ", _("SDL simple test displayer"), NULL);
	SDL_WM_SetCaption(title, "SingIt");
	g_free(title);

#	ifdef CODEDEBUG
	DEBUG(("  Start idle\n"));
#	endif

	TTF_Init();
	displayer_test_sdl_status.font = TTF_OpenFont(SINGIT_DATA_DIR "/vixar.ttf", 32);

	displayer_test_sdl_status.xmms_playing = FALSE;
	set_gtk_render_idle(TRUE);
}

static void displayer_test_sdl_finish(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_test_sdl_finish]\n"));
#	endif

	shutdown_gtk_render_idle();

	if (displayer_test_sdl_status.sdl_inuse == FALSE) {
		sdl_helper_blanking(BLANK_RESET);
		TTF_Quit();
		SDL_Quit();
	}

	displayer_test_sdl_status_finish();
}

static void displayer_test_sdl_show(SingitSong *cur_song)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_sdl_show]\n"));
#	endif

	if (!xmms_remote_is_paused(test_sdl_dp.xmms_session) &&
		(set_gtk_render_idle(FALSE)))
	{
		sdl_helper_map(TRUE);
	}
}

static void displayer_test_sdl_hide(SingitSong *cur_song)
{
	gboolean hide;

#	ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_sdl_hide]\n"));
#	endif

	hide = ((displayer_test_sdl_status.hideIfNotFound && !displayer_test_sdl_status.is_fullscreen) ||
		(displayer_test_sdl_status.hideIfNotFoundFullscreen && displayer_test_sdl_status.is_fullscreen));

	if (hide) { sdl_helper_map(FALSE); }
	else { set_gtk_render_idle(TRUE); }
}

static void displayer_test_sdl_set_time(guint time, SingitSong *cur_song, GList *real_next)
{
	static GList *last = (GList*) -1;
	const SDL_Color light_red = { 255, 150, 150, 0 };
	const SDL_Color light_gray = { 200, 200, 200, 0 };

	SingitSong *my_song;
	GList *current, *next;
	SDL_Rect rect;

	my_song = singit_song_attach(cur_song);

	if (my_song != displayer_test_sdl_status.song)
	{
#	ifdef CODEDEBUG
		DEBUG(("displayer_test_sdl.c [displayer_test_sdl_set_time]: New song\n"));
#	endif
		singit_song_detach(&displayer_test_sdl_status.song);
		displayer_test_sdl_status.song = singit_song_attach(my_song);
		last = (GList*) -1;

#	ifdef CODEDEBUG
		if (my_song)
			DEBUG(("  Filename: %s\n", my_song->song_filename));
#	endif
		displayer_test_sdl_status.singit_playing = singit_song_text_found(my_song);

		if (!displayer_test_sdl_status.singit_playing) {
			if (displayer_test_sdl_status.current_text != NULL) {
				SDL_FreeSurface(displayer_test_sdl_status.current_text);
				displayer_test_sdl_status.current_text = NULL;
			}
			if (displayer_test_sdl_status.next_text != NULL) {
				SDL_FreeSurface(displayer_test_sdl_status.next_text);
				displayer_test_sdl_status.next_text = NULL;
			}
		}
	}
	else {
		if (displayer_test_sdl_status.singit_playing &&
			!xmms_remote_is_paused(test_sdl_dp.xmms_session))
		{
			set_gtk_render_idle(FALSE);
		}
		else {
			set_gtk_render_idle(TRUE);
		}
	}

	if (displayer_test_sdl_status.screen == SDL_GetVideoSurface())
	{
//		singit_karaoke_data_set_time(displayer_test_sdl_status.skd, time);
		if ((my_song != NULL) &&
			singit_song_lyrics_found(my_song) &&
			(last != my_song->active_token))
		{
			if (my_song->active_token == NULL)
				{ current = last = my_song->first_token; }
			else { current = last = my_song->active_token; }
			next = singit_song_find_next_lyric_line(my_song, my_song->active_token, TRUE, NULL);

			if (displayer_test_sdl_status.current_text != NULL) {
				rect.x = 10;
				rect.y = 10;
				rect.w = displayer_test_sdl_status.current_text->w;
				rect.h = displayer_test_sdl_status.current_text->h;
				SDL_FreeSurface(displayer_test_sdl_status.current_text);
				SDL_FillRect(displayer_test_sdl_status.screen, &rect, 0);
			}
			if (displayer_test_sdl_status.next_text != NULL) {
				rect.x = 10;
				rect.y = 10;
				rect.w = displayer_test_sdl_status.next_text->w;
				rect.h = displayer_test_sdl_status.next_text->h
					+ TTF_FontLineSkip(displayer_test_sdl_status.font);
				SDL_FreeSurface(displayer_test_sdl_status.next_text);
				SDL_FillRect(displayer_test_sdl_status.screen, &rect, 0);
			}

			if (current && !singit_song_is_empty_item(my_song, current, FALSE)) {
				displayer_test_sdl_status.current_text = TTF_RenderText_Blended
					(displayer_test_sdl_status.font, tText(my_song, current), light_red);
			}
			else { displayer_test_sdl_status.current_text = NULL; }

			if (next && !singit_song_is_empty_item(my_song, next, FALSE)) {
				displayer_test_sdl_status.next_text = TTF_RenderText_Blended
					(displayer_test_sdl_status.font, tText(my_song, next), light_gray);
			}
			else { displayer_test_sdl_status.next_text = NULL; }
		}
		displayer_test_sdl_status.time = time;
		if (displayer_test_sdl_status.timeout == 0)
			{ displayer_test_sdl_event_thread_func(NULL); }
	}
	if (my_song != NULL)
		{ singit_song_detach(&my_song); }
}

static void displayer_test_sdl_config_update(const SingitConfigData *singit_config)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_test_sdl_config_update]\n"));
	#endif

	g_print("Config Update - SDL_WasInit(SDL_INIT_VIDEO): %i\n", SDL_WasInit(SDL_INIT_VIDEO));
}

static void displayer_test_sdl_playback_start(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_sdl_playback_start]\n"));
#	endif

	displayer_test_sdl_status.xmms_playing = TRUE;
}

static void displayer_test_sdl_playback_stop(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_sdl_playback_stop]\n"));
#	endif

	displayer_test_sdl_status.xmms_playing = FALSE;
	set_gtk_render_idle(TRUE);
}

static void sdl_test_handler_expose_video(void)
{
	SDL_UpdateRect(displayer_test_sdl_status.screen, 0, 0, 0, 0);
}

static void sdl_test_handler_key_down(void *pEvent)
{
	SDL_KeyboardEvent* pKeyEvent = (SDL_KeyboardEvent*)pEvent;
	gint key = pKeyEvent->keysym.sym;
	gint mod = pKeyEvent->keysym.mod;
	gint volume, value;
	gboolean draw;

	#ifdef CODEDEBUG
	DEBUG(("sdl_handlers.cpp [sdl_handler_key_down]\n"));
	#endif

	switch (key) {
	case SDLK_ESCAPE:
	case SDLK_q:
		if (displayer_test_sdl_status.is_fullscreen) {
			draw = CreateScreen(640, 480, !displayer_test_sdl_status.is_fullscreen);
		}
		gtk_idle_add (displayer_test_sdl_disable_func, NULL);
//		test_sdl_dp.disable(&test_sdl_dp);
		break;
	case SDLK_RETURN:
		if (mod & KMOD_ALT) {
/*
			if (displayer_test_sdl_status.timeout == 0) {
#	ifdef CODEDEBUG
				DEBUG(("  Start idle\n"));
#	endif
				displayer_test_sdl_status.timeout =
					gtk_timeout_add(50, displayer_sdl_event_thread_func, NULL);
			}
*/
			draw = CreateScreen(640, 480, !displayer_test_sdl_status.is_fullscreen);
			if (draw == TRUE) {
				SDL_ShowCursor(displayer_test_sdl_status.is_fullscreen ? SDL_DISABLE : SDL_ENABLE);
			}
		}
		break;
	case SDLK_a:
		if (mod & KMOD_CTRL) {
			GDK_THREADS_ENTER();
			singit_about_show();
			GDK_THREADS_LEAVE();
		}
		break;
	case SDLK_b:
		xmms_remote_playlist_next(test_sdl_dp.xmms_session);
		break;
	case SDLK_c:
		if ((mod & KMOD_CTRL) && !displayer_test_sdl_status.is_fullscreen) {
/*			GDK_THREADS_ENTER();
			displayer_sdl_config_show();
			GDK_THREADS_LEAVE();*/
		}
		else { xmms_remote_pause(test_sdl_dp.xmms_session); }
		break;
	case SDLK_e:
		if (mod & KMOD_CTRL) {
			GDK_THREADS_ENTER();
			singit_editor_show();
			GDK_THREADS_LEAVE();
		}
		break;
	case SDLK_i:
		if (mod & KMOD_CTRL) {
			GDK_THREADS_ENTER();
			singit_tag_manager_show();
			GDK_THREADS_LEAVE();
		}
		break;
	case SDLK_p:
		if (mod & KMOD_CTRL) {
			GDK_THREADS_ENTER();
			singit_config_show();
			GDK_THREADS_LEAVE();
		}
		break;
	case SDLK_v:
		xmms_remote_stop(test_sdl_dp.xmms_session);
		break;
	case SDLK_x:
		xmms_remote_play(test_sdl_dp.xmms_session);
		break;
	case SDLK_y:
		if (displayer_test_sdl_status.toggle_yz)
			{ xmms_remote_playlist_prev(test_sdl_dp.xmms_session); }
		break;
                case SDLK_z:
		if (!displayer_test_sdl_status.toggle_yz)
			{ xmms_remote_playlist_prev(test_sdl_dp.xmms_session); }
		break;
/*
	case SDLK_QUESTION:
	case SDLK_WORLD_63:
		if (gGF != NULL)
			{ gGF->ShowHelp(); }
		break;
*/
	case SDLK_KP_PLUS:
		volume = xmms_remote_get_main_volume(test_sdl_dp.xmms_session);
		if (volume < 100) { volume++; }
		xmms_remote_set_main_volume(test_sdl_dp.xmms_session, volume);
		break;
	case SDLK_KP_MINUS:
		volume = xmms_remote_get_main_volume(test_sdl_dp.xmms_session);
		if (volume > 0) { volume--; }
		xmms_remote_set_main_volume(test_sdl_dp.xmms_session, volume);
		break;
	case SDLK_KP_ENTER:
		if (displayer_test_sdl_status.jump_to_song == 0) { break; }
		if (xmms_remote_get_playlist_pos(test_sdl_dp.xmms_session) !=
			(displayer_test_sdl_status.jump_to_song - 1)) {
		xmms_remote_set_playlist_pos
			(test_sdl_dp.xmms_session, (displayer_test_sdl_status.jump_to_song - 1));
		}
	case SDLK_KP_PERIOD:
		displayer_test_sdl_status.jump_to_song = 0;
		break;
	case SDLK_KP0:
	case SDLK_KP1:
	case SDLK_KP2:
	case SDLK_KP3:
	case SDLK_KP4:
	case SDLK_KP5:
	case SDLK_KP6:
	case SDLK_KP7:
	case SDLK_KP8:
	case SDLK_KP9:
	case SDLK_0:
	case SDLK_1:
	case SDLK_2:
	case SDLK_3:
	case SDLK_4:
	case SDLK_5:
	case SDLK_6:
	case SDLK_7:
	case SDLK_8:
	case SDLK_9:
		value = key - SDLK_0;
		if (value < 0 || value > 9)
		value = key - SDLK_KP0;

		if (mod & KMOD_CTRL) {
			GDK_THREADS_ENTER();
			enable_plugin(((value + 9) % 10));
			GDK_THREADS_LEAVE();
		}
		else {
			displayer_test_sdl_status.jump_to_song = (displayer_test_sdl_status.jump_to_song * 10) + value;
			if (displayer_test_sdl_status.jump_to_song > xmms_remote_get_playlist_length(test_sdl_dp.xmms_session))
				{ displayer_test_sdl_status.jump_to_song = value; }
		}
		break;
	default:
		// printf("%x - %x\n", event->state, event->keyval);
//		g_print("%i\n", key);
		break;
        }
}

static void sdl_test_handler_resize(void *pEvent)
{
	gboolean draw = TRUE;
	SDL_ResizeEvent* pResizeEvent = (SDL_ResizeEvent*) pEvent;

	if (!displayer_test_sdl_status.is_fullscreen)
		{ draw = CreateScreen(pResizeEvent->w, pResizeEvent->h, FALSE); }
}
