/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <string.h>
#include <stdio.h>

#include <gtk/gtkbin.h>
#include <gtk/gtktext.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkscrolledwindow.h>

#include <gdk/gdkkeysyms.h>

#include "singit_debug.h"
#include "editor_singit_main.h"
#include "editor_singit_private.h"
#include "editor_plain_text.h"

enum {
	ARG_0,
	ARG_CURSOR_TEXT_OFFSET
};

extern EditorStatus editor_status;

static GtkBinClass *parent_class = NULL;

static void editor_plain_text_set_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void editor_plain_text_get_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void editor_plain_text_changed_event (GtkEditable *editable, gpointer user_data);
static void editor_plain_text_destroy (GtkObject *object);
static void editor_plain_text_draw (GtkWidget *widget, GdkRectangle *area);
static gint editor_plain_text_expose (GtkWidget *widget, GdkEventExpose *event);
static void editor_plain_text_size_request (GtkWidget *widget, GtkRequisition *requisition);
static void editor_plain_text_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static gboolean editor_plain_text_keypress(GtkWidget *widget, GdkEventKey *event, gpointer data);

static void editor_plain_text_class_init (EditorPlainTextClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = (GtkObjectClass*) class;
	widget_class = (GtkWidgetClass*) class;

	parent_class = gtk_type_class (gtk_bin_get_type ());

	gtk_object_add_arg_type ("EditorPlainText::cursoroffset",
			GTK_TYPE_UINT,
			GTK_ARG_READABLE,
			ARG_CURSOR_TEXT_OFFSET);

	object_class->set_arg = editor_plain_text_set_arg;
	object_class->get_arg = editor_plain_text_get_arg;
	object_class->destroy = editor_plain_text_destroy;
	widget_class->draw = editor_plain_text_draw;
	widget_class->expose_event = editor_plain_text_expose;
	widget_class->size_request = editor_plain_text_size_request;
	widget_class->size_allocate = editor_plain_text_size_allocate;
}

static void editor_plain_text_init (EditorPlainText *ept)
{
	GtkWidget *editor_plain_text_vscrollbar;
	GtkWidget *editor_plain_text_hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(ept), editor_plain_text_hbox);
	gtk_widget_show(editor_plain_text_hbox);

	ept->text = gtk_text_new (NULL, NULL);
	gtk_box_pack_start(GTK_BOX(editor_plain_text_hbox), ept->text, TRUE, TRUE, 0);
	gtk_text_set_editable(GTK_TEXT(ept->text), TRUE);
	gtk_text_set_line_wrap(GTK_TEXT(ept->text), FALSE);
	gtk_widget_set_usize(ept->text, 220, 120);
/*	gtk_widget_set_events(ept->text
		GDK_EXPOSURE_MASK |
		GDK_BUTTON_PRESS_MASK |
		GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_POINTER_MOTION_HINT_MASK);*/

	gtk_signal_connect(GTK_OBJECT(ept->text), "changed",
		GTK_SIGNAL_FUNC(editor_plain_text_changed_event), NULL);

	gtk_signal_connect(GTK_OBJECT(ept), "key-press-event",
		GTK_SIGNAL_FUNC(editor_plain_text_keypress), NULL);

	gtk_widget_show (ept->text);

	editor_plain_text_vscrollbar = gtk_vscrollbar_new (GTK_TEXT(ept->text)->vadj);
	gtk_box_pack_start(GTK_BOX(editor_plain_text_hbox),
		editor_plain_text_vscrollbar, FALSE, FALSE, 0);
	gtk_widget_show(editor_plain_text_vscrollbar);
}

GtkType editor_plain_text_get_type (void)
{
	static guint ept_type = 0;

	if (!ept_type) {
		GtkTypeInfo ept_info = {
			(gchar*) "EditorPlainText",
			sizeof (EditorPlainText),
			sizeof (EditorPlainTextClass),
			(GtkClassInitFunc) editor_plain_text_class_init,
			(GtkObjectInitFunc) editor_plain_text_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		ept_type = gtk_type_unique (gtk_bin_get_type(), &ept_info);
	}

	return ept_type;
}

static void editor_plain_text_destroy (GtkObject *object)
{
	EditorPlainText *ept;

	g_return_if_fail (ept = EDITOR_PLAIN_TEXT (object));

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void editor_plain_text_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	EditorPlainText *editor;

	editor = EDITOR_PLAIN_TEXT(object);

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_set_arg]\n"));
#endif

	switch (arg_id) {
	case ARG_CURSOR_TEXT_OFFSET:
		editor->cursor_text_offset = GTK_VALUE_UINT(*arg);
		break;
	default:
		break;
	}
}

static void editor_plain_text_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	EditorPlainText *editor;

	editor = EDITOR_PLAIN_TEXT(object);

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_get_arg]\n"));
#endif

	switch (arg_id) {
	case ARG_CURSOR_TEXT_OFFSET:
		GTK_VALUE_UINT(*arg) = editor->cursor_text_offset;
		break;
	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

static void editor_plain_text_draw (GtkWidget *widget, GdkRectangle *area)
{
	GtkBin *bin;
	GdkRectangle child_area;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_EDITOR_PLAIN_TEXT (widget));
	g_return_if_fail (area != NULL);

	if (GTK_WIDGET_DRAWABLE (widget)) {
		bin = GTK_BIN (widget);

		if (bin->child && gtk_widget_intersect (bin->child, area, &child_area))
			gtk_widget_draw (bin->child, &child_area);
	}
}

static gint editor_plain_text_expose (GtkWidget *widget, GdkEventExpose *event)
{
	GtkBin *bin;
	GdkEventExpose child_event;

	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (IS_EDITOR_PLAIN_TEXT (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	if (GTK_WIDGET_DRAWABLE (widget)) {
		bin = GTK_BIN (widget);

		child_event = *event;
		if (bin->child &&
			GTK_WIDGET_NO_WINDOW (bin->child) &&
			gtk_widget_intersect (bin->child, &event->area, &child_event.area))
			gtk_widget_event (bin->child, (GdkEvent*) &child_event);
	}

	return FALSE;
}

static void editor_plain_text_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
	GtkBin *bin;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_EDITOR_PLAIN_TEXT (widget));
	g_return_if_fail (requisition != NULL);

	bin = GTK_BIN (widget);

	requisition->width = (GTK_CONTAINER (widget)->border_width +
		GTK_WIDGET (widget)->style->klass->xthickness) * 2;

	requisition->height = (GTK_CONTAINER (widget)->border_width +
		GTK_WIDGET (widget)->style->klass->ythickness) * 2;

	if (bin->child && GTK_WIDGET_VISIBLE (bin->child)) {
		GtkRequisition child_requisition;
		gtk_widget_size_request (bin->child, &child_requisition);
	}
}

static void editor_plain_text_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	GtkBin *bin;
	GtkAllocation child_allocation;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_EDITOR_PLAIN_TEXT (widget));
	g_return_if_fail (allocation != NULL);

	bin = GTK_BIN (widget);

	if (GTK_WIDGET_MAPPED (widget) &&
		((widget->allocation.x != allocation->x) ||
		(widget->allocation.y != allocation->y) ||
		(widget->allocation.width != allocation->width) ||
		(widget->allocation.height != allocation->height)) &&
		(widget->allocation.width != 0) &&
		(widget->allocation.height != 0))
	{
		gtk_widget_queue_clear (widget);
	}

	widget->allocation = *allocation;

	if (bin->child && GTK_WIDGET_VISIBLE (bin->child)) {
		child_allocation.x = (GTK_CONTAINER(widget)->border_width +
			widget->style->klass->xthickness);
		child_allocation.width = MAX(1, (gint)allocation->width - child_allocation.x * 2);

		child_allocation.y = (GTK_CONTAINER(widget)->border_width +
			GTK_WIDGET(widget)->style->klass->ythickness);
		child_allocation.height = MAX (1, ((gint)allocation->height - child_allocation.y -
					 (gint)GTK_CONTAINER (widget)->border_width -
					 (gint)GTK_WIDGET (widget)->style->klass->ythickness));
		child_allocation.x += allocation->x;
		child_allocation.y += allocation->y;
		gtk_widget_size_allocate (bin->child, &child_allocation);
	}
}

static gboolean editor_plain_text_keypress(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	gboolean result = TRUE;

	switch(event->keyval) {
		case GDK_Escape:
			if (check_user_continue)
				singit_editor_hide();
			break;
		default:
			result = FALSE;
			break;
        }
        return result;
}

gchar *editor_plain_text_get_text(EditorPlainText *ept, gint *char_offset, gint start, gint end)
{
	gint temp;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_get_text]\n"));
#endif

	g_return_val_if_fail (IS_EDITOR_PLAIN_TEXT(ept), NULL);

	if (start < 0) {
		start = GTK_EDITABLE(ept->text)->selection_start_pos;
		end = GTK_EDITABLE(ept->text)->selection_end_pos;
	}
	if (start == end) { return NULL; }
	if (end < -1) { end = -1; }
	if ((end > 0) && (start > end)) {
		temp = start;
		start = end;
		end = temp;
	}
	if (char_offset) {
		*char_offset = gtk_editable_get_position(GTK_EDITABLE(ept->text));
	}

	return g_strdup(gtk_editable_get_chars(GTK_EDITABLE(ept->text), start, end));
}

void editor_plain_text_set_text(EditorPlainText *ept, gchar *text, gint char_offset)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_set_text]\n"));
#endif

	g_return_if_fail (IS_EDITOR_PLAIN_TEXT(ept));

	gtk_text_freeze(GTK_TEXT(ept->text));
	if (gtk_text_get_length(GTK_TEXT(ept->text)) > 0) {
		gtk_text_set_point(GTK_TEXT(ept->text), 0);
		gtk_text_forward_delete(GTK_TEXT(ept->text),
			gtk_text_get_length(GTK_TEXT(ept->text)));
	}
	if (text) {
		gtk_text_insert(GTK_TEXT(ept->text), NULL,
			&ept->text->style->black, NULL, text, -1);
		if (char_offset > (gint) strlen(text)) { char_offset = strlen(text); }
		gtk_text_set_point(GTK_TEXT(ept->text), char_offset);
	}
	gtk_text_thaw(GTK_TEXT(ept->text));
}

void editor_plain_text_xchg_selection(EditorPlainText *ept, gchar *text)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_xchg_selection]\n"));
#endif

	gtk_text_freeze(GTK_TEXT(ept->text));
	gtk_editable_delete_selection(GTK_EDITABLE(ept->text));
	gtk_text_insert(GTK_TEXT(ept->text), NULL,
		&ept->text->style->black, NULL, text, -1);
	gtk_text_thaw(GTK_TEXT(ept->text));
}

gboolean editor_plain_text_is_selected(EditorPlainText *ept)
{
	return (GTK_EDITABLE(ept->text)->selection_end_pos -
		GTK_EDITABLE(ept->text)->selection_start_pos);
}

static void editor_plain_text_changed_event (GtkEditable *editable, gpointer user_data)
{
	toggle_modified_state(TRUE);
}

GtkWidget* editor_plain_text_new(void)
{
	GtkWidget *editor;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_new]\n"));
#endif

	editor = gtk_type_new(TYPE_EDITOR_PLAIN_TEXT);
	gtk_widget_set_usize(editor, 300, 160);

	return editor;
}
