
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#undef VERSION
#define VERSION  "0.1"

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <unistd.h>	/* isatty() */
#include <png.h>	/* libpng header file:  includes zlib.h and setjmp.h */

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

typedef struct _jmpbuf_wrapper {

	jmp_buf jmpbuf;
}
jmpbuf_wrapper;

static jmpbuf_wrapper p2llPngJmpbufStruct;

static int  png2csource (FILE *infile, FILE *outfile);
static void p2llPngErrorHandler (png_structp png_ptr, png_const_charp msg);

static char **argv_ptr;

static void print_blurb(FILE *bout)
{
	fprintf(bout, "%s version %s, by Jan-Marek Glogowski.\n",
		argv_ptr[0], VERSION);
	fprintf(bout, "   Compiled with libpng %s; using libpng %s.\n",
		PNG_LIBPNG_VER_STRING, png_libpng_ver);
	fprintf(bout, "   Compiled with zlib %s; using zlib %s.\n\n",
		ZLIB_VERSION, zlib_version);
	fprintf(bout, "usage:  %s [options] <PNG_FILE>... > "
		"c-source.h\n", argv_ptr[0]);
	fprintf(bout, "   or:  ... | %s > c-source.h\n", argv_ptr[0]);
}

static int parse_args (gint *argc_p, gchar ***argv_p)
{
	guint argc = *argc_p;
	gchar **argv = *argv_p;
	guint i = 1;

	while (i < argc) {
		if (strcmp ("-h", argv[i]) == 0 ||
			 strcmp ("--help", argv[i]) == 0)
		{
			print_blurb (stderr);
			argv[i] = NULL;
			exit (0);
		}
		else if (strcmp ("-v", argv[i]) == 0 ||
			 strcmp ("--version", argv[i]) == 0)
		{
			print_blurb (stderr);
			argv[i] = NULL;
			exit (0);
		}
		else { break; }
		i++;
	}
	
//	fprintf(stderr, "%i / %i\n", i, argc);
	
	return i;
}

int main(int argc, char *argv[])
{
	FILE *infile, *outfile;
	int /* c, */ error = 0;
	gint files;

	argv_ptr = argv;

	files = parse_args(&argc, &argv);

#ifdef RISCOS
	outfile = stdout;
#else
	if ((outfile = fdopen(1, "w")) == (FILE *)NULL) {
		fprintf(stderr, "%s error:  cannot write to stdout\n", argv[0]);
		return 5;
	}
#endif

	if ((argc - files) == 0) {
		if (isatty(0) 
			// || ((c = getc(stdin)) == EOF)
			)
		{
			print_blurb(stderr);
			return 0;
		}

		// ungetc(c, stdin);

		// setmode/fdopen code borrowed from Info-ZIP's funzip.c (Mark Adler)
		// [HAVE_SETMODE is same macro used by NetPBM 9.x for Cygwin, etc.]

#ifdef HAVE_SETMODE   // DOS, FLEXOS, Human68k, NetWare, OS/2, Win32
# if (defined(__HIGHC__) && !defined(FLEXOS))
		setmode(stdin, _BINARY);
# else
		setmode(0, O_BINARY);   /* some buggy C libs require BOTH setmode() */
# endif          //  call AND fdopen() in binary mode :-(
#endif

#ifdef RISCOS
		infile = stdin;
#else
		if ((infile = fdopen(0, "rb")) == (FILE *)NULL) {
			fprintf(stderr, "%s error:  cannot find stdin\n", argv[0]);
			return 5;
		}
#endif
		goto process_png;
	}
	
	while (files < argc) {
		if ((infile = fopen(argv[files], "rb")) == (FILE *)NULL) {
			fprintf(stderr,
				"%s error: cannot open %s for reading\n", 
				argv[0], argv[files]);
			return 5;
		}

process_png:
		// output always goes to stdout (text mode, not binary)
		error = png2csource(infile, outfile);

		fclose(infile);

		files++;
	}

	fclose(outfile);

	return error;
}



/* This function is based on the Chapter 13 demo code in "PNG: The
 * Definitive Guide" (http://www.libpng.org/pub/png/pngbook.html).
 *
 * Return codes:
 *   0 = okey dokey
 *   1 = infile is not a PNG image
 *   2 = PNG image is wrong type or wrong size or has too many colors
 *   3 = libpng error of some sort (longjmp)
 *   4 = insufficient memory
 */

static int png2csource(FILE *infile, FILE *outfile)
{
	static int add_bogus_entry = FALSE;
	static png_byte sig[8] = {0, 0, 0, 0, 0, 0, 0, 0};
	png_structp png_ptr;
	png_infop info_ptr;
	png_uint_32 width, height, rowbytes;
	int bit_depth, color_type, num_palette;
	png_colorp palette;
	png_bytep p, image_data = NULL;
	png_bytepp row_pointers = NULL;
	int i;
	time_t numtime;
	char ascnumtime[32];


	// first do a quick check that the file really is a PNG image; could
	// have used slightly more general png_sig_cmp() function instead
	i = fread(sig, 1, 8, infile);
	if (i < 8 || !png_check_sig(sig, 8)) {
		fprintf(stderr, "%s error: not a PNG image\n", argv_ptr[0]);
		fprintf(stderr,
			"  (signature bytes = %02x %02x %02x %02x %02x %02x %02x %02x)\n",
			sig[0], sig[1], sig[2], sig[3], sig[4], sig[5], sig[6], sig[7]);
		return 1; // bad signature
	}

	png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
		&p2llPngJmpbufStruct, p2llPngErrorHandler, NULL);
	if (png_ptr == NULL) {
		fprintf(stderr,
			"%s error: can't allocate libpng main struct\n",
			argv_ptr[0]);
		return 4;
	}

	info_ptr = png_create_info_struct(png_ptr);
	if (info_ptr == NULL) {
		fprintf(stderr,
			"%s error: can't allocate libpng info struct\n", 
			argv_ptr[0]);
		png_destroy_read_struct(&png_ptr, NULL, NULL);
		return 4;
	}

	// setjmp() must be called in every non-callback function that calls a
	// PNG-reading libpng function
	if (setjmp(p2llPngJmpbufStruct.jmpbuf)) {
		fprintf(stderr,
			"%s error: setjmp returns error condition\n",
			argv_ptr[0]);
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		return 3;
	}

	png_init_io(png_ptr, infile);
	png_set_sig_bytes(png_ptr, 8);  // we already read the 8 signature bytes

	png_read_info(png_ptr, info_ptr);  // read all PNG info up to image data

	png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth, 
		&color_type, NULL, NULL, NULL);

	// GRR 20010922:  should allow 1- to 8-bit grayscale, too (but would have
	//  to count colors in 8-bit case)
	if (color_type != PNG_COLOR_TYPE_PALETTE) {
		fprintf(stderr, 
			"%s error: only palette PNGs supported\n",
			argv_ptr[0]);
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		return 2;
	}

	if (bit_depth < 8)
		png_set_packing(png_ptr);   // expand to 1 byte per pixel

	png_get_PLTE(png_ptr, info_ptr, &palette, &num_palette);

	if (num_palette > 255) {
		fprintf(stderr,
			"%s error: too many colors (%d); must be less than 224\n",
			argv_ptr[0], num_palette);
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		return 2;
	}

	if (num_palette == 214)
		add_bogus_entry = TRUE;   // 214 is special:  triggers stock version

	if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS)) {
		fprintf(stderr, 
			"%s warning: transparency not supported\n",
			argv_ptr[0]);
	}

	png_read_update_info(png_ptr, info_ptr);

	// allocate space for the PNG image data */
	rowbytes = png_get_rowbytes(png_ptr, info_ptr);
	if ((image_data = (png_bytep)malloc(rowbytes*height)) == NULL) {
		fprintf(stderr, 
			"%s error: can't allocate image data\n",
			argv_ptr[0]);
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		return 4;
	}
	if ((row_pointers = (png_bytepp)malloc(height*sizeof(png_bytep))) == NULL) {
		fprintf(stderr, 
			"%s error: can't allocate row pointers\n",
			argv_ptr[0]);
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		free(image_data);
		return 4;
	}

	// set the individual row_pointers to point at the correct offsets
	for (i = 0;  i < (int) height;  ++i)
		row_pointers[i] = image_data + i*rowbytes;

	png_read_image(png_ptr, row_pointers);   // read whole image...
	png_read_end(png_ptr, NULL);             // ...done!

	numtime = time((time_t *)NULL);
	strftime(ascnumtime, 32, "%Y/%m/%d %H:%M:%S", localtime(&numtime));

	p = image_data;
	for (i = 0;  i < 80*80;  ++i) {
		if (i > 0)
			fprintf(outfile, ",");
		if ((i & 7) == 0)
			fprintf(outfile, "\n ");
		fprintf(outfile, " 0x%02X", (*p++) + 0x20);
	}

	png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
	free(image_data);
	free(row_pointers);

	return 0;
}

static void p2llPngErrorHandler(png_structp png_ptr, png_const_charp msg)
{
	jmpbuf_wrapper  *jmpbuf_ptr;

	// This function, aside from the extra step of retrieving the "error
	// pointer" (below) and the fact that it exists within the application
	// rather than within libpng, is essentially identical to libpng's
	// default error handler.  The second point is critical:  since both
	// setjmp() and longjmp() are called from the same code, they are
	// guaranteed to have compatible notions of how big a jmp_buf is,
	// regardless of whether _BSD_SOURCE or anything else has (or has not)
	// been defined.

	fprintf(stderr, "png2linuxlogo:  fatal libpng error: %s\n", msg);
	fflush(stderr);

	jmpbuf_ptr = png_get_error_ptr(png_ptr);
	if (jmpbuf_ptr == NULL) {         /* we are completely hosed now */
		fprintf(stderr, "png2linuxlogo:  EXTREMELY fatal error: jmpbuf "
			"unrecoverable; terminating.\n");
		fflush(stderr);
		exit(99);
	}

	longjmp(jmpbuf_ptr->jmpbuf, 1);
}
