/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/***********************************************************************
*
*	Check thread and helper functions
*	This thread checks, if the contents of a file has changed
*	All funtions are inlined because they are used only
*	once and are just seperated for overview purposes
*
***********************************************************************/

#include <pthread.h>

#include <gtk/gtk.h>
#include <string.h>

// #include <time.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include "singit_debug.h"
#include "singit_config_gen.h"
#include "singit_config.h"
#include "singit_song.h"
#include "singit_song_private.h"

#include "singit_main.h"

#include "singit_file_info.h"


extern SingitConfigGen *singit_config;
extern SingitStatus singit_status;
extern VisPlugin singit_vp;

static SingitFileInfo** create_lyric_filenames(gchar *song_filename)
{
	gchar *buffer, *fileName, *fileBase;
	gchar **directories, **extensions;
        gint i = 0, j = 0;
	SingitFileInfo **result = NULL;
	GSList *names = NULL, *names_it;
	gint name_count = 0;

#	ifdef CODEDEBUG
	DEBUG(8, ("singit_main.c [create_lyric_filenames]\n"));
#	endif

	/*
	 * Get filename, directories and extensions
	 */
	fileBase = g_strdup(g_basename(song_filename));
	buffer = strrchr(fileBase, '.');
	if (buffer != NULL) { buffer[1] = '\0'; }

	singit_status.next_lyric_line = NULL;

	directories = g_strsplit(GET_SCD->basePath, ",", 0);
	extensions = g_strsplit(GET_SCD->lyricExtension, ",", 0);

	while (directories[i]) {
		j = 0;
		while (extensions[j]) {
			if (directories[i][0] == '~') {
				if (directories[i][1] == '~') {
					buffer = &directories[i][2];
					fileName = g_strconcat(g_dirname(song_filename), buffer, fileBase, extensions[j], NULL);
				}
				else {
					buffer = &directories[i][1];
					fileName = g_strconcat(g_get_home_dir(), buffer, fileBase, extensions[j], NULL);
				}
			}
			else { fileName = g_strdup(g_strconcat(directories[i], fileBase, extensions[j], NULL)); }
#			ifdef CODEDEBUG
			DEBUG(8, ("     %s\n", fileName));
#			endif

			names = g_slist_prepend(names, fileName);
			name_count++;
			j++;
		}
		i++;
	}
	g_strfreev(directories);
	g_strfreev(extensions);
	g_free(fileBase);

#	ifdef HAVE_ID3
	name_count++;
#	endif
	
	result = g_new(SingitFileInfo*, name_count+1);
	result[name_count] = NULL;
	name_count--;

	names_it = names;
	while (names_it != NULL) {
		result[name_count] = singit_file_info_new((gchar*) names_it->data, TRUE);
		g_free(names_it->data);
		names_it = g_slist_next(names_it);
		name_count--;
	}
	g_slist_free(names);

#	ifdef HAVE_ID3
	result[name_count] = singit_file_info_new(song_filename, TRUE);
#	endif
	
	return result;
}

static gboolean create_lyric_file_polls_and_load_song(SingitSong *check_song, SingitFileInfo **infos)
{
        gint i = 0;
	gboolean loaded = FALSE;

#	ifdef CODEDEBUG
	DEBUG(8, ("singit_main.c [create_lyric_file_polls]\n"));
#	endif

	/*
	 * Construct filenames and create IO channels
	 */
	i = 0;
	while ((infos[i] != NULL) && (loaded == FALSE))
	{
		loaded = singit_song_load_lyrics(check_song, infos[i]->name);
		i++;
	}
	if (loaded == TRUE) {
		i--;
		singit_file_info_reinit((SingitFileInfo*) check_song->file_info, 
			infos[i]->name, TRUE);
	}

	return loaded;
}

/*
 * We skip some of the cpu intensive file checks (file size / sha hash)
 */
static gboolean has_song_changed(SingitSong *cur_song, SingitFileInfo **file_infos,
	gint *new_length, gchar **new_filename)
{
#define SKIP_FILE_CHECKS 50
	static gint run_file_check = 1;
	static gboolean last_changed = FALSE;

	gchar *file_name = NULL;
	gint pos, length;
	gboolean lyric_changed = FALSE;
	
	g_return_val_if_fail(cur_song != NULL, FALSE);

	// Get all important values from xmms to check if the song changed
	pos = xmms_remote_get_playlist_pos(singit_vp.xmms_session);
	length = xmms_remote_get_playlist_time(singit_vp.xmms_session, pos);
	file_name = xmms_remote_get_playlist_file(singit_vp.xmms_session, pos);

	if (!file_name) { return FALSE; }

	lyric_changed = (length != (gint) cur_song->song_length);

	// Sometimes xmms deliveres the wrong song length (xxxx000)
	// Check if we had a change and the difference < 1000
	if (lyric_changed && last_changed && ((length - cur_song->song_length) < 1000)) {
		singit_song_set_length(cur_song, length);
		lyric_changed = FALSE;
	}

//	if (lyric_changed) { g_print("Test: old - %i / new - %i\n", cur_song->song_length, length); }
	if (!lyric_changed) {
		if (!cur_song->song_filename) { lyric_changed = TRUE; }
		else { lyric_changed = (strcmp(cur_song->song_filename, file_name) != 0); }
	}

	if (singit_status.text_update == TRUE)
	{
		run_file_check = 0;
		singit_status.text_update = FALSE;
	}

	if ((run_file_check == 0) && !lyric_changed) {
		pthread_mutex_lock(&singit_status.lyrics_rw_mutex);
		lyric_changed = singit_song_lyrics_changed(cur_song);
		if (!lyric_changed && (file_infos != NULL) && (singit_song_text_found(cur_song) == FALSE)) {
			lyric_changed = (singit_file_info_any_changed(file_infos, FALSE) != NULL);
		}
		pthread_mutex_unlock(&singit_status.lyrics_rw_mutex);
	}

	if (lyric_changed) {
		*new_filename = file_name;
		*new_length = length;
		run_file_check = 1;
		last_changed = TRUE;
       	}
	else { 
		g_free(file_name);
		run_file_check = (run_file_check + 1) % SKIP_FILE_CHECKS;
		last_changed = FALSE;
	}

	return lyric_changed;
#undef SKIP_FILE_CHECKS
}

/*
 * We have two song pointers
 * One contains the current song, the other one is used for checks
 * We don't need to alloc song "objects" for every check
 */
void *check_thread_func(void *args)
{
	enum {
		CUR_SONG = 0,
		LOAD_SONG
	};
	SingitSong* songs[2];
	gboolean lyrics_changed, song_loaded;
	gint time;
	SingitFileInfo **file_infos = NULL;
	gchar *new_filename;
	gint new_length;
//	struct timespec req = { 0, 50000000 }, rem;

	// Get two song objects
	GET_SCD->song = songs[CUR_SONG] = SINGIT_SONG(singit_song_new(NULL));
	songs[LOAD_SONG] = SINGIT_SONG(singit_song_new(NULL));

#	ifdef CODEDEBUG
	DEBUG(6, ("singit_main.c [check_thread_func] : Thread started\n"));
#	endif

	while (!singit_status.kill_threads)
	{
		time = xmms_remote_get_output_time(singit_vp.xmms_session);

		if (time >= 0) {
			lyrics_changed = has_song_changed
				(songs[CUR_SONG], file_infos, &new_length, &new_filename);
			if (!singit_status.kill_threads && lyrics_changed) {
				if (file_infos != NULL) 
					{ singit_file_info_free_list(file_infos); }

				file_infos = create_lyric_filenames(new_filename);

				pthread_mutex_lock(&singit_status.lyrics_rw_mutex);
				song_loaded = create_lyric_file_polls_and_load_song
					(songs[LOAD_SONG], file_infos);
				pthread_mutex_unlock(&singit_status.lyrics_rw_mutex);

				singit_song_set_length(songs[LOAD_SONG], new_length);
				singit_song_set_song_filename(songs[LOAD_SONG], new_filename);

				if (lyrics_changed == FALSE)
					singit_song_set_song_filename(songs[LOAD_SONG], new_filename);

				GET_SCD->song = songs[LOAD_SONG];
				singit_song_detach(&songs[CUR_SONG]);
				songs[CUR_SONG] = songs[LOAD_SONG];
				songs[LOAD_SONG] = SINGIT_SONG(singit_song_new(NULL));

				singit_status.next_lyric_line =
					(singit_song_lyrics_found(songs[CUR_SONG])) ? 
					inl_singit_song_get_next_token(songs[CUR_SONG]) : NULL;
				singit_status.song_changed = TRUE;
#				ifdef CODEDEBUG
				DEBUG(6, ("singit_main.c [check_thread_func] : New song\n"));
#				endif
			}
		}
		xmms_usleep(50000);
//		nanosleep(&req, &rem);
	}

	// Remove all song references
	GET_SCD->song = NULL;
	singit_song_detach(&songs[CUR_SONG]);
	songs[CUR_SONG] = songs[LOAD_SONG] = NULL;

	if (file_infos != NULL) 
		{ singit_file_info_free_list(file_infos); }

#	ifdef CODEDEBUG
	DEBUG(6, ("singit_main.c [check_thread_func] : Thread ended\n"));
#	endif

	singit_status.check_thread = 0;
        pthread_exit(0);
}
