/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "singit_debug.h"
#include "singit_plugin_data.h"

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#include <gdk/gdk.h>

static GtkObjectClass *parent_class = NULL;

static void displayer_plugin_data_class_init (DisplayerPluginDataClass *klass);
static void displayer_plugin_data_init (DisplayerPluginData *dpd);
static void displayer_plugin_data_destroy(GtkObject *object);

GtkType displayer_plugin_data_get_type (void)
{
	static GtkType displayer_plugin_data_type = 0;

	if (!displayer_plugin_data_type) {

		static const GtkTypeInfo displayer_plugin_data_info =
		{
			(gchar*) "DisplayerPluginData",
			sizeof (DisplayerPluginData),
			sizeof (DisplayerPluginDataClass),
			(GtkClassInitFunc) displayer_plugin_data_class_init,
			(GtkObjectInitFunc) displayer_plugin_data_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		displayer_plugin_data_type = gtk_type_unique(GTK_TYPE_OBJECT, &displayer_plugin_data_info);
	}

	return displayer_plugin_data_type;
}

static void displayer_plugin_data_class_init (DisplayerPluginDataClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class(GTK_TYPE_OBJECT);

	object_class->destroy = displayer_plugin_data_destroy;
}

static void displayer_plugin_data_init(DisplayerPluginData *dpd)
{
	dpd->displayer_list = NULL;
	dpd->enabled_list = NULL;
	dpd->visible_list = NULL;
	dpd->displayer_last = NULL;

	dpd->playback_started = FALSE;
	dpd->initialized = FALSE;

	pthread_mutex_init(&dpd->plugin_update_mutex, NULL);
	pthread_mutex_init(&dpd->init_finish_mutex, NULL);

	pthread_mutex_lock(&dpd->plugin_update_mutex);
}

static void displayer_plugin_data_destroy(GtkObject *object)
{
	DisplayerPluginData *dpd;

	g_return_if_fail(object != NULL);
	g_return_if_fail(IS_DISPLAYER_PLUGIN_DATA(object));

	dpd = DISPLAYER_PLUGIN_DATA(object);

	g_list_free(dpd->visible_list);
	g_list_free(dpd->enabled_list);
	g_list_free(dpd->displayer_list);

	pthread_mutex_destroy(&dpd->plugin_update_mutex);
	pthread_mutex_destroy(&dpd->init_finish_mutex);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

GtkObject *displayer_plugin_data_new(void)
{
	return gtk_type_new(TYPE_DISPLAYER_PLUGIN_DATA);
}

gboolean displayer_plugin_data_attach(DisplayerPluginData *dpd)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("displayer_plugin_data.c [displayer_plugin_data_attach] : "));
#endif

	if (dpd != NULL) {

		g_return_val_if_fail (IS_DISPLAYER_PLUGIN_DATA (dpd), FALSE);

		gtk_object_ref(GTK_OBJECT(dpd));

#ifdef CODEDEBUG
		DEBUG(DLV_ALL, ("Attached\n"));
#endif
		return TRUE;
	}
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("Failed\n"));
#endif
	return FALSE;
}

void displayer_plugin_data_detach(DisplayerPluginData **dpd)
{
	g_return_if_fail (dpd != NULL);
	g_return_if_fail (*dpd != NULL);
	g_return_if_fail (IS_DISPLAYER_PLUGIN_DATA(*dpd));

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("displayer_plugin_data.c [displayer_plugin_data_detach]\n"));
#endif

	gtk_object_unref(GTK_OBJECT(*dpd));
	*dpd = NULL;
}

inline gboolean   displayer_plugin_data_lock_init
	(DisplayerPluginData *dpd, gboolean *initialized)
{
	gboolean result = TRUE;

	g_return_val_if_fail(dpd != NULL, FALSE);

#ifdef CODEDEBUG
	DEBUG(6, ("displayer_plugin_data.c [displayer_plugin_data_lock_init] : "));
#endif

	result = (pthread_mutex_lock(&dpd->init_finish_mutex) == 0);

	if ((result == TRUE) && (initialized != NULL))
		*initialized = dpd->initialized;

#ifdef CODEDEBUG
	DEBUG(6, ("%s\n", (result == TRUE) ? "TRUE" : "FALSE"));
#endif
	
	return result;
}

inline gboolean displayer_plugin_data_lock_init_ext
	(DisplayerPluginData *dpd, gboolean finish, 
		gboolean has_lock, gboolean trylock)
{
	gboolean result = TRUE;

	g_return_val_if_fail(dpd != NULL, FALSE);

#ifdef CODEDEBUG
	DEBUG(6, ("displayer_plugin_data.c [displayer_plugin_data_lock_init_ext] : "));
#endif

	if (has_lock == FALSE) {
		if (trylock == TRUE)
			result = (pthread_mutex_trylock(&dpd->init_finish_mutex) == 0);
		else
			result = (pthread_mutex_lock(&dpd->init_finish_mutex) == 0);
	}

	if (result == TRUE) {
		result = (dpd->initialized == finish);
		if ((result == FALSE) && (has_lock == FALSE))
			pthread_mutex_unlock(&dpd->init_finish_mutex);
	}

#ifdef CODEDEBUG
	DEBUG(6, ("%s\n", (result == TRUE) ? "TRUE" : "FALSE"));
#endif
	
	return result;
}

inline void displayer_plugin_data_unlock_init(DisplayerPluginData *dpd)
{
	g_return_if_fail(dpd != NULL);
	g_return_if_fail(pthread_mutex_unlock(&dpd->init_finish_mutex) == 0);
#ifdef CODEDEBUG
	DEBUG(6, ("displayer_plugin_data.c [displayer_plugin_data_unlock_init] : Unlock\n"));
#endif
}

inline gboolean displayer_plugin_data_lock_plugins(DisplayerPluginData *dpd, gboolean trylock)
{
	gint lock_result;

	g_return_val_if_fail(dpd != NULL, FALSE);

	if (trylock == FALSE) {
		// Since we are in GDK the calls may be inside a GDK locked
		// callback - to prevent deadlock first release the GDK lock
		GDK_THREADS_LEAVE();
		lock_result = 
			pthread_mutex_lock(&dpd->plugin_update_mutex);
		GDK_THREADS_ENTER();

		g_return_val_if_fail(lock_result == 0, FALSE);
	}
	else {
		lock_result =
			pthread_mutex_trylock(&dpd->plugin_update_mutex);
		g_return_val_if_fail(
			(lock_result == 0) || (lock_result == EBUSY),
			FALSE);

#ifdef CODEDEBUG
		if (lock_result == 0)
			DEBUG(6, ("displayer_plugin_data.c [displayer_plugin_data_lock_plugins]: trylock\n"));
#endif
		return (lock_result == 0);
	}

#ifdef CODEDEBUG
	DEBUG(6, ("displayer_plugin_data.c [displayer_plugin_data_lock_plugins] : Lock\n"));
#endif
	return TRUE;
}

inline void displayer_plugin_data_unlock_plugins(DisplayerPluginData *dpd)
{
	g_return_if_fail(dpd != NULL);
	g_return_if_fail(pthread_mutex_unlock(&dpd->plugin_update_mutex) == 0);
#ifdef CODEDEBUG
	DEBUG(6, ("displayer_plugin_data.c [displayer_plugin_data_unlock_plugins] : Unlock\n"));
#endif
}

gboolean displayer_plugin_data_lock_lower(DisplayerPluginData *dpd, gboolean *initialized)
{
	gboolean real_initialized;

	g_return_val_if_fail(dpd != NULL, FALSE);
	g_return_val_if_fail(initialized != NULL, FALSE);

	if (displayer_plugin_data_lock_init(dpd, &real_initialized) == FALSE)
		{ return FALSE; }

	if (real_initialized == TRUE) {
		if (displayer_plugin_data_lock_plugins(dpd, FALSE) == FALSE) { 
			displayer_plugin_data_unlock_init(dpd);
			return FALSE; 
		}
		else { 
			displayer_plugin_data_unlock_init(dpd); 
			*initialized = TRUE;
		}
	}
	else { *initialized = FALSE; }

#ifdef CODEDEBUG
	DEBUG(6, (("displayer_plugin_data.c [displayer_plugin_data_lock_lower] : %s\n"),
		((real_initialized == TRUE) ? "plugins" : "init")));
#endif
	return TRUE;
}
