/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <ctype.h>
#include <sys/param.h>

#include <gtk/gtk.h>

#ifdef HAVE_ID3
#include "singit_id3lib_wrapper.h"
#endif

#include "singit_debug.h"
#include "singit_song.h"
#include "singit_song_private.h"
#include "singit_tools.h"
#include "singit_file_info.h"

enum {
	FREE,
	UPDATE,
	LAST_SIGNAL
};

static GtkObjectClass *parent_class = NULL;
static guint song_signals[LAST_SIGNAL] = { 0 };

// static gint counter = 0;

static void singit_song_class_init (SingitSongClass *klass);
static void singit_song_init (SingitSong *ssong);

//static void singit_song_set_arg(GtkObject *object, GtkArg *arg, guint arg_id);
//static void singit_song_get_arg(GtkObject *object, GtkArg *arg, guint arg_id);
static void singit_song_destroy(GtkObject *object);

GtkType singit_song_get_type (void)
{
	static GtkType singit_song_type = 0;

	if (!singit_song_type) {

		static const GtkTypeInfo singit_song_info =
		{
			(gchar*) "SingitSong",
			sizeof (SingitSong),
			sizeof (SingitSongClass),
			(GtkClassInitFunc) singit_song_class_init,
			(GtkObjectInitFunc) singit_song_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		singit_song_type = gtk_type_unique(GTK_TYPE_OBJECT, &singit_song_info);
	}

	return singit_song_type;
}

static void singit_song_class_init (SingitSongClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class(GTK_TYPE_OBJECT);

//	object_class->set_arg = singit_song_set_arg;
//	object_class->get_arg = singit_song_get_arg;
	object_class->destroy = singit_song_destroy;

//	gtk_object_add_arg_type ("SingitSong::attachments", GTK_TYPE_UINT, GTK_ARG_READABLE, ARG_ATTACHMENTS);

	song_signals[FREE] =
		gtk_signal_new("free",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitSongClass, free),
			gtk_marshal_NONE__POINTER,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_POINTER);

	song_signals[UPDATE] =
		gtk_signal_new("update",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitSongClass, update),
			gtk_marshal_NONE__NONE,
			GTK_TYPE_NONE, 0);

	klass->free = NULL;
	klass->update = NULL;

	gtk_object_class_add_signals (object_class, song_signals, LAST_SIGNAL);
}

static void singit_song_init (SingitSong *ssong)
{
	ssong->first_token = NULL;
	ssong->last_token = NULL;
	ssong->active_token = NULL;
	ssong->lyrics = NULL;
	ssong->song_length = 0;
	ssong->id3tag = NULL;
	ssong->file_info = singit_file_info_new(NULL, FALSE);

	ssong->song_filename = NULL;
	ssong->lyric_filename = NULL;
	ssong->delimiter = NULL;
	ssong->lyric_type = LT_NONE;
}

static void singit_song_destroy (GtkObject *object)
{
	SingitSong *ssong;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SINGIT_SONG (object));

	ssong = SINGIT_SONG (object);

//	if (ssong->config_data != NULL)
//		gtk_signal_emit (GTK_OBJECT(ssong), song_signals[FREE], ssong->config_data);

//	g_print ("Free Song\n");
	singit_song_clear(ssong);

	if (ssong->song_filename != NULL)
		{ g_free(ssong->song_filename); }
		
	if (ssong->file_info != NULL)
		{ singit_file_info_free((SingitFileInfo*) ssong->file_info); }

#ifdef HAVE_ID3
	ID3Tag_Delete(TAG(ssong));
#endif

/*	if (ssong->adjustment)
		gtk_signal_disconnect_by_data (GTK_OBJECT(ssong->adjustment), ssong);*/

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

GtkObject *singit_song_new (gchar* song_filename)
{
	SingitSong *ssong = gtk_type_new(TYPE_SINGIT_SONG);

//	counter++;
//	g_print ("New Song\n");

#	ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_song.c [singit_song_new]\n"));
#	endif

	ssong->file_info = (SingitFileInfo*)
		singit_file_info_new(song_filename, TRUE);
	
	if (song_filename) {
		ssong->song_filename = g_strdup(song_filename);
#		ifdef HAVE_ID3
		ssong->id3tag = ID3Tag_New();
		ID3Tag_Link_WRP(TAG(ssong), song_filename);
#		endif
	}
	else {
		ssong->song_filename = song_filename;
#		ifdef HAVE_ID3
		TAG(ssong) = NULL;
#		endif
	}
	return GTK_OBJECT(ssong);
}

void singit_song_clear(SingitSong* ssong)
{
	GList *token;
	gchar *song_filename;

#	ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_song.c [singit_song_clear]\n"));
#	endif

	g_return_if_fail (ssong != NULL);
	g_return_if_fail (IS_SINGIT_SONG (ssong));

	if (ssong->first_token) {
		token = ssong->first_token;
		while (token != NULL) {
			g_free((LToken*) (token->data));
			token = g_list_next(token);
		}
//		g_free((LToken*) (token->data));
		g_list_free(ssong->first_token);
	}
	if (ssong->lyrics) { g_strfreev(ssong->lyrics); }
	if (ssong->lyric_filename) { g_free(ssong->lyric_filename); }
	if (ssong->delimiter) { g_free(ssong->delimiter); }

	song_filename = ssong->song_filename;
	singit_song_init(ssong);
	ssong->song_filename = song_filename;
}

void singit_song_free(SingitSong* ssong)
{
#	ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_song.c [singit_song_free]\n"));
#	endif

	g_return_if_fail (ssong != NULL);
	g_return_if_fail (IS_SINGIT_SONG (ssong));

	gtk_object_destroy(GTK_OBJECT(ssong));
}

SingitSong* singit_song_attach(SingitSong *ssong)
{
#	ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_song.c [singit_song_attach] : "));
#	endif

	if (ssong != NULL) {

//		counter++;
//		g_print ("Attach Song: %i\n", counter);

		g_return_val_if_fail (IS_SINGIT_SONG (ssong), NULL);

		gtk_object_ref(GTK_OBJECT(ssong));

#		ifdef CODEDEBUG
		DEBUG(DLV_ALL, ("Attached\n"));
#		endif
		return ssong;
	}
#	ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("Failed\n"));
#	endif
	return NULL;
}

void singit_song_detach(SingitSong **ssong)
{
	SingitSong *_ssong;

	g_return_if_fail(ssong != NULL);
	_ssong = *ssong;
	if (_ssong == NULL) { return; }
	
	g_return_if_fail(IS_SINGIT_SONG(_ssong));

#	ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_song.c [singit_song_detach]\n"));
#	endif

//	counter--;
//	g_print ("Detach Song: %i\n", counter);

	gtk_object_unref(GTK_OBJECT(_ssong));
	_ssong = NULL;
}

gboolean singit_song_set_song_filename(SingitSong *ssong, gchar *filename)
{
	g_return_val_if_fail (ssong != NULL, FALSE);
	g_return_val_if_fail (IS_SINGIT_SONG (ssong), FALSE);

	if (ssong->song_filename != NULL)
		{ g_free(ssong->song_filename); }
	ssong->song_filename = g_strdup(filename);

	return TRUE;
}

inline gint inl_singit_song_is_time_ok(SingitSong *ssong, gint time)
{
	if (!ssong->first_token) { return 0; }
	if (ssong->active_token) {
		if (time < (gint) tTime(ssong->active_token)) { return -1; }
		if (!g_list_next(ssong->active_token)) { return 0; }
		if (time >= (gint) tTime(g_list_next(ssong->active_token))) { return 1; }
	}
	else
		{ if (time >= (gint) tTime(ssong->first_token)) { return 1; } }
	return 0;
}

gint singit_song_is_time_ok(SingitSong *ssong, gint time)
{
	return inl_singit_song_is_time_ok(ssong, time);
}

gint singit_song_find_line(SingitSong* ssong, gint line)
{
	GList *item = NULL;

#	ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_song.c [singit_song_find_line]\n"));
#	endif

	item = ssong->first_token;
	while (item != ssong->last_token) {
		if ((gint) tLine(item) == line) { return tLine(item); }
		item = g_list_next(item);
	}
	if ((gint) tLine(item) == line) { return tLine(item); }
	return -1;
}

gint singit_song_find_time(SingitSong* ssong, gint time)
{
	GList *item = NULL;

#	ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_song.c [singit_song_find_time]\n"));
#	endif

	item = ssong->first_token;
	while (item != ssong->last_token) {
		if ((gint) tTime(item) == time) { return tTime(item); }
		item = g_list_next(item);
	}
	if ((gint) tTime(item) == time) { return tTime(item); }
	return -1;
}

#define ta ((LToken*) a)
#define tb ((LToken*) b)
gint compare_token_by_time(gconstpointer a, gconstpointer b)
{
	if ((a == NULL) || (b == NULL)) { return 0; }

	if (ta->time < tb->time) { return -1; }
	if (ta->time > tb->time) { return 1; }

	return 0;
}

gint compare_token_by_rpos(gconstpointer a, gconstpointer b)
{
	if ((a == NULL) || (b == NULL)) { return 0; }

	if (ta->line < tb->line) { return 1; }
	if (ta->line > tb->line) { return -1; }
	
	if (ta->pos < ta->pos) { return 1; }
	if (ta->pos > ta->pos) { return -1; }

	if (ta->pos == 0) {
		if (ta->time < tb->time) { return -1; }
		if (ta->time > tb->time) { return 1; }
	}
	else {
		if (ta->time < tb->time) { return 1; }
		if (ta->time > tb->time) { return -1; }
	}

	return 0;
}

gint compare_token_by_pos(gconstpointer a, gconstpointer b)
{
	if ((a == NULL) || (b == NULL)) { return 0; }

	if (ta->line < tb->line) { return -1; }
	if (ta->line > tb->line) { return 1; }
	
	if (ta->pos < ta->pos) { return -1; }
	if (ta->pos > ta->pos) { return 1; }

	if (ta->pos == 0) {
		if (ta->time < tb->time) { return 1; }
		if (ta->time > tb->time) { return -1; }
	}
	else {
		if (ta->time < tb->time) { return -1; }
		if (ta->time > tb->time) { return 1; }
	}
		
	return 0;
}
#undef tb
#undef ta

inline void singit_song_get_id3_tag(SingitSong *ssong, gchar *filename)
{
#ifdef HAVE_ID3
	if (!TAG(ssong)) { TAG(ssong) = ID3Tag_New(); }
	else { ID3Tag_Clear(TAG(ssong)); }
	if (TAG(ssong)) { ID3Tag_Link_WRP(TAG(ssong), filename); }
#endif
}

gboolean singit_song_load_lyrics(SingitSong *ssong, gchar *filename)
{
	FILE *file;
	gboolean result;
	struct stat stats;

#	ifdef CODEDEBUG
	DEBUG(7, ("singit_song.c [singit_song_load_lyrics]\n"));
	DEBUG(8, ("     %s\n", filename));
#	endif

	if (!singit_song_attach(ssong)) { return FALSE; }
	singit_song_clear(ssong);

	if (stat(filename, &stats) == -1) { singit_song_detach(&ssong); return FALSE; }
	if (S_ISDIR(stats.st_mode)) { singit_song_detach(&ssong); return FALSE; }
	
	if (!(file = fopen(filename, "r"))) { singit_song_detach(&ssong); return FALSE; }
	fclose(file);

#	ifdef CODEDEBUG
	DEBUG(8, ("1: File found\n"));
#	endif

#	ifdef HAVE_ID3
	result = singit_song_load_id3v2xx_lyrics(ssong, filename);
	if (!result) {
#	endif
		result = singit_song_load_midi_lyrics(ssong, filename);
		if (!result)
			result = singit_song_load_from_text_file(ssong, filename);
#	ifdef HAVE_ID3
	}
#	endif
	
	singit_song_detach(&ssong);

	return result;
}

GList* singit_song_find_current_token(SingitSong *ssong, gint time, gint state)
{
	GList *item = NULL;

#	ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_song.c [singit_song_find_current_token] : "));
#	endif

	if (state == 0) { state = inl_singit_song_is_time_ok(ssong, time); }
	switch (state) {
	case 0: return ssong->active_token; break;
	case 1:
		item = inl_singit_song_get_next_token(ssong);
		while (item) {
			if ((gint) tTime(item) > time) { return g_list_previous(item); }
			item = g_list_next(item);
		}
		if (!item) { return ssong->last_token; }
		break;
	case -1:
		item = ssong->active_token;
		while (item) {
			if ((gint) tTime(item) <= time) { return item; }
			item = g_list_previous(item);
		}
		break;
	}
	return item;
}

gboolean singit_song_lyrics_changed(SingitSong *ssong)
{
#define INFO(ssong) ((SingitFileInfo*) ssong->file_info)
	FILE *file;
        gchar *stat_name;
        struct stat stats;
	gboolean result;

	if (ssong == NULL) { return TRUE; }
	if (ssong->lyric_filename == NULL) { stat_name = ssong->song_filename; }
	else { stat_name = ssong->lyric_filename; }
	if (stat(stat_name, &stats) == -1) { return singit_song_text_found(ssong); }
        if (!(file = fopen(stat_name, "r"))) { return singit_song_text_found(ssong); }
/*
	if ((ssong->lyric_type != LT_TAG) && (stats.st_size > 50000)) {
		fclose(file);
		return TRUE;
	}
*/
	result = (singit_file_info_changed(INFO(ssong), &stats, file, TRUE) > 0);

	fclose(file);

	return result;
#undef INFO
}

inline gint inl_singit_song_get_text_length(SingitSong *ssong)
{
	gint length = 0;
	GList *next = NULL;

	if (ssong->active_token == NULL) { return -2; }
	next = g_list_next(ssong->active_token);
	if (next == NULL) { return -1; }
	if (tLine(ssong->active_token) != tLine(next)) { return -1; }
	length = (tPos(next) - tPos(ssong->active_token));
	if (length > 0) { return length; }
	else { return -2; }
}

gint singit_song_get_text_length(SingitSong *ssong)
{
	return inl_singit_song_get_text_length(ssong);
}

inline gboolean singit_song_is_empty_item(SingitSong *ssong, GList *item, gboolean check_end)
{
	guint length = strlen(ssong->lyrics[tLine(item)]);
	
	if ((check_end == TRUE) && (tPos(item) == length)) 
		{ return TRUE; }
	if (length == 0) 
		{ return TRUE; }
	return FALSE;
}

GList* singit_song_find_next_lyric_line(SingitSong *ssong, GList *item, gboolean empty, guint *hops)
{
	GList *runner = item;
	guint run_hops = 0;
	if (!item) { runner = ssong->first_token; }
	if (!empty) {
		while (runner) {
			runner = g_list_next(runner);
			run_hops++;
			while (runner) {
				if (!singit_song_is_empty_item(ssong, runner, FALSE)) {
					if (!item) {
						if (hops) { *hops = run_hops; }
						return runner;
					}
					if (tLine(runner) != tLine(item)) {
						if (hops) { *hops = run_hops; }
						return runner;
					}
				}
				runner = g_list_next(runner);
				run_hops++;
			}
		}
	}
	else {
		while (runner) {
			runner = g_list_next(runner);
			run_hops++;
			if (runner) {
				if (!item) {
					if (hops) { *hops = run_hops; }
					return runner;
				}
				if (tLine(item) != tLine(runner)) {
					if (hops) { *hops = run_hops; }
					return runner;
				}
			}
		}
	}
	if (hops) { *hops = 0; }
	return NULL;
}

GList* singit_song_find_prev_lyric_line(SingitSong *ssong, GList *item, gboolean empty, guint *hops)
{
	GList *runner = item;
	guint run_hops = 0;
	if (!item) { return NULL; }
	if (!empty) {
		while (runner) {
			runner = g_list_previous(runner);
			run_hops++;
			while (runner) {
				if (!singit_song_is_empty_item(ssong, runner, FALSE)) {
					if (!item) {
						if (hops) { *hops = run_hops; }
						return runner;
					}
					if (tLine(runner) != tLine(item)) {
						if (hops) { *hops = run_hops; }
						return runner;
					}
				}
				runner = g_list_previous(runner);
				run_hops++;
			}
		}
	}
	else {
		while (runner) {
			runner = g_list_previous(runner);
			run_hops++;
			if (runner) {
				if (!item) {
					if (hops) { *hops = run_hops; }
					return runner;
				}
				if (tLine(item) != tLine(runner)) {
					if (hops) { *hops = run_hops; }
					return runner;
				}
			}
		}
	}
	if (hops) { *hops = 0; }
	return runner;
}

gint singit_song_check_sync_lyric_consistency(SingitSong *ssong)
{
	GList *cur_item, *prev_item;
	if (!ssong) { return -1; }
	if (!ssong->first_token) { return -1; }
	if (ssong->first_token == ssong->last_token) { return -1; }
	if (!singit_song_guess_sync_lyrics(ssong)) { return -1; }

	prev_item = ssong->first_token;

	// Don't check empty line items
	while ((prev_item) && (!singit_song_is_empty_item(ssong, prev_item, FALSE)))
		{ prev_item = g_list_next(prev_item); }

	cur_item = prev_item;
	while (cur_item) {
		if (tLine(prev_item) > tLine(cur_item)) { return tLine(cur_item); }
		if (tLine(prev_item) == tLine(cur_item))
			if (tPos(prev_item) > tPos(cur_item)) { return tLine(cur_item); }

		prev_item = cur_item;
		cur_item = g_list_next(cur_item);
		// Don't check empty line items
		while ((cur_item) && (!singit_song_is_empty_item(ssong, cur_item, FALSE)))
			{ cur_item = g_list_next(cur_item); }
	}
	return -1;
}

void singit_song_modify_overall_time(SingitSong *ssong, gint time)
{
	GList *cur_item;
	gint song_time;
	if (!ssong) { return; }

	cur_item = ssong->first_token;
	while (cur_item) {
		song_time = tTime(cur_item) + time;
		if (song_time < 0) { tTime(cur_item) = 0; }
		else { tTime(cur_item) = song_time; }
		cur_item = g_list_next(cur_item);
	}
}

// If any Item has an offset it's synced
gboolean singit_song_guess_sync_lyrics(SingitSong *ssong)
{
	GList *cur_item;
	if (!ssong) { return FALSE; }

	cur_item = ssong->first_token;
	while (cur_item) {
		if (tPos(cur_item) > 0) { return TRUE; }
		cur_item = g_list_next(cur_item);
	}
	return FALSE;
}

inline GList *inl_singit_song_get_next_token(SingitSong *ssong)
{
	if (ssong->active_token) { return g_list_next(ssong->active_token); }
	return ssong->first_token;
}

inline GList *inl_singit_song_get_prev_token(SingitSong *ssong)
{
	if (ssong->active_token) { return g_list_previous(ssong->active_token); }
	return NULL;
}

gboolean singit_song_extract_token(gchar *lyric_text, gint token_nr, gint *time)
{
	gint result_time, token_it = (token_nr + 1);
	gchar *token_pos = lyric_text;
	gboolean token_found = FALSE;

	g_return_val_if_fail(lyric_text == NULL, FALSE);
	g_return_val_if_fail(token_nr >= 0, FALSE);

	do {
		token_pos = strstr(token_pos, "[");
		if (extrakt_timetag_information(token_pos, &result_time) == TRUE)
		{
			token_it--;
		}
		if (token_pos != NULL) { token_pos++; }
	}
	while ((token_pos == NULL) && (token_it == 0));

	if (token_it == 0) {
		token_found = TRUE;
		if (time != NULL)
			{ *time = result_time; }
	}

	return token_found;
}

gchar* singit_song_remove_token(gchar *lyric_text, gint token_nr)
{
	gint result_time, token_it = (token_nr + 1);
	gchar *token_pos = lyric_text, *end_text, *result_text = NULL;
	gchar tmp_char;

	g_return_val_if_fail(lyric_text != NULL, NULL);
	g_return_val_if_fail(token_nr >= 0, NULL);

	while ((token_pos != NULL) && (token_it > 0))
	{
		token_pos = strstr(token_pos, "[");

		if (token_pos != NULL) {
			if (extrakt_timetag_information(token_pos, &result_time) == TRUE)
			{
				token_it--;
			}
			token_pos++;
		}
	}

	if (token_it == 0) {
		end_text = strstr(token_pos, "]");
		end_text++;
		token_pos--;
		tmp_char = token_pos[0];
		token_pos[0] = '\0';
		result_text = g_strconcat(lyric_text, end_text, NULL);
		token_pos[0] = tmp_char;
	}

	return result_text;
}
