/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


// By enabling this define there will be a stream write on every read and 
// the results will be displayed in debug mode 9
// #define TEST_WRITE_FROM_READ

#include <string.h>
#include <sys/stat.h>

#include "singit_debug.h"

#include "singit_id3lib_wrapper.h"

#include "singit_song.h"
#include "singit_song_private.h"
#include "singit_sha.h"
#include "singit_tools.h"
#include "singit_file_info.h"



static SingitSong* attach_id3tag(SingitSong *ssong, gchar **filename)
{
	SingitSong *my_song = singit_song_attach(ssong);
	struct stat stats;
	
	if (my_song == NULL) 
		{ return NULL; }

	if ((*filename) == NULL) 
		{ (*filename) = my_song->song_filename; }

	if (stat((*filename), &stats) == -1) { 
		singit_song_detach(&my_song);
		return NULL; 
	}

	if (TAG(my_song) == NULL) { 
		TAG(my_song) = ID3Tag_New();
		ID3Tag_Link_WRP(TAG(my_song), (*filename));
	}
	
	return my_song;
}

gboolean singit_song_load_id3v2xx_lyrics(SingitSong *ssong, gchar *filename)
{
	SingitSong *my_song;
	size_t nDataSize;
	ID3Frame *id3_frame;
	ID3Field *id3_field;
	gboolean result = FALSE;
	luint num_chars;
	gchar *dyn_text = NULL;
	uint32 timstamp_format = 0;
	struct stat stats;
#	ifdef CODEDEBUG
	int content_type = 0;
#	endif

#	ifdef CODEDEBUG
	DEBUG(8, ("singit_song_id3v2xx.c [singit_song_load_id3v2xx_lyrics]\n"));
#	endif

	my_song = singit_song_attach(ssong);
	if (my_song == NULL) 
		{ return FALSE; }

	if (filename == NULL) 
		{ filename = my_song->song_filename; }

	if (stat(filename, &stats) == -1) {
		singit_song_detach(&my_song);
		return FALSE;
	}

	if (TAG(my_song) == NULL)
		{ TAG(my_song) = ID3Tag_New(); }
	ID3Tag_Link_WRP(TAG(my_song), filename);
	
	if (ID3Tag_NumFrames(TAG(my_song)) <= 0) {
		singit_song_detach(&my_song);
		return FALSE;
	}

/*	g_print("%i\n", frm_size);
	
	if (!ID3Tag_HasTagType(TAG(my_song),
			ID3TT_ID3V2 |
			ID3TT_LYRICS3 |
			ID3TT_LYRICS3V2 |
			ID3TT_MUSICMATCH)
		)
	{
		ID3Tag_Delete(TAG(my_song));
		TAG(my_song) = NULL;
		singit_song_detach(my_song);
		return FALSE;
	}
*/

	if ((id3_frame = ID3Tag_FindFrameWithID(TAG(my_song), ID3FID_SYNCEDLYRICS)) != NULL) {
		if ((id3_field = ID3Frame_GetField(id3_frame, ID3FN_TIMESTAMPFORMAT)) != NULL) {
			timstamp_format = ID3Field_GetINT (id3_field); // (2 == nTimestamp) ? "ms" : "frames"
		}
#ifdef CODEDEBUG
		DEBUG_AREA(DLV_ALL, 
		if ((id3_field = ID3Frame_GetField(id3_frame, ID3FN_CONTENTTYPE)) != NULL) {
			content_type = ID3Field_GetINT (id3_field);
			switch (content_type) {
			case ID3CT_OTHER:    g_print("  id3 data type: Other\n"); break;
			case ID3CT_LYRICS:   g_print("  id3 data type: Lyrics\n"); break;
			case ID3CT_TEXTTRANSCRIPTION:     g_print("  id3 data type: Text transcription\n"); break;
			case ID3CT_MOVEMENT: g_print("  id3 data type: Movement/part name\n"); break;
			case ID3CT_EVENTS:   g_print("  id3 data type: Events\n"); break;
			case ID3CT_CHORD:    g_print("  id3 data type: Chord\n"); break;
			case ID3CT_TRIVIA:   g_print("  id3 data type: Trivia/'pop up' information\n"); break;
			}
		}
		);
#endif
		if ((id3_field = ID3Frame_GetField(id3_frame, ID3FN_DATA)) != NULL) {
			nDataSize = ID3Field_Size(id3_field);
			dyn_text = g_new(gchar, ID3Field_Size(id3_field));
			ID3Field_GetBINARY (id3_field, dyn_text, nDataSize);

			result = singit_song_read_id3v2_sync_stream
				(my_song, dyn_text, nDataSize);
		}
	}
	else {
		if ((id3_frame = ID3Tag_FindFrameWithID(TAG(my_song), ID3FID_UNSYNCEDLYRICS)) != NULL) {
			if ((id3_field = ID3Frame_GetField(id3_frame, ID3FN_TEXT)) != NULL) {
				dyn_text = g_new(gchar, ID3Field_Size(id3_field) + 1);
				dyn_text[ID3Field_Size(id3_field)] = 0;
				num_chars = ID3Field_GetASCII_WRP(id3_field, dyn_text, ID3Field_Size(id3_field));
			}
			result = singit_song_read_text_stream(my_song, dyn_text);
			g_free(dyn_text);
		}
	}

	if (result == TRUE) {
		singit_file_info_reinit((SingitFileInfo*) my_song->file_info, filename, TRUE);
		my_song->lyric_filename = g_strdup(filename);
		my_song->lyric_type = LT_TAG;
	}
	
	singit_song_detach(&my_song);

	return result;
}

gboolean singit_song_save_id3v2_lyrics(SingitSong *ssong, gchar *filename, gint tag_type)
{
	SingitSong *my_song = attach_id3tag(ssong, &filename);
	FILE *file;
	gpointer stream;
	guint slength;
	gboolean result = FALSE;

#ifdef CODEDEBUG
	DEBUG(8, ("singit_song_id3v2xx.c [singit_song_save_id3v2_lyrics]\n"));
#endif
	
	if (my_song == NULL) 
		{ return FALSE; }
	if (singit_song_text_found(my_song) == FALSE) 
		{ return FALSE; }

	if (singit_song_write_id3v2_sync_stream(ssong, &stream, &slength) == FALSE) 
		{ goto cleanup_song; }

	if (filename == NULL) 
		{ filename = my_song->lyric_filename; }

        if (!(file = fopen(filename, "w"))) 
		{ goto cleanup_stream; }

        if (fwrite(stream, 1, slength, file) == slength)
		{ result = TRUE; }
	fclose(file);
	
	if (my_song->lyric_filename == NULL) 
		{ my_song->lyric_filename = g_strdup(filename); }

cleanup_stream:
	g_free(stream);

cleanup_song:
	singit_song_detach(&my_song);
	return result;
}

gboolean singit_song_read_id3v2_sync_stream(SingitSong *ssong, gpointer stream, guint slength)
{
	SingitSong *my_song;
	gchar *stream_pos = NULL, *lyrics = NULL, *lyrics_pos = NULL;
	guint token_counter = 1, line_counter, len, lyrics_length, line_length;
	LToken *token = NULL;
	GList *list = NULL, *item;
	gint i;

#if defined CODEDEBUG && defined TEST_WRITE_FROM_READ
	gpointer test_stream;
	guint test_size;
#endif

#ifdef CODEDEBUG
	DEBUG(8, ("singit_song_id3v2xx.c [singit_song_read_id3v2_sync_stream]\n"));
#endif
	
	if ((stream == NULL) || (slength == 0))
		{ return FALSE; }

	my_song = singit_song_attach(ssong);
	if (my_song == NULL) 
		{ return FALSE; }

	lyrics = g_new(gchar, slength + 1);
	
	singit_song_clear(my_song);
	
	stream_pos = stream;
	lyrics_pos = lyrics;
	while (stream_pos < ((gchar*) stream) + slength) {
		len = strlen(stream_pos);
		if (len > 0) {
			memcpy(lyrics_pos, stream_pos, len);
			lyrics_pos += len;
		}
		stream_pos += len + 1;

		if (stream_pos < ((gchar*) stream) + slength) {
			token = g_malloc(sizeof(LToken));
			token->pos = 0;
			token->line = (lyrics_pos - lyrics);
			token_counter++;
			token->time = 0;
			for (i = 0; i < 4; i++) {
				token->time *= 256; // 2^8
				token->time += 0xFF & stream_pos[i];
			}

			item = g_list_alloc();
			item->data = token;
			if (list == NULL) { list = item; }
			else {
				list->next = item;
				item->prev = list;
				list = item;
			}
			stream_pos += 4;
		}
	}
	*lyrics_pos = '\0';

	item = my_song->first_token = g_list_first(list);
	my_song->last_token = list;
	my_song->lyrics = g_strsplit(lyrics, "\n", 0);
	my_song->lyric_lines = lines_count((const gchar**) my_song->lyrics);

	line_counter = 0;
	lyrics_length = strlen(my_song->lyrics[line_counter]);
	line_length = lyrics_length;
	while (item != NULL) {
		while (tLine(item) > lyrics_length) {
			line_counter++;
			line_length = strlen(my_song->lyrics[line_counter]);
			lyrics_length += (line_length + 1);
		}
		tPos(item) = line_length - (lyrics_length - tLine(item));
		tLine(item) = line_counter;
		item = g_list_next(item);
	}

	my_song->first_token = g_list_sort(my_song->first_token, compare_token_by_time);

#if defined CODEDEBUG && defined TEST_WRITE_FROM_READ
	DEBUG_AREA(DLV_ALL, 
		singit_song_write_id3v2_sync_stream(my_song, &test_stream, &test_size);
		debug("Write test sizes: %i / %i\n", slength, test_size);
		if (slength == test_size) {
			debug("Write test bcmp: %s (%i)\n", 
				(memcmp(test_stream, stream, slength) == 0) ? 
				"Succeeded" : "Failed", test_size);
		}
	);
#endif

	singit_song_detach(&my_song);
	
	return TRUE;
}

gboolean singit_song_write_id3v2_sync_stream(SingitSong *song, gpointer *stream, guint *slength)
{
	gchar *pos = NULL;
	GList *item;
	gint line = 0, i, offset = 0, length, count, tmp_time;

#ifdef CODEDEBUG
	DEBUG(8, ("singit_song_id3v2xx.c [singit_song_write_id3v2_sync_stream]\n"));
#endif

	if ((song == NULL) || (stream == NULL) || (slength == NULL))
		{ return FALSE; }
	if (!singit_song_text_found(song)) 
		{ return FALSE; }
	if (lines_info((const gchar**) song->lyrics, &length, &count) == FALSE)
		{ return FALSE; }

	(*stream) = pos = g_malloc
		(length + count + g_list_length(song->first_token) * 5); 

	song->first_token = g_list_sort(song->first_token, compare_token_by_pos);
	item = song->first_token;

	while ((item != NULL) || (song->lyrics[line] != NULL)) {
		if (item != NULL) {
			while (line < (gint) tLine(item)) {
				length = strlen(song->lyrics[line]);
				if (offset > 0) {
					if (offset < length) {
						memcpy(pos, song->lyrics[line] + offset, 
							length - offset);
						pos += (length - offset);
					}
					offset = 0;
				}
				else {
					if (length > 0) {
						memcpy(pos, song->lyrics[line], length);
						pos += length;
					}
				}
				pos[0] = 0x0A;
				pos++;
				line++;
			}

			if (offset < (gint) tPos(item)) {
				memcpy(pos, song->lyrics[line] + offset, tPos(item) - offset);
				pos += (tPos(item) - offset);
				offset = tPos(item);
			}

//			g_print("%i: %i\n", i, pos - (gchar*) (*stream));
			pos[0] = '\0';
			pos++;
//			memcpy(pos, &(tTime(item)), 4);
			tmp_time = tTime(item);
			for (i = 3; i >= 0; i--) {
//				pos[i] = 0xFF & (tTime(item) >> (8 * i));
//				pos[i] = tTime(item) % (i * 256));
				pos[i] = tmp_time & 0xFF;
				tmp_time /= 256;
			}
			pos += 4;
			offset = tPos(item);
		}
		else {
			length = strlen(song->lyrics[line]);
			if (offset > 0) {
				if (offset < length) {
					memcpy(pos, song->lyrics[line] + offset, length - offset);
					pos += (length - offset);
				}
				pos[0] = 0x0A;
				pos++;
				line++;
			}
			while (song->lyrics[line] != NULL) {
				length = strlen(song->lyrics[line]);
				if (length > 0) {
					memcpy(pos, song->lyrics[line], length);
					pos += length;
				}
				line++;
				if (song->lyrics[line] != NULL) {
					pos[0] = 0x0A;
					pos++;
				}
			}
		}
		item = g_list_next(item);
	}

	(*slength) = (gint) pos - (gint) (*stream);

	song->first_token = 
		g_list_sort(song->first_token, compare_token_by_time);
	
	return TRUE;
}
