/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_ERRNO_H
# include <errno.h>
#endif

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#include <string.h>
#include <stdio.h>

#include "singit_tools.h"

gchar* tools_replace_string(const gchar *input_string, 
	const gchar *token, const gchar *replacement)
{
	gchar *new_text = NULL, *new_text_pos, *pos_begin, *pos_end;
	gint token_len, input_len, replacement_len, maxstr_len, text_len;

	if (!input_string || !token || !replacement) { return NULL; }

/*	#ifdef CODEDEBUG
	DEBUG(("singit_tools.c [tools_replace_string]\n"));
	#endif*/

	token_len = strlen(token);
	input_len = strlen(input_string);
	if (token_len > input_len) { return NULL; }

	replacement_len = strlen(replacement);
	maxstr_len = (input_len / token_len + 1) * replacement_len + input_len;
	new_text = g_new(gchar, maxstr_len);
	new_text_pos = new_text;
	pos_begin = (gchar*) input_string;

	while (pos_begin) {
		pos_end = strstr(pos_begin, token);
		if (!pos_end) {
			text_len = strlen(pos_begin);
			if (text_len > 0) { memcpy(new_text_pos, pos_begin, text_len); }
			new_text_pos += text_len;
			new_text_pos[0] = '\0';
			pos_begin = NULL;
		}
		else {
			text_len = pos_end - pos_begin;
			if (text_len > 0) {
				memcpy(new_text_pos, pos_begin, text_len);
				new_text_pos += text_len;
				memcpy(new_text_pos, replacement, replacement_len);
				new_text_pos += replacement_len;
			}
			pos_begin = pos_end + token_len;
		}
	}

	// Just use exact strlen mem
	new_text_pos = new_text;
	new_text = g_strdup(new_text_pos);
	g_free(new_text_pos);

	return new_text;
}

gchar* tools_insert_string(const gchar *input_string, 
	const gchar *token, guint offset)
{
	gchar *new_text = NULL, *new_text_pos;
	gint token_len, input_len, testit;

	if (!input_string || !token) { return NULL; }
	input_len = strlen(input_string);
	testit = offset - 1;
	if (testit > input_len) { return NULL; }

/*	#ifdef CODEDEBUG
	DEBUG(("singit_tools.c [tools_insert_string]\n"));
	#endif*/

	if (offset == 0) { return g_strconcat(token, input_string, 0); }
	if ((offset-1) == (guint) input_len) { return g_strconcat(input_string, token, 0); }

	token_len = strlen(token);

	new_text_pos = new_text = g_new(gchar, token_len + input_len + 1);
	memcpy(new_text_pos, input_string, offset);
	new_text_pos += offset;
	memcpy(new_text_pos, token, token_len);
	new_text_pos += token_len;
	memcpy(new_text_pos, (input_string + offset), input_len - offset);
	new_text[token_len + input_len] = '\0';

	return new_text;
}

gboolean calc_line_intersection(gint a, guint la, gint b, guint lb, gint *r, guint *lr)
{
	gint ra = a + la;
	gint rb = b + lb;

	if ((ra < b) || (a > rb) || (rb < a) || (b > ra)) { return FALSE; }

	if (a > b) { *r = a; }
	else { *r = b; }

	if (ra > rb) { *lr = rb - *r; }
	else { *lr = ra - *r; }

	return TRUE;
}

gboolean calc_rect_intersection
	(gint left_s, gint top_s, guint width_s, guint height_s,
	 gint left_d, gint top_d, guint width_d, guint height_d,
	 gint *left_r, gint *top_r, guint *width_r, guint *height_r)
{
	gint right_s = left_s + width_s;
	gint right_d = left_d + width_d;
	gint bottom_s = top_s + height_s;
	gint bottom_d = top_d + height_d;
	gint right_r, bottom_r;

	if ((right_s < left_d) || (bottom_s < top_d) || (bottom_s < top_d) || (right_s < left_d))
		{ return FALSE; }

	if (left_s < left_d) { *left_r = left_d; }
	else { *left_r = left_s; }

	if (top_s < top_d) { *top_r = top_d; }
	else { *top_r = top_s; }

	if (right_s < right_d) { right_r = right_s; }
	else { right_r = right_d; }

	if (bottom_s < bottom_d) { bottom_r = bottom_s; }
	else { bottom_r = bottom_d; }

	*width_r = right_r - *left_r;
	*height_r = bottom_r - *top_r;

/*	printf("\nSource:     %i, %i, %i, %i\n", left_s, top_s, width_s, height_s);
	printf("Destinatin: %i, %i, %i, %i\n", left_d, top_d, width_d, height_d);
	printf("Result:     %i, %i, %i, %i\n\n", *left_r, *top_r, *width_r, *height_r); */

	return TRUE;
}

gint lines_length(const gchar **lines)
{
	gint i = 0;
	gint length = 0;
	
	if (lines == NULL) 
		{ return -1; }
	
	while (lines[i] != NULL) {
		length += strlen(lines[i]);
		i++;
	}
	
	return length;
}

gint lines_count(const gchar **lines)
{
	gint i = 0;
	
	if (lines == NULL) 
		{ return -1; }
	
	while (lines[i] != NULL) { i++; }

	return i;
}

gboolean lines_info(const gchar **lines, guint *length, guint *count)
{
	if ((lines == NULL) || (length == NULL) || (count == NULL))
		{ return FALSE; }

	(*count) = 0;
	(*length) = 0;
	while (lines[(*count)] != NULL) {
		(*length) += strlen(lines[(*count)]);
		(*count)++;
	}
	
	return TRUE;
}

gboolean create_abs_path(const gchar* path, __mode_t mode)
{
	gchar *cwd, **directories;
	gint i;
	gboolean result = TRUE;

	g_return_val_if_fail(path != NULL, FALSE);
	g_return_val_if_fail(g_path_is_absolute(path), FALSE);
	
	cwd = g_get_current_dir();

	// Does the path already exists ?
	if (chdir(path) == 0) {
		chdir(cwd);
		g_free(cwd);
		return TRUE;
	}

	chdir("/");

	directories = g_strsplit(path, "/", 0);
	i = 0;
	while (directories[i] != NULL) {
		if ((strlen(directories[i]) > 0) && (chdir(directories[i]) != 0)) {
			if (errno == ENOENT) {
				if (mkdir(directories[i], mode) != 0)
				{
					result = FALSE;
					break;
				}
				continue;
			}
			else {
				result = FALSE;
				break;
			}
		}
		i++;
	}
	g_strfreev(directories);
	
	chdir(cwd);
	g_free(cwd);
	
	return result;
}

/*
uint32 io::readLENumber(ID3_Reader& reader, size_t len)
{
	uint32 val = 0;
	for (size_t i = 0; i < len; i++)
	{
		if (reader.atEnd())
		{
			break;
		}
		val += (static_cast<uint32>(0xFF & reader.readChar()) << (i * 8));
	}
	return val;
}

uint32 io::readBENumber(ID3_Reader& reader, size_t len)
{
	uint32 val = 0;
	
	for (ID3_Reader::size_type i = 0; i < len && !reader.atEnd(); ++i)
	{
		val *= 256; // 2^8
		val += static_cast<uint32>(0xFF & reader.readChar());
	}
	return val;
}
*/

gchar* create_centered_string(const gchar *input, guint size)
{
	gint len = strlen(input);
	gchar *result;
	
	if (len < size) {
		gint add = (size - len) / 2;
		
		result = g_strnfill(size, ' ');
		memcpy(result + add, input, len);
	}
	else if (len > size) {
		result = g_strndup(input, size);
	}
	else { result = g_strdup(input); }
	
	return result;
}

void create_std_wgt_description(GtkWidget *widget, 
	const gchar* widget_var_name, gboolean show, GtkWidget *window)
{
	gtk_widget_set_name(widget, widget_var_name);
	gtk_widget_ref(widget);
	gtk_object_set_data_full(GTK_OBJECT(window), widget_var_name, widget,
		(GtkDestroyNotify) gtk_widget_unref);
	if (show == TRUE)
		gtk_widget_show(widget);
}

const gchar* extract_basename_noext(const gchar *filename, gint *len) 
{
	const char *basename, *ext;

	g_return_val_if_fail(len != NULL, NULL);
	
	if (filename != NULL) {
		basename = g_basename(filename);
		if (basename == NULL) 
			{ basename = filename; }

		ext = strrchr(basename, '.');
		if (ext == NULL) { *len = 0; }
		else { (*len) = ext - basename; }

		return basename;
	}

	return NULL;
}
