/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "singit_debug.h"

#include "singit_karaoke_data.h"
#include "singit_tools.h"

#define left_right_line_border ((skd->ball_diameter / 2) + 1)

/*
	signal marshaller to marshal a int result and two pointer arguments
*/
typedef gint (*GtkSignal_INT__POINTER_POINTER)(GtkObject *object, gpointer arg1, gpointer arg2, gpointer user_data);

static void gtk_marshal_INT__POINTER_POINTER(GtkObject* object, GtkSignalFunc func, gpointer func_data, GtkArg* args)
{
	GtkSignal_INT__POINTER_POINTER rfunc;
	gint *return_val;
	return_val = GTK_RETLOC_INT(args[2]);
	rfunc = (GtkSignal_INT__POINTER_POINTER) func;
	*return_val = (*rfunc)(object, GTK_VALUE_POINTER(args[0]), GTK_VALUE_POINTER(args[1]), func_data);
}

/*
	signal marshaller to marshal a bool result and a pointer and two integer arguments
*/
typedef gboolean (*GtkSignal_BOOL__POINTER_INT_INT)(GtkObject *object, gpointer arg1, gint arg2, gint arg3, gpointer user_data);

void gtk_marshal_BOOL__POINTER_INT_INT(GtkObject* object, GtkSignalFunc func, gpointer func_data, GtkArg* args)
{
	GtkSignal_BOOL__POINTER_INT_INT rfunc;
	gboolean *return_val;
	return_val = GTK_RETLOC_BOOL(args[3]);
	rfunc = (GtkSignal_BOOL__POINTER_INT_INT) func;
	*return_val = (*rfunc)(object, GTK_VALUE_POINTER(args[0]), GTK_VALUE_INT(args[1]), GTK_VALUE_INT(args[2]), func_data);
}

/*
 * Uncomment to enable extended debug messages
 */
// #define SKD_DEBUG

/*
 * Referesh enums for the calc_karaoke_widget_sizes funtion
 * They are used to set the refresh state (singit_karaoke_data_refresh)
 */
enum {

	SONG_CHG = 1 << 1,		// Song has changed
	FONT_CHG = 1 << 2,		// Font has changed
	LINE_CHG = 1 << 3		// No of lines has changed
};

void calc_karaoke_widget_sizes(SingitKaraokeData *skd, gint change);
void singit_karaoke_data_update_ball(SingitKaraokeData *skd, GdkRectangle *main_area);
void singit_karaoke_data_update_progess_bar(SingitKaraokeData *skd, gchar *text, GdkRectangle *main_area);
void skd_draw_line_rects(SingitKaraokeData *skd, GdkPixmap *pixmap, GdkGC *draw_gc);

enum {
	RENDER,
	OPTIMIZE,
	EXPOSE,
	NEW_BALL,
	NEW_VISUAL,
	LAST_SIGNAL
};

static GtkObjectClass *parent_class = NULL;
static guint skd_signals[LAST_SIGNAL] = { 0 };

static void singit_karaoke_data_class_init (SingitKaraokeDataClass *klass);
static void singit_karaoke_data_init (SingitKaraokeData *skd);

static void singit_karaoke_data_destroy(GtkObject *object);

GtkType singit_karaoke_data_get_type (void)
{
	static GtkType singit_karaoke_data_type = 0;

	if (!singit_karaoke_data_type) {

		static const GtkTypeInfo singit_karaoke_data_info =
		{
			(gchar*) "SingitKaraokeData",
			sizeof (SingitKaraokeData),
			sizeof (SingitKaraokeDataClass),
			(GtkClassInitFunc) singit_karaoke_data_class_init,
			(GtkObjectInitFunc) singit_karaoke_data_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		singit_karaoke_data_type = gtk_type_unique(GTK_TYPE_OBJECT, &singit_karaoke_data_info);
	}

	return singit_karaoke_data_type;
}

static void singit_karaoke_data_class_init (SingitKaraokeDataClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class(GTK_TYPE_OBJECT);

	skd_signals[RENDER] =
		gtk_signal_new("render",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitKaraokeDataClass, render),
			gtk_marshal_INT__POINTER_POINTER,
			GTK_TYPE_INT, 2,
			GTK_TYPE_POINTER,
			GTK_TYPE_POINTER);

	skd_signals[OPTIMIZE] =
		gtk_signal_new("optimize",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitKaraokeDataClass, optimize),
			gtk_marshal_NONE__POINTER,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_POINTER);

	skd_signals[EXPOSE] =
		gtk_signal_new("expose",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitKaraokeDataClass, expose),
			gtk_marshal_BOOL__POINTER_INT_INT,
			GTK_TYPE_BOOL, 3,
			GTK_TYPE_POINTER,
			GTK_TYPE_INT,
			GTK_TYPE_INT);

	skd_signals[NEW_BALL] =
		gtk_signal_new("new_ball",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitKaraokeDataClass, new_ball),
			gtk_marshal_NONE__INT,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_INT);

	skd_signals[NEW_VISUAL] =
		gtk_signal_new("new_visual",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitKaraokeDataClass, new_visual),
			gtk_marshal_NONE__INT_INT,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_INT,
			GTK_TYPE_INT);

	klass->render = NULL;
	klass->optimize = NULL;
	klass->expose = NULL;
	klass->new_ball = NULL;
	klass->new_visual = NULL;

	gtk_object_class_add_signals (object_class, skd_signals, LAST_SIGNAL);

	object_class->destroy = singit_karaoke_data_destroy;
}

static void singit_karaoke_data_init (SingitKaraokeData *skd)
{
#ifdef SKD_DEBUG
	g_print("skd_init\n");
#endif

	skd->song = NULL;

	// Progress bar positions (pbp)
	skd->pbp_start = 0;
	skd->pbp_start_last = 0;
	skd->pbp_offset = 0;
	skd->pbp_offset_last = 0;
	skd->pbp_offset_max = 0;

	// Ball y positions
	skd->ball_y_pos = 0;
	skd->ball_y_pos_last = 0;
	skd->ball_start = 0;
	skd->ball_offset_max = 0;

	skd->lines = 3;
	skd->top_lines = 1;
	skd->current = NULL;

	skd->use_ball = TRUE;

	skd->show_empty_lines = FALSE;
	skd->maxLineWidth = FALSE;
	skd->last_time = -1;
	skd->line_seperator_high = 5;
	skd->active_line_seperator_high = skd->line_seperator_high * 2;
	skd->line_info = NULL;

	skd->font = NULL;
	skd->centerLines = FALSE;
}

GtkObject *singit_karaoke_data_new(void)
{
	GtkObject *karaoke_data_object;

#ifdef SKD_DEBUG
	g_print("skd_new\n");
#endif

	karaoke_data_object = gtk_type_new(TYPE_SINGIT_KARAOKE_DATA);

	return karaoke_data_object;
}

static void singit_karaoke_data_free_line_info(StrLineInfo *line_info)
{
#ifdef SKD_DEBUG
	g_print("skd_free_line_data\n");
#endif

	g_return_if_fail (line_info != NULL);

	if (line_info->word_pixel != NULL) { g_free(line_info->word_pixel); }
	if (line_info->space_pixel != NULL) { g_free(line_info->space_pixel); }
	if (line_info->start_pixel != NULL) { g_free(line_info->start_pixel); }
	g_free(line_info);
}

static void singit_karaoke_data_destroy (GtkObject *object)
{
	SingitKaraokeData *skd;

#ifdef SKD_DEBUG
	g_print("skd_destroy\n");
#endif

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_DATA (object));

	skd = SINGIT_KARAOKE_DATA (object);

	singit_song_detach(&skd->song);

	if (skd->line_info != NULL)
		{ singit_karaoke_data_free_line_info(skd->line_info); }

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy (object);
}

/*
 * Splits the active line into word sizes, so the ball knows where to drop
 * This is needed because of different space sizes between the words
 *
 * Expl.:
 *
 * String: I'm here and there
 * Splits: I'm| |here| |and| |there|

 * You get 7 split position
 * The ball jumps from the start to the middle of the first and second position...
 */
static void singit_karaoke_data_fill_str_line_info(SingitKaraokeData *skd, gchar *in_text)
{
	gchar **splitted_line = NULL, *text_dup;
	gint counter;
	gint textpos;
	gint word_count, space_count;

#ifdef SKD_DEBUG
	g_print("skd_fill_str_line_slinfo\n");
#endif

/*
 * Checks for changed lines and if do free or new slinfo creation is required
 */
	g_return_if_fail(skd != NULL);

	if ((in_text != NULL) && (strlen(in_text) > 0)) {
		if (skd->line_info != NULL) {
			if ((skd->line_info->text != NULL) &&
				(strcmp(in_text, skd->line_info->text) == 0)) { return; }
			singit_karaoke_data_free_line_info(skd->line_info);
		}
	}
	else {
		if (skd->line_info != NULL) {
			singit_karaoke_data_free_line_info(skd->line_info);
			skd->line_info = NULL;
		}
		return;
	}

	/* count words */
	splitted_line = g_strsplit(in_text, " ", 0);
	word_count = 0;
	while (splitted_line[word_count] != 0) { word_count++; }

	if (word_count == 0) {
		g_strfreev(splitted_line);
		if (skd->line_info != NULL) {
			singit_karaoke_data_free_line_info(skd->line_info);
			skd->line_info = NULL;
		}
		return;
	}
	space_count = word_count - 1;

/*
 * Prepare new line slinfo
 */
	skd->line_info = g_malloc(sizeof(StrLineInfo));
	skd->line_info->text = in_text;
	skd->line_info->str_len = strlen(in_text);
	gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font, in_text, &skd->line_info->str_pixel);

	if (word_count > 0) {
		skd->line_info->word_pixel = g_malloc(sizeof(gint) * (word_count+1));
		skd->line_info->word_pixel[word_count] = 0;
		skd->line_info->start_pixel = g_malloc(sizeof(gint) * (word_count+1));
		skd->line_info->start_pixel[word_count] = 0;
	}
	else {
		skd->line_info->word_pixel = NULL;
		skd->line_info->start_pixel = NULL;
	}

	if (space_count > 0) {
		skd->line_info->space_pixel = g_malloc(sizeof(gint) * (space_count+1));
		skd->line_info->space_pixel[space_count] = 0;
	}
	else { skd->line_info->space_pixel = NULL; }

	counter = space_count;

/*
 * Fill word and space pixel lengths
 */
	textpos = skd->line_info->str_len;
	text_dup = g_strdup(in_text);

	while (counter >= 0) {
		text_dup[textpos] = '\0';

		// get the length of the text if not full length
//		if (textpos != slinfo->str_len)
			gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font, text_dup,
				&skd->line_info->start_pixel[counter]);

		// make the text one word shorter
		if (textpos >= 0) {
			textpos -= (strlen(splitted_line[counter]) + 1);
		}

		// get the length of the word
		gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font,
			(gchar*)(text_dup + textpos + 1), &skd->line_info->word_pixel[counter]);

		counter--;
	}
	g_strfreev(splitted_line);

	// calculate the spaces if we have some
	if (space_count > 0) {
		counter = 0;
		if (space_count > 1) {
			while (counter < space_count) {
				skd->line_info->space_pixel[counter] = skd->line_info->start_pixel[counter+1] -
					skd->line_info->start_pixel[counter] - skd->line_info->word_pixel[counter+1];
				counter++;
			}
		}
		skd->line_info->space_pixel[counter] = skd->line_info->str_pixel -
			skd->line_info->start_pixel[counter] - skd->line_info->word_pixel[counter+1];
	}
	g_free(text_dup);

#ifdef SKD_DEBUG
	counter = 0;
	while (skd->line_info->start_pixel[counter+1] != 0) {
		g_print("| %.4i |      ", skd->line_info->start_pixel[counter]);
		counter++;
	}
	g_print("| %.4i |\n", skd->line_info->start_pixel[counter]);
	counter = 0;
	while (skd->line_info->word_pixel[counter] != 0) {
		g_print("| %.4i ", skd->line_info->word_pixel[counter]);
		if (skd->line_info->space_pixel[counter] != 0)
			{ g_print("| %.4i ", skd->line_info->space_pixel[counter]); }
		counter++;
	}
	g_print("|\n");
#endif
}

/*
 * Get the y position of the ball dependant on the word split positions
 * and the x position of the progress bar
 */
static void singit_karaoke_data_set_curr_y_pos(SingitKaraokeData *skd)
{
/*
 * This is the smallest range the ball can jump
 * It's the pixel size of the characters "IM"
 */
#define SMALLEST_JUMP "IM"

	guint x_pbp_position;
	gint cur_word_pos, new_ball_pos_index = 0, min_size, new_pos = 0;
	gint real_max_height = skd->ball_line_height - skd->ball_diameter;
	gboolean found = FALSE;
	StrLineInfo *info;

	if ((skd->line_info == NULL) || ((skd->pbp_start == 0) && (skd->pbp_offset == 0))) {
		skd->ball_y_pos = real_max_height;
		return;
	}

	x_pbp_position = skd->pbp_start + skd->pbp_offset;
	if ((x_pbp_position < skd->ball_start) ||
		(x_pbp_position > (skd->ball_start + skd->ball_offset_max)))
	{
		info = skd->line_info;
		new_pos = 0;
		while ((found != TRUE) && info->start_pixel[new_ball_pos_index]) {
			if (info->start_pixel[new_ball_pos_index] > (gint) x_pbp_position)
			{
				if (new_ball_pos_index == 0) { new_pos = 0; }
				else { new_pos = info->start_pixel[new_ball_pos_index - 1]; }
				new_ball_pos_index--;
				found = TRUE;
			}
			new_ball_pos_index++;
		}

		if ((gint) skd->pbp_start > new_pos) { skd->ball_start= skd->pbp_start; }
		else { skd->ball_start = new_pos; }

		new_pos = 0;
		gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font, 
			SMALLEST_JUMP, &min_size);
		while ((new_pos < min_size) && info->start_pixel[new_ball_pos_index]) {
			new_pos = info->start_pixel[new_ball_pos_index] - skd->ball_start;
			if (((skd->pbp_offset_max + skd->pbp_start) < (skd->ball_start + new_pos)) &&
				((skd->pbp_offset_max - skd->pbp_offset) > (guint) min_size))
			{
				new_pos = skd->pbp_offset_max + skd->pbp_start - skd->ball_start;
			}
			new_ball_pos_index++;
		}
		skd->ball_offset_max = new_pos;
	}

	cur_word_pos = x_pbp_position - skd->ball_start;
	if (skd->ball_offset_max == 0) {  skd->ball_y_pos = real_max_height; }
	else {
		new_pos = 2 * (cur_word_pos / (double) skd->ball_offset_max) * real_max_height;
		if (new_pos < real_max_height) { new_pos = real_max_height - new_pos; }
		else { new_pos = new_pos - real_max_height; }
		skd->ball_y_pos = new_pos;
	}

//	g_print("Pos Y : %i\n", skd->ball_y_pos);
#undef SMALLEST_JUMP
}

/*
 *On variable sized fonts it's really hard to get the right
 * measurement of (multiple) spaces because it's dependant
 * of the prevous and next characters
 *
 * So we have to do some tricks
 */
static gint singit_karaoke_data_get_offset_pos(SingitKaraokeData *skd, gchar *text, gint pos)
{
	gint str_len;
	gint str_pixel;
	gchar *offset_pos;
	gchar old_char;

	gint start_pixel, end_pixel, start_pos, end_pos;

	if (pos == 0) { return 0; }

	str_len = strlen(text);
	gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font, text, &str_pixel);
	if (pos == str_len) { return str_pixel; }

	// If we have some spaces at the position in the string...
	if ((text[pos-1] == ' ') || (text[pos] == ' '))
	{
		// Find the next non-space character and measure the rest
		end_pos = pos;
		while ((end_pos < str_len) && (text[end_pos] == ' ')) { end_pos++; }
		if (end_pos < str_len) {
			gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font, &text[end_pos], &end_pixel);
		}
		else { end_pixel = 0; }

		// Find the prev non-space character and measure from the start
		start_pos = pos - 1;
		while ((start_pos > 0) && (text[start_pos] == ' ')) { start_pos--; }
		if (start_pos > 0) {
			offset_pos = text + start_pos + 1;
			old_char = offset_pos[0];
			offset_pos[0] = '\0';
			gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font, text, &start_pixel);
			offset_pos[0] = old_char;
		}
		else { start_pixel = 0; }

		// Calculate the percentual position
		if ((end_pos - start_pos) != str_len) {
			str_pixel = start_pixel + (str_pixel - start_pixel - end_pixel) *
				(pos - start_pos) / (end_pos - start_pos);
		}
	}
	else {
		// Do the simple measurement
		offset_pos = text + pos;
		old_char = offset_pos[0];
		offset_pos[0] = '\0';
		gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font, text, &str_pixel);
		offset_pos[0] = old_char;
	}

	return str_pixel;
}

/*
 * Clear the progress dependant data
 */
static gchar* singit_karaoke_data_reset_progress_data
	(SingitKaraokeData *skd, SingitSong* song, GList *item)
{
	gchar *text;

	g_return_val_if_fail(skd != NULL, NULL);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), NULL);
	g_return_val_if_fail(song != NULL, NULL);

	text = (item != NULL) ? tText(song, item) : NULL;

	skd->pbp_start_last = 0;
	skd->pbp_offset_last = 0;

	skd->ball_y_pos = skd->ball_line_height - skd->ball_diameter;
	skd->ball_y_pos_last = 0;
	skd->ball_start = 0;
	skd->ball_offset_max = 0;

	singit_karaoke_data_fill_str_line_info(skd, text);
	skd->line_offset =
		singit_karaoke_data_calc_line_offset(skd, FALSE, text);
	singit_karaoke_data_set_curr_y_pos(skd);

	return text;
}

/*
 * Calculate the karaoke data dependant on the current time
 */
void singit_karaoke_data_set_time (SingitKaraokeData *skd, guint time)
{
	GList *new_item = NULL, *new_item_next;
	SingitSong *my_song = NULL;
	gchar *text = NULL, *startPos;
	double multi = 0.0;
	/* Calculate the karaoke dependant exposure area */
	GdkRectangle area;
	gboolean new_has_offset, new_next_has_offset = FALSE;
	gint expose = 0;
	gboolean redrawn = FALSE;

	/* First the obvious security checks */
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

	if (skd->font == NULL) { return; }
	
	/* If time didn't change keep old values */
	if (skd->last_time == (gint) time) { return; }
	else { skd->last_time = time; }

	my_song = singit_song_attach(skd->song);
	if (my_song) {
		area.x = 0;
		area.y = 0;
		area.width = 0;
		area.height = 0;

		new_item = my_song->active_token;
		new_item_next = inl_singit_song_get_next_token(my_song);

		/* Do we have a new next token? */
		if ((new_item != NULL) && (new_item != skd->current))
		{
			text = tText(my_song, new_item);
			new_has_offset = (tPos(new_item) > 0);

			/* Does the token next to the new token have an offset? */
			if ((new_item_next != NULL) &&
				(tLine(new_item_next) == tLine(new_item)) &&
				(tPos(new_item_next) != strlen(text)) &&
				(tPos(new_item_next) > 0))
			{
				new_next_has_offset = TRUE;
			}

			/* Do we have any offsets? */
			if (new_has_offset || new_next_has_offset) {
				// Yes
				startPos = text;

				/* Dependant on the offset of the new token get the pbp start */
				if (new_has_offset) {
					startPos += tPos(new_item);
					skd->pbp_start =
						singit_karaoke_data_get_offset_pos(skd, text, tPos(new_item));
				}
				else { skd->pbp_start = 0; }

				/* Dependant on the offset of the next new token get the pbp max offset */
				if (new_next_has_offset) {
					skd->pbp_offset_max =
						singit_karaoke_data_get_offset_pos
						(skd, startPos, tPos(new_item_next) - tPos(new_item));
				}
				else {
					gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER],
						skd->font, startPos, &skd->pbp_offset_max);
				}
			}
			else {
				/* No offsets - start at zero and end at the text length */
				skd->pbp_start = 0;
				skd->ball_y_pos = skd->ball_line_height - skd->ball_diameter;
				gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font, text, &skd->pbp_offset_max);
			}
			skd->pbp_offset = 0;

			/* Do we have to redraw the whole widget? */
			if ((skd->current == NULL) || (tLine(new_item) != tLine(skd->current)))
			{
				if ((skd->current != g_list_previous(new_item)) || ((skd->current != NULL) &&
					((skd->show_empty_lines == TRUE) ||
						(singit_song_is_empty_item(my_song, skd->current, FALSE) == FALSE))))
				{
					expose |= SKD_EXPOSE_ALL;
					area.width = skd->visual_min_width;
					area.height = skd->visual_min_height;
					singit_karaoke_data_reset_progress_data(skd, my_song, new_item);
				}
			}
			skd->current = new_item;
		}

		// Do we have a current offset
		if (new_item != NULL) {
			// Calculate the current offset
			if (new_item != my_song->last_token) {
				multi = (time - tTime(new_item)) / (tTime(new_item_next) - tTime(new_item));
				if ((multi > 1.0) || (multi < 0.0)) { skd->pbp_offset = skd->pbp_offset_max; }
				else {
					skd->pbp_offset = skd->pbp_offset_max *
						(time - tTime(new_item)) / (tTime(new_item_next) - tTime(new_item));
				}
			}
			else { skd->pbp_offset = skd->pbp_offset_max; }
		}
		else {
			if ((my_song->first_token) && (new_item != skd->current) && ((expose & SKD_EXPOSE_ALL) == 0))
			{
				expose |= SKD_EXPOSE_ALL;
				area.width = skd->visual_min_width;
				area.height = skd->visual_min_height;
				singit_karaoke_data_reset_progress_data(skd, my_song, new_item);
			}
			skd->current = new_item;
		}

		if ((skd->pbp_offset_last != skd->pbp_offset)) {
			if (skd->use_ball) {
				expose |= SKD_EXPOSE_BALL;
				singit_karaoke_data_set_curr_y_pos(skd);
				singit_karaoke_data_update_ball(skd, &area);
			}
			if (new_item) {
				expose |= SKD_EXPOSE_PROGRESS;
				singit_karaoke_data_update_progess_bar
					(skd, tText(my_song, new_item), &area);
			}
		}
		else if (skd->use_ball) {
			singit_karaoke_data_set_curr_y_pos(skd);
			if (skd->ball_y_pos_last != skd->ball_y_pos) {
				expose |= SKD_EXPOSE_BALL;
				singit_karaoke_data_update_ball(skd, &area);
			}
		}

		if ((skd->current != NULL) && ((skd->show_empty_lines == TRUE) || (skd->use_ball == TRUE)) &&
			(singit_song_is_empty_item(my_song, skd->current, FALSE) == TRUE))
		{
			expose |= SKD_EXPOSE_TIMELINE;
		}

		if ((skd->freezers == 0) && (((area.width != 0) && (area.height != 0)) || (expose != 0))) {
			gtk_signal_emit(GTK_OBJECT(skd), skd_signals[EXPOSE], &area, expose, time, &redrawn);

			/* Just store the last settings if drawing was successfull */
			if (redrawn == TRUE) {
				skd->pbp_offset_last = skd->pbp_offset;
				skd->pbp_start_last = skd->pbp_start;
				skd->ball_y_pos_last = skd->ball_y_pos;
			}
		}

		singit_song_detach(&my_song);
	}
}

/*
 * Calculate the exposure area for the ball
 */
void singit_karaoke_data_update_ball(SingitKaraokeData *skd, GdkRectangle *main_area)
{
	gint pos_x, height;
	gint xpos[2];
	GdkRectangle area;
	gint calced_offset;

	calced_offset = singit_karaoke_data_get_line_offset(skd, TRUE);
	pos_x = skd->pbp_start_last + skd->pbp_offset_last + calced_offset;
	xpos[0] = xpos[1] = pos_x;
	height = skd->top_lines_height;

	if (inl_singit_song_get_next_token(skd->song)) {
		pos_x = skd->pbp_start + skd->pbp_offset + calced_offset;
		xpos[1] = pos_x;
	}

	if (main_area && (skd->freezers == 0)) {
		area.y = height;
		area.height = skd->ball_line_height + height;
		if (xpos[0] < xpos[1]) {
			area.x = xpos[0];
			area.width = xpos[1] - xpos[0] + skd->ball_diameter;
		}
		else {
			area.x = xpos[1];
			area.width = xpos[0] - xpos[1] + skd->ball_diameter;
		}
		area.x += calced_offset;
		gdk_rectangle_union(main_area, &area, main_area);
	}
}

/*
 * Calculate the exposure area of the progress bar
 */
void singit_karaoke_data_update_progess_bar(SingitKaraokeData *skd, gchar *text, GdkRectangle *main_area)
{
	GdkRectangle area;
	gint height;
	gint ranges[4];

	if (strlen(text) < 1) { return; }

	area.height = skd->text_line_height;

	height = skd->top_lines_height;
	if (skd->use_ball) { height += (skd->ball_line_height + skd->line_seperator_high); }

	ranges[0] = skd->pbp_start_last + skd->pbp_offset_last;
	ranges[1] = skd->pbp_start + skd->pbp_offset;
	ranges[2] = MIN(ranges[0], ranges[1]); // x
	ranges[3] = MAX(ranges[0], ranges[1]) - ranges[2]; // width

	if (main_area && (skd->freezers == 0)) {
		area.x = ranges[2] + singit_karaoke_data_get_line_offset(skd, FALSE);
		area.y = height;
		area.width = ranges[3];
		gdk_rectangle_union(main_area, &area, main_area);
	}
}

static void singit_karaoke_data_refresh (SingitKaraokeData *skd, gint add_changer)
{
	static gint changer = 0;
	GdkRectangle area;
	SingitSong *my_song;
	gboolean result;
	gchar *text;

	changer |= add_changer;

	if (skd->freezers != 0) { return; }
	if (!skd->song) { return; }

#ifdef SKD_DEBUG
	g_print("skd_refresh\n");
#endif

	calc_karaoke_widget_sizes(skd, changer);
	skd->current = NULL;
	area.x = 0;
	area.y = 0;
	area.width = skd->visual_min_width;
	area.height = skd->visual_min_height;

	my_song = singit_song_attach(skd->song);
	if (my_song != NULL) {
		text = singit_karaoke_data_reset_progress_data(skd, my_song, my_song->active_token);
		if (my_song->active_token)
			singit_karaoke_data_update_progess_bar(skd, text, FALSE);
		gtk_signal_emit(GTK_OBJECT(skd), skd_signals[EXPOSE], &area, SKD_EXPOSE_ALL, skd->last_time, &result);
		singit_song_detach(&my_song);
	}

	changer = 0;
}


/*****************************
 *
 * API - Setters and Getters
 *
 *****************************/

void singit_karaoke_data_set_song (SingitKaraokeData *skd, SingitSong *new_song)
{
	SingitSong *attached_song;

	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_song\n");
#endif

	attached_song = singit_song_attach(new_song);

	if (attached_song != skd->song) {
		singit_song_detach(&skd->song);
		skd->song = attached_song;

		skd->current = NULL;
		skd->pbp_start = skd->pbp_offset = 0;
		skd->pbp_start_last = skd->pbp_offset_last = 0;
		skd->last_time = -1;
		skd->ball_y_pos_last = skd->ball_y_pos = skd->ball_line_height - skd->ball_diameter;
		skd->ball_start = skd->ball_offset_max = 0;

		singit_karaoke_data_refresh(skd, SONG_CHG);
	}
	else { singit_song_detach(&attached_song); }
}

void singit_karaoke_data_set_lines(SingitKaraokeData *skd, guint value)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_lines\n");
#endif

	if ((value != skd->lines) && (value > 0)) {
		skd->lines = value;
		if (!(skd->top_lines < value))
			{ skd->top_lines = value - 1; }
		singit_karaoke_data_refresh(skd, LINE_CHG);
	}
}

guint singit_karaoke_data_get_lines(SingitKaraokeData *skd)
{
#ifdef SKD_DEBUG
	g_print("skd_get_lines\n");
#endif

	g_return_val_if_fail(skd != NULL, 0);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), 0);

	return skd->lines;
}

void singit_karaoke_data_set_toplines(SingitKaraokeData *skd, guint value)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_toplines\n");
#endif

	if ((skd->top_lines != value) && (skd->top_lines < (skd->lines-1))) {
		skd->top_lines = value;
		singit_karaoke_data_refresh(skd, 0);
	}
}

guint singit_karaoke_data_get_toplines(SingitKaraokeData *skd)
{
#ifdef SKD_DEBUG
	g_print("skd_get_toplines\n");
#endif

	g_return_val_if_fail(skd != NULL, 0);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), 0);

	return skd->top_lines;
}

void singit_karaoke_data_set_font(SingitKaraokeData *skd, const gpointer font, gint font_heigth)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));
	g_return_if_fail(font != NULL);
	g_return_if_fail(font_heigth > 0);

	skd->font = font;
	skd->font_heigth = font_heigth;

	singit_karaoke_data_refresh(skd, FONT_CHG);
}

void singit_karaoke_data_set_show_empty_lines (SingitKaraokeData *skd, const gboolean value)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_show_empty_lines\n");
#endif

	if (value != skd->show_empty_lines) {

		skd->show_empty_lines = value;
	}
}

void singit_karaoke_data_set_jumping_ball (SingitKaraokeData *skd, const gboolean value)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_jumping_ball\n");
#endif

	if (value != skd->use_ball) {

		skd->use_ball = value;
		singit_karaoke_data_refresh(skd, 0);
	}
}

void singit_karaoke_data_freeze(SingitKaraokeData *skd)
{ skd->freezers++; }

void singit_karaoke_data_thaw(SingitKaraokeData *skd)
{
	if (skd->freezers > 0) { skd->freezers--; }
	if (skd->freezers == 0) { singit_karaoke_data_refresh(skd, 0); }
}

void singit_karaoke_data_set_optimal_font (SingitKaraokeData *skd)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_optimal_font\n");
#endif

	if ((skd->song == NULL) || (singit_song_lyrics_found(skd->song) == FALSE))
		{ return; }

	singit_karaoke_data_freeze(skd);
//	set_optimal_font(skd);
	singit_karaoke_data_thaw(skd);
}

void singit_karaoke_data_set_centered_lines (SingitKaraokeData *skd, gboolean value)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_centered_lines\n");
#endif

	if (value != skd->centerLines) {
		skd->centerLines = value;
		singit_karaoke_data_refresh(skd, 0);
	}
}

gboolean singit_karaoke_data_get_centered_lines (SingitKaraokeData *skd)
{
	g_return_val_if_fail((skd != NULL), FALSE);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), FALSE);

#ifdef SKD_DEBUG
	g_print("skd_get_centered_lines\n");
#endif

	return skd->centerLines;
}



/*****************************
 *
 * Helper funtions
 *
 *****************************/

static guint get_max_line_width (SingitKaraokeData *skd)
{
	gint i = 0;
	guint strlength;
	guint line_width = 0;
	SingitSong *my_song = singit_song_attach(skd->song);

	if (my_song) {
		if (my_song->lyrics) {
			while (my_song->lyrics[i]) {
				strlength = strlen(my_song->lyrics[i]);
				if (strlength > 0) {
					gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font, my_song->lyrics[i], &strlength);
		  			if (strlength > line_width) { line_width = strlength; }
				}
				i++;
			}
		}
		singit_song_detach(&my_song);
	}
	return line_width;
}

void calc_karaoke_widget_sizes(SingitKaraokeData *skd, gint change)
{
	if (change & FONT_CHG) {

		skd->ball_line_height = skd->font_heigth;

		skd->text_line_height = skd->ball_line_height + 2;

		skd->ball_diameter = (skd->font_heigth / 2.5);
		if ((skd->ball_diameter % 2) == 0) { skd->ball_diameter++; }

		skd->text_left_offset = ((skd->ball_diameter / 2) + 1);

		skd->line_seperator_high = skd->font_heigth / 6 + 1;
		skd->active_line_seperator_high = skd->font_heigth / 3 + 1;

		gtk_signal_emit(GTK_OBJECT(skd), skd_signals[NEW_BALL], skd->ball_diameter);
	}

	skd->visual_min_height =
		skd->lines * (skd->text_line_height + skd->line_seperator_high) +
		2 * skd->active_line_seperator_high;
	if (skd->use_ball)
		skd->visual_min_height += skd->ball_line_height;
	else
		skd->visual_min_height -= skd->line_seperator_high;

	if (change & (FONT_CHG | SONG_CHG))
		{ skd->max_line_width = get_max_line_width(skd); }

	skd->visual_min_width = skd->max_line_width + 2 * skd->text_left_offset;

	skd->top_lines_height = skd->top_lines * (skd->text_line_height + skd->line_seperator_high)
		+ skd->active_line_seperator_high;

	gtk_signal_emit(GTK_OBJECT(skd), skd_signals[NEW_VISUAL], skd->visual_min_width, skd->visual_min_height);
}

gint singit_karaoke_data_get_line_offset(SingitKaraokeData *skd, gboolean ball)
{
	g_return_val_if_fail((skd != NULL), FALSE);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), FALSE);

	return (ball == FALSE) ? skd->line_offset : skd->line_offset - left_right_line_border;
}

gint singit_karaoke_data_calc_line_offset(SingitKaraokeData *skd, gboolean ball, gchar *text)
{
	gint size = 0, result;

	g_return_val_if_fail(skd != NULL, size);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), size);

	if (text == NULL) { return size; }

	if (skd->centerLines == FALSE) {
		result = (ball == FALSE) ? left_right_line_border : 0;
	}
	else {
		if (strlen(text) > 0)
			{ gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER], skd->font, text, &size); }

		result = (skd->max_line_width - size) / 2;
		if (result < left_right_line_border)
			{ result = left_right_line_border; }
		if (ball != FALSE)
			{ result -= left_right_line_border; }
	}
	return result;
}
/*
gboolean singit_karaoke_data_get_timeline_times(SingitKaraokeData *skd, gint time, gint *start, gint* max)
{
	SingitSong *my_song;
	GList * new_item, new_item_next;

	g_return_val_if_fail(skd != NULL);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd));
	g_return_val_if_fail(time >= 0);
	g_return_val_if_fail(max != NULL);
	g_return_val_if_fail(offset != NULL);

	my_song = singit_song_attach(skd->song);
	if (my_song) {
		new_item = my_song->active_token;
		new_item_next = inl_singit_song_get_next_token(my_song);

		singit_song_detach(my_song);
	}
}
*/
