/* SingIt Lyrics Displayer
 * Copyright (C) 2000, 2001 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 * colorbutton.c
 * original from LIBGIMP - Copyright (C) 1999 Sven Neumann
 * modified for SingIt - Copyright (C) 2000 Jan-Marek Glogowski
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/* TODO:
 *
 * handle bytes != 3|4 -- would we have to provide a special color
 *                        select dialog for that case? Another possibility
 *                        could be to hide or destroy all color-related
 *                        widgets in the gtk colorselector.
 */

#include <gtk/gtk.h>

#include "singit_macros.h"

#include "singit_wgt_colorbutton.h"

#define GIMP_CHECK_SIZE    8
#define GIMP_CHECK_SIZE_SM 4
#define GIMP_CHECK_DARK   0.4          /*  corresponds to GRAY_CHECKS as  */
#define GIMP_CHECK_LIGHT  0.6          /*  defined in app/image_render.c  */

static void  color_button_destroy         (GtkObject *object);
static void  color_button_clicked         (GtkButton *button);
static void  color_button_paint           (ColorButton *gcb);
static void  color_button_state_changed   (GtkWidget *widget,
						GtkStateType previous_state);

static void  color_button_dialog_ok       (GtkWidget *widget,
						gpointer data);
static void  color_button_dialog_cancel   (GtkWidget *widget,
						gpointer data);

static gint   color_button_menu_popup     (GtkWidget *widget,
						GdkEvent *event,
						gpointer data);
static gchar* color_button_menu_translate (const gchar *path,
						gpointer func_data);

static void  color_button_use_color       (gpointer callback_data,
						guint callback_action,
						GtkWidget *widget)
					{
					}

/* MenuItemText, NULL, Function, Nbr, NULL */
static GtkItemFactoryEntry menu_items[] = {
  { (gchar*) N_("/Use Color"), NULL, color_button_use_color, 2, NULL }
};
static gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);


enum {
  COLOR_CHANGED,
  LAST_SIGNAL
};

static guint color_button_signals[LAST_SIGNAL] = { 0 };


static GtkWidgetClass *parent_class = NULL;

static void
color_button_destroy (GtkObject *object)
{
  ColorButton *gcb;

  g_return_if_fail (gcb = COLOR_BUTTON (object));

  g_free (gcb->title);

  if (gcb->dialog)
    gtk_widget_destroy (gcb->dialog);

  g_free (gcb->dcolor);
  g_free (gcb->even);
  g_free (gcb->odd);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
color_button_class_init (ColorButtonClass *class)
{
  GtkObjectClass *object_class;
  GtkButtonClass *button_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass*) class;
  button_class = (GtkButtonClass*) class;
  widget_class = (GtkWidgetClass*) class;

  parent_class = gtk_type_class (gtk_widget_get_type ());

  color_button_signals[COLOR_CHANGED] =
    gtk_signal_new ("color_changed",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (ColorButtonClass,
				       color_changed),
		    gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);
  gtk_object_class_add_signals (object_class, color_button_signals,
				LAST_SIGNAL);
  class->color_changed = NULL;

  object_class->destroy       = color_button_destroy;
  button_class->clicked       = color_button_clicked;
  widget_class->state_changed = color_button_state_changed;
}


static void
color_button_init (ColorButton *gcb)
{
  gcb->double_color = FALSE;

  gcb->title        = NULL;
  gcb->bpp          = 0;
  gcb->color        = NULL;

  gcb->dcolor       = NULL;
  gcb->preview      = NULL;
  gcb->dialog       = NULL;

  gcb->even         = NULL;
  gcb->odd          = NULL;
}

GtkType
color_button_get_type (void)
{
  static guint gcb_type = 0;

  if (!gcb_type)
    {
      GtkTypeInfo gcb_info =
      {
	(gchar*) "ColorButton",
	sizeof (ColorButton),
	sizeof (ColorButtonClass),
	(GtkClassInitFunc) color_button_class_init,
	(GtkObjectInitFunc) color_button_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL
      };

      gcb_type = gtk_type_unique (gtk_button_get_type (), &gcb_info);
    }

  return gcb_type;
}

static GtkWidget *
_color_button_new (gboolean  double_color,
			gchar    *title,
			gint      width,
			gint      height,
			gpointer  color,
			gint      bpp)
{
  ColorButton *gcb;
  gint i;

  g_return_val_if_fail (width > 0 && height > 0, NULL);
  g_return_val_if_fail (bpp == 3 || bpp == 4, NULL);

  gcb = gtk_type_new (color_button_get_type ());

  gcb->double_color = double_color;

  gcb->title  = g_strdup (title);
  gcb->width  = width;
  gcb->height = height;
  gcb->color  = color;
  gcb->bpp    = bpp;

  gcb->dcolor = g_new (gdouble, 4);
  gcb->even   = g_new (guchar, 3 * width);
  gcb->odd    = g_new (guchar, 3 * width);

  if (double_color)
    {
      for (i = 0; i < bpp; i++)
	gcb->dcolor[i] = ((gdouble *) color)[i];
    }
  else
    {
      for (i = 0; i < bpp; i++)
	gcb->dcolor[i] = (gdouble) ((guchar *) color)[i] / 255.0;
    }

  if (bpp == 3)
    gcb->dcolor[3] = 1.0;

  gcb->preview = gtk_preview_new (GTK_PREVIEW_COLOR);
  gtk_signal_connect (GTK_OBJECT (gcb->preview), "destroy",
		      gtk_widget_destroyed, &gcb->preview);

  gtk_preview_size (GTK_PREVIEW (gcb->preview), width, height);
  gtk_container_add (GTK_CONTAINER (gcb), gcb->preview);
  gtk_widget_show (gcb->preview);
  color_button_paint (gcb);

  /* right-click opens a popup */
  gcb->item_factory = gtk_item_factory_new (GTK_TYPE_MENU, "<popup>", NULL);
  gtk_item_factory_set_translate_func (gcb->item_factory,
				       color_button_menu_translate,
	  			       NULL, NULL);
  gtk_item_factory_create_items (gcb->item_factory, nmenu_items, menu_items, gcb);
  gtk_signal_connect (GTK_OBJECT (gcb), "button_press_event",
		      GTK_SIGNAL_FUNC (color_button_menu_popup),
		      gcb);

  return (GTK_WIDGET (gcb));
}

/**
 * color_button_new:
 * @title: String that will be used as title for the color_selector.
 * @width: Width of the colorpreview in pixels.
 * @height: Height of the colorpreview in pixels.
 * @color: An array of guchar holding the color (RGB or RGBA)
 * @bpp: May be 3 for RGB or 4 for RGBA.
 *
 * Creates a new #ColorButton widget.
 *
 * This returns a button with a preview showing the color.
 * When the button is clicked a GtkColorSelectionDialog is opened.
 * If the user changes the color the new color is written into the
 * array that was used to pass the initial color and the "color_changed"
 * signal is emitted.
 *
 * Returns: Pointer to the new #ColorButton widget.
 */
GtkWidget *
color_button_new (gchar   *title,
		       gint     width,
		       gint     height,
		       guchar  *color,
		       gint     bpp)
{
  return _color_button_new (FALSE, title, width, height,
				 (gpointer) color, bpp);
}

/**
 * color_button_double_new:
 * @title: String that will be used as title for the color_selector.
 * @width: Width of the colorpreview in pixels.
 * @height: Height of the colorpreview in pixels.
 * @color: An array of gdouble holding the color (RGB or RGBA)
 * @bpp: May be 3 for RGB or 4 for RGBA.
 *
 * Creates a new #ColorButton widget.
 *
 * This returns a button with a preview showing the color.
 * When the button is clicked a GtkColorSelectionDialog is opened.
 * If the user changes the color the new color is written into the
 * array that was used to pass the initial color and the "color_changed"
 * signal is emitted.
 *
 * Returns: Pointer to the new ColorButton widget.
 */
GtkWidget *
color_button_double_new (gchar   *title,
			      gint     width,
			      gint     height,
			      gdouble *color,
			      gint     bpp)
{
  return _color_button_new (TRUE, title, width, height,
				 (gpointer) color, bpp);
}

/**
 * color_button_update:
 * @gcb: Pointer to a #ColorButton.
 *
 * Should be used after the color controlled by a #ColorButton
 * was changed. The color is then reread and the change is propagated
 * to the preview and the GtkColorSelectionDialog if one is open.
 */
void
color_button_update (ColorButton *gcb)
{
  gint i;

  g_return_if_fail (IS_COLOR_BUTTON (gcb));

  if (gcb->double_color)
    {
      for (i = 0; i < gcb->bpp; i++)
	gcb->dcolor[i] = ((gdouble *) gcb->color)[i];
    }
  else
    {
      for (i = 0; i < gcb->bpp; i++)
	gcb->dcolor[i] = (gdouble) ((guchar *) gcb->color)[i] / 255.0;
    }

  color_button_paint (gcb);

  if (gcb->dialog)
    gtk_color_selection_set_color (GTK_COLOR_SELECTION (GTK_COLOR_SELECTION_DIALOG (gcb->dialog)->colorsel),
				   gcb->dcolor);
}

static void
color_button_state_changed (GtkWidget    *widget,
				 GtkStateType  previous_state)
{
  g_return_if_fail (IS_COLOR_BUTTON (widget));

  if (!GTK_WIDGET_IS_SENSITIVE (widget) && COLOR_BUTTON (widget)->dialog)
    gtk_widget_hide (COLOR_BUTTON (widget)->dialog);

  if (GTK_WIDGET_CLASS (parent_class)->state_changed)
    (* GTK_WIDGET_CLASS (parent_class)->state_changed) (widget, previous_state);
}

static gint
color_button_menu_popup (GtkWidget *widget,
			      GdkEvent  *event,
			      gpointer   data)
{
  ColorButton *gcb;
  GdkEventButton *bevent;
  gint x;
  gint y;

  g_return_val_if_fail (IS_COLOR_BUTTON (data), FALSE);
  gcb = COLOR_BUTTON (data);

  if (event->type != GDK_BUTTON_PRESS)
    return FALSE;

  bevent = (GdkEventButton *) event;

  if (bevent->button != 3)
    return FALSE;

  gdk_window_get_origin (GTK_WIDGET (widget)->window, &x, &y);
  gtk_item_factory_popup (gcb->item_factory,
			  x + bevent->x, y + bevent->y,
			  bevent->button, bevent->time);

  return (TRUE);
}

static void
color_button_clicked (GtkButton *button)
{
  ColorButton *gcb;
  GtkColorSelection *colorsel;

  g_return_if_fail (IS_COLOR_BUTTON (button));
  gcb = COLOR_BUTTON (button);

  if (!gcb->dialog)
    {
      gcb->dialog = gtk_color_selection_dialog_new (gcb->title);
      colorsel = GTK_COLOR_SELECTION (GTK_COLOR_SELECTION_DIALOG (gcb->dialog)->colorsel);
      gtk_color_selection_set_opacity (colorsel, (gcb->bpp == 4));
      gtk_color_selection_set_color (colorsel, gcb->dcolor);
      gtk_widget_destroy (GTK_COLOR_SELECTION_DIALOG (gcb->dialog)->help_button);
      gtk_container_set_border_width (GTK_CONTAINER (gcb->dialog), 2);

      gtk_signal_connect (GTK_OBJECT (gcb->dialog), "destroy",
			  (GtkSignalFunc) gtk_widget_destroyed, &gcb->dialog);
      gtk_signal_connect (GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG (gcb->dialog)->ok_button),
			  "clicked",
			  (GtkSignalFunc) color_button_dialog_ok, gcb);
      gtk_signal_connect (GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG (gcb->dialog)->cancel_button),
			  "clicked",
			  (GtkSignalFunc) color_button_dialog_cancel, gcb);
      gtk_window_position (GTK_WINDOW (gcb->dialog), GTK_WIN_POS_MOUSE);
    }
  gtk_color_selection_set_color (GTK_COLOR_SELECTION (GTK_COLOR_SELECTION_DIALOG (gcb->dialog)->colorsel),
				 gcb->dcolor);
  gtk_widget_show (gcb->dialog);
}

static void
color_button_paint (ColorButton *gcb)
{
  gint x, y, i;
  gdouble c0, c1;
  guchar *p0, *p1;

  g_return_if_fail (IS_COLOR_BUTTON (gcb));

  p0 = gcb->even;
  p1 = gcb->odd;

  if (gcb->bpp == 3)
    {
      for (x = 0; x < gcb->width; x++)
	{
	  for (i = 0; i < 3; i++)
	    *p0++ = gcb->dcolor[i] * 255.999;
	}
      for (y = 0; y < gcb->height; y++)
	gtk_preview_draw_row (GTK_PREVIEW (gcb->preview), gcb->even,
			      0, y, gcb->width);
    }
  else  /* gcb->bpp == 4 */
    {
      for (x = 0; x < gcb->width; x++)
	{
	  if ((x / GIMP_CHECK_SIZE_SM) & 1)
	    {
	      c0 = GIMP_CHECK_LIGHT;
	      c1 = GIMP_CHECK_DARK;
	    }
	  else
	    {
	      c0 = GIMP_CHECK_DARK;
	      c1 = GIMP_CHECK_LIGHT;
	    }
	  for (i = 0; i < 3; i++)
	    {
	      *p0++ = (c0 + (gcb->dcolor[i] - c0) * gcb->dcolor[3]) * 255.999;
	      *p1++ = (c1 + (gcb->dcolor[i] - c1) * gcb->dcolor[3]) * 255.999;
	    }
	}
      for (y = 0; y < gcb->height; y++)
	{
	  if ((y / GIMP_CHECK_SIZE_SM) & 1)
	    gtk_preview_draw_row (GTK_PREVIEW (gcb->preview), gcb->odd,
				  0, y, gcb->width);
	  else
	    gtk_preview_draw_row (GTK_PREVIEW (gcb->preview), gcb->even,
				  0, y, gcb->width);
	}
    }

  gtk_widget_queue_draw (gcb->preview);
}

static void
color_button_dialog_ok (GtkWidget *widget,
			     gpointer   data)
{
  ColorButton *gcb;
  gboolean color_changed = FALSE;
  gint i;

  g_return_if_fail (IS_COLOR_BUTTON (data));
  gcb = COLOR_BUTTON (data);

  gtk_color_selection_get_color (GTK_COLOR_SELECTION (GTK_COLOR_SELECTION_DIALOG (gcb->dialog)->colorsel), gcb->dcolor);

  if (gcb->double_color)
    {
      for (i = 0; i < gcb->bpp; i++)
	{
	  if (gcb->dcolor[i] != ((gdouble *) gcb->color)[i])
	    color_changed = TRUE;
	  ((gdouble *) gcb->color)[i] = gcb->dcolor[i];
	}
    }
  else
    {
      guchar new_color[4];

      for (i = 0; i < gcb->bpp; i++)
	{
	  new_color[i] = gcb->dcolor[i] * 255.999;
	  if (new_color[i] != ((guchar *) gcb->color)[i])
	    color_changed = TRUE;
	  ((guchar *) gcb->color)[i] = new_color[i];
	}
    }

  gtk_widget_hide (gcb->dialog);

  if (color_changed)
    {
      color_button_paint (gcb);
      gtk_signal_emit (GTK_OBJECT (gcb),
		       color_button_signals[COLOR_CHANGED]);
    }
}

static void
color_button_dialog_cancel (GtkWidget *widget,
				 gpointer   data)
{
  ColorButton *gcb;

  g_return_if_fail (IS_COLOR_BUTTON (data));
  gcb = COLOR_BUTTON (data);

  gtk_widget_hide (gcb->dialog);
}

static gchar *
color_button_menu_translate (const gchar *path,
				  gpointer     func_data)
{
  return (gettext (path));
}
