/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <X11/Xlib.h>
#include <gtk/gtk.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>

#include "singit/macros.h"
#include "singit/control.h"
#include "singit/displayer_plugin.h"
#include "singit/wgt_karaoke.h"

#include "dkwn_mouse_handling.h"
#include "displayer_karaoke_window_status.h"

static void displayer_karaoke_window_init(void);
static void displayer_karaoke_window_finish(void);

static void displayer_karaoke_window_set_song(SingitSong *cur_song);
static void displayer_karaoke_window_set_time(guint time, GList *real_next);
static void displayer_karaoke_window_config_update(void);

DisplayerPlugin karaoke_window_dp =
{
	NULL,
	NULL,
	0,
	0,
	NULL,

	"Karaoke window displayer",

	displayer_karaoke_window_init,
	displayer_karaoke_window_finish,
	displayer_karaoke_window_config_update,

	displayer_karaoke_window_set_song,
	displayer_karaoke_window_set_time,

	NULL,
	NULL,
	NULL,
	NULL,

	NULL,
	NULL
};

DIPSPLAYER_SYMBOL(libdisplayer_karaoke_window, karaoke_window_dp)

static GtkWidget *displayer_karaoke_window_win = NULL;
static GtkWidget
		*singit_vbox,
			*singit_hbox,
			*singit_timestamp_status_bar,
			*singit_song_gtk_event_box, *singit_song_status_bar;

static GtkTooltips
		*singit_tooltips;

static gboolean displayer_karaoke_window_win_keypress(GtkWidget * w, GdkEventKey * event, gpointer data);

static void set_jumping_ball(void)
{
	if (STATUS->song &&
		STATUS->guess_synced_lyrics)
	{
		singit_karaoke_widget_set_jumping_ball(
			SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
			(singit_song_guess_sync_lyrics(STATUS->song) &&
			STATUS->show_jumping_ball));
	}
	else {
		singit_karaoke_widget_set_jumping_ball(
			SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
			STATUS->show_jumping_ball);
	}
}

static void set_statusbar_messages(gint time)
{
	gint new_message_count;
	gchar timeStamp[10];

	if (!displayer_karaoke_window_win) { return; }

	timeStamp[9] = '\0';
	new_message_count = (time / 1000);
	if (new_message_count < 0) { new_message_count = 0; }
	else
		if (!(time % 1000 == 0)) { new_message_count++; }
	if (STATUS->messageCounter == new_message_count) { return; }

	if (STATUS->timestamp_context_id != 0) {
		while (STATUS->messageCounter > new_message_count) {
			gtk_statusbar_pop(GTK_STATUSBAR(singit_timestamp_status_bar), STATUS->timestamp_context_id);
			STATUS->messageCounter--;
		}
	}
	else {
		STATUS->timestamp_context_id =
			gtk_statusbar_get_context_id(GTK_STATUSBAR(singit_timestamp_status_bar), "timestamps");
		STATUS->messageCounter = 0;
	}
	while (STATUS->messageCounter < new_message_count) {
		STATUS->messageCounter++;
		sprintf(timeStamp, " - %.2i:%.2i ", STATUS->messageCounter / 60, STATUS->messageCounter % 60);
		gtk_statusbar_push
			(GTK_STATUSBAR(singit_timestamp_status_bar), STATUS->timestamp_context_id, timeStamp);
	}
}

static void set_status_filename(gchar *fileName)
{
	gchar *tmp;

	g_return_if_fail(displayer_karaoke_window_win != NULL);

	if (STATUS->song_context_id) {
		gtk_statusbar_pop(GTK_STATUSBAR(singit_song_status_bar),
			STATUS->song_context_id);
	}
	else {
		STATUS->song_context_id =
			gtk_statusbar_get_context_id
				(GTK_STATUSBAR(singit_song_status_bar), "filename");
	}

	tmp = g_strconcat(" ", fileName, NULL);
	gtk_statusbar_push(GTK_STATUSBAR(singit_song_status_bar),
		STATUS->song_context_id, tmp);
	gtk_tooltips_set_tip(singit_tooltips, singit_song_gtk_event_box, fileName, NULL);
	g_free(tmp);
}

#define CTRL_SESSION	karaoke_window_dp.singit_session
#define CTRL_CFG_DATA	status

static void displayer_karaoke_window_config_update(void)
{
	gchar *temp_str;
	KaraokeWindowNewStatus *status = STATUS;

#define wgt_set_color(color) \
	temp_str = singit_remote_cfg_color \
		(karaoke_window_dp.singit_session, color); \
	singit_karaoke_widget_set_color \
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), color, temp_str); \
	g_free(temp_str);

	REMOTE_CFG_V(debug_enabled, print_debug);
	REMOTE_CFG(toggle_yz);
	REMOTE_CFG(guess_synced_lyrics);
	REMOTE_CFG(show_jumping_ball);
	REMOTE_CFG(hide_if_not_found);
	REMOTE_CFG(auto_resize);

	g_return_if_fail(GTK_WIDGET_REALIZED(displayer_karaoke_window_win));

	singit_karaoke_widget_freeze
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));

	// Set background color in fullscreen mode and store it
	if (status->fullscreen_background_color)
		g_free(STATUS->fullscreen_background_color);
	status->fullscreen_background_color = singit_remote_cfg_color
		(karaoke_window_dp.singit_session, skwc_background);
	if (status->is_fullscreen == TRUE) {
		singit_karaoke_widget_set_color(
			SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
			skwc_background,
			status->fullscreen_background_color);
	}

	wgt_set_color(skwc_normal_text);
	wgt_set_color(skwc_normal_bar);
	wgt_set_color(skwc_progress_text);
	wgt_set_color(skwc_progress_bar);

	temp_str = singit_remote_cfg_lyric_font_name
		(karaoke_window_dp.singit_session);
	singit_karaoke_widget_set_font
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), temp_str);
	g_free(temp_str);

	singit_karaoke_widget_set_time_indicators
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
		singit_remote_cfg_show_time_indicator
		(karaoke_window_dp.singit_session));

	set_jumping_ball();
	singit_karaoke_widget_thaw
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));
}

#undef CTRL_CFG_DATA
#undef CTRL_SESSION


void displayer_karaoke_window_set_time(guint time, GList *real_next)
{
	SingitSong *my_song = NULL;
	GList *item;

#ifdef CODECODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_set_time]\n"));
#endif

	if (displayer_karaoke_window_win && GTK_WIDGET_VISIBLE(displayer_karaoke_window_win) &&
		(gdk_window_is_visible(displayer_karaoke_window_win->window)) &&
		(gdk_window_is_viewable(displayer_karaoke_window_win->window)))
	{
		my_song = singit_song_attach(STATUS->song);
		if (!my_song)
			{ return; }

		GDK_THREADS_ENTER();
		if (STATUS->update_song)
		{
			if (singit_song_lyrics_found(my_song)) {
				set_status_filename(my_song->lyric_filename);
			}
			singit_karaoke_widget_freeze
				(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));
			singit_karaoke_widget_set_song
				(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), my_song);
			singit_karaoke_widget_set_time
				(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), time);

			singit_song_detach(&(STATUS->song));
			STATUS->song = my_song;
			set_jumping_ball();

			singit_karaoke_widget_thaw
				(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));
			STATUS->update_song = FALSE;
		}
		else {
			singit_karaoke_widget_set_time
				(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), time);
			if (singit_song_lyrics_found(my_song)) {
				item = (real_next) ? real_next : my_song->last_token;
				if (item) { set_statusbar_messages(tTime(item) - time); }
			}
			singit_song_detach(&my_song);
		}
		GDK_THREADS_LEAVE();
	}
}

static void dkwn_change_screen_mode(gboolean fullscreen)
{
	static gint x, y, w = -1, h;

#ifdef CODECODEDEBUG
	DEBUG(("displayer_karaoke_window.c [dkwn_change_screen_mode]\n"));
#endif

	if (fullscreen == STATUS->is_fullscreen)
		{ return; }

	if (fullscreen == TRUE) {
		gdk_window_set_decorations(displayer_karaoke_window_win->window, 0);

		gdk_window_get_root_origin
			(displayer_karaoke_window_win->window, &x, &y);
		gdk_window_get_size
			(displayer_karaoke_window_win->window, &w, &h);
//		g_print("Save: %i %i %i %i\n", x, y, w, h);

		gtk_widget_hide(displayer_karaoke_window_win);

//		gtk_container_border_width(GTK_CONTAINER(displayer_karaoke_window_win), 0);
		gtk_window_set_policy(GTK_WINDOW(displayer_karaoke_window_win),
			FALSE, FALSE, TRUE);
		gtk_window_set_title(GTK_WINDOW(displayer_karaoke_window_win),
			_("SingIt full screen mode"));

		fullscreen_hide_mouse_enable();
		gtk_window_set_modal(GTK_WINDOW(displayer_karaoke_window_win), TRUE);
		gdk_keyboard_grab(displayer_karaoke_window_win->window,
			TRUE, GDK_CURRENT_TIME);
		gtk_grab_add(displayer_karaoke_window_win);
		gtk_widget_grab_focus(displayer_karaoke_window_win);

		if (STATUS->fullscreen_background_color != NULL) {
			singit_karaoke_widget_set_color(
				SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
				skwc_background,
				STATUS->fullscreen_background_color);
		}

		gtk_widget_show(displayer_karaoke_window_win);

		gtk_widget_set_usize(displayer_karaoke_window_win,
			gdk_screen_width(), gdk_screen_height());
		gtk_widget_set_uposition(displayer_karaoke_window_win, 0, 0);
	}
	else {
		gtk_widget_hide(displayer_karaoke_window_win);

		gdk_window_set_decorations(displayer_karaoke_window_win->window,
			GDK_DECOR_ALL);

		gtk_window_set_title(GTK_WINDOW(displayer_karaoke_window_win),
			SINGIT_VERSION_STRING);

		gtk_window_set_modal(GTK_WINDOW(displayer_karaoke_window_win), FALSE);
		gdk_keyboard_ungrab(GDK_CURRENT_TIME);
		gtk_grab_remove(displayer_karaoke_window_win);
		fullscreen_hide_mouse_disable();

		gtk_window_set_policy(GTK_WINDOW(displayer_karaoke_window_win),
			FALSE, FALSE, TRUE);

		singit_karaoke_widget_set_color_gdk(
			SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
			skwc_background,
			&(gtk_widget_get_style(STATUS->karaoke_widget)
				->bg[GTK_STATE_NORMAL]));

		gtk_widget_show(displayer_karaoke_window_win);

		gtk_window_set_policy(GTK_WINDOW(displayer_karaoke_window_win),
			TRUE, FALSE, TRUE);

		if (STATUS->auto_resize)
			gtk_widget_set_usize
				(displayer_karaoke_window_win, -1, -1);
		else
			gtk_widget_set_usize
				(displayer_karaoke_window_win, w, h);
		gtk_widget_set_uposition(displayer_karaoke_window_win, x, y);
	}

	STATUS->is_fullscreen = fullscreen;
}

static gboolean displayer_karaoke_window_win_keypress(GtkWidget * w, GdkEventKey * event, gpointer data)
{
	gint volume, value;
	SingitSong *cur_song = singit_song_attach(STATUS->song);

#ifdef CODECODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_win_keypress]\n"));
#endif

        switch(event->keyval) {
                case GDK_Escape: // exit plugin
			karaoke_window_dp.disable(&karaoke_window_dp);
                        break;
		case GDK_a:
			if (event->state & GDK_CONTROL_MASK) {
				singit_remote_show_about_dlg
					(karaoke_window_dp.singit_session);
			}
			break;
                case GDK_b:
                        xmms_remote_playlist_next(karaoke_window_dp.xmms_session);
                        break;
                case GDK_c:
			if (event->state & GDK_CONTROL_MASK) {  }
			else { xmms_remote_pause(karaoke_window_dp.xmms_session); }
                        break;
		case GDK_e:
			if (event->state & GDK_CONTROL_MASK) {
				singit_remote_show_editor_dlg
					(karaoke_window_dp.singit_session);
			}
			break;
		case GDK_i:
			if (event->state & GDK_CONTROL_MASK) {
				singit_remote_show_tag_manager_dlg
					(karaoke_window_dp.singit_session);
			}
			break;
		case GDK_p:
			if (event->state & GDK_CONTROL_MASK) {
				singit_remote_show_prefs_dlg
					(karaoke_window_dp.singit_session);
			}
			break;
		case GDK_v:
                        xmms_remote_stop(karaoke_window_dp.xmms_session);
                        break;
                case GDK_x:
                        xmms_remote_play(karaoke_window_dp.xmms_session);
                        break;
		case GDK_y:
			if (STATUS->toggle_yz)
				{ xmms_remote_playlist_prev(karaoke_window_dp.xmms_session); }
			break;
                case GDK_z:
			if (!STATUS->toggle_yz)
				{ xmms_remote_playlist_prev(karaoke_window_dp.xmms_session); }
                        break;
		case GDK_KP_Add:
			volume = xmms_remote_get_main_volume(karaoke_window_dp.xmms_session);
			if (volume < 100) { volume++; }
			xmms_remote_set_main_volume(karaoke_window_dp.xmms_session, volume);
			break;
		case GDK_KP_Subtract:
			volume = xmms_remote_get_main_volume(karaoke_window_dp.xmms_session);
			if (volume > 0) { volume--; }
			xmms_remote_set_main_volume(karaoke_window_dp.xmms_session, volume);
			break;
		case GDK_KP_Enter:
			if (STATUS->jump_to_song == 0) { break; }
			if (xmms_remote_get_playlist_pos(karaoke_window_dp.xmms_session) !=
				(STATUS->jump_to_song - 1)) {
					xmms_remote_set_playlist_pos(karaoke_window_dp.xmms_session,
						(STATUS->jump_to_song - 1));
			}
		case GDK_KP_Decimal:
			STATUS->jump_to_song = 0;
			break;
		case GDK_KP_0:
		case GDK_KP_1:
		case GDK_KP_2:
		case GDK_KP_3:
		case GDK_KP_4:
		case GDK_KP_5:
		case GDK_KP_6:
		case GDK_KP_7:
		case GDK_KP_8:
		case GDK_KP_9:
		case GDK_0:
		case GDK_1:
		case GDK_2:
		case GDK_3:
		case GDK_4:
		case GDK_5:
		case GDK_6:
		case GDK_7:
		case GDK_8:
		case GDK_9:
			value = event->keyval - GDK_0;
			if (value < 0 || value > 9)
				value = event->keyval - GDK_KP_0;

			if (event->state & GDK_CONTROL_MASK) {
				enable_plugin(((value + 9) % 10));
			}
			else {
				STATUS->jump_to_song =
					(STATUS->jump_to_song * 10) + value;
				if (STATUS->jump_to_song > xmms_remote_get_playlist_length
					(karaoke_window_dp.xmms_session)) { STATUS->jump_to_song = value; }
			}
			break;
		case GDK_Return:
			if (event->state & GDK_MOD1_MASK) {
				dkwn_change_screen_mode
					(!STATUS->is_fullscreen);
			}
			break;
                default:
			// printf("%x - %x\n", event->state, event->keyval);
			singit_song_detach(&cur_song);
			return FALSE;
        }

	singit_song_detach(&cur_song);

	return TRUE;
}

void displayer_karaoke_window_finish(void)
{
//	gint posX, posY;

#ifdef CODECODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_finish]\n"));
#endif

	if (displayer_karaoke_window_win) {
		fullscreen_hide_mouse_finish();
//		if (displayer_karaoke_window_win->window) {
//			gdk_window_get_root_origin(displayer_karaoke_window_win->window, &posX, &posY);
//			singit_config_save_plugin_position("karaoke_window", posX, posY, -1, -1);
//		}
		gtk_widget_destroy(displayer_karaoke_window_win);
	}

	karaoke_window_status_unref(NULL) ;
}

static gint karaoke_window_delete_event( GtkWidget *widget, GdkEvent  *event, gpointer data )
{
	karaoke_window_dp.disable(&karaoke_window_dp);
	return (FALSE);
}

void displayer_karaoke_window_init(void)
{
//	gint posX, posY;

	g_return_if_fail(displayer_karaoke_window_win == NULL);
	g_return_if_fail(karaoke_window_status_ref() != NULL);

#ifdef CODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_init]\n"));
#endif

	singit_tooltips = gtk_tooltips_new();

	displayer_karaoke_window_win = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_realize(displayer_karaoke_window_win);
	gtk_window_set_title(GTK_WINDOW(displayer_karaoke_window_win), SINGIT_VERSION_STRING);
	gtk_window_set_policy(GTK_WINDOW(displayer_karaoke_window_win), TRUE, FALSE, TRUE);
	gtk_widget_add_events(displayer_karaoke_window_win, GDK_POINTER_MOTION_MASK);

	gtk_signal_connect(GTK_OBJECT(displayer_karaoke_window_win),
		"destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &displayer_karaoke_window_win);
	gtk_signal_connect(GTK_OBJECT(displayer_karaoke_window_win),
		"delete_event", GTK_SIGNAL_FUNC(karaoke_window_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(displayer_karaoke_window_win),
		"key-press-event", GTK_SIGNAL_FUNC(displayer_karaoke_window_win_keypress), NULL);
	gtk_container_border_width(GTK_CONTAINER(displayer_karaoke_window_win), 5);
	gtk_object_set_data (GTK_OBJECT(displayer_karaoke_window_win),
		"tooltips", singit_tooltips);

	singit_vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(displayer_karaoke_window_win), singit_vbox);
	gtk_widget_show(singit_vbox);

	STATUS->karaoke_widget = singit_karaoke_widget_new();
	gtk_widget_add_events(STATUS->karaoke_widget, GDK_POINTER_MOTION_MASK);
	gtk_box_pack_start(GTK_BOX(singit_vbox), STATUS->karaoke_widget, TRUE, TRUE, 0);
	singit_karaoke_widget_set_show_empty_lines (
		SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), TRUE);
	singit_karaoke_widget_set_lines (SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), 5);
	singit_karaoke_widget_set_toplines (SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), 2);

	// We want to stick to the style background color - anything else looks mad ;-)
	singit_karaoke_widget_set_color_gdk(
		SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
		skwc_background,
		&(gtk_widget_get_style(STATUS->karaoke_widget)
			->bg[GTK_STATE_NORMAL]));
	gtk_widget_show(STATUS->karaoke_widget);

	singit_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(singit_vbox), singit_hbox, FALSE, FALSE, 1);
	gtk_widget_show(singit_hbox);

	singit_timestamp_status_bar = gtk_statusbar_new();
	gtk_box_pack_start (GTK_BOX(singit_hbox), singit_timestamp_status_bar, FALSE, FALSE, 0);
	gtk_widget_show(singit_timestamp_status_bar);

	singit_song_gtk_event_box = gtk_event_box_new();
	gtk_container_add (GTK_CONTAINER(singit_hbox), singit_song_gtk_event_box);
	gtk_widget_show (singit_song_gtk_event_box);

	singit_song_status_bar = gtk_statusbar_new();
	gtk_container_add (GTK_CONTAINER (singit_song_gtk_event_box), singit_song_status_bar);
	gtk_widget_show(singit_song_status_bar);

	STATUS->timestamp_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(singit_timestamp_status_bar), "timestamps");
	gtk_statusbar_push(GTK_STATUSBAR(singit_timestamp_status_bar),
		STATUS->timestamp_context_id, "         ");

//	if (singit_config_load_plugin_position("karaoke_window", &posX, &posY, NULL, NULL)) {
//		gtk_widget_set_uposition(displayer_karaoke_window_win, posX, posY);
//	}
//	else
		gtk_window_set_position
			(GTK_WINDOW(displayer_karaoke_window_win), GTK_WIN_POS_CENTER);

	fullscreen_hide_mouse_init(STATUS->karaoke_widget);
}
/*
static void displayer_karaoke_window_set_song(SingitSong *new_song)
{
	SingitSong *tmp_song;

	tmp_song = STATUS->song;
	STATUS->song = singit_song_attach(new_song);
	singit_song_detach(&tmp_song);

	if (!(displayer_karaoke_window_win) &&
		GTK_WIDGET_REALIZED(displayer_karaoke_window_win))
	{
		return;
	}

	if (singit_song_lyrics_found(new_song) ==
		GTK_WIDGET_VISIBLE(displayer_karaoke_window_win))
	{
		return;
	}

	if (singit_song_lyrics_found(new_song) == FALSE) {
		if ((STATUS->hide_if_not_found &&
				!STATUS->is_fullscreen) ||
			(STATUS->hide_if_not_found_fullscreen &&
				STATUS->is_fullscreen))
		{
			gtk_widget_hide(displayer_karaoke_window_win);
		}
	}

	gtk_widget_show(displayer_karaoke_window_win);

	STATUS->update_song = TRUE;
}
*/
static void displayer_karaoke_window_set_song(SingitSong *new_song)
{
	SingitSong *tmp_song;

	g_return_if_fail(STATUS != NULL);

	tmp_song = STATUS->song;
	STATUS->song = singit_song_attach(new_song);
	singit_song_detach(&tmp_song);

	g_return_if_fail(GTK_WIDGET_REALIZED(displayer_karaoke_window_win));

	set_status_filename
		((new_song == NULL) ? NULL : new_song->lyric_filename);

	singit_karaoke_widget_freeze
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));
	singit_karaoke_widget_set_song
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), new_song);
	set_jumping_ball();
	singit_karaoke_widget_thaw
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));

	if (new_song && singit_song_lyrics_found(new_song)) {
		if (!GTK_WIDGET_VISIBLE(displayer_karaoke_window_win))
			{ gtk_widget_show(displayer_karaoke_window_win); }
		else { gdk_window_raise(displayer_karaoke_window_win->window); }
	}
	else {
		if (GTK_WIDGET_VISIBLE(displayer_karaoke_window_win)) {
			gtk_widget_hide(displayer_karaoke_window_win);
		}
	}
}
