#include "EgOSUtils.h"

#include "CEgFileSpec.h"
#include "CEgErr.h"
#include "UtilStr.h"

#include <stdlib.h>
#include <time.h>
#include <math.h>

#include "V3.h"

#ifdef EG_MAC
#include <Resources.h>
#include <Sound.h>
#include <Fonts.h>
#include <Power.h>
#include <CFURL.h>
#include <CFURLAccess.h>
#include <Gestalt.h>
#include "RequestVideo.h"

#if TARGET_API_MAC_CARBON
#include <Navigation.h>
#endif
#endif

#if EG_MAC || WIN_QUICKTIME_PRESENT
#include <Movies.h>
#include <QTML.h>
#endif

#ifdef EG_WIN
#include <windows.h>
#endif

#ifdef EG_POSIX
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#endif

#ifdef EG_SDL
#include <SDL/SDL_mouse.h>
#include <SDL/SDL_timer.h>
#endif

#ifdef EG_X
#include <X11/Xlib.h>
#endif


int		EgOSUtils::sXdpi				= 72;
int		EgOSUtils::sYdpi				= 72;
int		EgOSUtils::sXSize				= 640;
int		EgOSUtils::sYSize				= 480;
long		EgOSUtils::sLastCursor			= -1;
long		EgOSUtils::sLastCursorChange 	= -1;
CEgFileSpec 	EgOSUtils::sAppSpec;
char		EgOSUtils::sInitialDir[ INITAL_DIR_STRLEN + 2 ];
//XStrList	EgOSUtils::sFontList( ASA_OrderImportant );
bool		EgOSUtils::sCanPreventSleep	= false;
bool		EgOSUtils::sQTAvailable		= false;



void EgOSUtils::ShowFileErr( const UtilStr* inName, char* inErrMsg, bool wasReading ) {

	UtilStr	s;

	if ( wasReading )
		s.Assign( "Error reading : " );
	else
		s.Assign( "Error writing : " );
	s.Append( inErrMsg );

#ifdef EG_MAC
	s.Insert( 14, "" );
#else
	s.Insert( 14, "\"\"" );
#endif

	s.Insert( 15, inName );
	ShowMsg( s );
}

void EgOSUtils::ShowFileErr( const UtilStr* inName, CEgErr& inErr, bool wasReading ) {

	UtilStr	msg;

	inErr.GetErrStr( msg );
	ShowFileErr( inName, msg.getCStr(), wasReading );
}

void EgOSUtils::ShowFileErr( const CEgFileSpec& inFileSpec, CEgErr& inErr, bool wasReading ) {

	UtilStr	msg, fileName;

	inFileSpec.GetFileName( fileName, false );
	inErr.GetErrStr( msg );
	ShowFileErr( &fileName, msg.getCStr(), wasReading );
}

void EgOSUtils::Initialize( void* inModuleInstance ) {

	srand( clock() );

#if EG_MAC || WIN_QUICKTIME_PRESENT

#if WIN_QUICKTIME_PRESENT
	if ( ::InitializeQTML( 0 ) == noErr )
		sQTAvailable = true;
#elif EG_MAC
	sQTAvailable = true;
#endif

#if GFORCE
	::EnterMovies();
#endif

#endif

#if EG_MAC
	#pragma unused( inModuleInstance )
	OSStatus	status;
	FCBPBRec	fcbPBRec;
	Str255		appName;
	FSSpec		spec;

	short fileID = ::CurResFile();

	fcbPBRec.ioCompletion	= nil;
	fcbPBRec.ioFCBIndx		= 0;
	fcbPBRec.ioVRefNum		= 0;
	fcbPBRec.ioRefNum		= fileID;
	fcbPBRec.ioNamePtr		= appName;

	status = ::PBGetFCBInfoSync(&fcbPBRec);

	::FSMakeFSSpec(fcbPBRec.ioFCBVRefNum, fcbPBRec.ioFCBParID, "\p", &spec );
	sAppSpec.Assign( &spec, 0 );

	// See if power manager is around to call IdleUpdate()
	sCanPreventSleep = false;
	{
		OSStatus	result;
		OSErr		theErr;
		UInt32  	mask;

		theErr = ::Gestalt( gestaltPowerMgrAttr, &result );

		if ( noErr == theErr ) {

			mask = ( 1 << gestaltPMgrExists ) | ( 1 << gestaltPMgrCPUIdle );

			if ( mask == (result & mask) ) {

				if ( (UniversalProcPtr) IsProcessorCyclingEnabled != (UniversalProcPtr) kUnresolvedCFragSymbolAddress ) {

					if ( (UniversalProcPtr) IdleUpdate != (UniversalProcPtr) kUnresolvedCFragSymbolAddress ) {

						sCanPreventSleep = IsProcessorCyclingEnabled();

						EnableProcessorCycling( false );
					}
				}
			}
		}
	}
	#endif

#ifdef EG_WIN
	HDC	hDC = ::CreateDC( "DISPLAY", nil, nil, nil );
	if ( hDC ) {
		sXdpi = ::GetDeviceCaps( hDC, LOGPIXELSX );
		sYdpi = ::GetDeviceCaps( hDC, LOGPIXELSY );
		::DeleteDC( hDC );
	}

	char path[ 700 ];
	long len = ::GetModuleFileName( (HINSTANCE) inModuleInstance, path, 699 );
	if ( len ) {
		UtilStr fdir( path, len );
		fdir.Keep( fdir.FindLastInstanceOf( '\\' ) );
		sAppSpec.Assign( fdir.getCStr(), 0 );
	}

	// Init timer accuracy...
	///::timeBeginPeriod( 1 );

	// Remember the initial dir
	::GetCurrentDirectory( INITAL_DIR_STRLEN, sInitialDir );
#endif

	// For mac, sXspi and sYdpi are already correct by default


#if EG_MAC
	VideoRequestRec	requestRec;
/*
	unsigned long	theHorizontalRequest;
	unsigned long	theVerticalRequest;

	theHorizontalRequest = minHorizontalRequest;
	theVerticalRequest = minVerticalRequest; */

	requestRec.screenDevice = ::GetMainDevice();
	sXSize	= (*(*(requestRec.screenDevice))->gdPMap)->bounds.right;	// main screen is always zero offset (bounds.left == 0)
	sYSize	= (*(*(requestRec.screenDevice))->gdPMap)->bounds.bottom;	// main screen is always zero offset (bounds.top == 0)
#endif

#if EG_X
	Display* plugin_display = XOpenDisplay(NULL);
	int plugin_screen = DefaultScreen (plugin_display);
	sXSize = DisplayWidth (plugin_display, plugin_screen);
	sYSize = DisplayHeight (plugin_display, plugin_screen);
	XCloseDisplay (plugin_display);
#endif

}

void EgOSUtils::Shutdown() {

#if EG_WIN

	// Restore the initial dir
	::SetCurrentDirectory( sInitialDir );

	// Init timer accuracy...
	//::timeEndPeriod( 1 );
#endif

#if EG_MAC
	if ( sCanPreventSleep ) {
		EnableProcessorCycling( true );
	}
#endif

#if WIN_QUICKTIME_PRESENT
	if ( sQTAvailable )
		::TerminateQTML();
#endif
}


bool EgOSUtils::AskSaveAs( const char* inPrompt, const char* inDefaultName, CEgFileSpec& outSpec, long inFileType ) {

	UtilStr inName( inDefaultName );

	return AskSaveAs( inPrompt, inName, outSpec, inFileType );
}

bool EgOSUtils::AskSaveAs( const char* inPrompt, const UtilStr& inDefaultName, CEgFileSpec& outSpec, long inFileType ) {

	int	doSave = false;
	UtilStr prompt( inPrompt );

#ifdef EG_MAC
	OSErr               anErr = noErr;
	NavReplyRecord      reply;
	NavDialogOptions    dialogOptions;
	OSType              fileTypeToSave = inFileType;
	OSType              creatorType = MCC4_TO_INT("????");

	anErr = ::NavGetDefaultDialogOptions( &dialogOptions );
	if ( anErr == noErr ) {

		prompt.copyTo( dialogOptions.message, 255 );
		inDefaultName.copyTo( dialogOptions.savedFileName, 255 );

		anErr = ::NavPutFile( nil, &reply, &dialogOptions, nil, fileTypeToSave, creatorType, nil ); //, fileTypeToSave, creatorType );
		if (anErr == noErr && reply.validRecord)
		{
			AEKeyword   theKeyword;
			DescType    actualType;
			Size        actualSize;
			FSSpec      documentFSSpec;

			anErr = AEGetNthPtr(&(reply.selection), 1, typeFSS,
				&theKeyword, &actualType,
				&documentFSSpec, sizeof(documentFSSpec),
				&actualSize );
			if (anErr == noErr)
			{
				outSpec.Assign( &documentFSSpec, inFileType );
				doSave = true;

				anErr = NavCompleteSave( &reply, kNavTranslateInPlace );

			}
			::NavDisposeReply(&reply);
		}
	}
#endif

#if EG_WIN
	unsigned char	c;
	UtilStr			winFilter;
	OPENFILENAME 	info;
	char			pathName[ 601 ];

	inDefaultName.copyTo( pathName, 600 );

	// See page 519 of Vol 5 in Win32 Ref for descrip of the lpstrFilter scheme.
	winFilter.Append( "/0*" );

	// Append the ext mask...
	for ( int d = 0; d <= 24; d += 8 ) {			// Go thru each byte in ID num
		c = ((inFileType << d) >> 24 );
		winFilter.Append( (char) c );
	}
	winFilter.Append( (char) 0 );					// Windows exptects an extra NUL to end the filter

	memset( &info, 0, sizeof(OPENFILENAME) );
	info.lStructSize = sizeof(OPENFILENAME);
	info.lpstrFilter		= winFilter.getCStr();
	info.lpstrFile			= pathName;
	info.nMaxFile			= 600;
	info.lpstrTitle			= prompt.getCStr();
	if ( ::GetSaveFileName( &info ) ) {
		outSpec.Assign( pathName );
		doSave = true;
	}
#endif

	return doSave;
}

bool EgOSUtils::AskOpen( const char* inPrompt, CEgFileSpec& outSpec, long inTypeMask ) {

	bool didOpen = false;

#if EG_MAC
	UtilStr prompt( inPrompt );
	StandardFileReply	macFileReply;
	SFTypeList			typeList;

	// clear the rec
	for ( int i = 0; i < sizeof( macFileReply ); i++ )
		( (char*) &macFileReply )[ i ] = 0;

	typeList[0] = inTypeMask;

	::StandardGetFile( nil, inTypeMask ? 1 : 0, typeList, &macFileReply );

	NavDialogOptions    dialogOptions;
	AEDesc              defaultLocation;
	OSErr               anErr = noErr;

	//  Specify default options for dialog box
	anErr = ::NavGetDefaultDialogOptions( &dialogOptions );
	if (anErr == noErr)
	{
		//  Adjust the options to fit our needs
		//  Set default location option
		dialogOptions.dialogOptionFlags |= kNavSelectDefaultLocation;

		//  Clear preview option
		dialogOptions.dialogOptionFlags ^= kNavAllowPreviews;

		// make descriptor for default location
		anErr = AECreateDesc( typeFSS, defaultLocationfssPtr,
			sizeof(*defaultLocationfssPtr),
			&defaultLocation );
		if (anErr == noErr)
		{
			// Get 'open' resource. A nil handle being returned is OK,
			// this simply means no automatic file filtering.
			NavTypeListHandle typeList = (NavTypeListHandle)
				GetResource( MCC4_TO_INT("open"), 128);
			NavReplyRecord reply;

			// Call NavGetFile() with specified options and
			// declare our app-defined functions and type list
			anErr = NavGetFile ( &defaultLocation, &reply,
				&dialogOptions, eventProc, nil, filterProc,
                                typeList, nil);
			if (anErr == noErr && reply.validRecord)
			{
				//  Deal with multiple file selection
				long    count;

				anErr = AECountItems(&(reply.selection), &count);
				// Set up index for file list
				if (anErr == noErr)
				{
					longindex;

					for (index = 1; index <= count; index++)
					{
						AEKeyword   theKeyword;
						DescType    actualType;
						Size        actualSize;
						FSSpec      documentFSSpec;

						// Get a pointer to selected file
						anErr = AEGetNthPtr(&(reply.selection), index,
							typeFSS, &theKeyword,
							&actualType,&documentFSSpec,
							sizeof(documentFSSpec),
							&actualSize);
						if (anErr == noErr)
						{
							anErr = DoOpenFile(&documentFSSpec);
						}
					}
				}
				//  Dispose of NavReplyRecord, resources, descriptors
				anErr = NavDisposeReply(&reply);
			}
			if (typeList != NULL) {
				ReleaseResource( (Handle)typeList);
			}
			(void) AEDisposeDesc(&defaultLocation);
		}
	}
	DisposeRoutineDescriptor(eventProc);
	DisposeRoutineDescriptor(filterProc);

	return anErr;
/*
	#if TARGET_API_MAC_CARBON // NOT WORKING YET
		NavDialogOptions		mNavOptions;
		NavReplyRecord			mNavReply;
		mNavOptions.version = kNavDialogOptionsVersion;
		mNavReply.version = kNavReplyRecordVersion;

		::NavGetDefaultDialogOptions(&mNavOptions);

		OSErr err = ::NavChooseFile(
						NULL,
						&mNavReply,
						&mNavOptions,
						NULL,
						NULL,
						NULL,
						NULL,
						0L);							// User Data

		if ( err == noErr )
		{
			AEDesc	specDesc;
			specDesc.descriptorType = typeNull;
			specDesc.dataHandle     = nil;

			AEDescList	selectedItems = mNavReply.selection;
			err = ::AECoerceDesc(&selectedItems, typeFSS, &specDesc);

			if ( err == noErr )
			{
				outSpec.Assign(*(FSSpec**) specDesc.dataHandle);

				didOpen = true;
			}

			NavDisposeReply(&mNavReply);
		}
	#else
		::StandardGetFile( nil, inTypeMask ? 1 : 0, typeList, &macFileReply );
	//UDesktop::Activate();
*/
	if ( macFileReply.sfGood ) {
		outSpec.Assign( &macFileReply.sfFile );
		didOpen = true;
	}
#endif

#if EG_WIN
	OPENFILENAME info;
	char pathName[ 401 ];

	#pragma unused( inTypeMask )

	info.lStructSize = sizeof(OPENFILENAME);
	info.hwndOwner = NULL;
	info.hInstance = 0;
	info.lpstrFilter = nil; //"All Files\0*.*\0\0";
	info.lpstrCustomFilter = 0;
	info.nMaxCustFilter = 0;
	info.nFilterIndex = 0;
	info.lpstrFile = pathName;
	info.nMaxFile = 400;
	info.lpstrFileTitle = nil;
	info.nMaxFileTitle = 0;
	info.lpstrInitialDir = nil;
	info.lpstrTitle = inPrompt;
	info.Flags = OFN_HIDEREADONLY;

	info.nFileOffset = 0;
	info.nFileExtension = 0;
	info.lpstrDefExt = nil;
	info.lCustData = 0;
	info.lpfnHook = nil;
	info.lpTemplateName = nil;

	if ( ::GetOpenFileName( &info ) ) {
		didOpen = true;
		outSpec.Assign( pathName );
	}

	/*
	unsigned char	c;
	UtilStr		winFilter;
	OPENFILENAME 	paramBlk;
	char			pathName[ 601 ];

	// See page 519 of Vol 5 in Win32 Ref for descrip of the lpstrFilter scheme.
	winFilter.Append( "/0*" );

	// Append the ext mask...
	for ( int d = 0; d <= 24; d += 8 ) {			// Go thru each byte in ID num
		c = ((inTypeMask << d) >> 24 );
		winFilter.Append( (char) c );
	}
	winFilter.Append( (char) 0 );					// Windows exptects an extra NUL to end the filter

	memset( &paramBlk, 0, sizeof(OPENFILENAME) );
	paramBlk.lStructSize = sizeof(OPENFILENAME);
	paramBlk.lpstrFilter		= winFilter.getCStr();
	paramBlk.lpstrFile			= pathName;
	paramBlk.nMaxFile			= 600;
	paramBlk.lpstrTitle			= prompt.getCStr();
	if ( ::GetOpenFileName( &paramBlk ) ) {
		outSpec.Assign( pathName );
		didOpen = true;
	}*/
#endif

	return didOpen;
}

bool EgOSUtils::AreYouSure( const UtilStr& inMsg ) {

	int ans = 1;

#ifdef EG_MAC
	::ParamText( inMsg.getPasStr(), "\p", "\p", "\p");
	//UDesktop::Deactivate();
	ans = ::CautionAlert( 2000, nil );
	//UDesktop::Activate();

	return ans == 1; //answer_Save;
#elif EG_WIN
	ans = ::MessageBox( nil, inMsg.getCStr(), "Examgen Message", MB_ICONEXCLAMATION | MB_YESNO | MB_SETFOREGROUND | MB_TASKMODAL );
	return ans == IDYES;
#else
	return (ans == 1);
#endif
}


bool EgOSUtils::AreYouSure( const char* inMsg ) {

	UtilStr	msg( inMsg );
	return AreYouSure( msg );
}

int EgOSUtils::AskSaveChanges( const char* inName ) {

	UtilStr name( inName );
	return AskSaveChanges( name );
}

int EgOSUtils::AskSaveChanges( const UtilStr& inName ) {

	int ans = 1;

#ifdef EG_MAC
	::ParamText( inName.getPasStr(), "\p", "\p", "\p" );
	//UDesktop::Deactivate();
	ans = ::CautionAlert( 2001, nil );
	//UDesktop::Activate();

	return 2 - ans;
#elif EG_WIN
	UtilStr	msg( "Save changes to \"" );
	msg.Append( inName );
	msg.Append( "\" before closing?" );
	ans = ::MessageBox( nil, msg.getCStr(), "Examgen Message", MB_ICONEXCLAMATION | MB_YESNOCANCEL | MB_SETFOREGROUND | MB_TASKMODAL );
	if ( ans == IDYES )
		return 1;
	else if ( ans == IDNO )
		return -1;
	else
		return 0;
#else
	return ans;
#endif
}


void EgOSUtils::PreventSleep() {

#if EG_MAC
	if ( sCanPreventSleep )
		IdleUpdate();
#endif
}

void EgOSUtils::SpinCursor() {

	long time = clock();

	if ( sLastCursorChange == -1 )
		sLastCursorChange = time;
	else if ( time - CLOCKS_PER_SEC / 3 > sLastCursorChange ) {					// Every 1/3 second...
#ifdef EG_MAC
		Handle cursHndl;
		sLastCursor		= ( sLastCursor + 1 ) % 8;							// 8 Cursors
		cursHndl		= ::GetResource( MCC4_TO_INT("CURS"), 6500 + sLastCursor );		// 6500 = Base ID
		sLastCursorChange	= time;
		if ( ! cursHndl )
			cursHndl = (Handle) ::GetCursor( watchCursor );
		if ( cursHndl )
			::SetCursor( (Cursor*) *cursHndl );
#endif

#ifdef EG_WIN
		SetCursor( ::LoadCursor( nil, IDC_WAIT ) );
		sLastCursor = 1;
#endif
	}
}

void EgOSUtils::ResetCursor() {


	if ( sLastCursor != -1 ) {

#if EG_MAC
		::InitCursor();
#endif

#ifdef EG_WIN
		while ( ::ShowCursor( true ) < 0 ) { }
		::SetCursor( ::LoadCursor( nil, IDC_ARROW ) );
#endif

		sLastCursor = -1;
	}

	sLastCursorChange = -1;
}

void EgOSUtils::ShowCursor() {

#ifdef EG_MAC
	::ShowCursor();
#elif defined EG_WIN
	while ( ::ShowCursor( true ) < 0 ) { }
#elif defined EG_SDL
	SDL_ShowCursor(1);
#endif

}

void EgOSUtils::HideCursor() {

#ifdef EG_MAC
	::HideCursor();
#elif defined EG_WIN
	while ( ::ShowCursor( false ) >= 0 ) { }
#elif defined EG_SDL
	SDL_ShowCursor(0);
#endif
}

void EgOSUtils::FindExtension( const UtilStr& inFileName, UtilStr* outName, UtilStr* outExtension ) {

	long pos = inFileName.FindLastInstanceOf( '.' );	// Find where ext begins

	if ( pos > 0 )
		pos--;
	else
		pos = inFileName.length();

	if ( outName ) {
		if ( &inFileName != outName )
			outName -> Assign( inFileName.getCStr(), pos );
		else
			outName -> Keep( pos );
	}

	if ( outExtension ) {
		if ( &inFileName != outExtension )
			outExtension -> Assign( inFileName.getCStr() + pos );
		else
			outExtension -> Trunc( pos, false );
	}
}

long EgOSUtils::FindExtension( const UtilStr* inFileName ) {

	long i, ext = 0;
	char c, *s;

	if ( inFileName ) {
		i = inFileName -> FindLastInstanceOf( '.' );

		if ( i > 0 ) {
			s = inFileName -> getCStr() + i - 1;
			for ( i = 0; i < 4 && s[ i ]; i++ ) {

				// Convert to uppercase
				c = s[ i ];
				if ( c >= 'a' && c <= 'z' )
					c -= 32;

				// Tack the char to the extension long
				ext = ( ext << 8 ) | c;
			}
		}
	}

	return ext;
}


bool EgOSUtils::GetNextFile( const CEgFileSpec& folderSpec, CEgFileSpec& outSpec, long inFlags, long* ioIdx ) {

	bool	doFolders 	= inFlags & OSU_DO_FOLDERS;
	bool	doAll		= doFolders && ( inFlags & OSU_DO_FILES );
	bool	ok;

#ifdef EG_MAC
	OSErr				err;
	HFileInfo			pb;
	FSSpec				spec;
	long				parID;
	Str255				str;
	bool				resolveAliases	= ( inFlags & OSU_RESOLVE_ALIASES );
	Boolean				isFolder, wasAliased;

	// Goal: find the dirID of the folder specified by folderSpec (and use that as the folder to scan)
	folderSpec.GetFileName() -> copyTo( str, 255 );
	pb.ioCompletion 		= nil;
	pb.ioNamePtr			= str;
	pb.ioVRefNum			= ( (FSSpec*) folderSpec.OSSpec() ) -> vRefNum;
	pb.ioDirID 				= ( (FSSpec*) folderSpec.OSSpec() ) -> parID;
	pb.ioFDirIndex 			= 0;
	::PBGetCatInfoSync( (CInfoPBRec*) &pb );

	// parID now contains the dir ID of folderSpec
	parID = pb.ioDirID;


	pb.ioFDirIndex = *ioIdx;

	ok = false;
	do {
		pb.ioDirID 	= parID;
		err = ::PBGetCatInfoSync( (CInfoPBRec*) &pb );

		if ( err == noErr && ( (pb.ioFlFndrInfo.fdFlags & fInvisible) == 0)  /*&& pb.ioFlFndrInfo.fdCreator == cEGCreator*/ ) {
			if ( ( pb.ioFlAttrib & ioDirMask ) == 0 )
				ok = false;
			if ( doFolders )
				ok = ! ok;
			if ( ok || doAll ) {
				::FSMakeFSSpec( pb.ioVRefNum, pb.ioFlParID, str, &spec );
				if ( resolveAliases )
					err = ::ResolveAliasFile( &spec, true, &isFolder, &wasAliased );
				outSpec.Assign( &spec, pb.ioFlFndrInfo.fdType );
			}
		}

		pb.ioFDirIndex++;
		*ioIdx = pb.ioFDirIndex;
	} while ( err == noErr && ! ok );

#elif EG_WIN
	WIN32_FIND_DATA		fileData;
	static HANDLE		hSearch;
	UtilStr			name;
	bool			isDir, tryAgain;

	do {
		if ( *ioIdx == OSU_FIRST_FILE ) {
			name.Assign( (char*) folderSpec.OSSpec() );
			if ( name.getChar( name.length() ) == FILE_SYS_SEP_CHAR )
				name.Trunc( 1 );
			ok = SetCurrentDirectory( name.getCStr() );
			if ( ok ) {
				hSearch = ::FindFirstFile( "*.*", &fileData );
				ok = hSearch != INVALID_HANDLE_VALUE;
			}
		}
		else
			ok = ::FindNextFile( hSearch, &fileData );

		(*ioIdx)++;

		if ( ok ) {
			name.Assign( fileData.cFileName );
			isDir = ::GetFileAttributes( fileData.cFileName ) & FILE_ATTRIBUTE_DIRECTORY;
			if ( isDir == doFolders || doAll ) {
				tryAgain = name.compareTo( "." ) == 0 || name.compareTo( ".." ) == 0;
				outSpec.Assign( folderSpec );
				if ( isDir )
					name.Append( FILE_SYS_SEP_CHAR );
				outSpec.Rename( name ); }
			else
				tryAgain = true;
		}
	} while ( ok && tryAgain );

#elif EG_POSIX
	UtilStr name;
	bool    isDir, tryAgain = false;
	UtilStr fullname;

	static DIR *d = NULL;
	struct dirent *de;

	ok = true;
	do {
		if ( (*ioIdx == OSU_FIRST_FILE) ) {
			if (d != NULL) {
				closedir(d);
				d = NULL;
			}
			doAll = false;
			name.Assign( (char*) folderSpec.OSSpec() );
			if ( name.getChar( name.length() ) == FILE_SYS_SEP_CHAR )
				name.Trunc( 1 );
			d = opendir(name.getCStr());
			if (d == NULL)
				{ return false; }
		}

		de = readdir(d);
		if (de == NULL)
			{ return false; }
		name.Assign(de->d_name);
		struct stat statdata;
		fullname.Assign( (char*) folderSpec.OSSpec() );
		if ( fullname.getChar( fullname.length() ) != FILE_SYS_SEP_CHAR )
			fullname.Append( FILE_SYS_SEP_CHAR );
		fullname.Append(de->d_name);
		ok = (stat(fullname.getCStr(), &statdata) == 0);

		if (!tryAgain)
			(*ioIdx)++;

		if ( ok ) {
			isDir = (S_ISDIR(statdata.st_mode));
			if ( isDir == doFolders || doAll ) {
				tryAgain = name.compareTo( "." ) == 0 || name.compareTo( ".." ) == 0;
				if (tryAgain)
					{ continue; }
				outSpec.Assign( folderSpec );
				if ( isDir )
					name.Append( FILE_SYS_SEP_CHAR );
				outSpec.Rename( name );
			}
			else
				tryAgain = true;
		}
	}
	while ( ok && tryAgain );
#endif

	return ok;
}


void EgOSUtils::Beep() {

#ifdef EG_MAC
	::SysBeep( 200 );
#endif

#ifdef EG_WIN
	MessageBeep(0);
#endif
}

long EgOSUtils::CurTimeMS() {

#if EG_WIN
	return ::timeGetTime();
#elif EG_MAC
	return ::TickCount() * 16;
#elif EG_SDL
	return SDL_GetTicks();
#elif EG_POSIX
	struct timeval tv;
	struct timezone tz;

	tz.tz_minuteswest = 0;
	tz.tz_dsttime = 0;
	gettimeofday(&tv, &tz);

	return ((tv.tv_sec * 1000) & 0x7fffffff) + tv.tv_usec / 1000;
#endif
}

void EgOSUtils::GetMouse( Point& outPt ) {

#if EG_MAC
	::GetMouse( &outPt );
	::LocalToGlobal( &outPt );
#elif EG_WIN
	POINT p;
	::GetCursorPos( &p );
	outPt.h = p.x;
	outPt.v = p.y;
#elif EG_SDL
	int x, y;
	SDL_GetMouseState(&x, &y);
	outPt.h = x;
	outPt.v = y;
#endif
}


void EgOSUtils::ShowMsg( const UtilStr& inMsg ) {

#ifdef EG_MAC
	//UDesktop::Deactivate();
	::ParamText( inMsg.getPasStr(), "\p", "\p", "\p");
	::StopAlert( 2002, nil );
	//#pragma rem back in!
	//UDesktop::Activate();
#endif

#ifdef WIN32
	::MessageBox( nil, inMsg.getCStr(), "Message", MB_ICONEXCLAMATION | MB_OK | MB_SETFOREGROUND | MB_APPLMODAL );
	//ZafMessageWindow* w = new ZafMessageWindow( "Message", ZAF_EXCLAMATION_ICON, ZAF_DIALOG_OK, ZAF_DIALOG_OK, inMsg.getCStr() );
	//zafWindowManager -> Add( w );
	//w -> Control();
#endif
}

void EgOSUtils::ShowMsg( const char* inMsg ) {

	UtilStr msg( inMsg );

	ShowMsg( msg );
}


#if EG_MAC
#include <Processes.h>
#include <AppleEvents.h>


//----------------------------------------------------------------------------
// BuildOpenDocumentsEvent
//
// General utility to turn a ProcessSerialNumber and a list of FSSpecs into
// an 'odoc' AppleEvent with the ProcessSerialNumber as the target
// application.  Used by SendOpenDocumentEventToProcess, and
// LaunchApplicationWithDocument.
//----------------------------------------------------------------------------
OSErr BuildOpenDocumentEvent(
	ProcessSerialNumber		*targetPSN,
	const FSSpec 			*theSpecArray,
	const short				numOfSpecs,
	AppleEvent				*odocAppleEvent );
OSErr BuildOpenDocumentEvent(
	ProcessSerialNumber		*targetPSN,
	const FSSpec 			*theSpecArray,
	const short				numOfSpecs,
	AppleEvent				*odocAppleEvent)
{
	OSErr			retCode;
	AppleEvent		theAppleEvent;
	AEDesc			targetAddrDesc, docDesc;
	AEDescList		docDescList;
	AliasHandle		docAlias;
	short			counter;
	FSSpecPtr		specIterator;

	// to simplify cleanup, ensure that handles are nil to start
	targetAddrDesc.dataHandle	= nil;
	theAppleEvent.dataHandle	= nil;
	docDescList.dataHandle		= nil;
	docDesc.dataHandle			= nil;
	docAlias					= nil;

	// create an address descriptor based on the serial number of
	// the target process
	retCode = AECreateDesc(typeProcessSerialNumber, (Ptr) targetPSN,
		sizeof(ProcessSerialNumber), &targetAddrDesc);
	if (retCode != noErr) goto Bail;

	// make a descriptor list containing just a descriptor with an
	// alias to the document
	retCode = AECreateList(nil, 0, false, &docDescList);
	if (retCode != noErr) goto Bail;

	// start at the beginning of the FSSpec list, and start adding
	// them to the document list descriptor

	// NOTE: we need to make sure we dispose of the aliases and the
	// AE descriptor in this loop, otherwise there will be memory
	// leaks.
	specIterator = (FSSpecPtr)theSpecArray;
	for (counter = 0; counter < numOfSpecs; counter ++) {

		retCode = NewAlias(nil, &specIterator[counter], &docAlias);
		if (retCode != noErr) goto Bail;

		HLock((Handle) docAlias);
		retCode = AECreateDesc(typeAlias, (Ptr) *docAlias,
		GetHandleSize((Handle) docAlias), &docDesc);
		HUnlock((Handle) docAlias);

		if (retCode != noErr) goto Bail;
		// the alias is now in the AEDescriptor, so dispose of it
		DisposeHandle((Handle)docAlias); docAlias = nil;

		retCode = AEPutDesc(&docDescList, 0, &docDesc);
		if (retCode != noErr) goto Bail;

		// the alias is now in the AE document list, so dispose of it
		retCode = AEDisposeDesc(&docDesc);
		if (retCode != noErr) goto Bail;

	}

	// now make the 'odoc' AppleEvent descriptor and insert the
	// document descriptor list as the direct object
	retCode = AECreateAppleEvent(kCoreEventClass, kAEOpenDocuments,
		&targetAddrDesc, kAutoGenerateReturnID, kAnyTransactionID,
		&theAppleEvent);
	if (retCode != noErr) goto Bail;

	retCode = AEPutParamDesc(&theAppleEvent, keyDirectObject, &docDescList);
	if (retCode != noErr) goto Bail;

	*odocAppleEvent = theAppleEvent;

Bail:
	// dispose of everything that was allocated, except the return AE
	if (targetAddrDesc.dataHandle != nil)  (void) AEDisposeDesc(&targetAddrDesc);
	if (docDescList.dataHandle != nil)  (void) AEDisposeDesc(&docDescList);
	if (docDesc.dataHandle != nil)  (void) AEDisposeDesc(&docDesc);
	if (docAlias != nil)  DisposeHandle((Handle) docAlias);

	return retCode;

}





// FindApplicationFromDocument uses the Desktop Database to
// locate the creator application for the given document
//
// this routine will first check the desktop database of the disk
// containing the document, then the desktop database of all local
// disks, then the desktop databases of all server volumes
// (so up to three passes will be made)

OSErr FindApplicationFromDocument( const FSSpec* documentFSSpecPtr, FSSpec* appFSSpecPtr );
OSErr FindApplicationFromDocument( const FSSpec* documentFSSpecPtr, FSSpec* appFSSpecPtr ) {
	enum { documentPass, localPass, remotePass, donePass } volumePass;
	DTPBRec desktopParams;
	HParamBlockRec hfsParams;
	FInfo documentFInfo;
	short volumeIndex;
	Boolean foundFlag;
	GetVolParmsInfoBuffer volumeInfoBuffer;
	OSErr retCode;

	// dkj 12/94 initialize flag to false (thanks to Peter Baral for pointing out this bug)
	foundFlag = false;

	// verify the document file exists and get its creator type
	retCode = FSpGetFInfo(documentFSSpecPtr, &documentFInfo);
	if (retCode != noErr) goto Bail;

	volumePass = documentPass;
	volumeIndex = 0;

	do {

		// first, find the vRefNum of the volume whose Desktop Database
		// we're checking this time

		// if we're on the initial pass (documentPass) just use
		// the vRefNum of the document itself

		if (volumePass == documentPass)

			desktopParams.ioVRefNum = documentFSSpecPtr->vRefNum;

		// otherwise, find the vRefNum of the next volume appropriate
		// for this pass

		else {

			volumeIndex++;

			// convert the volumeIndex into a vRefNum

			hfsParams.volumeParam.ioNamePtr = nil;
			hfsParams.volumeParam.ioVRefNum = 0;
			hfsParams.volumeParam.ioVolIndex = volumeIndex;
			retCode = PBHGetVInfoSync(&hfsParams);

			// a nsvErr indicates that the current pass is over
			if (retCode == nsvErr) goto SkipThisVolume;
			if (retCode != noErr) goto Bail;

			// since we handled the document volume during the documentPass,
			// skip it if we have hit that volume again

			if (hfsParams.volumeParam.ioVRefNum == documentFSSpecPtr->vRefNum)
				goto SkipThisVolume;

			// call GetVolParms to determine if this volume is a server
			// (a remote volume)

			hfsParams.ioParam.ioBuffer = (Ptr) &volumeInfoBuffer;
			hfsParams.ioParam.ioReqCount = sizeof(GetVolParmsInfoBuffer);
			retCode = PBHGetVolParmsSync(&hfsParams);
			if (retCode != noErr) goto Bail;

			// if the vMServerAdr field of the volume information buffer
			// is zero, this is a local volume; skip this volume
			// if it's local on a remote pass or remote on a local pass

			if ((volumeInfoBuffer.vMServerAdr != 0) !=
				(volumePass == remotePass)) goto SkipThisVolume;

			// okay, now we've found the vRefNum for our desktop database call

			desktopParams.ioVRefNum = hfsParams.volumeParam.ioVRefNum;
		}

		// find the path refNum for the desktop database for
		// the volume we're interested in

		desktopParams.ioNamePtr = nil;

		retCode = PBDTGetPath(&desktopParams);
		if (retCode == noErr && desktopParams.ioDTRefNum != 0) {

			// use the GetAPPL call to find the preferred application
			// for opening any document with this one's creator

			desktopParams.ioIndex = 0;
			desktopParams.ioFileCreator = documentFInfo.fdCreator;
			desktopParams.ioNamePtr = appFSSpecPtr->name;
			retCode = PBDTGetAPPLSync(&desktopParams);

			if (retCode == noErr) {

				// okay, found it; fill in the application file spec
				// and set the flag indicating we're done

				appFSSpecPtr->parID = desktopParams.ioAPPLParID;
				appFSSpecPtr->vRefNum = desktopParams.ioVRefNum;
				foundFlag = true;

			}
		}

	SkipThisVolume:

		// if retCode indicates a no such volume error or if this
		// was the first pass, it's time to move on to the next pass

		if (retCode == nsvErr || volumePass == documentPass) {

			// Inc volumePass
			switch ( volumePass ) {
				case documentPass:		volumePass = localPass;		break;
				case localPass:			volumePass = remotePass;	break;
				case remotePass:		volumePass = donePass;		break;
			}
			volumeIndex = 0;
		}

	} while (foundFlag == false && volumePass != donePass);

Bail:
	return retCode;
}

/*

//----------------------------------------------------------------------------
// LaunchApplicationWithDocument
//
// given an application and any number of documents,
// LaunchApplicationWithDocument launches the application and passes the
// application an OpenDocuments event for the document(s)
//----------------------------------------------------------------------------
OSErr LaunchApplicationWithDocument(
	const FSSpec		*applicationFSSpecPtr,
	const FSSpec 		*theSpecArray,
	const short			numOfSpecs );
OSErr LaunchApplicationWithDocument(
	const FSSpec		*applicationFSSpecPtr,
	const FSSpec 		*theSpecArray,
	const short			numOfSpecs)
{
	OSErr retCode;
	LaunchParamBlockRec launchParams;
	AppleEvent theAppleEvent;
	AEDesc launchParamDesc;
	ProcessSerialNumber targetPSN;

	// to simplify cleanup, ensure that handles are nil to start
	launchParams.launchAppParameters	= nil;
	theAppleEvent.dataHandle			= nil;
	launchParamDesc.dataHandle			= nil;

	if (theSpecArray != nil) {

		// because 'odoc' events require a address descriptor, I just
		// grab the PSN for the current process.  It doesn't matter what
		// it is, because it's never used.
		(void) GetCurrentProcess( &targetPSN );

		// build an 'odoc' event given the array of FSSpecs and the ProcessSerialNumber
		retCode = BuildOpenDocumentEvent(&targetPSN, theSpecArray, numOfSpecs, &theAppleEvent);

		if (retCode == noErr) {

			// coerce the AppleEvent into app parameters, for _LaunchApplication
			retCode = AECoerceDesc(&theAppleEvent, typeAppParameters, &launchParamDesc);
			if (retCode != noErr) goto Bail;

			// fill in the launch parameter block, including the
			// Apple event, and make the launch call
			HLock((Handle) launchParamDesc.dataHandle);
			launchParams.launchAppParameters =
				(AppParametersPtr) *(launchParamDesc.dataHandle);

		}

	}

	launchParams.launchBlockID		= extendedBlock;
	launchParams.launchEPBLength	= extendedBlockLen;
	launchParams.launchFileFlags	= launchNoFileFlags;
	launchParams.launchControlFlags	= launchContinue;
	launchParams.launchAppSpec		= (FSSpecPtr)applicationFSSpecPtr;

	retCode = LaunchApplication(&launchParams);

Bail:
	// dispose of everything that was allocated
	if (theAppleEvent.dataHandle != nil)     (void) AEDisposeDesc(&theAppleEvent);
	if (launchParamDesc.dataHandle != nil)   (void) AEDisposeDesc(&launchParamDesc);

	return retCode;
}


*/
#endif


bool EgOSUtils::LaunchDoc( const CEgFileSpec& inSpec, long inCreator_ID ) {

#if EG_MAC
/*	OSErr retCode;
	FSSpec* docSpecPtr = (FSSpec*) inSpec.OSSpec();
	FSSpec applSpec;

	//look for open app!!
	retCode = FindApplicationFromDocument( docSpecPtr, &applSpec );
	if ( retCode == noErr )
		retCode = LaunchApplicationWithDocument( &applSpec, docSpecPtr, 1 );

	return retCode == noErr;
	*/
	return false;
#elif EG_WIN
	char path[ 401 ];
	::GetShortPathName( (char*) inSpec.OSSpec(), path, 400 );

	return LaunchURL( path, inCreator_ID );
#else
	return false;
#endif
}


#if EG_MAC
#include <InternetConfig.h>
#endif

bool EgOSUtils::LaunchHTMLFile( const CEgFileSpec& inHTMLFile, long inCreator_ID ) {

	bool result = false;

#if EG_WIN
	result = LaunchDoc( inHTMLFile, inCreator_ID );
#endif

#if EG_MAC

	// Strip off the volume name and
	long pos, osVers;

	UtilStr pathName, osXName;

	inHTMLFile.GetPathname( pathName );
	pathName.Replace( "/", "%2F" );
	pathName.Replace( " ", "%20" );
	pos = pathName.FindNextInstanceOf( 0, ':' );
	pathName.Replace( ':', '/' );

	// Get OS vers...
	::Gestalt( gestaltSystemVersion, &osVers );

	// In Mac OS X, we first assume the pathname is onthe default volume.  If it's not found there,
	// we assume the volume isn't the default vol and we use the 'Volumes' path.
	if ( osVers >= 0x0A00 ) {
		osXName.Assign( pathName.getCStr() + pos );
		osXName.Prepend( "file:///" );
		result = LaunchURL(  osXName, inCreator_ID );
		if ( ! result ) {
			osXName.Assign( "file:///Volumes/" );
			osXName.Append( pathName );
			result = LaunchURL(  osXName, inCreator_ID );
		} }

	else {

		/*
		It was too hellish to update MacOS's Universal hdrs to 3.4 w/ CW Pro 6.
		FSRef newRef;
		CFURLRef myURL;
		::FSpMakeFSRef( (FSSpec*) inHTMLFile.OSSpec(), &newRef );
		myURL = CFURLCreateFromFSRef( kCFAllocatorDefault, &newRef );
		LSOpenCFURLRef();
		CFURLDestroyResource( myURL, &err ); */

		pathName.Prepend( "file:///" );
		result = LaunchURL( pathName, inCreator_ID );
	}

#endif

	return result;
}

bool EgOSUtils::LaunchURL( const char* inURL, long inCreator_ID ) {


#if EG_MAC
	OSStatus err;
	ICInstance inst;
	long startSel, endSel;

	long len = 0;
	while ( inURL[ len ] )
		len++;

	err = ::ICStart( &inst, inCreator_ID );           // Use your creator code if you have one!
	if ( err == noErr ) {

#if ! TARGET_API_MAC_CARBON
		err = ::ICFindConfigFile( inst, 0, nil );
#endif

		if ( err == noErr ) {
			startSel	= 0;
			endSel		= len;
			err = ::ICLaunchURL( inst, "\p", inURL, endSel, &startSel, &endSel );

			UtilStr s;
			s.Assign( (long) err );
		}
		::ICStop( inst );
	}
	return err == noErr;

#elif EG_WIN
	UtilStr commandLine( "Start " );
	commandLine.Append( inURL );

	PROCESS_INFORMATION process_info;
	STARTUPINFO info;
	info.cb = sizeof( STARTUPINFO );
	info.lpDesktop = NULL;
	info.lpTitle = NULL;
	info.dwFlags = STARTF_USESHOWWINDOW;
	info.wShowWindow = SW_HIDE;
	info.cbReserved2 = 0;
	info.lpReserved2 = NULL;

	long retCode = ::CreateProcess(
		NULL,
		commandLine.getCStr(),
		NULL,	// pointer to process security attributes
		NULL,	// pointer to thread security attributes
		FALSE,
		HIGH_PRIORITY_CLASS,	// creation flags
		NULL,	// pointer to new environment block
		NULL,	// pointer to current directory name
		&info,	// pointer to STARTUPINFO
		&process_info 	// pointer to PROCESS_INFORMATION
	);

	return retCode != 0;
#else
	return false;
#endif
}

long EgOSUtils::Rnd( long min, long max ) {

	long maxRnd 	= RAND_MAX;
	long retNum 	= rand() * ( max - min + 1 ) / maxRnd + min;

	if ( retNum >= max )
		return max;
	else
		return retNum;
}

unsigned long EgOSUtils::RevBytes( unsigned long inNum ) {


	return ( inNum << 24 ) | ( ( inNum & 0xFF00 ) << 8 ) | ( ( inNum & 0xFF0000 ) >> 8 ) | ( inNum >> 24 );
}

int EgOSUtils::CalcIntensity( const PixPalEntry& inC ) {

#if EG_MAC
	return CalcIntensity( inC.rgb );
#else
	return CalcIntensity( inC.rgbRed, inC.rgbGreen, inC.rgbBlue );
#endif
}

int EgOSUtils::CalcContrast( int inR, int inG, int inB, int inR2, int inG2, int inB2 ) {

	V3 c1, c2;

	/* Contrast of colors A and B are related to the spacial distance between the points
	defined by the HSV cone (the ROYGBIV colorwheel is the base and the taper is black).  However, our
	eyes are more sensitive to green than red than blue.  So when we map each color to a modified HSV cone,
	where the hues are weighted differently (see CalcIntensity).  */

	CalcContrast( inR, inG, inB, c1 );
	CalcContrast( inR2, inG2, inB2, c2 );

	return 100 * c1.dist( c2 );
}

void EgOSUtils::CalcContrast( int inR, int inG, int inB, V3& outContrast ) {

	float h, s, v;

	RGB2HSV( (inR / 255.0), (inG / 255.0), (inB / 255.0), h, s, v );

	outContrast.mX = v * s * cos( 6.283185 * h );
	outContrast.mY = v * s * sin( 6.283185 * h );
	outContrast.mZ = v * CalcIntensity( inR, inG, inB ) / 255.0;
}



void EgOSUtils::CalcContrast( const PixPalEntry& inC, V3& outContrast ) {

	#if EG_MAC
	CalcContrast( inC.rgb, outContrast );
	#else
	CalcContrast( inC.rgbRed, inC.rgbGreen, inC.rgbBlue, outContrast );
	#endif
}



int EgOSUtils::CalcContrast( const PixPalEntry& inC1, const PixPalEntry& inC2 ) {

#if EG_MAC
	return CalcContrast( inC1.rgb, inC2.rgb );
#else
	return CalcContrast( inC1.rgbRed, inC1.rgbGreen, inC1.rgbBlue,
				inC2.rgbRed, inC2.rgbGreen, inC2.rgbBlue );
#endif
}

void EgOSUtils::CalcInverted( const RGBColor& inC, RGBColor& outInverted ) {

	outInverted.red		= 0xFFFF - inC.red;
	outInverted.green	= 0xFFFF - inC.green;
	outInverted.blue	= 0xFFFF - inC.blue;

}

#define __SET_RGB( R, G, B ) 	\
	if ( R < 0 )				\
		outRGB.red = 0;			\
	else if ( R <= 0xFFFF )		\
		outRGB.red = R;			\
	else						\
		outRGB.red = 0xFFFF;	\
	if ( G < 0 )				\
		outRGB.green = 0;		\
	else if ( G <= 0xFFFF )		\
		outRGB.green = G;		\
	else						\
		outRGB.green = 0xFFFF;	\
	if ( B < 0 )				\
		outRGB.blue = 0;		\
	else if ( B <= 0xFFFF )		\
		outRGB.blue = B;		\
	else						\
		outRGB.blue = 0xFFFF;	\
	break;


#define __BoundsChk( var ) \
	if ( var < 0 )	var = 0;		\
	else if ( var > 1.0 ) var = 1;


void EgOSUtils::HSV2RGB( float H, float S, float V, RGBColor& outRGB ) {


	// H is given on [0, 1] or WRAPPED. S and V are given on [0, 1].
	// RGB are each returned on [0, 1].
	long hexQuadrant, m, n, v;
	H = ( H - floor( H ) ) * 6;  // Wrap the Hue angle around 1.0, then find quadrant

	hexQuadrant = H;
	float f = H - hexQuadrant;

	// Check val and sat bounds
	__BoundsChk( S )
	__BoundsChk( V )

	if ( ! ( hexQuadrant & 1 ) )
		f = 1 - f; // hexQuadrant i is even

	V *= 65535.0;
	v = V;
	m = V * (1 - S);
	n = V * (1 - S * f);

	switch ( hexQuadrant ) {
		case 1: __SET_RGB( n, v, m );
		case 2: __SET_RGB( m, v, n );
		case 3: __SET_RGB( m, n, v );
		case 4: __SET_RGB( n, m, v );
		case 5: __SET_RGB( v, m, n );
		default:
			__SET_RGB( v, n, m );
	}
}

void EgOSUtils::RGB2HSV( float R, float G, float B, float& outH, float& outS, float& outV ) {

	// RGB are each on [0, 1]. S, V, and are returned on [0, 1]
	float v;
	bool hasHue;

	// Check RGB bounds
	__BoundsChk( R )
	__BoundsChk( G )
	__BoundsChk( B )

	// Calc val (the component MAX)
	if ( R >= G && R >= B )
		v = R;
	else if ( G >= R && G >= B )
		v = G;
	else
		v = B;

	outV = v;

	// See if we have black
	if ( v <= 0 ) {
		outH = 0;
		outS = 0;

		return;
	}

	// If we have an undefined hue (ie, a grey)
	if ( R == G && R == B && G == B ) {
		outH = 0;
		hasHue = false; }
	else
		hasHue = true;


	// Is R, G, or B the min?
	if ( R <= G && R <= B ) {
		if ( hasHue )
			outH = .5 - ( G - B ) / ( ( v - R ) * 6.0 );
		outS = ( v - R ) / v; }
	else if ( G <= R && G <= B ) {
		if ( hasHue )
			outH = .8333333333 - ( B - R ) / ( ( v - G ) * 6.0 );
		outS = ( v - G ) / v; }
	else {
		if ( hasHue )
			outH = .1666666666 - ( R - G ) / ( ( v - B ) * 6.0 );
		outS = ( v - B ) / v;
	}
}


/*
#define RETURN_HSV(h, w, b) {HSV.H = h; HSV.S = s; HSV.V = v; return HSV;}

#define RETURN_RGB(r, g, b) {RGB.R = r; RGB.G = g; RGB.B = b; return RGB;}

#define UNDEFINED -1

// Theoretically, hue 0 (pure red) is identical to hue 6 in these transforms. Pure
// red always maps to 6 in this implementation. Therefore UNDEFINED can be
// defined as 0 in situations where only unsigned numbers are desired.

typedef struct {float R, G, B;} RGBType;

typedef struct {float H, S, V;} HSVType;

HSVType

RGB_to_HSV( RGBType RGB ) {

            // RGB are each on [0, 1]. S and V are returned on [0, 1] and H is
            // returned on [0, 6]. Exception: H is returned UNDEFINED if S==0.
            float R = RGB.R, G = RGB.G, B = RGB.B, v, x, f;
            int i;
            HSVType HSV;
            x = min(R, G, B);
            v = max(R, G, B);
            if(v == x) RETURN_HSV(UNDEFINED, 0, v);
            f = (R == x) ? G - B : ((G == x) ? B - R : R - G);
            i = (R == x) ? 3 : ((G == x) ? 5 : 1);
            RETURN_HSV(i - f /(v - x), (v - x)/v, v);


}


RGBType

HSV_to_RGB( HSVType HSV ) {

           // H is given on [0, 6] or UNDEFINED. S and V are given on [0, 1].
           // RGB are each returned on [0, 1].
           float h = HSV.H, s = HSV.S, v = HSV.V, m, n, f;
           int i;
           RGBType RGB;
           if(h == UNDEFINED) RETURN_RGB(v, v, v);
           i = floor(h);
           f = h - i;
           if(!(i & 1)) f = 1 - f; // if i is even
           m = v * (1 - s);
           n = v * (1 - s * f);
           switch (i) {
                     case 6:
                     case 0: RETURN_RGB(v, n, m);
                     case 1: RETURN_RGB(n, v, m);
                     case 2: RETURN_RGB(m, v, n)
                     case 3: RETURN_RGB(m, n, v);
                     case 4: RETURN_RGB(n, m, v);
                     case 5: RETURN_RGB(v, m, n);
           }


}
*/
