
#include "MemFileSys.h"

#include "XLongList.h"
#include "MemFileSysDir.h"
#include "EgOSUtils.h"

#include "CEgOStream.h"
#include "CEgIStream.h"

MemFileSys::MemFileSys( bool inUniqueTitles ) :
	GeneralFileSys( inUniqueTitles ),
	mSys() {

	mNextID = 55;

	MemFileItem* root = new MemFileItem();
	root -> mFlags = FILE_SYS_FOLDER;
	root -> mDir = new MemFileSysDir;
	root -> mName = nil;
	root -> mParent = FILE_SYS_INVALID_ID;

	mSys.PutOwned( FILE_SYS_ROOT_ID, root );

#ifdef EG_DEBUG
	mItemCount = 1;
#endif
}

MemFileSys::~MemFileSys() {

	internalDelete( FILE_SYS_ROOT_ID, false );

	// Halt the program if a memory leak is detected
#ifdef EG_DEBUG
	if ( mItemCount != 0 ) {
		 long x = *((long*) nil );
	}
#endif
}

FileResult MemFileSys::Catalog( FileObj inParentID, XLongList& outList ) {

	MemFileItem*	parent;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( GetFolder( inParentID, &parent ) ) {

		parent -> mDir -> Catalog( outList );
		return FILE_SYS_SUCCESS;
	}

	return FILE_SYS_FAIL;
}

bool MemFileSys::GetFolder( FileObj inFolder, MemFileItem** outFolder ) {

	bool ok = false;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( mSys.Get( inFolder, (void**) outFolder ) ) {

		// If we don't have a folder, return an error otherwise, procees to the sub head
		if ( (*outFolder) -> mFlags & FILE_SYS_FOLDER )
			ok = true;
		else
			ThrowErr( FILE_SYS_FOLDER_NEEDED, inFolder );
	}
	else
		ThrowErr( FILE_SYS_FNF, inFolder, "Folder not found" );

	return ok;
}

FileResult MemFileSys::GetItemID( FileObj inParentID, const UtilStr& inName, FileObj& outID ) {

	MemFileItem*	parent;
	long		result;
	bool		ok = false;

	outID = FILE_SYS_INVALID_ID;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( GetFolder( inParentID, &parent ) ) {

		result = parent -> mDir -> LookupName( &inName, outID );

		// Was just one title found?
		if ( result == MEM_FILE_SYS_UNIQUE )
			ok = true;
		else {
			UtilStr msg( "\"" );
			msg.Append( inName );

			// Was there more than one item with that title? Note: outID contains one of the items with that title
			if ( result == MEM_FILE_SYS_DUPE ) {
				msg.Append( "\" already exists" );
				ThrowErr( FILE_SYS_AMBIGUOUS_NAME, inParentID, msg.getCStr() );
			}

			// Or was the title not found?  :^(
			else  {
				msg.Append( "\" not found" );

				ThrowErr( FILE_SYS_FNF, inParentID, msg.getCStr() );
			}
		}
	}

	return ( ok ) ? FILE_SYS_SUCCESS : FILE_SYS_FAIL;
}

const UtilStr* MemFileSys::Read( FileObj inID ) {

	MemFileItem*	item;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( mSys.Get( inID, (void**) &item ) ) {
		return &item -> mData;
	}
	else
		ThrowErr( FILE_SYS_FNF, inID );

	return nil;
}

void* MemFileSys::Get( FileObj inID ) {

	MemFileItem*	item;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( mSys.Get( inID, (void**) &item ) ) {
		return item -> mData.getCStr();
	}
	else
		ThrowErr( FILE_SYS_FNF, inID );

	return nil;
}

FileResult MemFileSys::Read( FileObj inID, UtilStr& outBuf ) {

	const UtilStr* data = Read( inID );

	outBuf.Assign( data );

	return ( data ) ? FILE_SYS_SUCCESS : FILE_SYS_FAIL;
}

FileResult MemFileSys::WriteData( FileObj inID, void* inData, long inNumBytes ) {

	MemFileItem*	item;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( mSys.Get( inID, (void**) &item ) ) {

		// If we have an item that contains user-readable data, then proceed
		item -> mData.Assign( inData, inNumBytes );
		return FILE_SYS_SUCCESS;   }
	else
		ThrowErr( FILE_SYS_FNF, inID );

	return FILE_SYS_FAIL;
}

bool MemFileSys::TryReplace( MemFileItem* inDestParent, const UtilStr& inName, bool inReplace ) {

	long result;
	FileObj	existingItem;
	bool ok = true;

	// Look for items with identical titles...
	result = inDestParent -> mDir -> LookupName( &inName, existingItem );

	// Does an item with that title already exist?
	if ( result == MEM_FILE_SYS_UNIQUE || result == MEM_FILE_SYS_DUPE ) {

		// If we have permission to replace the existing item, then we replace it (by deleteing it)
		if ( inReplace )
			Delete( existingItem );

		// If we don't have permission to replace and names aren't allowed to be identical, then bail
		else if ( mUniqueNames ) {
			ThrowErr( FILE_SYS_ALREADY_EXISTS, existingItem );
			ok = false;
		}
	}

	return ok;
}

FileResult MemFileSys::Move( FileObj inID, FileObj inNewParent, bool inReplace ) {

	MemFileItem* newParent, *oldParent, *item;

	// If the file and folder there?
	if ( mSys.Get( inID, (void**) &item ) ) {
		if ( GetFolder( inNewParent, &newParent ) ) {

			// Can we proceed to write the file?
			if ( TryReplace( newParent, item -> mName, inReplace ) ) {

				// Tell the new parent about the new item (and its ID) and store the ptr to the name str
				item -> mName = newParent -> mDir -> AddName( item -> mName, inID );

				// Remove the item's title (and ID) from its parent's ID lookup table
				mSys.Get( item -> mParent, (void**) &oldParent );
				oldParent -> mDir -> RemoveName( item -> mName, inID );

				// The item now has a new parent
				item -> mParent = inNewParent;

				return FILE_SYS_SUCCESS;
			}
		}
	}
	else
		ThrowErr( FILE_SYS_FNF, inID );

	return FILE_SYS_FAIL;
}


long MemFileSys::GetInfo( FileObj inID, long inParamID, FileResult* outResult ) {

	MemFileItem* item;
	bool ok = false;
	long outParam = 0;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( mSys.Get( inID, (void**) &item ) ) {

		if (inParamID == MCC4_TO_INT("flag")) {
			outParam = item -> mFlags;
			ok = true;
		}
		else if (inParamID == MCC4_TO_INT("size")) {
			outParam = item -> mData.length();
			ok = true;
		}
		else if (inParamID == MCC4_TO_INT("subs")) {
			outParam = 0;
			if ( item -> mFlags & FILE_SYS_FOLDER  )
				outParam = item -> mDir -> NumEntries();
		}
		else if (inParamID == MCC3_TO_INT("ext")) {
			outParam = EgOSUtils::FindExtension( item -> mName );
		}
		else {
			ThrowErr( FILE_SYS_PARAM_NOT_FOUND, inID );
		}
	}
	else
		ThrowErr( FILE_SYS_FNF, inID );


	if ( outResult )
		*outResult = ok ? FILE_SYS_SUCCESS : FILE_SYS_FAIL;

	return outParam;
}

FileResult MemFileSys::SetInfo( FileObj inID, long inParamID, long inParam ) {

	MemFileItem* item;
	bool ok = false;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( mSys.Get( inID, (void**) &item ) ) {

		if (inParamID == MCC4_TO_INT("flag")) {
			item -> mFlags = item -> mFlags & ( ~ FILE_SYS_USER_ACCESSABLE_FLAGS );
			item -> mFlags |= inParam & FILE_SYS_USER_ACCESSABLE_FLAGS;
			ok = true;
		}
		else {
			ThrowErr( FILE_SYS_PARAM_NOT_FOUND, inID );
		}
	}
	else
		ThrowErr( FILE_SYS_FNF, inID );

	return ok ? FILE_SYS_SUCCESS : FILE_SYS_FAIL;
}

FileResult MemFileSys::Create( FileObj inParentID, const UtilStr& inName, FileObj& outID, long inFlags ) {

	MemFileItem*	parent, *item;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( GetFolder( inParentID, &parent ) ) {

		if ( TryReplace( parent, inName, inFlags & FILE_SYS_REPLACE ) ) {

			// Make a new item...
			item = new MemFileItem;
			item -> mParent = inParentID;

			// Tell the parent about the new item (and its ID)
			outID = mNextID;	mNextID++;
			item -> mName = parent -> mDir -> AddName( &inName, outID );

			// Make a unique ID in MemFileSys
			mSys.PutOwned( outID, item );

			// Apply flags...
			if ( inFlags & FILE_SYS_FOLDER ) {
				item -> mFlags = FILE_SYS_FOLDER | FILE_SYS_DATA;
				item -> mDir = new MemFileSysDir;  }
			else
				item -> mFlags = FILE_SYS_DATA;
			if ( inFlags & FILE_SYS_INVISIBLE )
				item -> mFlags |= FILE_SYS_INVISIBLE;

#if EG_DEBUG
			mItemCount++;
#endif

			return FILE_SYS_SUCCESS;
		}
	}

	// Return the existing item if it's a already_exists error
	if ( mLastErr.mErrorID == FILE_SYS_ALREADY_EXISTS )
		outID = mLastErr.mItemID;
	else
		outID = FILE_SYS_INVALID_ID;

	return FILE_SYS_FAIL;
}

FileResult MemFileSys::Delete( FileObj inID ) {

	// Don't let the user delete the root.  Stupid users.
	if ( inID != FILE_SYS_ROOT_ID )
		return internalDelete( inID, true );

	ThrowErr( FILE_SYS_ROOT_LOCKED, inID );

	return FILE_SYS_FAIL;
}

FileResult MemFileSys::DeleteContents( FileObj inFolder ) {

	MemFileItem* folder;

	if ( GetFolder( inFolder, &folder ) ) {

		// Delete everything inside the item
		if ( folder -> mDir )
			folder -> mDir -> RemoveAll( this );

		return FILE_SYS_SUCCESS;
	}

	return FILE_SYS_FAIL;
}

FileResult MemFileSys::internalDelete( FileObj inID, bool inUpdateParent ) {

	MemFileItem*	item, *parent;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( mSys.Get( inID, (void**) &item ) ) {

		// Delete everything inside the item
		if ( item -> mDir )
			item -> mDir -> RemoveAll( this );

		if ( inUpdateParent ) {

			// Remove the item's title (and ID) from its parent's ID lookup table
			mSys.Get( item -> mParent, (void**) &parent );
			parent -> mDir -> RemoveName( item -> mName, inID );
		}

		// Remove the item's ID from MemFileSys's ID lookup table (since the item is owned, the item is also deleted )
		mSys.Remove( inID );

		#if EG_DEBUG
		mItemCount--;
		#endif

		return FILE_SYS_SUCCESS;
	}

	ThrowErr( FILE_SYS_FNF, inID );

	return FILE_SYS_FAIL;
}

FileResult MemFileSys::GetParent( FileObj inID, FileObj& outParent ) {

	MemFileItem* item;

	// The root has no parent
	if ( inID == FILE_SYS_ROOT_ID )
		ThrowErr( FILE_SYS_NO_ROOT_PARENT, inID );

	// Use the hashtable to fetch the MemFileItem in O(1) time
	else if ( mSys.Get( inID, (void**) &item ) ) {

		outParent = item -> mParent;
		return FILE_SYS_SUCCESS; }
	else
		ThrowErr( FILE_SYS_FNF, inID );

	outParent = FILE_SYS_INVALID_ID;
	return FILE_SYS_FAIL;
}

FileResult MemFileSys::SetName( FileObj inID, const UtilStr& inName ) {

	MemFileItem* parent, *item;
	long result;
	FileObj existingItem;

	// If item and parent info...
	if ( mSys.Get( inID, (void**) &item ) ) {
		if ( mSys.Get( item -> mParent, (void**) &parent ) ) {

			// If dupe names aren't allowed, check for an existing item the new name we want
			if ( mUniqueNames ) {
				result = parent -> mDir -> LookupName( item -> mName, existingItem );

				// Does an item with that title already exist?  Bail if so.
				if ( result == MEM_FILE_SYS_UNIQUE ) {
					ThrowErr( FILE_SYS_ALREADY_EXISTS, existingItem );
					return FILE_SYS_FAIL;
				}
			}

			// Remove the old name from the same parent's name table
			parent -> mDir -> RemoveName( item -> mName, inID );

			// Add the new name to the parents name table and save the ptr to the allocated name
			item -> mName = parent -> mDir -> AddName( &inName, inID );

			return FILE_SYS_SUCCESS;
		} }
	else
		ThrowErr( FILE_SYS_FNF, inID );

	return FILE_SYS_FAIL;
}

const UtilStr* MemFileSys::GetName( FileObj inID ) {

	MemFileItem *item;

	if ( mSys.Get( inID, (void**) &item ) )
		return item -> mName;
	else
		ThrowErr( FILE_SYS_FNF, inID );

	return nil;
}

void MemFileSys::WriteToStream( FileObj inID, CEgOStream* inOStream ) {

	MemFileItem*	item;
	long num, i, flags = 0;

	// Use the hashtable to fetch the MemFileItem in O(1) time
	if ( mSys.Get( inID, (void**) &item ) ) {

		flags = item -> mFlags;
		inOStream -> PutLong( inID );
		inOStream -> PutLong( flags );
		inOStream -> PutLong( item -> mParent );

		if ( inID != FILE_SYS_ROOT_ID ) {
			item -> mName -> WriteTo( inOStream );
			item -> mData.WriteTo( inOStream );
		}

		if ( flags & FILE_SYS_FOLDER ) {
			XLongList catalog;

			if ( Catalog( inID, catalog ) ) {
				num = catalog.Count();
				inOStream -> PutLong( num );
				for ( i = 1; i <= num && inOStream -> noErr(); i++ ) {
					WriteToStream( catalog.Fetch( i ), inOStream );
				}
			}
		} }
	else
		ThrowErr( FILE_SYS_FNF, inID );
}

void MemFileSys::ReadFromStream( CEgIStream* inStream ) {

	DeleteContents( FILE_SYS_ROOT_ID );

	ReadFromStream_Internal( inStream );
}

void MemFileSys::ReadFromStream_Internal( CEgIStream* inStream ) {

	MemFileItem*	item;
	long flags, num, i = mNextID;
	FileObj itemID;
	UtilStr name;

	mNextID 	= inStream -> GetLong();
	flags 		= inStream -> GetLong();
	itemID 		= inStream -> GetLong();

	// This is only true if we're corrupt or damaged
	if ( mNextID < FILE_SYS_ROOT_ID ) {
		mNextID = i;
		flags = 0; }

	else if ( mNextID > FILE_SYS_ROOT_ID ) {

		// Create the memfile item in this file sys
		name.ReadFrom( inStream );
		Create( itemID, name, itemID, flags | FILE_SYS_REPLACE );

		// Use the hashtable to fetch the MemFileItem in O(1) time
		if ( mSys.Get( itemID, (void**) &item ) ) {

			item -> mFlags = flags;
			item -> mData.ReadFrom( inStream );
		}
	}

	if ( flags & FILE_SYS_FOLDER ) {
		num = inStream -> GetLong();

		for ( i = 0; i < num && inStream -> noErr(); i++ ) {
			ReadFromStream_Internal( inStream );
		}
	}
}

MemFileItem::MemFileItem() {

	mParent = 0;
	mFlags = 0;
	mDir = nil;
}

MemFileItem::~MemFileItem() {

	// We allocate these on our own so we're responsible for them
	if ( mDir )
		delete mDir;
}
