/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2003 Nicolas Adenis-Lamarre (aspegic500@free.fr)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "displayer_xosd_status.h"
#include <stdlib.h>
#include <string.h>

#include "displayer_xosd_config.h"

// Since the sdl_geforce is a singleton, keep this global static object
static SingitSingleton       *xosd_status = NULL;
static SingitSingletonClass  *parent_class = NULL;

static void xosd_status_class_init (XosdStatusClass *klass);
static void xosd_status_init (XosdStatus *sstat);
static void xosd_status_destroy(GtkObject *object);
static gboolean xosd_status_after_init(SingitSingleton *single);

GtkType xosd_status_get_type (void)
{
	static GtkType xosd_status_type = 0;

	if (!xosd_status_type) {

		static const GtkTypeInfo xosd_status_info =
		{
			(gchar*) "XosdStatus",
			sizeof (XosdStatus),
			sizeof (XosdStatusClass),
			(GtkClassInitFunc) xosd_status_class_init,
			(GtkObjectInitFunc) xosd_status_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		xosd_status_type = gtk_type_from_name("XosdStatus");
		if ((xosd_status_type == GTK_TYPE_NONE) ||
			(xosd_status_type == GTK_TYPE_INVALID))
		{
			xosd_status_type = gtk_type_unique
				(TYPE_SINGIT_SINGLETON, &xosd_status_info);
		}
		else { parent_class = gtk_type_class(TYPE_SINGIT_SINGLETON); }
	}

	return xosd_status_type;
}

static void xosd_status_class_init (XosdStatusClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	SingitSingletonClass *singleton_class =
		(SingitSingletonClass*) klass;

	parent_class = gtk_type_class(TYPE_SINGIT_SINGLETON);

	object_class->destroy = xosd_status_destroy;
	singleton_class->after_init = xosd_status_after_init;
}

static void xosd_status_init(XosdStatus *sstat)
{
	sstat->m_song=NULL;

	sstat->m_osd = NULL;

	sstat->m_config = NULL;

	sstat->displayed_line = NULL;
}

static void xosd_status_destroy(GtkObject *object)
{
	XosdStatus *sstat;

	g_return_if_fail(object != NULL);
	g_return_if_fail(IS_XOSD_STATUS (object));

	sstat = XOSD_STATUS(object);

	xosd_destroy(sstat->m_osd);

	if (sstat->displayed_line)
		g_free(sstat->displayed_line);

	singit_song_detach(&sstat->m_song);

	singit_config_gen_detach(&sstat->m_config);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

static gboolean xosd_status_after_init(SingitSingleton *single)
{
	g_return_val_if_fail(single != NULL, FALSE);
	g_return_val_if_fail(IS_XOSD_STATUS(single), FALSE);

	if ((XOSD_STATUS(single)->m_osd = xosd_create(2)) == NULL) {
		free(XOSD_STATUS(single)->m_osd);
		return FALSE;
	}

	XOSD_STATUS(single)->m_config =
		displayer_xosd_config_new();
	singit_config_gen_load(XOSD_STATUS(single)->m_config);

	if (XOSD_STATUS(single)->m_config == NULL) {
		free(XOSD_STATUS(single)->m_osd);
	}

	return (XOSD_STATUS(single)->m_config != NULL);
}

SingitSingleton* xosd_status_ref(void)
{
	gboolean init;
	SingitSingleton *status = singit_singleton_construct
		(&xosd_status, TYPE_XOSD_STATUS, &init);
	if (init == TRUE)
		{ return status; }
	return singit_singleton_attach(status);
}

void xosd_status_unref(SingitSingleton** single)
{
	g_return_if_fail(xosd_status != NULL);

	if (single == NULL) {
		SingitSingleton* unref_single = xosd_status;
		singit_singleton_detach(&unref_single);
	}
	else {
		g_return_if_fail(*single == xosd_status);
		singit_singleton_detach(single);
	}
}

SingitSingleton* xosd_status_noref(void)
{
	if (xosd_status == NULL)
		{ return NULL; }

	g_return_val_if_fail(IS_XOSD_STATUS(xosd_status), NULL);

	return xosd_status;
}

int displayer_xosd_status_set_color(XosdStatus *osd, char *color) {
  char color_tmp[8]; /* rgb:xx/xx/xx -> #xxxxxx */

  color_tmp[0]='#';
  color_tmp[1]=color[4];
  color_tmp[2]=color[5];
  color_tmp[3]=color[7];
  color_tmp[4]=color[8];
  color_tmp[5]=color[10];
  color_tmp[6]=color[11];
  color_tmp[7]='\0';

  return xosd_set_colour(STATUS->m_osd, color_tmp);
}

inline int displayer_xosd_status_get_colour (XosdStatus *osd, int *red, int *green, int *blue){
  return xosd_get_colour(STATUS->m_osd, red, blue, green);
}

inline int displayer_xosd_status_set_font(XosdStatus *osd, char *font){
  return xosd_set_font(STATUS->m_osd, font);
}

char* displayer_xosd_status_get_font(XosdStatus *osd){
  char *font;

  if(xosd_get_font(STATUS->m_osd, &font) != 0)
    return NULL;

  return font;
}

inline int displayer_xosd_status_show(XosdStatus *osd){
  return xosd_show(STATUS->m_osd);
}

inline int displayer_xosd_status_hide(XosdStatus *osd){
  return xosd_hide(STATUS->m_osd);
}

int displayer_xosd_status_toggle(XosdStatus *osd){
  if(xosd_is_onscreen(STATUS->m_osd) == 0)
    return xosd_show(STATUS->m_osd);
  else
    return xosd_hide(STATUS->m_osd);
}

inline SingitSong* displayer_xosd_status_get_song(XosdStatus *osd){
  return STATUS->m_song;
}

inline void displayer_xosd_status_set_song(XosdStatus *osd, SingitSong *song){
  STATUS->m_song=song;
}

inline int displayer_xosd_status_get_alignement(XosdStatus *osd, xosd_align *align){
  return xosd_get_align(STATUS->m_osd, align);
}

inline int displayer_xosd_status_set_alignement(XosdStatus *osd, xosd_align align){
  return xosd_set_align(STATUS->m_osd, align);
}

inline int displayer_xosd_status_get_position(XosdStatus *osd, xosd_pos *position){
  return xosd_get_pos(STATUS->m_osd, position);
}

inline int displayer_xosd_status_set_position(XosdStatus *osd, xosd_pos position){
  return xosd_set_pos (STATUS->m_osd, position);
}

inline int displayer_xosd_status_set_v_offset(XosdStatus *osd, int offset){
  return xosd_set_vertical_offset (STATUS->m_osd, offset);
}

inline int displayer_xosd_status_set_h_offset(XosdStatus *osd, int offset){
  return xosd_set_horizontal_offset (STATUS->m_osd, offset);
}

inline int displayer_xosd_status_set_shadow_offset(XosdStatus *osd, int offset){
  return xosd_set_shadow_offset (STATUS->m_osd, offset);
}

inline int displayer_xosd_status_update(XosdStatus *osd)
{
	SingitSong *cur_song =
		singit_song_attach(displayer_xosd_status_get_song(osd));
	gchar *cur_line;
	gint result = 0;
	gboolean update = FALSE;

	// Most of the following stuff should be handled inside xosd
	// The xosd shouldn't recreate everything, if the displayed
	// string didn't really change
	// Since I didn't investigated all code, I check
	// if the string changed before setting a new one
	if (cur_song != NULL) {
		if (cur_song->active_token != NULL) {
			cur_line = tText(cur_song, cur_song->active_token);
			if (!(STATUS->displayed_line &&
				strcmp(cur_line, STATUS->displayed_line) == 0))
			{
				if (STATUS->displayed_line)
					g_free(STATUS->displayed_line);
				STATUS->displayed_line = g_strdup(cur_line);
				update = TRUE;
			}
		}
		else if (STATUS->displayed_line) {
			g_free(STATUS->displayed_line);
			STATUS->displayed_line = NULL;
			update = TRUE;
		}

		if (update == TRUE)
			result = xosd_display(STATUS->m_osd, 0,
				XOSD_string_extended, STATUS->displayed_line, 0, -1);

		singit_song_detach(&cur_song);
	}
	else {
		if (STATUS->displayed_line) {
			g_free(STATUS->displayed_line);
			STATUS->displayed_line = NULL;
		}
	}

	return (result == 0) ? 0 : -1;
}
