/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>

#include "singit_debug.h"
#include "singit/macros.h"
#include "singit_tools.h"

#include "singit_tag_manager.h"
#include "mpeg_header_info.h"

#include "singit_tag_manager_private.h"
#include "singit_id3lib_wrapper.h"

static GtkWidget *mpeg_file_info_values_label;

static void fill_mpeg_info_page(MPEGHeaderInfo *info)
{
	static const gchar *channel_mode_str[] = {
		N_("Stereo"),
		N_("Joint stereo"),
		N_("Dual channel"),
		N_("Single channel")
	};

	static const gchar *bool_str[] = {
		N_("no"),
		N_("yes")
	};

	static const gchar *emphasis_str[] = {
		N_("None"),
		N_("50/15 ms"),
		"",
		N_("CCIT J.17")
	};

	gchar *cat_str = NULL, *append_str = NULL, *temp_str = NULL;

	if (info->mpeg_25) { cat_str = g_strdup("\n2.5\n"); }
	else { cat_str = g_strdup_printf("\n%i\n", info->mpeg_version); }

	append_str = g_strdup_printf("%i\n\n%i\n%i Bytes\n%i:%i\n\n%i kB/s\n%i Hz\n", info->mpeg_layer,
		info->frames, info->filesize, info->length / 60000, info->length / 1000 % 60,
		info->bitrate, info->samplerate);

	temp_str = cat_str;
	cat_str = g_strconcat(temp_str, append_str, NULL);
	g_free(append_str);
	g_free(temp_str);

	temp_str = cat_str;
	if (info->channel_mode < 0 || info->channel_mode > 3) { append_str = g_strdup(""); }
	else { append_str = g_strdup(_(channel_mode_str[info->channel_mode])); }
	cat_str = g_strconcat(temp_str, append_str, "\n", NULL);
	g_free(append_str);
	g_free(temp_str);

	temp_str = cat_str;
	if (info->emphasis < 0 || info->emphasis > 3) { append_str = g_strdup(""); }
	else { append_str = g_strdup(_(emphasis_str[info->emphasis])); }
	cat_str = g_strconcat(temp_str, append_str, "\n\n", NULL);
	g_free(append_str);
	g_free(temp_str);

	append_str = g_strdup_printf("%s\n%s\n%s",
		(info->error_protection != FALSE) ? _(bool_str[1]) : _(bool_str[0]),
		(info->copyright != FALSE) ? _(bool_str[1]) : _(bool_str[0]),
		(info->original != FALSE) ? _(bool_str[1]) : _(bool_str[0]));

	temp_str = cat_str;
	cat_str = g_strconcat(temp_str, append_str, NULL);
	g_free(append_str);
	g_free(temp_str);

	gtk_label_set_text(GTK_LABEL(mpeg_file_info_values_label), cat_str);
	g_free(cat_str);
}

ID3Tag* get_mpeg_file_info(gchar *file_name,
	gboolean *is_writeable, MPEGHeaderInfo **info)
{
	ID3Tag *id3tag;

	g_return_val_if_fail(file_name != NULL, NULL);
	g_return_val_if_fail(info != NULL, NULL);

	if (!mpeg_header_info_detect_by_content(file_name))
		{ return NULL; }

	id3tag = ID3Tag_New();
	ID3Tag_Link_WRP(id3tag, file_name);

	(*info) = mpeg_header_info_new(file_name);

	return id3tag;
}

void append_mpeg_info_page(GtkWidget *notebook,
	GtkWidget *window, ID3Tag *id3, MPEGHeaderInfo *info)
{
	GtkWidget *mpeg_file_info_table, *mpeg_file_info_label,
		*mpeg_file_info_properties_label;

	gchar *cat_str = NULL;

	g_return_if_fail(notebook != NULL);
	g_return_if_fail(window != NULL);
	g_return_if_fail(id3 != NULL);
	g_return_if_fail(info != NULL);

	g_return_if_fail(GTK_IS_NOTEBOOK(notebook));
	g_return_if_fail(GTK_IS_WINDOW(window));
	g_return_if_fail(IS_MPEG_HEADER_INFO(info));

	mpeg_file_info_table = gtk_table_new (1, 2, FALSE);
	create_std_wgt_description(mpeg_file_info_table,
			"mpeg_file_info_table", TRUE, window);
	gtk_container_set_border_width (GTK_CONTAINER (mpeg_file_info_table), 1);
	gtk_table_set_row_spacings (GTK_TABLE (mpeg_file_info_table), 3);
	gtk_table_set_col_spacings (GTK_TABLE (mpeg_file_info_table), 2);

	mpeg_file_info_label = gtk_label_new (_("file information"));
	create_std_wgt_description(mpeg_file_info_label,
			"mpeg_file_info_label", TRUE, window);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), mpeg_file_info_table, mpeg_file_info_label);

	cat_str = g_strconcat("MPEG\n    ", _("Version"), ":\n    ",
		_("Layer"), ":\n\n", _("Frames"), ":\n", _("Filesize"), ":\n",
		_("Length"), ":\n\n", _("Bit rate"), ":\n", _("Sample rate"),
		":\n", _("Channel mode"), ":\n", _("Emphasis"), ":\n\n",
		_("Error protection (CRC)"), ":\n", _("Copyright"), ":\n",
		_("Original"), ":",  NULL);

	mpeg_file_info_properties_label = gtk_label_new(cat_str);
	gtk_label_set_justify(GTK_LABEL(mpeg_file_info_properties_label), GTK_JUSTIFY_LEFT);
	g_free(cat_str);
	create_std_wgt_description(mpeg_file_info_properties_label,
			"mpeg_file_info_properties_label", TRUE, window);
	gtk_table_attach(GTK_TABLE (mpeg_file_info_table),
		mpeg_file_info_properties_label, 0, 1, 0, 1,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 2);

	mpeg_file_info_values_label = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(mpeg_file_info_values_label), GTK_JUSTIFY_LEFT);
	create_std_wgt_description(mpeg_file_info_values_label,
		"mpeg_file_info_values_label", TRUE, window);
	gtk_table_attach(GTK_TABLE (mpeg_file_info_table),
		mpeg_file_info_values_label, 1, 2, 0, 1,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 2);

	fill_mpeg_info_page(info);
}
