/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "singit_debug.h"

#include "singit/karaoke_data.h"
#include "singit_tools.h"

#include "singit_marshallers.h"

#define left_right_line_border ((skd->ball_diameter / 2) + 1)

/*
 * Uncomment to enable extended debug messages
 */
// #define SKD_DEBUG

/*
 * Referesh enums for the calc_karaoke_widget_sizes funtion
 * They are used to set the refresh state (singit_karaoke_data_refresh)
 */
enum {

	SONG_CHG = 1 << 1,	// Song has changed
	FONT_CHG = 1 << 2,	// Font has changed
	LINE_CHG = 1 << 3,	// No of lines has changed
	OPTM_CHG = 1 << 4,	// Any data for font optimization changed
};

static void calc_karaoke_widget_sizes(SingitKaraokeData *skd);
static void singit_karaoke_data_optimize_font_int(SingitKaraokeData *skd);
static void singit_karaoke_data_update_ball(SingitKaraokeData *skd, GdkRectangle *main_area);
static void singit_karaoke_data_update_progess_bar(SingitKaraokeData *skd, gchar *text, GdkRectangle *main_area);

enum {
	RENDER,
	OPTIMIZE,
	EXPOSE,
	NEW_BALL,
	NEW_VISUAL,
	LAST_SIGNAL
};

static GtkObjectClass *parent_class = NULL;
static guint skd_signals[LAST_SIGNAL] = { 0 };

static void singit_karaoke_data_class_init (SingitKaraokeDataClass *klass);
static void singit_karaoke_data_init (SingitKaraokeData *skd);

static void singit_karaoke_data_destroy(GtkObject *object);

static void str_line_info_free(StrLineInfo* sli);

GtkType singit_karaoke_data_get_type (void)
{
	static GtkType singit_karaoke_data_type = 0;

	if (!singit_karaoke_data_type) {

		static const GtkTypeInfo singit_karaoke_data_info =
		{
			(gchar*) "SingitKaraokeData",
			sizeof (SingitKaraokeData),
			sizeof (SingitKaraokeDataClass),
			(GtkClassInitFunc) singit_karaoke_data_class_init,
			(GtkObjectInitFunc) singit_karaoke_data_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		singit_karaoke_data_type = gtk_type_unique(GTK_TYPE_OBJECT, &singit_karaoke_data_info);
	}

	return singit_karaoke_data_type;
}

static void singit_karaoke_data_class_init (SingitKaraokeDataClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class(GTK_TYPE_OBJECT);

	skd_signals[RENDER] =
		gtk_signal_new("render",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitKaraokeDataClass, render),
			gtk_marshal_INT__POINTER_POINTER,
			GTK_TYPE_INT, 2,
			GTK_TYPE_POINTER,
			GTK_TYPE_POINTER);

	skd_signals[OPTIMIZE] =
		gtk_signal_new("optimize",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitKaraokeDataClass, optimize),
			gtk_marshal_POINTER__INT_INT,
			GTK_TYPE_POINTER, 2,
			GTK_TYPE_INT,
			GTK_TYPE_INT);

	skd_signals[EXPOSE] =
		gtk_signal_new("expose",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitKaraokeDataClass, expose),
			gtk_marshal_BOOL__POINTER_INT_INT,
			GTK_TYPE_BOOL, 3,
			GTK_TYPE_POINTER,
			GTK_TYPE_INT,
			GTK_TYPE_INT);

	skd_signals[NEW_BALL] =
		gtk_signal_new("new_ball",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitKaraokeDataClass, new_ball),
			gtk_marshal_NONE__INT,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_INT);

	skd_signals[NEW_VISUAL] =
		gtk_signal_new("new_visual",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitKaraokeDataClass, new_visual),
			gtk_marshal_NONE__INT_INT,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_INT,
			GTK_TYPE_INT);

	klass->render = NULL;
	klass->optimize = NULL;
	klass->expose = NULL;
	klass->new_ball = NULL;
	klass->new_visual = NULL;

	gtk_object_class_add_signals (object_class, skd_signals, LAST_SIGNAL);

	object_class->destroy = singit_karaoke_data_destroy;
}

static void singit_karaoke_data_init (SingitKaraokeData *skd)
{
#ifdef SKD_DEBUG
	g_print("skd_init\n");
#endif

	skd->song = NULL;

	// Progress bar positions (pbp)
	skd->pbp_start = 0;
	skd->pbp_start_last = 0;
	skd->pbp_offset = 0;
	skd->pbp_offset_last = 0;
	skd->pbp_offset_max = 0;

	// Ball y positions
	skd->ball_y_pos = 0;
	skd->ball_y_pos_last = 0;
	skd->ball_start = 0;
	skd->ball_offset_max = 0;

	skd->lines = 3;
	skd->top_lines = 1;
	skd->current = NULL;

	skd->use_ball = TRUE;

	skd->max_line_nr = -1;
	skd->max_line_pixel = -1;

	skd->show_empty_lines = FALSE;
	skd->last_time = -1;
	skd->line_seperator_high = 5;
	skd->active_line_seperator_high = skd->line_seperator_high * 2;
	skd->line_info = NULL;

	skd->font = NULL;
	skd->centerLines = FALSE;

	skd->freezers = 0;
	skd->thaw_refresh = 0;
}

GtkObject *singit_karaoke_data_new(void)
{
	GtkObject *karaoke_data_object;

#ifdef SKD_DEBUG
	g_print("skd_new\n");
#endif

	karaoke_data_object = gtk_type_new(TYPE_SINGIT_KARAOKE_DATA);

	return karaoke_data_object;
}

static void singit_karaoke_data_destroy (GtkObject *object)
{
	SingitKaraokeData *skd;

#ifdef SKD_DEBUG
	g_print("skd_destroy\n");
#endif

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_DATA (object));

	skd = SINGIT_KARAOKE_DATA (object);

	singit_song_detach(&skd->song);

	if (skd->line_info != NULL)
		{ str_line_info_free(skd->line_info); }

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy (object);
}

static StrLineInfo* str_line_info_new
	(gchar *text, gint pixel_len, gint word_count)
{
	StrLineInfo *result;

#ifdef SKD_DEBUG
	g_print("str_line_info_new\n");
#endif
	result = g_malloc(sizeof(StrLineInfo));

	if (result != NULL) {
		result->text = text;
		result->str_len = strlen(text);
		result->str_pixel = pixel_len;

		if (word_count > 0) {
			result->word_pixel = g_malloc(sizeof(gint) * (word_count + 1));
			result->word_pixel[word_count] = 0;
			result->start_pixel = g_malloc(sizeof(gint) * (word_count + 1));
			result->start_pixel[word_count] = pixel_len;
			result->start_pixel[0] = 0;

			result->jump_radius = g_malloc(sizeof(gdouble) * word_count);
			result->jump_radius[word_count - 1] = 0;
			result->jump_angle = g_malloc(sizeof(gdouble) * word_count);
			result->jump_angle[word_count - 1] = 0;

			result->space_pixel = g_malloc(sizeof(gint) * word_count);
			result->space_pixel[word_count - 1] = 0;
		}
		else {
			result->word_pixel = NULL;
			result->start_pixel = NULL;

			result->jump_radius = NULL;
			result->jump_angle = NULL;

			result->space_pixel = NULL;
		}
	}

	return result;
}

static void str_line_info_free(StrLineInfo* sli)
{
#ifdef SKD_DEBUG
	g_print("str_line_info_free\n");
#endif
	g_return_if_fail(sli != NULL);

	if (sli->word_pixel != NULL) { g_free(sli->word_pixel); }
	if (sli->space_pixel != NULL) { g_free(sli->space_pixel); }
	if (sli->start_pixel != NULL) { g_free(sli->start_pixel); }
	if (sli->jump_radius != NULL) { g_free(sli->jump_radius); }
	if (sli->jump_angle != NULL) { g_free(sli->jump_angle); }

	g_free(sli);
}

/*
 * Check if the StrLineInfo has to be updated
 * Also set the required data, if needed
 */
static inline gboolean str_line_info_need_update(StrLineInfo** sli,
	gchar *text, gchar ***splitted_line, gint *word_count)
{
	g_return_val_if_fail(sli != NULL, FALSE);
	g_return_val_if_fail(splitted_line != NULL, FALSE);
	g_return_val_if_fail(word_count != NULL, FALSE);

	if ((text != NULL) && (strlen(text) > 0)) {
		if ((*sli) != NULL) {
			if (((*sli)->text != NULL) &&
				(strcmp(text, (*sli)->text) == 0))
			{
				return FALSE;
			}
			str_line_info_free(*sli);
			(*sli) = NULL;
		}
	}
	else {
		if ((*sli) != NULL) {
			str_line_info_free(*sli);
			(*sli) = NULL;
		}
		return FALSE;
	}

	(*splitted_line) = g_strsplit(text, " ", 0);
	(*word_count) = lines_count
		((const gchar**) (*splitted_line));

	if (word_count == 0) {
		g_strfreev((*splitted_line));
		if ((*sli) != NULL) {
			str_line_info_free(*sli);
			(*sli) = NULL;
		}
		return FALSE;
	}

	return TRUE;
}

/*
 * Splits the active line into word sizes, so the ball knows where to drop
 * This is needed because of different space sizes between the words
 *
 * Expl.:
 *
 * String: I'm here and there
 * Splits: |I'm |here |and |there|
 *
 * You get 5 start positions, 4 space and word lengths
 * The ball jumps from the start to the middle of the first and second position...
 */
static void singit_karaoke_data_fill_str_line_info
	(SingitKaraokeData *skd, gchar *in_text)
{
	gchar **splitted_line = NULL, *text_dup;
	gint i;
	gint textpos;
	gint word_count, space_count;
	gint pixel_len;

	g_return_if_fail(skd != NULL);

#ifdef SKD_DEBUG
	g_print("skd_fill_str_line_slinfo\n");
#endif

// * Checks for changed lines and if do free or new slinfo creation is required
	if (str_line_info_need_update(&skd->line_info,
		in_text, &splitted_line, &word_count) == FALSE)
	{
		return;
	}

	// Set the line_info to
	skd->line_info = NULL;
	g_return_if_fail(skd->font != NULL);

	space_count = word_count - 1;

// * Prepare new line slinfo
	gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER],
			skd->font, in_text, &pixel_len);

	skd->line_info = str_line_info_new
		(in_text, pixel_len, word_count);

// * Fill word pixel lengths
	i = word_count;
	textpos = skd->line_info->str_len;
	text_dup = g_strdup(in_text);

	while (i > 0) {
		text_dup[textpos] = '\0';

		// get the length of the text if not full length
		gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER],
			skd->font, text_dup, &skd->line_info->start_pixel[i]);

		// make the text one word shorter
		i--;
		textpos -= (strlen(splitted_line[i]));

		// get the length of the word
		gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER],
			skd->font, (gchar*)(text_dup + textpos),
			&skd->line_info->word_pixel[i]);
		textpos--;
	}
	g_strfreev(splitted_line);
	g_free(text_dup);

// * Correct start pixels and fill space pixel lengths
	for (i = 0; i < space_count; i++) {
		skd->line_info->start_pixel[i+1] =
			skd->line_info->start_pixel[i+2] -
			skd->line_info->word_pixel[i+1];

		skd->line_info->space_pixel[i] =
			skd->line_info->start_pixel[i+1] -
			skd->line_info->start_pixel[i] -
			skd->line_info->word_pixel[i];
	}

	gint jump_distance = 0;
	gdouble half_distance;
	gint height;
	gdouble beta;
	gdouble x, rad;

// * Set jump radias and angles *
	for (i = 0; i < word_count; i++) {
		jump_distance = skd->line_info->start_pixel[i+1] -
			skd->line_info->start_pixel[i];

		// * For more information see "docs/karaoke-data" *
		height = MIN((jump_distance - skd->ball_diameter) / 2,
			skd->ball_line_height - skd->ball_diameter);
		height = MAX(height,
			(skd->ball_line_height - skd->ball_diameter) / 2);

		half_distance = jump_distance / 2.0;
		beta = atan(half_distance / height);
		x = hypot(half_distance, height);
		skd->line_info->jump_radius[i] = rad =
			(x / 2.0) / cos(beta);
		skd->line_info->jump_angle[i] =
			acos((rad - height) / rad);
	}

#ifdef SKD_DEBUG
	g_print("ST: ");
	for (i = 0; skd->line_info->start_pixel[i] != pixel_len; i++) {
		g_print("| %.4i ", skd->line_info->start_pixel[i]);
		if (skd->line_info->start_pixel[i+1] != pixel_len)
			g_print("       ");
		else
			g_print("| %.4i", skd->line_info->start_pixel[i+1]);
	}
	g_print("\nWD: ");
	for (i = 0; skd->line_info->word_pixel[i] != 0; i++)
		{ g_print("| %.4i |      ", skd->line_info->word_pixel[i]); }
	g_print("\nSP:  ");
	for (i = 0; skd->line_info->space_pixel[i] != 0; i++)
		{ g_print("      | %.4i |", skd->line_info->space_pixel[i]); }
	g_print("\n");
#endif
#undef SMALLEST_JUMP
}

/*
 * Get the y position of the ball dependant on the word split positions
 * and the x position of the progress bar
 */
static void singit_karaoke_data_set_curr_y_pos(SingitKaraokeData *skd)
{
	guint x_pbp_position;
	gint cur_word_pos;
	gint real_max_height = skd->ball_line_height - skd->ball_diameter;
	StrLineInfo *info = skd->line_info;

	static gint j = 0;
	gint i;

	// If we have no line info set the ball to the ground
	if ((skd->line_info == NULL) ||
		((skd->pbp_start == 0) && (skd->pbp_offset == 0)))
	{
		skd->ball_y_pos = real_max_height;
		return;
	}

	x_pbp_position = skd->pbp_start + skd->pbp_offset;

	// If we are still in th range of the last calculation, skip it
	if ((x_pbp_position < skd->ball_start) ||
		(x_pbp_position > (skd->ball_start + skd->ball_offset_max)))
	{
		i = 0;
		while (1) {
			if (info->jump_radius[i] > 0.0)
				{ j = i; }
			if (info->start_pixel[i] > (gint) x_pbp_position)
			{
				if (i == 0) {
					j = i;
				}
				i--;
				if (i < j)
					j = i;

				skd->ball_start = info->start_pixel[j];

				skd->ball_offset_max =
					info->start_pixel[i+1] -
					skd->ball_start;
				break;
			}
			if (info->start_pixel[i] == info->str_pixel) {
				skd->ball_start = skd->ball_offset_max = 0;
				break;
			}
			i++;
		}
	}

	if (skd->ball_offset_max != 0) {
		cur_word_pos = x_pbp_position - skd->ball_start;

		// Why is there a difference between signed end unsigned??? ;-(
		gdouble beta = info->jump_angle[j] /
			(gint) skd->ball_offset_max *
			((gint) skd->ball_offset_max - 2 * cur_word_pos);

		skd->ball_y_pos = (int)
			rint((1 - cos(beta)) * info->jump_radius[j])
			+ real_max_height -
			((1 - cos(info->jump_angle[j])) * info->jump_radius[j]);
//		g_print("%i / %i / %f / %f\n", i, skd->ball_y_pos,
//			(1 - cos(beta)) * info->jump_radius[j],
//			(1 - cos(info->jump_angle[j])) * info->jump_radius[j]);
	}
	else { skd->ball_y_pos = real_max_height; }
}

/*
 *On variable sized fonts it's really hard to get the right
 * measurement of (multiple) spaces because it's dependant
 * of the prevous and next characters
 *
 * So we have to do some tricks
 */
static gint singit_karaoke_data_get_offset_pos(SingitKaraokeData *skd, gchar *text, gint pos)
{
	gint str_len;
	gint str_pixel;
	gchar *offset_pos;
	gchar old_char;
	gint start_pixel, end_pixel, start_pos, end_pos;

	g_return_val_if_fail(text != NULL, 0);
	g_return_val_if_fail(pos >= 0, 0);

	if (pos == 0)
		{ return 0; }

	str_len = strlen(text);
	gtk_signal_emit(GTK_OBJECT(skd),
		skd_signals[RENDER], skd->font, text, &str_pixel);
	if (pos == str_len)
		{ return str_pixel; }

	// If we have some spaces at the position in the string...
	if ((text[pos-1] == ' ') || (text[pos] == ' '))
	{
		// Find the next non-space character and measure the rest
		end_pos = pos;
		while ((end_pos < str_len) && (text[end_pos] == ' ')) { end_pos++; }
		if (end_pos < str_len) {
			gtk_signal_emit(GTK_OBJECT(skd),
				skd_signals[RENDER], skd->font,
				&text[end_pos], &end_pixel);
		}
		else { end_pixel = 0; }

		// Find the prev non-space character and measure from the start
		start_pos = pos - 1;
		while ((start_pos > 0) && (text[start_pos] == ' ')) { start_pos--; }
		if (start_pos > 0) {
			offset_pos = text + start_pos + 1;
			old_char = offset_pos[0];
			offset_pos[0] = '\0';
			gtk_signal_emit(GTK_OBJECT(skd),
				skd_signals[RENDER],
				skd->font, text, &start_pixel);
			offset_pos[0] = old_char;
		}
		else { start_pixel = 0; }

		// Calculate the percentual position
		if ((end_pos - start_pos) != str_len) {
			str_pixel = start_pixel + (str_pixel - start_pixel - end_pixel) *
				(pos - start_pos) / (end_pos - start_pos);
		}
	}
	else {
		// Do the simple measurement
		offset_pos = text + pos;
		old_char = offset_pos[0];
		offset_pos[0] = '\0';
		gtk_signal_emit(GTK_OBJECT(skd),
			skd_signals[RENDER], skd->font, text, &str_pixel);
		offset_pos[0] = old_char;
	}

	return str_pixel;
}

/*
 * Clear the progress dependant data
 */
static gchar* singit_karaoke_data_reset_progress_data
	(SingitKaraokeData *skd, SingitSong* song, GList *item)
{
	gchar *text;

	g_return_val_if_fail(skd != NULL, NULL);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), NULL);
	g_return_val_if_fail(song != NULL, NULL);

	text = (item != NULL) ? tText(song, item) : NULL;

	skd->pbp_start_last = 0;
	skd->pbp_offset_last = 0;

	skd->ball_y_pos = skd->ball_line_height - skd->ball_diameter;
	skd->ball_y_pos_last = 0;
	skd->ball_start = 0;
	skd->ball_offset_max = 0;

	singit_karaoke_data_fill_str_line_info(skd, text);
	skd->line_offset =
		singit_karaoke_data_calc_line_offset(skd, FALSE, text);
	singit_karaoke_data_set_curr_y_pos(skd);

	return text;
}

/*
 * Calculate the karaoke data dependant on the current time
 */
void singit_karaoke_data_set_time (SingitKaraokeData *skd, guint time)
{
	GList *new_item = NULL, *new_item_next;
	SingitSong *my_song = NULL;
	gchar *text = NULL, *startPos;
	double multi = 0.0;

	// * Calculate the karaoke dependant exposure area *
	GdkRectangle area;
	gboolean new_has_offset, new_next_has_offset = FALSE;
	gint expose = 0;
	gboolean redrawn = FALSE;

	// * First a few security checks *
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));
	g_return_if_fail(skd->font != NULL);

	// * If time didn't change keep old values *
	if (skd->last_time == (gint) time) { return; }
	else { skd->last_time = time; }

	my_song = singit_song_attach(skd->song);
	if (my_song) {
		area.x = 0;
		area.y = 0;
		area.width = 0;
		area.height = 0;

		new_item = my_song->active_token;
		new_item_next = inl_singit_song_get_next_token(my_song);

		// * Do we have a new next token? *
		if ((new_item != NULL) && (new_item != skd->current))
		{
			text = tText(my_song, new_item);
			new_has_offset = (tPos(new_item) > 0);

			// * Does the token next to the new token have an offset? *
			if ((new_item_next != NULL) &&
				(tLine(new_item_next) == tLine(new_item)) &&
				(tPos(new_item_next) != strlen(text)) &&
				(tPos(new_item_next) > 0))
			{
				new_next_has_offset = TRUE;
			}

			// * Do we have any offsets? *
			if (new_has_offset || new_next_has_offset) {
				// Yes
				startPos = text;

				// * Dependant on the offset of the new token get the pbp start *
				if (new_has_offset) {
					startPos += tPos(new_item);
					skd->pbp_start =
						singit_karaoke_data_get_offset_pos(skd, text, tPos(new_item));
				}
				else { skd->pbp_start = 0; }

				// * Dependant on the offset of the next new token get the pbp max offset *
				if (new_next_has_offset) {
					skd->pbp_offset_max =
						singit_karaoke_data_get_offset_pos
						(skd, startPos, tPos(new_item_next) - tPos(new_item));
				}
				else {
					gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER],
						skd->font, startPos, &skd->pbp_offset_max);
				}
			}
			else {
				// * No offsets - start at zero and end at the text length *
				skd->pbp_start = 0;
				skd->ball_y_pos =
					skd->ball_line_height - skd->ball_diameter;
				gtk_signal_emit(GTK_OBJECT(skd),
					skd_signals[RENDER], skd->font,
					text, &skd->pbp_offset_max);
			}
			skd->pbp_offset = 0;

			// * Do we have to redraw the whole widget? *
			if ((skd->current == NULL) || (tLine(new_item) != tLine(skd->current)) ||
				(skd->show_empty_lines == TRUE) ||
				(singit_song_is_empty_item(my_song, skd->current, TRUE) == FALSE))
			{
				expose |= SKD_EXPOSE_ALL;
				area.width = skd->visual_min_width;
				area.height = skd->visual_min_height;
				singit_karaoke_data_reset_progress_data
					(skd, my_song, new_item);
			}
/*			else if (tPos(new_item) != tPos(skd->current)) {
				expose |= SKD_EXPOSE_PROGRESS;
				singit_karaoke_data_reset_progress_data
					(skd, my_song, new_item);
			}*/
			skd->current = new_item;
		}

		// Do we have a current offset
		if (new_item != NULL) {
			// Calculate the current offset
			if (new_item != my_song->last_token) {
				multi = (time - tTime(new_item)) / (tTime(new_item_next) - tTime(new_item));
				if ((multi > 1.0) || (multi < 0.0)) { skd->pbp_offset = skd->pbp_offset_max; }
				else {
					skd->pbp_offset = skd->pbp_offset_max *
						(time - tTime(new_item)) / (tTime(new_item_next) - tTime(new_item));
				}
			}
			else { skd->pbp_offset = skd->pbp_offset_max; }
		}
		else {
			if ((my_song->first_token) && (new_item != skd->current) && ((expose & SKD_EXPOSE_ALL) == 0))
			{
				expose |= SKD_EXPOSE_ALL;
				area.width = skd->visual_min_width;
				area.height = skd->visual_min_height;
				singit_karaoke_data_reset_progress_data(skd, my_song, new_item);
			}
			skd->current = new_item;
		}

		if ((skd->pbp_offset_last != skd->pbp_offset)) {
			if (skd->use_ball) {
				expose |= SKD_EXPOSE_BALL;
				singit_karaoke_data_set_curr_y_pos(skd);
				singit_karaoke_data_update_ball(skd, &area);
			}
			if (new_item) {
				expose |= SKD_EXPOSE_PROGRESS;
				singit_karaoke_data_update_progess_bar
					(skd, tText(my_song, new_item), &area);
			}
		}
		else if (skd->use_ball) {
			singit_karaoke_data_set_curr_y_pos(skd);
			if (skd->ball_y_pos_last != skd->ball_y_pos) {
				expose |= SKD_EXPOSE_BALL;
				singit_karaoke_data_update_ball(skd, &area);
			}
		}

		if ((skd->current != NULL) && ((skd->show_empty_lines == TRUE) || (skd->use_ball == TRUE)) &&
			(singit_song_is_empty_item(my_song, skd->current, FALSE) == TRUE))
		{
			expose |= SKD_EXPOSE_TIMELINE;
		}

		if ((skd->freezers == 0) && (((area.width != 0) && (area.height != 0)) || (expose != 0))) {
			gtk_signal_emit(GTK_OBJECT(skd), skd_signals[EXPOSE], &area, expose, time, &redrawn);

			// * Just store the last settings if drawing was successfull *
			if (redrawn == TRUE) {
				skd->pbp_offset_last = skd->pbp_offset;
				skd->pbp_start_last = skd->pbp_start;
				skd->ball_y_pos_last = skd->ball_y_pos;
			}
		}

		singit_song_detach(&my_song);
	}
}

/*
 * Calculate the exposure area for the ball
 */
static void singit_karaoke_data_update_ball(SingitKaraokeData *skd, GdkRectangle *main_area)
{
	gint pos_x, height;
	gint xpos[2];
	GdkRectangle area;
	gint calced_offset;

	calced_offset = singit_karaoke_data_get_line_offset(skd, TRUE);
	pos_x = skd->pbp_start_last + skd->pbp_offset_last + calced_offset;
	xpos[0] = xpos[1] = pos_x;
	height = skd->top_lines_height;

	if (inl_singit_song_get_next_token(skd->song)) {
		pos_x = skd->pbp_start + skd->pbp_offset + calced_offset;
		xpos[1] = pos_x;
	}

	if (main_area && (skd->freezers == 0)) {
		area.y = height;
		area.height = skd->ball_line_height + height;
		if (xpos[0] < xpos[1]) {
			area.x = xpos[0];
			area.width = xpos[1] - xpos[0] + skd->ball_diameter;
		}
		else {
			area.x = xpos[1];
			area.width = xpos[0] - xpos[1] + skd->ball_diameter;
		}
		area.x += calced_offset;
		gdk_rectangle_union(main_area, &area, main_area);
	}
}

/*
 * Calculate the exposure area of the progress bar
 */
static void singit_karaoke_data_update_progess_bar(SingitKaraokeData *skd, gchar *text, GdkRectangle *main_area)
{
	GdkRectangle area;
	gint height;
	gint ranges[4];

	if (strlen(text) < 1) { return; }

	area.height = skd->text_line_height;

	height = skd->top_lines_height;
	if (skd->use_ball) {
		height += (skd->ball_line_height + skd->line_seperator_high);
	}

	ranges[0] = skd->pbp_start_last + skd->pbp_offset_last;
	ranges[1] = skd->pbp_start + skd->pbp_offset;
	ranges[2] = MIN(ranges[0], ranges[1]); // x
	ranges[3] = MAX(ranges[0], ranges[1]) - ranges[2]; // width

	if (main_area && (skd->freezers == 0)) {
		area.x = ranges[2] +
			singit_karaoke_data_get_line_offset(skd, FALSE);
		area.y = height;
		area.width = ranges[3];
		gdk_rectangle_union(main_area, &area, main_area);
	}
}

static void singit_karaoke_data_refresh (SingitKaraokeData *skd, gint add_changer)
{
	GdkRectangle area;
	SingitSong *my_song;
	gboolean result;
	gchar *text;

	skd->thaw_refresh |= add_changer;

	if (skd->freezers != 0) { return; }
	if (!skd->song) { return; }

#ifdef SKD_DEBUG
	g_print("skd_refresh\n");
#endif

	calc_karaoke_widget_sizes(skd);

	skd->current = NULL;
	area.x = 0;
	area.y = 0;
	area.width = skd->visual_min_width;
	area.height = skd->visual_min_height;

	my_song = singit_song_attach(skd->song);
	if (my_song != NULL) {
		text = singit_karaoke_data_reset_progress_data
			(skd, my_song, my_song->active_token);

		if (my_song->active_token)
			singit_karaoke_data_update_progess_bar(skd, text, FALSE);

		gtk_signal_emit(GTK_OBJECT(skd), skd_signals[EXPOSE],
			&area, SKD_EXPOSE_ALL, skd->last_time, &result);

		singit_song_detach(&my_song);
	}

	skd->thaw_refresh = 0;
}


/*****************************
 *
 * API - Setters and Getters
 *
 *****************************/

void singit_karaoke_data_set_song (SingitKaraokeData *skd, SingitSong *new_song)
{
	SingitSong *attached_song;

	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_song\n");
#endif
	attached_song = singit_song_attach(new_song);

	if (attached_song != skd->song) {
		singit_song_detach(&skd->song);
		skd->song = attached_song;

		skd->current = NULL;
		skd->pbp_start = skd->pbp_offset = 0;
		skd->pbp_start_last = skd->pbp_offset_last = 0;
		skd->last_time = -1;
		skd->ball_y_pos_last = skd->ball_y_pos =
			skd->ball_line_height - skd->ball_diameter;
		skd->ball_start = skd->ball_offset_max = 0;

		singit_karaoke_data_refresh(skd, SONG_CHG);
	}
	else { singit_song_detach(&attached_song); }
}

void singit_karaoke_data_set_lines(SingitKaraokeData *skd, guint value)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_lines\n");
#endif

	if ((value != skd->lines) && (value > 0)) {
		skd->lines = value;
		if (!(skd->top_lines < value))
			{ skd->top_lines = value - 1; }
		singit_karaoke_data_refresh(skd, LINE_CHG);
	}
}

guint singit_karaoke_data_get_lines(SingitKaraokeData *skd)
{
#ifdef SKD_DEBUG
	g_print("skd_get_lines\n");
#endif

	g_return_val_if_fail(skd != NULL, 0);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), 0);

	return skd->lines;
}

void singit_karaoke_data_set_toplines(SingitKaraokeData *skd, guint value)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_toplines\n");
#endif

	if ((skd->top_lines != value) && (skd->top_lines < (skd->lines-1))) {
		skd->top_lines = value;
		singit_karaoke_data_refresh(skd, 0);
	}
}

guint singit_karaoke_data_get_toplines(SingitKaraokeData *skd)
{
#ifdef SKD_DEBUG
	g_print("skd_get_toplines\n");
#endif

	g_return_val_if_fail(skd != NULL, 0);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), 0);

	return skd->top_lines;
}

void singit_karaoke_data_set_font(SingitKaraokeData *skd, const gpointer font, gint font_heigth)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));
	g_return_if_fail(font != NULL);
	g_return_if_fail(font_heigth > 0);

	skd->font = font;
	skd->font_heigth = font_heigth;

	singit_karaoke_data_refresh(skd, FONT_CHG);
}

void singit_karaoke_data_set_show_empty_lines (SingitKaraokeData *skd, const gboolean value)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_show_empty_lines\n");
#endif

	if (value != skd->show_empty_lines) {

		skd->show_empty_lines = value;
	}
}

void singit_karaoke_data_set_jumping_ball (SingitKaraokeData *skd, const gboolean value)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_jumping_ball\n");
#endif

	if (value != skd->use_ball) {

		skd->use_ball = value;
		singit_karaoke_data_refresh(skd, 0);
	}
}

void singit_karaoke_data_freeze(SingitKaraokeData *skd)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

	skd->freezers++;
}

void singit_karaoke_data_thaw(SingitKaraokeData *skd)
{
	if (skd->freezers > 0) { skd->freezers--; }
	if (skd->freezers == 0) { singit_karaoke_data_refresh(skd, 0); }
}

void singit_karaoke_data_set_centered_lines (SingitKaraokeData *skd, gboolean value)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_centered_lines\n");
#endif

	if (value != skd->centerLines) {
		skd->centerLines = value;
		singit_karaoke_data_refresh(skd, 0);
	}
}

gboolean singit_karaoke_data_get_centered_lines (SingitKaraokeData *skd)
{
	g_return_val_if_fail((skd != NULL), FALSE);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), FALSE);

#ifdef SKD_DEBUG
	g_print("skd_get_centered_lines\n");
#endif

	return skd->centerLines;
}

void singit_karaoke_data_set_optimize_font(SingitKaraokeData *skd, gboolean optimize)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_get_get_optimize_font\n");
#endif

	if (optimize != skd->optimize_font) {
		skd->optimize_font = optimize;
		if (skd->optimize_font == TRUE)
			singit_karaoke_data_refresh(skd, OPTM_CHG);
	}
}

gboolean singit_karaoke_data_get_optimize_font(SingitKaraokeData *skd)
{
	g_return_val_if_fail((skd != NULL), FALSE);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), FALSE);

#ifdef SKD_DEBUG
	g_print("skd_get_get_optimize_font\n");
#endif

	return skd->optimize_font;
}

void singit_karaoke_data_optimize_font(SingitKaraokeData *skd)
{
	gpointer font;
	gint x = 0, y = 0;

	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_optimize_font\n");
#endif

	if ((skd->song == NULL) || (singit_song_text_found(skd->song) == FALSE))
		{ return; }

	gtk_signal_emit(GTK_OBJECT(skd), skd_signals[OPTIMIZE],
		x, y, skd->song->lyrics[skd->max_line_nr], &font);
}

void singit_karaoke_data_set_drawing_area(SingitKaraokeData *skd, gint w, gint h)
{
	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));

#ifdef SKD_DEBUG
	g_print("skd_set_optimize_font\n");
#endif

	if ((skd->drawing_area_width != w) ||
		(skd->drawing_area_height != h))
	{
		skd->drawing_area_width = w;
		skd->drawing_area_height = h;
		if (skd->optimize_font == TRUE)
			singit_karaoke_data_refresh(skd, OPTM_CHG);
	}
}



/*****************************
 *
 * Helper funtions
 *
 *****************************/

static guint get_max_line_pixel (SingitKaraokeData *skd, gint *line)
{
	gint i = 0, line_nr = -1;
	guint strlength;
	guint line_width = 0;
	SingitSong *my_song = singit_song_attach(skd->song);

	if (my_song != NULL) {
		if (singit_song_text_found(my_song) == FALSE)
			{ goto bail_out_song; }
	}
	else { goto bail_out; }

	while (my_song->lyrics[i]) {
		strlength = strlen(my_song->lyrics[i]);
		if (strlength > 0) {
			gtk_signal_emit(GTK_OBJECT(skd), skd_signals[RENDER],
				skd->font, my_song->lyrics[i], &strlength);
  			if (strlength > line_width) {
				line_width = strlength;
				line_nr = i;
			}
		}
		i++;
	}

	if (line != NULL)
		*line = line_nr;

bail_out_song:
	singit_song_detach(&my_song);
bail_out:
	return line_width;
}

static void singit_karaoke_data_optimize_font_int(SingitKaraokeData *skd)
{
	gpointer font;
	gint x = 0, y = 0;
	SingitKaraokeDataClass *klass =
		(SingitKaraokeDataClass*) GTK_OBJECT(skd)->klass;

	if ((skd->song == NULL) ||
		(singit_song_text_found(skd->song) == FALSE) ||
		(skd->drawing_area_width <= 0) ||
		(skd->drawing_area_height <= 0) ||
		(klass->optimize == NULL))
	{
		return;
	}

	gtk_signal_emit(GTK_OBJECT(skd), skd_signals[OPTIMIZE],
		x, y, skd->song->lyrics[skd->max_line_nr], &font);
}

static void calc_karaoke_widget_sizes(SingitKaraokeData *skd)
{
	// First optimize the font - most calculated data will depend on it
	if (skd->thaw_refresh & OPTM_CHG)
		singit_karaoke_data_optimize_font_int(skd);

	if (skd->thaw_refresh & FONT_CHG) {

		skd->ball_line_height = skd->font_heigth;

		skd->text_line_height = skd->ball_line_height + 2;

		skd->ball_diameter = (skd->font_heigth / 2.5);
		if ((skd->ball_diameter % 2) == 0) { skd->ball_diameter++; }

		skd->text_left_offset = ((skd->ball_diameter / 2) + 1);

		skd->line_seperator_high = skd->font_heigth / 6 + 1;
		skd->active_line_seperator_high = skd->font_heigth / 3 + 1;

		gtk_signal_emit(GTK_OBJECT(skd),
			skd_signals[NEW_BALL], skd->ball_diameter);
	}

	skd->visual_min_height =
		skd->lines * (skd->text_line_height + skd->line_seperator_high) +
		2 * skd->active_line_seperator_high;
	if (skd->use_ball)
		skd->visual_min_height += skd->ball_line_height;
	else
		skd->visual_min_height -= skd->line_seperator_high;

	if (skd->thaw_refresh & (FONT_CHG | SONG_CHG)) {
		skd->max_line_pixel =
			get_max_line_pixel(skd, &skd->max_line_nr);
	}

	skd->visual_min_width = skd->max_line_pixel + 2 * skd->text_left_offset;

	skd->top_lines_height = skd->top_lines *
		(skd->text_line_height + skd->line_seperator_high)
			+ skd->active_line_seperator_high;

	gtk_signal_emit(GTK_OBJECT(skd), skd_signals[NEW_VISUAL],
			skd->visual_min_width, skd->visual_min_height);
}

gint singit_karaoke_data_get_line_offset
	(SingitKaraokeData *skd, gboolean ball)
{
	g_return_val_if_fail((skd != NULL), FALSE);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), FALSE);

	return (ball == FALSE) ? skd->line_offset : skd->line_offset - left_right_line_border;
}

gint singit_karaoke_data_calc_line_offset
	(SingitKaraokeData *skd, gboolean ball, gchar *text)
{
	gint size = 0, result;

	g_return_val_if_fail(skd != NULL, size);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd), size);

	if (text == NULL) { return size; }

	if (skd->centerLines == FALSE) {
		result = (ball == FALSE) ? left_right_line_border : 0;
	}
	else {
		if (strlen(text) > 0) {
			gtk_signal_emit(GTK_OBJECT(skd),
				skd_signals[RENDER], skd->font, text, &size);
		}

		result = (skd->max_line_pixel - size) / 2;
		if (result < left_right_line_border)
			{ result = left_right_line_border; }
		if (ball != FALSE)
			{ result -= left_right_line_border; }
	}
	return result;
}
/*
gboolean singit_karaoke_data_get_timeline_times(SingitKaraokeData *skd, gint time, gint *start, gint* max)
{
	SingitSong *my_song;
	GList * new_item, new_item_next;

	g_return_val_if_fail(skd != NULL);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_DATA(skd));
	g_return_val_if_fail(time >= 0);
	g_return_val_if_fail(max != NULL);
	g_return_val_if_fail(offset != NULL);

	my_song = singit_song_attach(skd->song);
	if (my_song) {
		new_item = my_song->active_token;
		new_item_next = inl_singit_song_get_next_token(my_song);

		singit_song_detach(my_song);
	}
}
*/
