

/*****************************************************************
Fourier calibration
*/
#include <gtk/gtk.h>
#ifdef USE_IMLIB
#include <gdk_imlib.h>
#else
#include <gdk-pixbuf/gdk-pixbuf.h>
#endif
#include "gtk-meta.h"

#include <math.h>

#include <complex>
#include <stdlib.h>

#include "fourier.hh"

//#define GOOD_TRANSLATION_MATCH(A)  ((A) >= 0.9)
#define BEST_TRANSLATION_MATCH(A)  ((A) >= 0.9995)


#define M2PI (2*M_PI)

static inline std::complex<double> expI(double a)
{
  std::complex<double>  e( cos(a), sin(a));
  return e;
}

#define size 16
#define M2PIS  (2*M_PI/(double)size)
#define SIZEARG 

#define F(I,J) [(J)+(size)*(I)]

double *the_cos=new  double[size];
std::complex<double> *the_exp=new  std::complex<double>[size*size];

void fourier_init_()
{
  int i,j;
  for (i=0; i < size ; i++)
    the_cos[i]=cos( M_PI * (double)(i+size)/(double)(size*4)  );
  for (i=0; i < size ; i++)
    for (j=0; j < size ; j++)
      the_exp F(i,j) = expI( - M2PIS * (double)(i*j) );
}

std::complex<double> * fourier(SIZEARG
			       GdkPixbuf *pixbuf,
			       int x, int y)
{
  int width, height, rowstride;
  guchar *pixels;
  guint n_channels = gdk_pixbuf_get_n_channels (pixbuf);
  
  g_assert(the_cos[1]); //will trigger if fourier_init was not called
  
  g_assert (gdk_pixbuf_get_colorspace (pixbuf) == GDK_COLORSPACE_RGB);
  g_assert (gdk_pixbuf_get_bits_per_sample (pixbuf) == 8);
  //g_assert (gdk_pixbuf_get_has_alpha (pixbuf));
  g_assert ( (size & 1 ) == 0);

  width = gdk_pixbuf_get_width (pixbuf);
  height = gdk_pixbuf_get_height (pixbuf);

  x-=size/2;  y-=size/2;

  int  leftx=0, rightx=size, lefty=0, righty=size;  
  if(x<0) { leftx=-x;}
  if(y<0) { lefty=-y;}
  if( x+size >=  width  ) { rightx=width -x-1; }
  if( y+size >=  height ) { righty=height-y-1; }

  //  g_assert (x >= 0 && x < width-sizex);  g_assert (y >= 0 && y < height-sizey);
 
  rowstride = gdk_pixbuf_get_rowstride (pixbuf);
  pixels = gdk_pixbuf_get_pixels (pixbuf);

  std::complex<double> *fourier = new std::complex<double>[size*size];
  double *image = new double[size*size];

  //  std::complex<double> I(0,1);
  int i,j,l,k;
  if (n_channels >= 3)
    for (l=leftx; l < rightx ; l++)
      for (k=lefty; k < righty ; k++) {
	guchar *  p=pixels + (k+y) * rowstride + (l+x) * n_channels;
	int v= (int)(*p) + (int)(*(p+1))+ (int)(*(p+2));
	image F(l,k)= (double)v  * the_cos[k] * the_cos[l];
      }
  else
    if (n_channels == 1)
      for (l=leftx; l < rightx ; l++)
	for (k=lefty; k < righty ; k++) {
	  guchar *  p=pixels + (k+y) * rowstride + (l+x);
	  image F(l,k)= (double)(*p) * the_cos[k] * the_cos[l];
	}
    else throw "wrong number of channels in pixbuf";

#if 0
  //original formula
  for (i=0; i < size ; i++)
    for (j=0; j < size ; j++)
	for (l=leftx; l < rightx ; l++)
	  for (k=lefty; k < righty ; k++)
	  fourier F(i,j)  +=
	    expI ( - M2PIS * (i*l +j*k) ) * image F(l,k);
#else
  // faster formula
  std::complex<double> *fouriertmp = new std::complex<double>[size*size];
  for (i=0; i < size ; i++)
    for (l=leftx; l < rightx ; l++)
      for (k=lefty; k < righty ; k++) 
	fouriertmp F(i,k) += the_exp F( i,l) * image F(l,k);
  for (i=0; i < size ; i++)
    for (j=0; j < size ; j++)      
	for (k=lefty; k < righty ; k++) 
	  fourier F(i,j)  += the_exp F( j,k) * fouriertmp F(i,k);
  delete fouriertmp;
#endif
  return fourier;
}


static double energy(//const int size, 
		     std::complex<double> *src)
{
  int i,j;
  double e=0; 
  std::complex<double>  v;
  for (i=0; i < size ; i++)
    for (j=0; j < size ; j++) {
      v=src F(i,j);       
      e += v.real()*v.real() + v.imag()*v.imag() ;
    }
  return e;
}

static double distance(//const int size,
	      std::complex<double> *src,
	      std::complex<double> *dst,
	      double x,double y)
{
  int i,j;
  double e=0; 
  std::complex<double> //I(0,1),
     a,b,v;
  for (i=0; i < size ; i++)
    for (j=0; j < size ; j++) { 
      a=(src F(i,j)) ;
      b=(dst F(i,j)) *  expI( x* (i/(double)size)) *
	expI(  y *(j/(double)size));
      v=a-b;
      e += v.real()*v.real() + v.imag()*v.imag() ;
    }
  return e;
}


static double correlation(//const int size,
	      std::complex<double> *src,
	      std::complex<double> *dst,
	      double x,double y)
{
  int i,j;
  double e=0; 
  std::complex<double> I(0,1) , a,b,v;
  for (i=0; i < size ; i++)
    for (j=0; j < size ; j++) {
      a=(src F(i,j)) ;
      b=(dst F(i,j)) *  expI( M2PIS * (x * i + y * j ));
      v= a * conj(b);   
      e += v.real();
    }
  return e;
}

static double correlation_dx(//const int size,
	      std::complex<double> *src,
	      std::complex<double> *dst,
	      double x,double y)
{
  int i,j;
  double e=0; 
  std::complex<double> I(0,1) , a,b,v;
  for (i=0; i < size ; i++)
    for (j=0; j < size ; j++) {
      a=(src F(i,j)) ;
      b=(dst F(i,j)) * expI( M2PIS * ( x * i + y * j) ) * M2PIS * I *(double)i;
      v= a * conj(b);   
      e += v.real();
    }
  return e;
}

static double correlation_dy(//const int size,
	      std::complex<double> *src,
	      std::complex<double> *dst,
	      double x,double y)
{
  int i,j;
  double e=0; 
  std::complex<double> I(0,1) , a,b,v;
  for (i=0; i < size ; i++)
    for (j=0; j < size ; j++) {
      a=(src F(i,j)) ;
      b=(dst F(i,j)) * expI( M2PIS * ( x * i + y * j) ) * M2PIS * I *(double)j;
      v= a * conj(b);   
      e += v.real();
    }
  return e;
}

#ifdef __WIN32__
double drand48()
{
  return (double)rand() / (double)RAND_MAX;
}
#endif

double search_translation(//const int size,
			  std::complex<double> src[],
			  std::complex<double> dst[],
			  double &x,double &y
			  //double span
			  )
{
  const double span=3.0;
  static int success_random=0, success_gradient=0;
  std::complex<double> I(0,1);
  double   e = correlation(SIZEARG src,dst,x,y) , orige=e, 
    nx=x, ny=y ,ne=e;
  double f= sqrt( energy(SIZEARG src)) * sqrt( energy(SIZEARG dst));
  if(f< size*size) {
    g_message("(not enough energy %g for auto adjust)",f);
    return 0;
  }
  //  if( BEST_TRANSLATION_MATCH(e/f))    return 10;      

#ifdef DEBUG
#define PRINT(R,A)    fprintf(stderr,\
    "%c%5d : grad %d rand %d corr %+2.5g nx %+1.3g ny %+1.3g %c",\
            	    R,lp,success_gradient,success_random,ne/f,nx,ny,A);    
#else
#define PRINT(R,A)
#endif
#define STEPS 200

  int lp=STEPS, lq;

  PRINT('-','\n')

  while (lp>0 ) {    
    lq=6;
    while(lp>0 && lq>0) {
      lp--;lq--;
      nx +=  (drand48()-0.5)*span/3.;
      ny +=  (drand48()-0.5)*span/3.;
      nx=CLAMP(nx,-span,span);
      ny=CLAMP(ny,-span,span);
      
      ne = correlation(SIZEARG src,dst,nx,ny);
      PRINT(' ','\r')
      if (e<ne)   {
	success_random++; PRINT('*','\n')
	x=nx; y=ny; e=ne;
	if( BEST_TRANSLATION_MATCH(e/f)) {
	  return (e-orige) / f ;
	}
	break;
      }
    }
    lq=5;
    while(lp>0 && lq>0) {
      lp--;lq--;
      nx = x + correlation_dx(SIZEARG     src,dst,x,y) /20./f;
      ny = y + correlation_dy(SIZEARG     src,dst,x,y) /20./f;
      nx=CLAMP(nx,-span,span);
      ny=CLAMP(ny,-span,span);
      ne = correlation(SIZEARG src,dst,nx,ny);
      PRINT(' ','\r')
      if (e<ne)   {
	success_gradient++; PRINT('>','\n')
	x=nx; y=ny; e=ne;
	if( BEST_TRANSLATION_MATCH(e/f))
	  return (e-orige) / f ;
      } else break;
    }
    //#ifdef DEBUG

    //#endif
  }
  PRINT('-','\n')

  return (e-orige) / f ;
}

gboolean detect_translation_(GdkPixbuf *src, double sx, double sy,
			GdkPixbuf *dst, double dx, double dy,
			 // new suggested destination 
			double *nx,double *ny)
{

  //  int size=10;
  //size=MIN( size  , MIN( MIN(x,width-x) , MIN(y,height-y)  ));

  int isx=(int)floor(sx), isy=(int)floor(sy),
    idx=(int)floor(dx), idy=(int)floor(dy);
  double dsx=sx-isx , dsy=sy-isy , ddx=dx-idx , ddy=dy-idy;

  double tx = ddx-dsx,     ty = ddy-dsy;

  std::complex<double>  
    * srcf =fourier(SIZEARG src, isx , isy) ,
    * dstf =fourier(SIZEARG dst, idx , idy) ;

  double v=search_translation(SIZEARG    srcf, dstf, tx, ty);

  *nx =   tx-(ddx-dsx) + dx;
  *ny =   ty-(ddy-dsy) + dy;
  if( v>0.0001 ) {
    g_message ("autoadjust %3g %3g to %3g %3g (corr+= %.3g)",dx,dy,*nx,*ny,v);
    return TRUE;
  }  else {
    g_message ("autoadjust %3g %3g stays at %3g %3g ",dx,dy,*nx,*ny);
    return FALSE;
  }
}

extern "C" { 
gboolean detect_translation(GdkPixbuf *src, double sx, double sy,
			  GdkPixbuf *dst, double dx, double dy,
			  // new suggested destination 
			  double *nx,double *ny)
  {
    return detect_translation_(src,sx,sy,    dst,dx,dy,  	nx,ny);
  }

  void fourier_init() {fourier_init_();}
}

