#       xorgabstraction.py -- Core class of X-Kit's parser
#       
#       Copyright 2008 Alberto Milone <albertomilone@alice.it>
#       
#       This program is free software; you can redistribute it and/or modify
#       it under the terms of the GNU General Public License as published by
#       the Free Software Foundation; either version 2 of the License, or
#       (at your option) any later version.
#       
#       This program is distributed in the hope that it will be useful,
#       but WITHOUT ANY WARRANTY; without even the implied warranty of
#       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#       GNU General Public License for more details.
#       
#       You should have received a copy of the GNU General Public License
#       along with this program; if not, write to the Free Software
#       Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
#       MA 02110-1301, USA.
import XKit.xutils
import XKit.xorgparser
import os
import re
import tempfile
from subprocess import Popen,PIPE
import sys

import xorgmanparser

xorgTypes = {u'xv-id': None,
             u'clockchip-type': None,
             u'horizsync-range': 'number',
             u'red-gamma green-gamma blue-gamma': 'number',
             u'rate': 'number',
             u'frequency': 'number',
             u'value': None,
             u'vdisp vsyncstart vsyncend vtotal': 'number',
             u'vendor': None,
             u'id': None,
             u'width height': 'number',
             u'x y': None,
             u'monitor': None,
             u'clock ...': 'number',
             u'rev': None,
             u'modesection-id': None,
             u'bus-id': None,
             u'bool': 'boolean',
             u'layout-id': None,
             u'monitor-id': None,
             u'ramdac-type': None,
             u'...': None,
             u'visual-name': None,
             u'mode-description': None,
             u'string': None,
             u'red green blue': 'number',
             u'mem': 'number',
             u'device-id': None,
             u'vertrefresh-range': 'number',
             u'hdisp hsyncstart hsyncend htotal': 'number',
             u'path': None,
             u'freq': 'number',
             u'baseaddress': None,
             u'red-weight green-weight blue-weight': 'number',
             u'rotation': None,
             u'name': None,
             u'speed-8 speed-16 speed-24 speed-32': 'number',
             u'when': 'number',
             u'xdim ydim': 'number',
             u'bpp': 'number',
             u'boolean': 'boolean',
             u'command': None,
             u'x0 y0': 'number',
             u'time': 'number',
             u'chipset': None,
             u'model': None,
             u'depth': 'number'}

class XorgAbstraction:
    def __init__(self, xorgSource=None, xorgDestination=None):
        
        self.optionsBlacklist = ['Vendorname']
        
        standardPath = '/etc/X11/xorg.conf'
        self.xorgSource = xorgSource or standardPath
        self.xorgDestination = xorgDestination or standardPath
        
        try:
            self.xparser = XKit.xutils.XUtils(self.xorgSource)
        #if xorg.conf is missing or broken
        except(IOError, XKit.xorgparser.ParseException):
            #print >> sys.stderr, 'invalid xorg.conf. STARTING FROM SCRATCH'
            #start from scratch
            self.xparser = XKit.xutils.XUtils()
        
        self.fillCardsDetails()
        
        self.getAllDrivers()

    def getXorgDevices(self):
        devlist = []
        
        devices = self.xparser.identifiers['Device']
        if len(devices) > 0:
            it = 0
            for couple in devices:
                devlist.append(devices[it][0])
                it += 1
        return devlist

    def detection(self):
        '''
        Detect the models of the graphics cards
        and store them in self.cardIds
        '''
        self.cardIds = []
        p1 = Popen(['lspci', '-n'], stdout=PIPE)
        p = p1.communicate()[0].split('\n')
        indentifier1 = re.compile('.*0300: *(.+):(.+) \(.+\)')
        indentifier2 = re.compile('.*0300: *(.+):(.+)')
        for line in p:
            m1 = indentifier1.match(line)
            m2 = indentifier2.match(line)
            if m1:
                id1 = m1.group(1).strip().upper()
                id2 = m1.group(2).strip().upper()
                id = id1 + id2
                self.cardIds.append(id)
            elif m2:
                id1 = m2.group(1).strip().upper()
                id2 = m2.group(2).strip().upper()
                id = id1 + id2
                self.cardIds.append(id)

    def getAllDrivers(self):
        '''
        get all the available open source drivers
        + fglrx and nvidia
        '''
        alldrivers = {}.fromkeys(['nvidia', 'fglrx'])
        aliasesPath = '/usr/share/xserver-xorg/pci/'
        files = os.listdir(aliasesPath)
        for file in files:
            driver = file[:file.find('.ids')]
            alldrivers.setdefault(driver)
        try:
            del alldrivers['apm']
        except KeyError:
            pass
        self.allDrivers = alldrivers.keys()
    
    def fillCardsDetails(self):
        '''Get cards details from XKit and the pci ids'''
        self.detection()
        self.cards = self.getXorgDevices()
        
    def fillXorgDetails(self, globalDetails, optionsRef, manOptions):
        '''Fill globalDetails with details on cards and sections'''
        # globalDetails = {
        #                       card_index: {
        #                                    'cardName': card_name,
        #                                    'listStore': object,
        #                                    'screen': screen_position (in XKit),
        #                                    'monitor': monitor_position (in XKit),
        #                                    'device': device_position (in XKit),
        #                                    'driver': driver in the combobox,
        #                                   }
        #                      }
        
        # Start filling globalDetails
        
        # ServerFlags at 0
        it = 0
        globalDetails.setdefault(it, {})
        globalDetails[it]['cardName'] = ''
        
        # All the detected cards
        it += 1
        for card in self.cards:
            #print 'CARD', card
            #self.deviceList.append([card])
            globalDetails.setdefault(it, {})
            globalDetails[it]['cardName'] = card
            optionsRef[it] = {}
            it += 1
        
        
        if len(self.cards) > 0:
            # If one or more Device sections are detected by xkit
            it = 0
            for card in self.cards:
                try:
                    driver = self.xparser.getDriver('Device', it)
                except XKit.xorgparser.OptionException:
                    driver = None
                
                if driver == None:
                    driver = self.getDriverPciId(it)
                
                # Fill globalDetails
                for cardIndex in globalDetails:
                    if globalDetails[cardIndex]['cardName'] == card:
                        globalDetails[cardIndex]['driver'] = driver
                        globalDetails[cardIndex]['device'] = it
                        break
                
                # FIXME: should I really set the driver at this point???
                #self.xparser.setDriver('Device', driver, it)
                it += 1
            
            
            
            if len(self.xparser.globaldict['Screen']) > 0:
                # Make sure that each Device section has its own Screen section
                it = 0
                for screen in self.xparser.globaldict['Screen']:
                    '''
                    There can be no screen or monitor section
                    '''
                    
                    screenReferences = self.xparser.getReferences('Screen', it, reflist=['Device', 'Monitor'])
                    if screenReferences['Device']:
                        deviceName = screenReferences['Device'][0]
                    else:
                        # the Screen section is not used by any of the Device sections
                        # therefore it will be ignored
                        continue
                    
                    monitor = ''
                    if screenReferences['Monitor']:
                        monitorName = screenReferences['Monitor'][0]
                        try:
                            monitor = self.xparser.getPosition('Monitor', monitorName)
                        except xorgparser.IdentifierException:
                            pass
                    
                    
                    
                    
                    # if the Screen section has no reference to a Monitor section
                    # create a new Monitor section
                    if monitor == '':
                        screenName = self.xparser.identifiers['Screen'][it][0]
                        monitorName = screenName.replace(' Screen', ' Monitor') + ' XKit' + str(it)
                        
                        # Create a new Monitor section
                        monitor = self.xparser.makeSection('Monitor', identifier=monitorName)
                        
                        # Add a reference to the new Monitor section in the Screen section
                        self.xparser.addReference('Screen', 'Monitor', monitorName, position=screen)
                        
                    
                    # Fill globalDetails
                    for cardIndex in globalDetails:
                        if globalDetails[cardIndex]['cardName'] == deviceName:
                            globalDetails[cardIndex]['screen'] = screen
                            globalDetails[cardIndex]['monitor'] = monitor
                            break

                        
                    it += 1
                
                
                
#                it = 0
#                for card in self.cards:
#                    # If there are cards without Screen section
#                    for cardIndex in globalDetails:
#                        # if globalDetails[cardIndex].get('screen') == None
#                        # or globalDetails[cardIndex].get('screen') == ''
#                        if globalDetails[cardIndex]['cardName'] == card and \
#                        globalDetails[cardIndex].get('screen'):
#                            
#                            deviceName = card
#                            screenName = deviceName.replace(' Video', '').replace(' Device', '') + 'Screen XKit' + str(it)
#                            monitorName = screenName.replace('Screen', 'Monitor')
#                            
#                            screen = self.xparser.makeSection('Screen', identifier=screenName)
#                            monitor = self.xparser.makeSection('Monitor', identifier=monitorName)
#                            
#                            self.xparser.addReference('Screen', 'Device', deviceName, position=screen)
#                            self.xparser.addReference('Screen', 'Monitor', monitorName, position=screen)
#                            
#                            # Fill globalDetails
#                            globalDetails[cardIndex]['screen'] = screen
#                            globalDetails[cardIndex]['monitor'] = monitor
#                            break

#                    it += 1
            else:
                # Create as many Screen sections as the number of Device sections.
                # Make sure that each Device section has its own Screen section
                it = 0
                for card in self.cards:
                    deviceName = card
                    screenName = deviceName.replace(' Video', '').replace(' Device', '') + ' Screen XKit' + str(it)
                    monitorName = screenName.replace('Screen', 'Monitor')
                    
                    screen = self.xparser.makeSection('Screen', identifier=screenName)
                    monitor = self.xparser.makeSection('Monitor', identifier=monitorName)
                    
                    self.xparser.addReference('Screen', 'Device', deviceName, position=screen)
                    self.xparser.addReference('Screen', 'Monitor', monitorName, position=screen)
                    
                    # Fill globalDetails
                    for cardIndex in globalDetails:
                        if globalDetails[cardIndex]['cardName'] == card:
                            globalDetails[cardIndex]['screen'] = screen
                            globalDetails[cardIndex]['monitor'] = monitor
                            break
                    
                    it += 1

            
        else:
            # No Device section is detected by xkit
            it = 0
            for pciId in self.cardIds:
                deviceName = 'Configured Video Device ' + str(it)
                screenName = deviceName.replace('Video', 'Screen')
                monitorName = deviceName.replace('Video', 'Monitor')
                device = self.xparser.makeSection('Device', identifier=deviceName)
                screen = self.xparser.makeSection('Screen', identifier=screenName)
                monitor = self.xparser.makeSection('Monitor', identifier=monitorName)
                
                self.cards.append(deviceName)
                
                self.xparser.addReference('Screen', 'Device', deviceName, position=screen)
                self.xparser.addReference('Screen', 'Monitor', monitorName, position=screen)
                
                #self.usedScreensByPos[deviceName] = screen
                #self.usedScreensByID[deviceName] = screenName
                
                driver = self.getDriverPciId(it)
                #self.usedDrivers[it] = driver
                #self.xparser.setDriver('Device', driver, device)
                
                # Fill globalDetails
                cardIndex = len(globalDetails.keys())
                globalDetails[cardIndex] = {}
                globalDetails[cardIndex]['cardName'] = deviceName
                globalDetails[cardIndex]['screen'] = screen
                globalDetails[cardIndex]['monitor'] = monitor
                globalDetails[cardIndex]['device'] = device
                globalDetails[cardIndex]['driver'] = driver
                
                
                it += 1
    
        # Fill optionsRef
        self.fillXorgOptionsReference(globalDetails, optionsRef, manOptions)
    
    
    def getDriverPciId(self, cardposition):
        '''
        /usr/share/misc/pci.ids
        
        Look up the pci id in /usr/share/xserver-xorg/pci/
        and return the appropriate driver.
        '''
        aliasesPath = '/usr/share/xserver-xorg/pci/'
        files = os.listdir(aliasesPath)
        
        #print self.cardIds
        
        try:
            cardId = self.cardIds[cardposition]
            #print 'CARD ID =', cardId
            for file in files:
                a = open(aliasesPath + file, 'r')
                driver = file[:file.find('.ids')]
                lines = a.readlines()
                for line in lines:
                    if line.find(cardId) != -1:
                        return driver
                a.close()
        except IndexError:
            pass
        return 'fbdev'#fallback on fbdev
    
    def getIdentifier(self, section, position, defaults=None):
        '''Try to get the identifier of a section and return defaults if none can be found.'''
        try:
            identifier = self.xparser.getIdentifier(section, position)
        except (XKit.xorgparser.IdentifierException, TypeError):
            identifier = defaults
        return identifier
    
    def addOption(self, section, option, value, optiontype=None, position=None, reference=None, prefix='"'):
        '''Do some magic and call XKit's method with the same name.'''
        # Exceptions should be caught here
        self.xparser.addOption(section, option, value, optiontype, position, reference, prefix)
        return True
    
    def removeOption(self, section, option, value=None, position=None, reference=None):
        '''Do some magic and call XKit's method with the same name.'''
        self.xparser.removeOption(section, option, value, position, reference)
        return True
    
    def addSubOption(self, section, identifier, option, value, optiontype=None, position=None):
        '''Do some magic and call XKit's method with the same name.'''
        self.xparser.addSubOption(section, identifier, option, value, optiontype, position)
        return True
        
    def removeSubOption(self, section, identifier, option, position=None):
        '''Remove an option from a subsection.'''
        self.xparser.removeSubOption(section, identifier, option, position)
        return True
    
    def setDriver(self, section, driver, globalDetails, index):
        '''Do some magic and call XKit's method with the same name and set the driver in globalDetails too
        
        globalDetails is the data structure used as a reference in the UI
        index is a key from globalDetails which also represents the position in the 1st treeview
        '''
        try:
            self.xparser.setDriver('Device', driver, globalDetails[index]['device'])
        # If ServerFlags
        except (TypeError, KeyError):
            return False
        else:
            globalDetails[index]['driver'] = driver
            return True
    
    def writeFile(self):
        '''Write settings to a tempfile which will be used by policykit mechanism'''
        # Write to a temporary file
        myTempFile = tempfile.NamedTemporaryFile(mode='w')
        self.xparser.writeFile(myTempFile.name)
        
        return myTempFile
    
    
    def hasValidType(self, value, valueType):#value, valueType, optionType, section):
        '''See if the value matches the type in xorgTypes
        
        Return:
          bool isValid
          bool needsPrefix - e.g. '"' or no prefix at all
        '''
        upperType = xorgTypes.get(valueType)
        
        boolean = re.compile(r'^(?:1|0|true|false|yes|no|enable|disable|on|off)$', re.IGNORECASE)
        number = re.compile(r'^(?:\d+|\d+\.\d+|(\d+\s*){1,4}|(\d+\.\d+\s*){1,4}|\d+\.\d+-\d+\.\d+|\d+-\d+)$')

        onestring = re.compile(r'^[a-zA-Z]+$')#any 1-word string
        anystring = re.compile(r'^([a-zA-Z]+\s*){1,10}$')#any string from 1 to 10 words
        anything = re.compile(r'.*')#any string from 1 to 10 words
        if upperType == 'boolean':
            pattern = boolean
        elif upperType == 'number':
            pattern = number
        else:
            pattern = anything
        
        return (pattern.match(value), (pattern != number))


    def __cleanOption(self, option):
        '''
        clean the option and return all its components in a list
        '''
        optbits = []
        optbit = ''
        it = 0
        quotation = 0
        optcount = option.count('"')
        if optcount > 0:#dealing with a section option
            for i in option:
                if not i.isspace():
                    if i == '"':
                        quotation += 1
                    else:
                        optbit += i
                else:    
                    if quotation % 2 != 0:
                        optbit += i
                        
                    else:
                        if len(optbit) > 0:
                            optbits.append(optbit)
                            optbit = ''
                        
                if it == len(option) - 1:
                    if optbit != '':
                        optbits.append(optbit)
                it += 1            
        else:#dealing with a subsection option
            for i in option:
                if not i.isspace():
                    optbit += i
                else:    
                    if len(optbit) > 0:
                        optbits.append(optbit)
                        optbit = ''
                        
                if it == len(option) - 1:
                    if optbit != '':
                        optbits.append(optbit)
                it += 1
        return optbits


    def fillXorgOptionsReference(self, globalDetails, optionsRef, manOptions):
        '''Fill the self.optionsRef with the options detected by XKit'''
        self.usedOptions = {}
        
        # Sections from which data will be extracted
        toInspect = ['Device', 'Screen', 'Monitor']
        # Options we don't want to show
        blacklist = ['device', 'driver', 'identifier', 'screen', 'monitor']
        
        # Position in the left-most treeview
        it = 0
        
        '''
        ServerFlags options
        '''
        optionsRef[it] = {}
        
        sectionLength = len(self.xparser.globaldict['ServerFlags'])
        if sectionLength > 1:
            # Merge all the serverflags sections into one
            
            for elem in range(1, sectionLength):
                for option in self.xparser.globaldict['ServerFlags'][elem]:
                    self.xparser.globaldict['ServerFlags'][0].append(option)
            self.xparser.globaldict['ServerFlags'] = {0: self.xparser.globaldict['ServerFlags'][0]}
        
        
        
        section = 'ServerFlags'
        # Create an empty ServerFlags section if none exists
        if len(self.xparser.globaldict[section]) == 0:
            self.xparser.makeSection(section)
        
        for line in self.xparser.globaldict['ServerFlags'][0]:
            if not line.strip().startswith('#'):
                pound = line.find('#')
                if pound != -1:
                    line = line[0: pound].strip()
                    
                line = self.__cleanOption(line.strip())
                if line[0].lower() == 'option':
                    del line[0]
                    optionType = 'Option'
                else:
                    optionType = None
                
                optName = line[0]
                if optName.lower() not in [x.lower() for x in blacklist]:
                    #print 'optName', optName
                    optValue = ' '.join(line[1:])
                    optionsRef[it].setdefault(optName, {})
                    optionsRef[it][optName]['optionType'] = optionType
                    optionsRef[it][optName]['section'] = section
                    optionsRef[it][optName]['position'] = 0
                    optionsRef[it][optName]['value'] = optValue
                    try:
                        optionsRef[it][optName]['valueType'] = manOptions.get(optName).get('valueType')
                        optionsRef[it][optName]['description'] = manOptions.get(optName).get('description')
                    except AttributeError:
                        #print 'FAILED with', optName
                        #print 'MANOPTIONS\n', str(manOptions)
                        optionsRef[it][optName]['valueType'] = None
                        optionsRef[it][optName]['description'] = ''#'Description N/A'
        
        
        
        
        # Increment it by 1 as ServerFlags is at it=0
        it += 1
        # Deal with Device and device related options
        # and fill optionsRef
        for card in self.cards:

            optionsRef[it] = {}
            
            # Get the content of the SubSections in the Screen sections
            subIt = 0
            for sub in self.xparser.globaldict['SubSection']:
                subSection = self.xparser.globaldict['SubSection'][sub]
                position = subSection['position']
                section = subSection['section']
                identifier = subSection['identifier']
                if section == 'Screen' and position == globalDetails[it]['screen']:
                    options = subSection['options']
                    for line in options:
                        if not line.strip().startswith('#'):
                            pound = line.find('#')
                            if pound != -1:
                                line = line[0: pound].strip()
                                
                            line = self.__cleanOption(line.strip())
                            if line[0].lower() == 'option':
                                del line[0]
                                optionType = 'Option'
                            else:
                                optionType = ''
                            
                            optName = line[0]
#                            if optName.lower() not in [x.lower() for x in blacklist]:
                                #print 'optName', optName
                            
                            optValue = ' '.join(line[1:])
                            if optValue == '':
                                optValue = 'True'
                            #print repr(optValue)
                            optionsRef[it].setdefault(optName, {})
                            optionsRef[it][optName]['optionType'] = optionType
                            optionsRef[it][optName]['section'] = identifier
                            optionsRef[it][optName]['position'] = self.cards.index(card)#it
                            optionsRef[it][optName]['value'] = optValue
                            try:
                                optionsRef[it][optName]['valueType'] = manOptions.get(optName).get('valueType')
                                optionsRef[it][optName]['description'] = manOptions.get(optName).get('description')
                            except AttributeError:
                                optionsRef[it][optName]['valueType'] = None
                                optionsRef[it][optName]['description'] = ''#'Description N/A'
                
                subIt += 1                    
            
            # Add the options from the Device, Screen and Montor sections
            # related to the device defined by the "it" iterator    
            for section in toInspect:
                sectName = section.lower()
                
                sectIt = globalDetails[it].get(sectName)
                
                if sectIt == None:
                    #print 'section %s SKIPPED' % (section)
                    continue

                
                for line in self.xparser.globaldict[section][sectIt]:
                    if not line.strip().startswith('#'):
                        pound = line.find('#')
                        if pound != -1:
                            line = line[0: pound].strip()
                            
                        line = self.__cleanOption(line.strip())
                        if line[0].lower() == 'option':
                            del line[0]
                            optionType = 'Option'
                        else:
                            optionType = ''
                        
                        optName = line[0]
                        if optName.lower() not in [x.lower() for x in blacklist]:
                            #print 'optName', optName
                                                     
                            optValue = ' '.join(line[1:])
                            if optValue == '':
                                optValue = 'True'
                            #print repr(optValue)
                            optionsRef[it].setdefault(optName, {})
                            optionsRef[it][optName]['optionType'] = optionType
                            optionsRef[it][optName]['section'] = section
                            optionsRef[it][optName]['position'] = sectIt#it
                            optionsRef[it][optName]['value'] = optValue
                            try:
                                optionsRef[it][optName]['valueType'] = manOptions.get(optName).get('valueType')
                                optionsRef[it][optName]['description'] = manOptions.get(optName).get('description')
                            except AttributeError:
                                optionsRef[it][optName]['valueType'] = None
                                optionsRef[it][optName]['description'] = ''#'Description N/A'
            it += 1
        #print 'OPTIONSREF is\n', str(optionsRef)
        
    def getManOptions(self, sectionFilter):
        '''Get a data structure out of the Xorg man page'''
        return xorgmanparser.makeDict(sectionFilter)
