/*-
# MOTIF/X-BASED BARREL(tm)
#
#  xbarrel.c
#
###
#
#  Copyright (c) 2003 - 2005	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/11/15 X/Windows
*/

#ifdef WINVER
#include "BarrelP.h"
#include "wbarrel.h"
#define TITLE "wbarrel"

static BarrelRec widget;
static HWND Wnd;

#ifndef SCOREPATH
#define SCOREPATH "c:\\Windows"
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(Wnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->barrel.started = b
#else
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef VMS
#include <unixlib.h>
#define getlogin() cuserid(NULL)
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#if HAVE_FCNTL_H
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtSetArg(arg[0], XtNstart, b); XtSetValues(w, arg, 1)
#include "Barrel.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/barrel.xpm"
#endif
#include "icons/barrel.xbm"
#ifndef SCOREPATH
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif

#ifdef HAVE_MOTIF
static const char aboutHelp[] = {
	"Version 7.1.3\n"
	"Send bugs (reports or fixes) to the author: "
	"David Bagley <bagleyd@tux.org>\n"
	"The latest version is at: "
	"http://www.tux.org/~bagleyd/puzzles.html\n"
};
static const char optionsHelp[] = {
	"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]] "
	"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
	"[-{foreground|fg} {color}] [-{background|bg} {color}] "
	"[-{border|bd} {color}] [-tile {color}]\n"
	"[-face{0|1|2|3|4|5} {color}] [-[no]orient] [-[no]pairs] "
	"[-[no]install] [-picture {filename}] [-delay msecs]\n"
	"[-[no]sound] [-bumpsound {filename}] [-movesound {filename}] "
	"[-base {int}] [-{font|fn} {fontname}] [-username {string}]\n"
};
#endif
#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
	"The Barrel, the Billion Barrel, or Wonderful Barrel also "
	"seen as the Cylinder Puzzle of Japanese origin.  The puzzle\n"
	"contains 23 beads of 6 different colors.  4 each of 5 "
	"different colors arranged in 5 rows and 3 black beads making\n"
	"3 rows one unit longer.  A system of plungers slide all "
	"three of the long rows left and right.  Two of the plungers\n"
	"sit together and the other (the middle plunger) is "
	"separate from the other two.  There are 2 separate wheels or\n"
	"disks that rotate about the center axis.\n"
};
static const char featuresHelp[] = {
	"Press \"mouse-left\" button to move a tile. Release "
	"\"mouse-left\" button in the same row would move the tile towards\n"
	"the space (unless there is no space in that row).  "
	"Release on a different row will move the tiles up or down.\n"
	"\n"
	"Click \"mouse-center\" button, or press \"P\" or \"p\" "
	"keys, to toggle the practice mode (in practice mode the\n"
	"record should say \"practice\").  This is good for learning "
	"moves and experimenting.\n"
	"\n"
	"Click \"mouse-right\" button, or press \"R\" or \"r\" "
	"keys, to randomize the puzzle.\n"
	"\n"
	"Press \"O\" or \"o\" keys to toggle orient mode.  Slightly "
	"more challenging on (default is off).\n"
	"\n"
	"Press \"I\" or \"i\" keys to toggle the pairs mode.  All "
	"the tiles rotate in pairs.  This is considerably\n"
	"more challenging on (default is on).\n"
	"\n"
	"\"S\" or \"s\" keys reserved for the auto-solver "
	"(not implemented).\n"
	"\n"
	"Press \"U\" or \"u\" keys to undo a move.\n"
	"\n"
	"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
	"\n"
	"Press \"W\" or \"w\" keys to save a puzzle.\n"
	"\n"
	"Press \">\" or \".\" keys to speed up the movement of tiles.\n"
	"\n"
	"Press \"<\" or \",\" keys to slow down the movement of tiles.\n"
	"\n"
	"Press \"@\" key to toggle the sound.\n"
	"\n"
	"Press \"Esc\" key to hide program.\n"
	"\n"
	"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
	"\n"
	"Use the key pad, \"R\" keys, or arrow keys to move without "
	"the mouse.\n"
	"Key pad is defined for Barrel as:\n"
	"  8    Up\n"
	"  ^\n"
	"4< >6  Left, Right\n"
	"  v\n"
	"  2    Down\n"
	"\n"
	"Use the shift key and the up or down arrow keys to rotate "
	"all the tiles.\n"
};
static const char referencesHelp[] = {
	"L. E. Horden, Sliding Piece Puzzles (Recreations in "
	"Mathematics Series), Oxford University\n"
	"Press 1986, Plate X.\n"
	"James G Nourse, The Simple Solutions to Cubic, Puzzles, "
	"Bantam Books, New York, November 1981, pp 25-42.\n"
	"John Ewing & Czes Kosniowski, Puzzle it Out: "
	"Cubes Groups and Puzzles, Cambridge University Press,\n"
	"New York, 1982, pp 50-54?\n"
};
#endif
static const char solveHelp[] = {
	"Auto-solver: sorry, not implemented.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "barrel.scores"
#endif

#define NEVER -1
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord barrelRecord[2][2];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
static Widget moves, record, message;
static Widget orientSwitch, pairsSwitch, practiceSwitch;
#if 0
static Widget tile, face;
#endif
static Widget speed;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
#else
static char titleDsp[TITLELEN] = "";
#endif
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char usernameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap barrelIcon = None;
static Widget topLevel, barrel;
static Arg arg[3];
static char *progDsp;
static char usernameDsp[USERNAMELEN] = "";
#endif

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(arg[0], XmNlabelString, xmstr);
	XtSetValues(w, arg, 1);
}
#else
static void
PrintState(
#ifndef WINVER
Widget w,
#endif
Boolean pairs, int moves, char *msg)
{
	if (pairs)
		(void) sprintf(titleDsp, "%s: norm@ (%d/%s) - %s",
			progDsp, moves, recordDsp, msg);
	else
		(void) sprintf(titleDsp, "%s: ind@ (%d/%s) - %s",
			progDsp, moves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(Wnd, (LPSTR) titleDsp);
#else
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
#endif
}
#endif

static void
InitRecords(void)
{
	int k, l;

	for (l = 0; l < 2; l++)
		for (k = 0; k < 2; k++) {
			barrelRecord[l][k].score = NEVER;
			(void) strncpy(barrelRecord[l][k].name,
				NOACCESS, USERNAMELEN);
		}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int k, l, n;
	char username[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "r")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (l = 0; l < 2; l++)
		for (k = 0; k < 2; k++) {
			(void) fscanf(fp, "%d %s\n", &n, username);
			if (n <= barrelRecord[l][k].score ||
					barrelRecord[l][k].score <= NEVER) {
				barrelRecord[l][k].score = n;
				(void) strncpy(barrelRecord[l][k].name,
					username, USERNAMELEN);
			}
		}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int k, l;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "w")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockfile[FILENAMELEN];

		(void) strncpy(lockfile, name, FILENAMELEN - 6);
		(void) strcat(lockfile, ".lock");
		while (((lfd = open(lockfile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
			  "Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
			  "Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (l = 0; l < 2; l++) {
			for (k = 0; k < 2; k++) {
				(void) fprintf(fp, "%d %s\n",
					barrelRecord[l][k].score,
					barrelRecord[l][k].name);
			}
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockfile);
#endif
		(void) fclose(fp);
	}
	free(lname);
	free(fname);
}

static void
PrintRecord(Boolean orient, Boolean pairs, Boolean practice)
{
	int k = (orient) ? 1 : 0, l = (pairs) ? 1 : 0;

	if (practice)
		(void) strncpy(recordDsp, "practice", MESSAGELEN);
#if 0
	else if (tiles > MAXTILES) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	}
#endif
	else if (barrelRecord[l][k].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			barrelRecord[l][k].score,
			barrelRecord[l][k].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static Boolean
HandleSolved(int counter, Boolean orient, Boolean pairs)
{
	int k = (orient) ? 1 : 0, l = (pairs) ? 1 : 0;

	if ((counter < barrelRecord[l][k].score ||
			 barrelRecord[l][k].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		barrelRecord[l][k].score = counter;
		(void) strncpy(barrelRecord[l][k].name, usernameDsp,
			USERNAMELEN);
		WriteRecords();
		PrintRecord(orient, pairs, False);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
BarrelWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	Boolean orient, pairs, practice;
	char *username;

#ifdef WINVER
	InitializeBarrel(w, brush);

	w->barrel.started = FALSE;
	orient = w->barrel.orient;
	pairs = w->barrel.pairs;
	practice = w->barrel.practice;
	username = w->barrel.username;
#else
	XtVaGetValues(w,
		XtNuserName, &username,
		XtNorient, &orient,
		XtNpairs, &pairs,
		XtNpractice, &practice, NULL);
#ifdef HAVE_MOTIF
#if 0
	if (tiles > MAXTILES)
		XtVaSetValues(tile, XmNmaximum, tiles, NULL);
	XmScaleSetValue(tile, tiles);
	XmScaleSetValue(face, faces);
#endif
	XmToggleButtonSetState(orientSwitch, orient, True);
	XmToggleButtonSetState(pairsSwitch, pairs, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
	{
		int delay;

		XtVaGetValues(barrel, XtNdelay, &delay, NULL);
		XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay - 1);
	}
#endif
#endif
	SET_STARTED(w, False);
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(usernameDsp, username, USERNAMELEN);
#endif
	if (!strcmp(username, "") || !strcmp(username, "(null)") ||
	    !strcmp(username, NOACCESS) || !strcmp(username, NOBODY)) {
		/* The NOACCESS is not necessary, but it may stop silliness. */
#ifdef WINVER
		(void) strncpy(usernameDsp, username, USERNAMELEN);
#else
		(void) sprintf(usernameDsp, "%s", getlogin());
		if (!strcmp(usernameDsp, "") ||
				!strcmp(usernameDsp, "(null)") ||
				!strcmp(usernameDsp, NOACCESS) ||
				!strcmp(usernameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(usernameDsp, "%s", "guest");
#endif
	}
	PrintRecord(orient, pairs, practice);
#ifndef HAVE_MOTIF
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		pairs, movesDsp, messageDsp);
#endif
}

#ifdef WINVER
void
SetBarrel(BarrelWidget w, int reason)
#else
static void
CallbackBarrel(Widget w, caddr_t clientData, barrelCallbackStruct * callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	Boolean orient, pairs, practice, cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	orient = w->barrel.orient;
	pairs = w->barrel.pairs;
	practice = w->barrel.practice;
	cheat = w->barrel.cheat;
#else
	XtVaGetValues(w,
		XtNorient, &orient,
		XtNpairs, &pairs,
		XtNpractice, &practice,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case BARREL_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
			break;
#ifndef WINVER
		case BARREL_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(practiceDialog);
#else
			XtSetArg(arg[0], XtNmenu, 9); /* menu choice */
			XtSetValues(barrel, arg, 1);
#endif
			break;
		case BARREL_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
			XtSetValues(barrel, arg, 1);
#endif
			break;
#endif
		case BARREL_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else

#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case BARREL_RESTORE:
			if (practice) {
				(void) strncpy(recordDsp, "practice", MESSAGELEN);
#ifdef HAVE_MOTIF
				PrintState(record, recordDsp);
#endif
			}
			movesDsp = 0;
			break;
		case BARREL_RESET:
			movesDsp = 0;
			break;
		case BARREL_BLOCKED:
			(void) strncpy(messageDsp, "Blocked", MESSAGELEN);
			break;
		case BARREL_SPACE:
#if 0
			/* Too annoying */
			(void) strncpy(messageDsp, "A space can not slide",
				MESSAGELEN);
#endif
			break;
		case BARREL_IGNORE:
			(void) strncpy(messageDsp, "Randomize to start",
				MESSAGELEN);
			break;
		case BARREL_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case BARREL_CONTROL:
			return;
		case BARREL_SOLVED:
			if (practice)
				movesDsp = 0;
			else if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", usernameDsp);
			else if (HandleSolved(movesDsp, orient, pairs))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", usernameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
			SET_STARTED(w, False);
			break;
		case BARREL_PRACTICE:
			movesDsp = 0;
			practice = !practice;
			if (!practice)
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			PrintRecord(orient, pairs, practice);
#ifdef WINVER
			w->barrel.practice = practice;
			w->barrel.started = False;
#else
			XtSetArg(arg[0], XtNpractice, practice);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#endif
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
			break;
		case BARREL_RANDOMIZE:
			movesDsp = 0;
#ifdef WINVER
			w->barrel.practice = practice;
			w->barrel.started = False;
#else
			XtSetArg(arg[0], XtNpractice, practice);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#endif
			break;
		case BARREL_ORIENT:
			movesDsp = 0;
			orient = !orient;
			PrintRecord(orient, pairs, practice);
#ifdef WINVER
			w->barrel.orient = orient;
#else
			XtSetArg(arg[0], XtNorient, orient);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(orientSwitch, orient, True);
#endif
#endif
			break;
		case BARREL_PAIRS:
			movesDsp = 0;
			pairs = !pairs;
			PrintRecord(orient, pairs, practice);
#ifdef WINVER
			w->barrel.pairs = pairs;
#else
			XtSetArg(arg[0], XtNpairs, pairs);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(pairsSwitch, pairs, True);
#endif
#endif
			break;
#if 0
		case BARREL_DEC_X:
			movesDsp = 0;
			tiles--;
			PrintRecord(tiles, faces, orient, pairs, practice);
#ifdef WINVER
			w->barrel.tiles = tiles;
#else
			XtSetArg(arg[0], XtNtiles, tiles);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(tile, tiles);
			if (tiles >= MAXTILES)
				XtVaSetValues(tile, XmNmaximum, tiles, NULL);
#endif
#endif
			break;
		case BARREL_INC_X:
			movesDsp = 0;
			tiles++;
			PrintRecord(tiles, faces, orient, pairs, practice);
#ifdef WINVER
			w->barrel.tiles = tiles;
#else
			XtSetArg(arg[0], XtNtiles, tiles);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			if (tiles > MAXTILES)
				XtVaSetValues(tile, XmNmaximum, tiles, NULL);
			XmScaleSetValue(tile, tiles);
#endif
#endif
			break;
		case BARREL_DEC_Y:
			movesDsp = 0;
			faces--;
			PrintRecord(tiles, faces, orient, pairs, practice);
#ifdef WINVER
			w->barrel.faces = faces;
#else
			XtSetArg(arg[0], XtNfaces, faces);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(face, faces);
#endif
#endif
			break;
		case BARREL_INC_Y:
			movesDsp = 0;
			faces++;
			PrintRecord(tiles, faces, orient, pairs, practice);
#ifdef WINVER
			w->barrel.faces = faces;
#else
			XtSetArg(arg[0], XtNfaces, faces);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(face, faces);
#endif
			break;
#endif
#endif
		case BARREL_COMPUTED:
			SET_STARTED(w, False);
			break;
		case BARREL_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
#ifdef HAVE_MOTIF
		case BARREL_SPEED:
			{
				int oldDelay, delay;

				XtVaGetValues(barrel, XtNdelay, &delay, NULL);
				oldDelay = delay;
				if (delay > MAXSPEED - MINSPEED)
					delay = MAXSPEED - MINSPEED;
				if (delay < 0)
					delay = 0;
				if (delay != oldDelay) {
					XtSetArg(arg[0], XtNdelay, delay);
					XtSetValues(w, arg, 1);
				}
				XmScaleSetValue(speed,
					MAXSPEED + MINSPEED - delay - 1);
			}
			return;
#endif
	}
#ifdef HAVE_MOTIF
	PrintState(message, messageDsp);
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
#else
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		pairs, movesDsp, messageDsp);
#endif
}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;
	static Boolean mousePressed = False;

	Wnd = widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.barrel.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterBarrel(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.barrel.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeaveBarrel(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyBarrel(brush);
			break;
		case WM_SIZE:
			ResizeBarrel(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposeBarrel(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			RandomizeBarrel(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			mousePressed = True;
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectBarrel(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_MOUSEMOVE:
			if (!(wParam & MK_LBUTTON)) {
				if (mousePressed) {
					mousePressed = False;
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ReleaseBarrel(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
					(void) ReleaseDC(hWnd, widget.core.hDC);
				}
				break;
			}
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			MotionBarrel(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			mousePressed = False;
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleaseBarrel(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
		case WM_MOUSEWHEEL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			{
				int zDelta = ((short) HIWORD(wParam));
				POINT cursor, origin;

				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				if (zDelta > (WHEEL_DELTA >> 1)) {
					MoveBarrelInput(&widget, cursor.x - origin.x, TOP,
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
				} else if (zDelta < -(WHEEL_DELTA >> 1)) {
					MoveBarrelInput(&widget, cursor.x - origin.x, BOTTOM,
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
				}
			}
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#endif
		case WM_COMMAND:
			switch LOWORD((wParam)) {
				case IDM_GET:
					GetBarrel(&widget);
					ResizeBarrel(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WriteBarrel(&widget);
					break;
				case IDM_EXIT:
					DestroyBarrel(brush);
					break;
				case IDM_HIDE:
					HideBarrel(&widget);
					break;
				case IDM_CLEAR:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ClearBarrel(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoBarrel(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					RandomizeBarrel(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_PRACTICE:
					PracticeBarrel(&widget);
					SizeBarrel(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolveBarrel(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_ORIENT:
					OrientizeBarrel(&widget);
					SizeBarrel(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_PAIRS:
					(void) PairsBarrel(&widget);
					SizeBarrel(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SPEED:
					SpeedBarrel(&widget);
					break;
				case IDM_SLOW:
					SlowBarrel(&widget);
					break;
				case IDM_SOUND:
					SoundBarrel(&widget);
					break;
				case IDM_LEFT:
				case IDM_RIGHT:
				case IDM_SHIFT_UP:
				case IDM_SHIFT_DOWN:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					(void) MoveBarrelInput(&widget, 0,
							(int) LOWORD(wParam) - IDM_SHIFT_UP, !((int) wParam % 2));
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UP:
				case IDM_DOWN:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveBarrelInput(&widget, cursor.x - origin.x,
								(int) LOWORD(wParam) - IDM_UP, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
						"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
						"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
						"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#if 0
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#else
		512 + 10, 512 + 10 + 34,
#endif
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char * programName)
{
	(void) fprintf(stderr, "usage: %s\n", programName);
	(void) fprintf(stderr,
		"\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
		"\t[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n");
	(void) fprintf(stderr,
		"\t[-{foreground|fg} {color}] [-{background|bg} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{border|bd} {color}] [-tile {color}]\n");
	(void) fprintf(stderr,
		"\t[-face{0|1|2|3|4|5} {color}]\n");
#if 0
	(void) fprintf(stderr,
		"\t[-tiles {int}] [-faces {int}]\n");
#endif
	(void) fprintf(stderr,
		"\t[-[no]orient] [-[no]pairs]\n");
	(void) fprintf(stderr,
		"\t[-[no]install] [-picture {filename}] [-delay msecs] [-[no]sound]\n");
	(void) fprintf(stderr,
		"\t[-bumpsound {filename}] [-movesound {filename}]\n");
	(void) fprintf(stderr,
		"\t[-base {int}] [-{font|fn} {fontname}] [-username {string}]\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*barrel.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*barrel.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*barrel.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*barrel.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*barrel.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*barrel.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*barrel.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*barrel.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*barrel.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*barrel.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-tile", (char *) "*barrel.tileColor", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*barrel.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*barrel.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*barrel.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*barrel.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*barrel.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*barrel.faceColor5", XrmoptionSepArg, NULL},
#if 0
	{(char *) "-tiles", (char *) "*barrel.tiles", XrmoptionSepArg, NULL},
	{(char *) "-faces", (char *) "*barrel.faces", XrmoptionSepArg, NULL},
#endif
	{(char *) "-orient", (char *) "*barrel.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*barrel.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-pairs", (char *) "*barrel.pairs", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopairs", (char *) "*barrel.pairs", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-install", (char *) "*barrel.install", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noinstall", (char *) "*barrel.install", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-picture", (char *) "*barrel.picture", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*barrel.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*barrel.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*barrel.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bumpsound", (char *) "*barrel.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-movesound", (char *) "*barrel.moveSound", XrmoptionSepArg, NULL},
	{(char *) "-base", (char *) "*barrel.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*barrel.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*barrel.font", XrmoptionSepArg, NULL},
	{(char *) "-username", (char *) "*barrel.userName", XrmoptionSepArg, NULL}
};

#ifdef HAVE_MOTIF
static void
CallbackBarrelPractice(Widget w, XtPointer clientData,
		XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 9); /* menu choice */
		XtSetValues(barrel, arg, 1);
	}
}

static void
CallbackBarrelRandomize(Widget w, XtPointer clientData,
		XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
		XtSetValues(barrel, arg, 1);
	}
}

#if 0
static void
TileSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int tiles = cbs->value, faces, old;
	Boolean orient, pairs, practice;

	XtVaGetValues(barrel,
		XtNtiles, &old,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNpairs, &pairs,
		XtNpractice, &practice, NULL);
	if (old == tiles)
		return;
	XtVaSetValues(barrel,
		XtNtiles, tiles, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, pairs, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
FaceSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int tiles, faces = cbs->value, old;
	Boolean orient, pairs, practice;

	XtVaGetValues(barrel,
		XtNtiles, &tiles,
		XtNfaces, &old,
		XtNorient, &orient,
		XtNpairs, &pairs,
		XtNpractice, &practice, NULL);
	if (old == faces)
		return;
	XtVaSetValues(barrel,
		XtNfaces, faces, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, pairs, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}
#endif

static void
OrientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	Boolean orient = cbs->set, pairs, practice, oldOrient;
	char *picture;

	XtVaGetValues(barrel,
		XtNorient, &oldOrient,
		XtNpairs, &pairs,
		XtNpicture, &picture,
		XtNpractice, &practice, NULL);
	if (oldOrient == orient)
		return;
	if (!orient && picture && *picture) {
		XmToggleButtonSetState(orientSwitch, oldOrient, True);
		return;
	}
	XtVaSetValues(barrel,
		XtNorient, orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(orient, pairs, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PairsToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	Boolean orient, pairs = cbs->set, practice, oldPairs;

	XtVaGetValues(barrel,
		XtNorient, &orient,
		XtNpairs, &oldPairs,
		XtNpractice, &practice, NULL);
	if (oldPairs == pairs)
		return;
	XtVaSetValues(barrel,
		XtNpairs, pairs, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(orient, pairs, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PracticeToggle(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct * cbs)
{
	Boolean orient, pairs, practice = cbs->set, oldPract;

	XtVaGetValues(barrel,
		XtNorient, &orient,
		XtNpairs, &pairs,
		XtNpractice, &oldPract, NULL);
	if (oldPract == practice)
		return;
	XtVaSetValues(barrel,
		XtNpractice, practice,
		XtNstart, False, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(orient, pairs, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGELEN);
	PrintState(message, messageDsp);
}

static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int delay = MAXSPEED + MINSPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(barrel,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(barrel, XtNdelay, delay, NULL);
	}
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtSetArg(arg[0], XtNmenu, val);
	XtSetValues(barrel, arg, 1);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;
	XtSetArg(arg[0], XtNmenu, val + 3); /* GWQ */
	XtSetValues(barrel, arg, 1);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, speedRowCol, switchRowCol;
	Widget messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, randomizeString, solveString;
	XmString orientString, pairsString, practiceString;
	XmString speedString, slowString, soundString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Barrel",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) barrel_xpm, &barrelIcon, NULL,
			&xpmAttributes);
	}
	if (barrelIcon == (Pixmap) NULL)
#endif
		barrelIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) barrel_bits,
			barrel_width, barrel_height);
	XtSetArg(arg[0], XtNiconPixmap, barrelIcon);
#ifdef HAVE_MOTIF
	/* not XmEXPLICIT */
	XtSetArg(arg[1], XmNkeyboardFocusPolicy, XmPOINTER);
	XtSetValues(topLevel, arg, 2);
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "Solve");
	orientString = XmStringCreateSimple((char *) "Orientize");
	pairsString = XmStringCreateSimple((char *) "PAirs");
	practiceString = XmStringCreateSimple((char *) "Practice");
	speedString = XmStringCreateSimple((char *) ">Speed");
	slowString = XmStringCreateSimple((char *) "<Slow");
	soundString = XmStringCreateSimple((char *) "@Sound");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, pairsString, 'A', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(orientString);
	XmStringFree(pairsString);
	XmStringFree(practiceString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) CallbackBarrelPractice);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackBarrelRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move tile", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

#if 0
	tilesRowCol = XtVaCreateManagedWidget("tilesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	tile = XtVaCreateManagedWidget("tile",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Tiles", 6,
		XmNminimum, MINTILES,
		XmNmaximum, MAXTILES,
		XmNvalue, DEFAULTTILES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(tile, XmNvalueChangedCallback,
		(XtCallbackProc) TileSlider, (XtPointer) NULL);
	face = XtVaCreateManagedWidget("face",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Faces", 6,
		XmNminimum, MINFACES,
		XmNmaximum, MAXFACES,
		XmNvalue, DEFAULTFACES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(face, XmNvalueChangedCallback,
		(XtCallbackProc) FaceSlider, (XtPointer) NULL);
#endif
	speedRowCol = XtVaCreateManagedWidget("speedRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	speed = XtVaCreateManagedWidget("Animation Speed",
		xmScaleWidgetClass, speedRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - DEFAULTDELAY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTORIENT, NULL);
	XtAddCallback(orientSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) OrientToggle, (XtPointer) NULL);
	pairsSwitch = XtVaCreateManagedWidget("Pairs rigid",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTPAIRS, NULL);
	XtAddCallback(pairsSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PairsToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTPRACTICE, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PracticeToggle, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Barrel! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);

	barrel = XtCreateManagedWidget("barrel",
		barrelWidgetClass, mainPanel, NULL, 0);
#else
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(topLevel, arg, 2);
	barrel = XtCreateManagedWidget("barrel",
		barrelWidgetClass, topLevel, NULL, 0);
#endif
	XtAddCallback(barrel, XtNselectCallback,
		(XtCallbackProc) CallbackBarrel, (XtPointer) NULL);
	Initialize(barrel);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(barrel), (unsigned int) AnyButton, AnyModifier,
		XtWindow(barrel), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(barrel),
		XCreateFontCursor(XtDisplay(barrel), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
