/*-
# MOTIF/X-BASED MISSING LINK(tm)
#
#  xmlink.c
#
###
#
#  Copyright (c) 1994 - 2005	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/11/15 X/Windows
  Version 5: 95/10/01 Xt/Motif
  Version 4: 94/08/30 Xt
*/

#ifdef WINVER
#include "MlinkP.h"
#include "wmlink.h"
#define TITLE "wmlink"

static MlinkRec widget;
static HWND Wnd;

#ifndef SCOREPATH
#define SCOREPATH "c:\\Windows"
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(Wnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->mlink.started = b
#else
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef VMS
#include <unixlib.h>
#define getlogin() cuserid(NULL)
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#if HAVE_FCNTL_H
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtSetArg(arg[0], XtNstart, b); XtSetValues(w, arg, 1)
#include "Mlink.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/mlink.xpm"
#endif
#include "icons/mlink.xbm"
#ifndef SCOREPATH
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif

#ifdef HAVE_MOTIF
static const char aboutHelp[] = {
	"Version 7.1.3\n"
	"Send bugs (reports or fixes) to the author: "
	"David Bagley <bagleyd@tux.org>\n"
	"The latest version is at: "
	"http://www.tux.org/~bagleyd/puzzles.html\n"
};
static const char optionsHelp[] = {
	"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]] "
	"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
	"[-{foreground|fg} {color}] [-{background|bg} {color}] "
	"[-{border|bd} {color}] [-tile {color}]\n"
	"[-face{0|1|2|3|4|5|6|7} {color}] [-tiles {int}] "
	"[-faces {int}] [-[no]orient] [-[no]pairs]\n"
	"[-[no]install] [-picture {filename}] [-delay msecs] "
	"[-[no]sound] [-bumpsound {filename}] [-movesound {filename}]\n"
	"[-base {int}] [-{font|fn} {fontname}] [-username {string}]\n"
};
#endif
#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
	"Ideal's The Missing Link, a 4x4 puzzle,  riding on the wave "
	"of Rubik's Cube craze.  There are  many variations on this\n"
	"theme, including the \"6x6\" puzzle and Babylon Tower.  What "
	"makes the Missing Link a harder puzzle to solve is that the\n"
	"2 center positions rotate together.\n"
};
static const char featuresHelp[] = {
	"Press \"mouse-left\" button to move a tile. Release "
	"\"mouse-left\" button in the same row would move the tile towards\n"
	"the space (unless there is no space in that row).  "
	"Release on a different row will move the tiles up or down.\n"
	"\n"
	"Click \"mouse-center\" button, or press \"P\" or \"p\" "
	"keys, to toggle the practice mode (in practice mode the\n"
	"record should say \"practice\").  This is good for learning "
	"moves and experimenting.\n"
	"\n"
	"Click \"mouse-right\" button, or press \"R\" or \"r\" "
	"keys, to randomize the puzzle.\n"
	"\n"
	"Press \"O\" or \"o\" keys to toggle orient mode.  Slightly "
	"more challenging on (default is off).\n"
	"\n"
	"Press \"M\" or \"m\" keys to toggle the middle mode.  All "
	"the middle tiles rotate together.  This is considerably\n"
	"more challenging on (default is on).\n"
	"\n"
	"\"S\" or \"s\" keys reserved for the auto-solver "
	"(not implemented).\n"
	"\n"
	"Press \"U\" or \"u\" keys to undo a move.\n"
	"\n"
	"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
	"\n"
	"Press \"W\" or \"w\" keys to save a puzzle.\n"
	"\n"
	"Press \"C\" or \"c\" keys to clear a puzzle.\n"
	"\n"
	"Press \">\" or \".\" keys to speed up the movement of tiles.\n"
	"\n"
	"Press \"<\" or \",\" keys to slow down the movement of tiles.\n"
	"\n"
	"Press \"@\" key to toggle the sound.\n"
	"\n"
	"Press \"Esc\" key to hide program.\n"
	"\n"
	"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
	"\n"
	"Use the key pad, \"R\" keys, or arrow keys to move without "
	"the mouse.\n"
	"Key pad is defined for Missing Link as:\n"
	"  8    Up\n"
	"  ^\n"
	"4< >6  Left, Right\n"
	"  v\n"
	"  2    Down\n"
	"\n"
	"Use the shift key and the up or down arrow keys to rotate "
	"all the tiles.\n"
	"\n"
	"Use the control key and the keypad or arrow keys to reduce "
	"or enlarge the complexity of the puzzle.\n"
};
static const char referencesHelp[] = {
	"L. E. Horden, Sliding Piece Puzzles (Recreations in "
	"Mathematics Series), Oxford University\n"
	"Press 1986, Plate X.\n"
	"you are about to find the solution: The Missing Link, "
	"Ideal Toy Corporation, 1981.\n"
};
#endif
static const char solveHelp[] = {
	"Auto-solver: sorry, only implemented "
	"for tiles = 4 and faces = 4.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "mlink.scores"
#endif

#define MAXTILES 8
#define NEVER -1
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord mlinkRecord[2][2][MAXFACES - MINFACES + 1]
[MAXTILES - MINTILES + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
static Widget moves, record, message;
static Widget orientSwitch, middleSwitch, practiceSwitch, tile, face, speed;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
#else
static char titleDsp[TITLELEN] = "";
#endif
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char usernameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap mlinkIcon = None;
static Widget topLevel, mlink;
static Arg arg[3];
static char *progDsp;
static char usernameDsp[USERNAMELEN] = "";
#endif

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(arg[0], XmNlabelString, xmstr);
	XtSetValues(w, arg, 1);
}
#else
static void
PrintState(
#ifndef WINVER
Widget w,
#endif
int tiles, int faces, Boolean middle, int moves, char *msg)
{
	if (middle)
		(void) sprintf(titleDsp, "%s: %dx%d norm@ (%d/%s) - %s",
			progDsp, tiles, faces, moves, recordDsp, msg);
	else
		(void) sprintf(titleDsp, "%s: %dx%d ind@ (%d/%s) - %s",
			progDsp, tiles, faces, moves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(Wnd, (LPSTR) titleDsp);
#else
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
#endif
}
#endif

static void
InitRecords(void)
{
	int i, j, k, l;

	for (l = 0; l < 2; l++)
		for (k = 0; k < 2; k++)
			for (j = 0; j < MAXFACES - MINFACES + 1; j++)
				for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
					mlinkRecord[l][k][j][i].score = NEVER;
					(void) strncpy(mlinkRecord[l][k][j][i].name,
						NOACCESS, USERNAMELEN);
				}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int i, j, k, l, n;
	char username[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "r")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (l = 0; l < 2; l++)
		for (k = 0; k < 2; k++)
			for (j = 0; j < MAXFACES - MINFACES + 1; j++)
				for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
					(void) fscanf(fp, "%d %s\n", &n, username);
					if (n <= mlinkRecord[l][k][j][i].score ||
					    mlinkRecord[l][k][j][i].score <= NEVER) {
						mlinkRecord[l][k][j][i].score = n;
						(void) strncpy(mlinkRecord[l][k][j][i].name,
							username, USERNAMELEN);
					}
				}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i, j, k, l;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "w")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockfile[FILENAMELEN];

		(void) strncpy(lockfile, name, FILENAMELEN - 6);
		(void) strcat(lockfile, ".lock");
		while (((lfd = open(lockfile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
			  "Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
			  "Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (l = 0; l < 2; l++) {
			for (k = 0; k < 2; k++) {
				for (j = 0; j < MAXFACES - MINFACES + 1; j++) {
					for (i = 0; i < MAXTILES - MINTILES + 1; i++)
						(void) fprintf(fp, "%d %s\n",
							mlinkRecord[l][k][j][i].score, mlinkRecord[l][k][j][i].name);
					(void) fprintf(fp, "\n");
				}
				(void) fprintf(fp, "\n");
			}
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockfile);
#endif
		(void) fclose(fp);
	}
	free(lname);
	free(fname);
}

static void
PrintRecord(int tiles, int faces, Boolean orient, Boolean middle,
		Boolean practice)
{
	int i = tiles - MINTILES, j = faces - MINFACES;
	int k = (orient) ? 1 : 0, l = (middle) ? 1 : 0;

	if (practice)
		(void) strncpy(recordDsp, "practice", MESSAGELEN);
	else if (tiles > MAXTILES) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	} else if (mlinkRecord[l][k][j][i].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			mlinkRecord[l][k][j][i].score,
			mlinkRecord[l][k][j][i].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static Boolean
HandleSolved(int counter, int tiles, int faces, Boolean orient, Boolean middle)
{
	int i = tiles - MINTILES, j = faces - MINFACES;
	int k = (orient) ? 1 : 0, l = (middle) ? 1 : 0;

	if (tiles <= MAXTILES && (counter < mlinkRecord[l][k][j][i].score ||
			 mlinkRecord[l][k][j][i].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		mlinkRecord[l][k][j][i].score = counter;
		(void) strncpy(mlinkRecord[l][k][j][i].name, usernameDsp,
			USERNAMELEN);
		if (tiles < 4 || faces < 2) {
			mlinkRecord[!l][k][j][i].score = counter;
			(void) strncpy(mlinkRecord[!l][k][j][i].name,
				usernameDsp, USERNAMELEN);
		}
		WriteRecords();
		PrintRecord(tiles, faces, orient, middle, False);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
MlinkWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int tiles, faces;
	Boolean orient, middle, practice;
	char *username;

#ifdef WINVER
	InitializeMlink(w, brush);

	tiles = w->mlink.tiles;
	faces = w->mlink.faces;
	orient = w->mlink.orient;
	middle = w->mlink.middle;
	practice = w->mlink.practice;
	username = w->mlink.username;
#else
	XtVaGetValues(w,
		XtNuserName, &username,
		XtNtiles, &tiles,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNmiddle, &middle,
		XtNpractice, &practice, NULL);
#ifdef HAVE_MOTIF
	if (tiles > MAXTILES)
		XtVaSetValues(tile, XmNmaximum, tiles, NULL);
	XmScaleSetValue(tile, tiles);
	XmScaleSetValue(face, faces);
	XmToggleButtonSetState(orientSwitch, orient, True);
	XmToggleButtonSetState(middleSwitch, middle, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
	{
		int delay;

		XtVaGetValues(mlink, XtNdelay, &delay, NULL);
		XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay - 1);
	}
#endif
#endif
	SET_STARTED(w, False);
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(usernameDsp, username, USERNAMELEN);
#endif
	if (!strcmp(username, "") || !strcmp(username, "(null)") ||
	    !strcmp(username, NOACCESS) || !strcmp(username, NOBODY)) {
		/* The NOACCESS is not necessary, but it may stop silliness. */
#ifdef WINVER
		(void) strncpy(usernameDsp, username, USERNAMELEN);
#else
		(void) sprintf(usernameDsp, "%s", getlogin());
		if (!strcmp(usernameDsp, "") ||
				!strcmp(usernameDsp, "(null)") ||
				!strcmp(usernameDsp, NOACCESS) ||
				!strcmp(usernameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(usernameDsp, "%s", "guest");
#endif
	}
	PrintRecord(tiles, faces, orient, middle, practice);
#ifndef HAVE_MOTIF
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		tiles, faces, middle, movesDsp, messageDsp);
#endif
}

#ifdef WINVER
void
SetMlink(MlinkWidget w, int reason)
#else
static void
CallbackMlink(Widget w, caddr_t clientData, mlinkCallbackStruct * callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int tiles, faces;
	Boolean orient, middle, practice, cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	tiles = w->mlink.tiles;
	faces = w->mlink.faces;
	orient = w->mlink.orient;
	middle = w->mlink.middle;
	practice = w->mlink.practice;
	cheat = w->mlink.cheat;
#else
	XtVaGetValues(w,
		XtNtiles, &tiles,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNmiddle, &middle,
		XtNpractice, &practice,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case MLINK_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
			break;
#ifndef WINVER
		case MLINK_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(practiceDialog);
#else
			XtSetArg(arg[0], XtNmenu, 9); /* menu choice */
			XtSetValues(mlink, arg, 1);
#endif
			break;
		case MLINK_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
			XtSetValues(mlink, arg, 1);
#endif
			break;
#endif
		case MLINK_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case MLINK_RESTORE:
			if (practice) {
				(void) strncpy(recordDsp, "practice", MESSAGELEN);
#ifdef HAVE_MOTIF
				PrintState(record, recordDsp);
#endif
			}
			movesDsp = 0;
			break;
		case MLINK_RESET:
			movesDsp = 0;
			break;
		case MLINK_BLOCKED:
			(void) strncpy(messageDsp, "Blocked", MESSAGELEN);
			break;
		case MLINK_SPACE:
#if 0
			/* Too annoying */
			(void) strncpy(messageDsp, "A space can not slide",
				MESSAGELEN);
#endif
			break;
		case MLINK_IGNORE:
			(void) strncpy(messageDsp, "Randomize to start",
				MESSAGELEN);
			break;
		case MLINK_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case MLINK_CONTROL:
			return;
		case MLINK_SOLVED:
			if (practice)
				movesDsp = 0;
			else if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", usernameDsp);
			else if (HandleSolved(movesDsp, tiles, faces, orient, middle))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", usernameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
			SET_STARTED(w, False);
			break;
		case MLINK_PRACTICE:
			movesDsp = 0;
			practice = !practice;
			if (!practice)
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
			w->mlink.practice = practice;
			w->mlink.started = False;
#else
			XtSetArg(arg[0], XtNpractice, practice);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#endif
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
			break;
		case MLINK_RANDOMIZE:
			movesDsp = 0;
#ifdef WINVER
			w->mlink.practice = practice;
			w->mlink.started = False;
#else
			XtSetArg(arg[0], XtNpractice, practice);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#endif
			break;
		case MLINK_ORIENT:
			movesDsp = 0;
			orient = !orient;
			PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
			w->mlink.orient = orient;
#else
			XtSetArg(arg[0], XtNorient, orient);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(orientSwitch, orient, True);
#endif
#endif
			break;
		case MLINK_MIDDLE:
			movesDsp = 0;
			middle = !middle;
			PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
			w->mlink.middle = middle;
#else
			XtSetArg(arg[0], XtNmiddle, middle);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(middleSwitch, middle, True);
#endif
#endif
			break;
		case MLINK_DEC_X:
			movesDsp = 0;
			tiles--;
			PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
			w->mlink.tiles = tiles;
#else
			XtSetArg(arg[0], XtNtiles, tiles);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(tile, tiles);
			if (tiles >= MAXTILES)
				XtVaSetValues(tile, XmNmaximum, tiles, NULL);
#endif
#endif
			break;
		case MLINK_INC_X:
			movesDsp = 0;
			tiles++;
			PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
			w->mlink.tiles = tiles;
#else
			XtSetArg(arg[0], XtNtiles, tiles);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			if (tiles > MAXTILES)
				XtVaSetValues(tile, XmNmaximum, tiles, NULL);
			XmScaleSetValue(tile, tiles);
#endif
#endif
			break;
		case MLINK_DEC_Y:
			movesDsp = 0;
			faces--;
			PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
			w->mlink.faces = faces;
#else
			XtSetArg(arg[0], XtNfaces, faces);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(face, faces);
#endif
#endif
			break;
		case MLINK_INC_Y:
			movesDsp = 0;
			faces++;
			PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
			w->mlink.faces = faces;
#else
			XtSetArg(arg[0], XtNfaces, faces);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(face, faces);
#endif
#endif
			break;
		case MLINK_COMPUTED:
			SET_STARTED(w, False);
			break;
		case MLINK_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
#ifdef HAVE_MOTIF
		case MLINK_SPEED:
			{
				int oldDelay, delay;

				XtVaGetValues(mlink, XtNdelay, &delay, NULL);
				oldDelay = delay;
				if (delay > MAXSPEED - MINSPEED)
					delay = MAXSPEED - MINSPEED;
				if (delay < 0)
					delay = 0;
				if (delay != oldDelay) {
					XtSetArg(arg[0], XtNdelay, delay);
					XtSetValues(w, arg, 1);
				}
				XmScaleSetValue(speed,
					MAXSPEED + MINSPEED - delay - 1);
			}
			return;
#endif
	}
#ifdef HAVE_MOTIF
	PrintState(message, messageDsp);
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
#else
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		tiles, faces, middle, movesDsp, messageDsp);
#endif
}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;
	static Boolean mousePressed = False;

	Wnd = widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.mlink.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterMlink(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.mlink.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeaveMlink(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyMlink(brush);
			break;
		case WM_SIZE:
			ResizeMlink(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposeMlink(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			RandomizeMlink(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			mousePressed = True;
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectMlink(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_MOUSEMOVE:
			if (!(wParam & MK_LBUTTON)) {
				mousePressed = False;
				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
				ReleaseMlink(&widget, LOWORD(lParam), HIWORD(lParam),
					((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
				(void) ReleaseDC(hWnd, widget.core.hDC);
				break;
			}
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			MotionMlink(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			mousePressed = False;
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleaseMlink(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
		case WM_MOUSEWHEEL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			{
				int zDelta = ((short) HIWORD(wParam));
				POINT cursor, origin;

				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				if (zDelta > (WHEEL_DELTA >> 1)) {
					MoveMlinkInput(&widget, cursor.x - origin.x, TOP,
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
					if (GetKeyState(VK_CONTROL) >> 1) {
						SizeMlink(&widget);
						(void) InvalidateRect(hWnd, NULL, TRUE);
					}
				} else if (zDelta < -(WHEEL_DELTA >> 1)) {
					MoveMlinkInput(&widget, cursor.x - origin.x, BOTTOM,
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
					if (GetKeyState(VK_CONTROL) >> 1) {
						SizeMlink(&widget);
						(void) InvalidateRect(hWnd, NULL, TRUE);
					}
				}
			}
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#endif
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetMlink(&widget);
					ResizeMlink(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WriteMlink(&widget);
					break;
				case IDM_EXIT:
					DestroyMlink(brush);
					break;
				case IDM_HIDE:
					HideMlink(&widget);
					break;
				case IDM_CLEAR:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ClearMlink(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoMlink(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					RandomizeMlink(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_PRACTICE:
					(void) PracticeMlink(&widget);
					SizeMlink(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolveMlink(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_ORIENT:
					OrientizeMlink(&widget);
					SizeMlink(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_MIDDLE:
					(void) MiddleMlink(&widget);
					SizeMlink(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SPEED:
					SpeedMlink(&widget);
					break;
				case IDM_SLOW:
					SlowMlink(&widget);
					break;
				case IDM_SOUND:
					SoundMlink(&widget);
					break;
				case IDM_LEFT:
				case IDM_RIGHT:
				case IDM_SHIFT_UP:
				case IDM_SHIFT_DOWN:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					(void) MoveMlinkInput(&widget, 0,
							(int) LOWORD(wParam) - IDM_SHIFT_UP, !((int) wParam % 2), FALSE);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UP:
				case IDM_DOWN:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveMlinkInput(&widget, cursor.x - origin.x,
								(int) LOWORD(wParam) - IDM_UP, FALSE, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}
					break;
				case IDM_DECY:
				case IDM_INCX:
				case IDM_INCY:
				case IDM_DECX:
					MoveMlinkInput(&widget, 0,
						(int) LOWORD(wParam) - IDM_DECY, FALSE, TRUE);
					SizeMlink(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
					 	"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
					 	"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
					 	"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#if 0
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#else
		512 + 10, 512 + 10 + 34,
#endif
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char * programName)
{
	(void) fprintf(stderr, "usage: %s\n", programName);
	(void) fprintf(stderr,
		"\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
		"\t[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n");
	(void) fprintf(stderr,
		"\t[-{foreground|fg} {color}] [-{background|bg} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{border|bd} {color}] [-tile {color}]\n");
	(void) fprintf(stderr,
		"\t[-face{0|1|2|3|4|5|6|7} {color}]\n");
	(void) fprintf(stderr,
		"\t[-tiles {int}] [-faces {int}] [-[no]orient] [-[no]middle]\n");
	(void) fprintf(stderr,
		"\t[-[no]install] [-picture {filename}] [-delay msecs] [-[no]sound]\n");
	(void) fprintf(stderr,
		"\t[-bumpsound {filename}] [-movesound {filename}]\n");
	(void) fprintf(stderr,
		"\t[-base {int}] [-{font|fn} {fontname}] [-username {string}]\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*mlink.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*mlink.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*mlink.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*mlink.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*mlink.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*mlink.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*mlink.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*mlink.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*mlink.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*mlink.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-tile", (char *) "*mlink.tileColor", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*mlink.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*mlink.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*mlink.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*mlink.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*mlink.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*mlink.faceColor5", XrmoptionSepArg, NULL},
	{(char *) "-face6", (char *) "*mlink.faceColor6", XrmoptionSepArg, NULL},
	{(char *) "-face7", (char *) "*mlink.faceColor7", XrmoptionSepArg, NULL},
	{(char *) "-tiles", (char *) "*mlink.tiles", XrmoptionSepArg, NULL},
	{(char *) "-faces", (char *) "*mlink.faces", XrmoptionSepArg, NULL},
	{(char *) "-orient", (char *) "*mlink.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*mlink.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-middle", (char *) "*mlink.middle", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomiddle", (char *) "*mlink.middle", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-install", (char *) "*mlink.install", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noinstall", (char *) "*mlink.install", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-picture", (char *) "*mlink.picture", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*mlink.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*mlink.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*mlink.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bumpsound", (char *) "*mlink.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-movesound", (char *) "*mlink.moveSound", XrmoptionSepArg, NULL},
	{(char *) "-base", (char *) "*mlink.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*mlink.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*mlink.font", XrmoptionSepArg, NULL},
	{(char *) "-username", (char *) "*mlink.userName", XrmoptionSepArg, NULL}
};

#ifdef HAVE_MOTIF
static void
CallbackMlinkPractice(Widget w, XtPointer clientData,
		XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 9); /* menu choice */
		XtSetValues(mlink, arg, 1);
	}
}

static void
CallbackMlinkRandomize(Widget w, XtPointer clientData,
		XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
		XtSetValues(mlink, arg, 1);
	}
}

static void
TileSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int tiles = cbs->value, faces, old;
	Boolean orient, middle, practice;

	XtVaGetValues(mlink,
		XtNtiles, &old,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNmiddle, &middle,
		XtNpractice, &practice, NULL);
	if (old == tiles)
		return;
	XtVaSetValues(mlink,
		XtNtiles, tiles, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, middle, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
FaceSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int tiles, faces = cbs->value, old;
	Boolean orient, middle, practice;

	XtVaGetValues(mlink,
		XtNtiles, &tiles,
		XtNfaces, &old,
		XtNorient, &orient,
		XtNmiddle, &middle,
		XtNpractice, &practice, NULL);
	if (old == faces)
		return;
	XtVaSetValues(mlink,
		XtNfaces, faces, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, middle, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
OrientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int tiles, faces;
	Boolean orient = cbs->set, middle, practice, oldOrient;
	char *picture;

	XtVaGetValues(mlink,
		XtNtiles, &tiles,
		XtNfaces, &faces,
		XtNorient, &oldOrient,
		XtNmiddle, &middle,
		XtNpicture, &picture,
		XtNpractice, &practice, NULL);
	if (oldOrient == orient)
		return;
	if (!orient && picture && *picture) {
		XmToggleButtonSetState(orientSwitch, oldOrient, True);
		return;
	}
	XtVaSetValues(mlink,
		XtNorient, orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, middle, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
MiddleToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int tiles, faces;
	Boolean orient, middle = cbs->set, practice, oldMid;

	XtVaGetValues(mlink,
		XtNtiles, &tiles,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNmiddle, &oldMid,
		XtNpractice, &practice, NULL);
	if (oldMid == middle)
		return;
	XtVaSetValues(mlink,
		XtNmiddle, middle, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, middle, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PracticeToggle(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct * cbs)
{
	int tiles, faces;
	Boolean orient, middle, practice = cbs->set, oldPract;

	XtVaGetValues(mlink,
		XtNtiles, &tiles,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNmiddle, &middle,
		XtNpractice, &oldPract, NULL);
	if (oldPract == practice)
		return;
	XtVaSetValues(mlink,
		XtNpractice, practice,
		XtNstart, False, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, middle, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGELEN);
	PrintState(message, messageDsp);
}

static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int delay = MAXSPEED + MINSPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(mlink,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(mlink, XtNdelay, delay, NULL);
	}
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtSetArg(arg[0], XtNmenu, val);
	XtSetValues(mlink, arg, 1);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;
	XtSetArg(arg[0], XtNmenu, val + 3); /* GWQ */
	XtSetValues(mlink, arg, 1);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, tilesRowCol, speedRowCol, switchRowCol;
	Widget messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, randomizeString, solveString;
	XmString orientString, middleString, practiceString;
	XmString speedString, slowString, soundString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Mlink",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) mlink_xpm, &mlinkIcon, NULL,
			&xpmAttributes);
	}
	if (mlinkIcon == (Pixmap) NULL)
#endif
		mlinkIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) mlink_bits,
			mlink_width, mlink_height);
	XtSetArg(arg[0], XtNiconPixmap, mlinkIcon);
#ifdef HAVE_MOTIF
	/* not XmEXPLICIT */
	XtSetArg(arg[1], XmNkeyboardFocusPolicy, XmPOINTER);
	XtSetValues(topLevel, arg, 2);
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "Solve");
	orientString = XmStringCreateSimple((char *) "Orientize");
	middleString = XmStringCreateSimple((char *) "Middle");
	practiceString = XmStringCreateSimple((char *) "Practice");
	speedString = XmStringCreateSimple((char *) ">Speed");
	slowString = XmStringCreateSimple((char *) "<Slow");
	soundString = XmStringCreateSimple((char *) "@Sound");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, middleString, 'M', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(orientString);
	XmStringFree(middleString);
	XmStringFree(practiceString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) CallbackMlinkPractice);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackMlinkRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move tile", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	tilesRowCol = XtVaCreateManagedWidget("tilesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	tile = XtVaCreateManagedWidget("tile",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Tiles", 6,
		XmNminimum, MINTILES,
		XmNmaximum, MAXTILES,
		XmNvalue, DEFAULTTILES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(tile, XmNvalueChangedCallback,
		(XtCallbackProc) TileSlider, (XtPointer) NULL);
	face = XtVaCreateManagedWidget("face",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Faces", 6,
		XmNminimum, MINFACES,
		XmNmaximum, MAXFACES,
		XmNvalue, DEFAULTFACES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(face, XmNvalueChangedCallback,
		(XtCallbackProc) FaceSlider, (XtPointer) NULL);
	speedRowCol = XtVaCreateManagedWidget("speedRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	speed = XtVaCreateManagedWidget("Animation Speed",
		xmScaleWidgetClass, speedRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - DEFAULTDELAY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTORIENT, NULL);
	XtAddCallback(orientSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) OrientToggle, (XtPointer) NULL);
	middleSwitch = XtVaCreateManagedWidget("Middle rigid",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTMIDDLE, NULL);
	XtAddCallback(middleSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) MiddleToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTPRACTICE, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PracticeToggle, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Missing Link! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);

	mlink = XtCreateManagedWidget("mlink",
		mlinkWidgetClass, mainPanel, NULL, 0);
#else
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(topLevel, arg, 2);
	mlink = XtCreateManagedWidget("mlink",
		mlinkWidgetClass, topLevel, NULL, 0);
#endif
	XtAddCallback(mlink, XtNselectCallback,
		(XtCallbackProc) CallbackMlink, (XtPointer) NULL);
	Initialize(mlink);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(mlink), (unsigned int) AnyButton, AnyModifier,
		XtWindow(mlink), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(mlink),
		XCreateFontCursor(XtDisplay(mlink), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
