/*-
# X-BASED HEXAGONS
#
#  Hexagons.c
#
###
#
#  Copyright (c) 1994 - 2005	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Hexagons */

#include "file.h"
#include "rngs.h"
#include "sound.h"
#include "HexagonsP.h"

#ifndef PICTURE
#if 1
#define PICTURE ""
#else
#ifdef WINVER
#define PICTURE "picture"
#else
#ifdef HAVE_XPM
#define PICTURE "./mandrill.xpm"
#else
#define PICTURE "./mandrill.xbm"
#endif
#endif
#endif
#endif

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "whexagons.ini"
#endif

#define SECTION "setup"
#else
#include "picture.h"

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static Boolean SetValuesHexagons(Widget current, Widget request, Widget renew);
static void QuitHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void DestroyHexagons(Widget old);
static void ResizeHexagons(HexagonsWidget w);
static void SizeHexagons(HexagonsWidget w);
static void InitializeHexagons(Widget request, Widget renew);
static void ExposeHexagons(Widget renew, XEvent * event, Region region);
static void HideHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void SelectHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void ReleaseHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeHexagonsMaybe(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeHexagons2(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void GetHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void WriteHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void ClearHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void UndoHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void SolveHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void ModeHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void SpeedHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void SlowHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void SoundHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void EnterHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void LeaveHexagons(HexagonsWidget w, XEvent * event, char **args, int nArgs);

static void MoveHexagonsTl(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void MoveHexagonsTop(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void MoveHexagonsTr(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void MoveHexagonsLeft(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void MoveHexagonsRight(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void MoveHexagonsBl(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void MoveHexagonsBottom(HexagonsWidget w, XEvent * event, char **args, int nArgs);
static void MoveHexagonsBr(HexagonsWidget w, XEvent * event, char **args, int nArgs);

static char defaultTranslationsHexagons[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <KeyPress>r: Randomize()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>o: Mode()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 <KeyPress>@: Sound()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListHexagons[] =
{
	{(char *) "Quit", (XtActionProc) QuitHexagons},
	{(char *) "Hide", (XtActionProc) HideHexagons},
	{(char *) "MoveTl", (XtActionProc) MoveHexagonsTl},
	{(char *) "MoveTop", (XtActionProc) MoveHexagonsTop},
	{(char *) "MoveTr", (XtActionProc) MoveHexagonsTr},
	{(char *) "MoveLeft", (XtActionProc) MoveHexagonsLeft},
	{(char *) "MoveRight", (XtActionProc) MoveHexagonsRight},
	{(char *) "MoveBl", (XtActionProc) MoveHexagonsBl},
	{(char *) "MoveBottom", (XtActionProc) MoveHexagonsBottom},
	{(char *) "MoveBr", (XtActionProc) MoveHexagonsBr},
	{(char *) "Select", (XtActionProc) SelectHexagons},
	{(char *) "Release", (XtActionProc) ReleaseHexagons},
	{(char *) "Randomize", (XtActionProc) RandomizeHexagons},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizeHexagonsMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizeHexagons2},
	{(char *) "Get", (XtActionProc) GetHexagons},
	{(char *) "Write", (XtActionProc) WriteHexagons},
	{(char *) "Clear", (XtActionProc) ClearHexagons},
	{(char *) "Undo", (XtActionProc) UndoHexagons},
	{(char *) "Solve", (XtActionProc) SolveHexagons},
	{(char *) "Mode", (XtActionProc) ModeHexagons},
	{(char *) "Speed", (XtActionProc) SpeedHexagons},
	{(char *) "Slow", (XtActionProc) SlowHexagons},
	{(char *) "Sound", (XtActionProc) SoundHexagons},
	{(char *) "Enter", (XtActionProc) EnterHexagons},
	{(char *) "Leave", (XtActionProc) LeaveHexagons}
};

static XtResource resourcesHexagons[] =
{
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(HexagonsWidget, hexagons.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(HexagonsWidget, hexagons.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(HexagonsWidget, hexagons.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(HexagonsWidget, hexagons.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(HexagonsWidget, hexagons.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNtileColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(HexagonsWidget, hexagons.tileColor),
	 XtRString, (caddr_t) "gray75" /*XtDefaultForeground*/},
	{XtNtileBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(HexagonsWidget, hexagons.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultBackground*/},
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(HexagonsWidget, core.width),
	 XtRString, (caddr_t) "259"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(HexagonsWidget, core.height),
	 XtRString, (caddr_t) "200"},
	{XtNsizeX, XtCSizeX, XtRInt, sizeof (int),
	 XtOffset(HexagonsWidget, hexagons.sizeX),
	 XtRString, (caddr_t) "3"}, /* DEFAULTTILESX */
	{XtNsizeY, XtCSizeY, XtRInt, sizeof (int),
	 XtOffset(HexagonsWidget, hexagons.sizeY),
	 XtRString, (caddr_t) "3"}, /* DEFAULTTILESY */
	{XtNcorners, XtCCorners, XtRBoolean, sizeof (Boolean),
	 XtOffset(HexagonsWidget, hexagons.corners),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULTCORNERS */
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(HexagonsWidget, hexagons.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(HexagonsWidget, hexagons.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNinstall, XtCInstall, XtRBoolean, sizeof (Boolean),
	 XtOffset(HexagonsWidget, hexagons.install),
	 XtRString, (caddr_t) "FALSE"},
	{XtNpicture, XtCPicture, XtRString, sizeof (String),
	 XtOffset(HexagonsWidget, hexagons.picture),
	 XtRString, (caddr_t) PICTURE},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(HexagonsWidget, hexagons.delay),
	 XtRString, (caddr_t) "10"}, /* DEFAULTDELAY */
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(HexagonsWidget, hexagons.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNbumpSound, XtCBumpSound, XtRString, sizeof (String),
	 XtOffset(HexagonsWidget, hexagons.bumpSound),
	 XtRString, (caddr_t) BUMPSOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(HexagonsWidget, hexagons.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNbase, XtCBase, XtRInt, sizeof (int),
	 XtOffset(HexagonsWidget, hexagons.base),
	 XtRString, (caddr_t) "10"}, /* DEFAULTBASE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(HexagonsWidget, hexagons.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(HexagonsWidget, hexagons.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(HexagonsWidget, hexagons.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(HexagonsWidget, hexagons.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(HexagonsWidget, hexagons.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(HexagonsWidget, hexagons.select),
	 XtRCallback, (caddr_t) NULL}
};

HexagonsClassRec hexagonsClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Hexagons",	/* class name */
		sizeof (HexagonsRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeHexagons,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListHexagons,	/* actions */
		XtNumber(actionsListHexagons),	/* num actions */
		resourcesHexagons,	/* resources */
		XtNumber(resourcesHexagons),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyHexagons,		/* destroy */
		(XtWidgetProc) ResizeHexagons,	/* resize */
		(XtExposeProc) ExposeHexagons,	/* expose */
		(XtSetValuesFunc) SetValuesHexagons,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsHexagons,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass hexagonsWidgetClass = (WidgetClass) & hexagonsClassRec;

void
SetHexagons(HexagonsWidget w, int reason)
{
	hexagonsCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(HexagonsWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->hexagons.fontInfo) {
		XUnloadFont(XtDisplay(w), w->hexagons.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->hexagons.fontInfo);
	}
	if ((w->hexagons.fontInfo = XLoadQueryFont(display,
			w->hexagons.font)) == NULL) {
		(void) sprintf(buf,
			"Can not open %s font.\nAttempting %s font as alternate\n",
			w->hexagons.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->hexagons.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Can not open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->hexagons.fontInfo) {
		w->hexagons.digitOffset.x = XTextWidth(w->hexagons.fontInfo, "8", 1)
			/ 2;
		w->hexagons.digitOffset.y = w->hexagons.fontInfo->max_bounds.ascent
			/ 2;
	} else
#endif
	{
		w->hexagons.digitOffset.x = 3;
		w->hexagons.digitOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "hexagons.log"
#endif

static Point hexagonUnit[MAXORIENT][7] =
{
	{
		{0, 0},
		{2, 0},
		{1, 1},
		{-1, 1},
		{-2, 0},
		{-1, -1},
		{1, -1}},
	{
		{0, 0},
		{1, 1},
		{0, 2},
		{-1, 1},
		{-1, -1},
		{0, -2},
		{1, -1}}
};
static Point hexagonList[MAXORIENT][7];

static void
CheckTiles(HexagonsWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->hexagons.sizeX < MINTILES) {
		intCat(&buf1,
			"Smallest number of hexagons in X direction out of bounds, use at least ",
			MINTILES);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTTILESX);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->hexagons.sizeX = DEFAULTTILESX;
	}
	if (w->hexagons.sizeY < MINTILES) {
		intCat(&buf1,
			"Largest number of hexagons in Y direction out of bounds, use at least ",
			MINTILES);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTTILESY);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->hexagons.sizeY = DEFAULTTILESY;
	}
#if 0
	if (w->hexagons.delay < 0) {
		intCat(&buf1, "Delay can not be negative (",
			w->hexagons.delay);
		stringCat(&buf2, buf1, "), taking absolute value");
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
		w->hexagons.delay = -w->hexagons.delay;
	}
#endif
	if (w->hexagons.base < MINBASE || w->hexagons.base > MAXBASE) {
		intCat(&buf1, "Base out of bounds, use ", MINBASE);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAXBASE);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTBASE);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->hexagons.base = DEFAULTBASE;
	}
}

/* This is fast for small i, a -1 is returned for negative i. */
static int
Sqrt(const int i)
{
	int j = 0;

	while (j * j <= i)
		j++;
	return (j - 1);
}

static void
swap(int *a, int *b)
{
	int c;

	c = *b;
	*b = *a;
	*a = c;
}

#ifdef DEBUG
static int
PositionInRow(HexagonsWidget w, int position, int posRow)
{
	/* Untested */
	return (posRow <= w->hexagons.sizeY - 1) ?
		(position - w->hexagons.sizeX * posRow -
		posRow * (posRow - 1) / 2) :
		(position - w->hexagons.sizeY * (posRow - w->hexagons.sizeY) -
		4 * w->hexagons.sizeX * (w->hexagons.sizeX - 1) / 2 +
		(2 * w->hexagons.sizeX - posRow - 2) *
		(2 * w->hexagons.sizeX - posRow - 1) / 2 - 1);
}
#endif

static int
PositionFromRow(HexagonsWidget w, int rowPosition, int posRow)
{
	return (posRow <= w->hexagons.sizeY - 1) ?
		(w->hexagons.sizeX * posRow + posRow * (posRow - 1) / 2 +
		rowPosition) :
		(w->hexagons.sizeY * (2 * posRow - w->hexagons.sizeY + 1) -
		w->hexagons.sizeX * (posRow + 1) + 4 *
		w->hexagons.sizeX * (w->hexagons.sizeX - 1) / 2 -
		(2 * w->hexagons.sizeX - posRow - 2) *
		(2 * w->hexagons.sizeX - posRow - 1) / 2 + 1 + rowPosition);
}

static int
ToPosition(HexagonsWidget w, int row, int trbl)
{
	return ((row < w->hexagons.sizeY) ?
		PositionFromRow(w, trbl, row) :
		PositionFromRow(w, trbl, row) - row + w->hexagons.sizeY - 1);
}

int
Row(HexagonsWidget w, const int pos)
{
	return (pos <= w->hexagons.sizeCenter) ?
		(1 + Sqrt(1 + 8 * (pos + w->hexagons.sizeX *
		(w->hexagons.sizeX - 1) / 2))) / 2 - w->hexagons.sizeX :
		w->hexagons.sizeX + 2 * w->hexagons.sizeY - 2 -
		(1 + Sqrt(1 + 8 * (w->hexagons.sizeSize - 1 +
		w->hexagons.sizeX * (w->hexagons.sizeX - 1) / 2 - pos))) / 2;
}

/* Passing row so there is no sqrt calculation again */
int
TrBl(HexagonsWidget w, const int pos, const int posRow)
{
	return (pos <= w->hexagons.sizeCenter) ?
		(pos + w->hexagons.sizeX * (w->hexagons.sizeX - 1) / 2) -
		(posRow + w->hexagons.sizeX) *
		(posRow + w->hexagons.sizeX - 1) / 2 :
		w->hexagons.sizeY + w->hexagons.sizeX - 2 -
		(w->hexagons.sizeSize - 1 +
		w->hexagons.sizeX * (w->hexagons.sizeX - 1) / 2 - pos -
		(2 * w->hexagons.sizeY + w->hexagons.sizeX - posRow - 2) *
		(2 * w->hexagons.sizeY + w->hexagons.sizeX - posRow - 3) / 2);
}

int
TlBr(HexagonsWidget w, const int pos, const int posRow)
{
	return (pos <= w->hexagons.sizeCenter) ?
		-(1 + pos + w->hexagons.sizeX * (w->hexagons.sizeX - 1) / 2) +
		(posRow + w->hexagons.sizeX + 1) *
		(posRow + w->hexagons.sizeX) / 2 :
		w->hexagons.sizeY + w->hexagons.sizeX - 1 +
		(w->hexagons.sizeSize - 1 +
		w->hexagons.sizeX * (w->hexagons.sizeX - 1) / 2 - pos -
		(2 * w->hexagons.sizeY + w->hexagons.sizeX - posRow - 1) *
		(2 * w->hexagons.sizeY + w->hexagons.sizeX - posRow - 2) / 2);
}

static int
CartesianX(HexagonsWidget w, int pos, int row)
{
	return w->hexagons.tileSize.x / 4 - 1 + (2 * TrBl(w, pos, row) +
		w->hexagons.sizeY - row) * w->hexagons.offset.x / 2 - 2 +
		w->hexagons.puzzleOffset.x;
}

static int
CartesianY(HexagonsWidget w, int row)
{
	return row * (3 * w->hexagons.tileSize.y / 2 + w->hexagons.delta.y) +
		w->hexagons.delta.y + w->hexagons.puzzleOffset.y;
}

static int
TileNFromSpace(HexagonsWidget w, int n, int rowType, int direction)
{
	int pos = w->hexagons.spacePosition[HIGH];

	if (rowType == ROW) /* This one is easy */
		return (pos + ((direction == HIGH) ? -n : n));
	else {
		int row = Row(w, pos);
		int trbl = TrBl(w, pos, row);
		int offset;

		if (direction == HIGH) {
			offset = -n;
		} else {
			offset = n;
		}
		if (rowType == TRBL)
			return ToPosition(w, row + offset, trbl);
		else if (rowType == TLBR)
			return ToPosition(w, row + offset, trbl + offset);
		else /* rowType == ROW */
			return (pos + offset);
	}

}

Boolean
CheckSolved(const HexagonsWidget w)
{
	int i;

	for (i = 1; i < w->hexagons.sizeSize -
			((w->hexagons.corners) ? CORNERS : NOCORN); i++)
		if (w->hexagons.tileOfPosition[i - 1] != i)
			return FALSE;
	return TRUE;
}

static Boolean
bresenhamLine(int startX, int startY, int endX, int endY,
		int * pixX, int * pixY, int * pixNo)
{
	int pix = 0;
	int ex = endX - startX;
	int ey = endY - startY;
	int dx, dy, error;

	(*pixNo)++;
	if (ex > 0) {
		dx = 1;
	} else if (ex < 0) {
		dx = -1;
		ex = -ex;
	} else {
		dx = 0;
	}
	if (ey > 0) {
		dy = 1;
	} else if (ey < 0) {
		dy = -1;
		ey = -ey;
	} else {
		dy = 0;
	}
	*pixX = startX;
	*pixY = startY;
	if (ex > ey) {
		error = 2 * ey - ex;
		while (pix != *pixNo) {
			if (error >= 0) {
				error -= 2 * ex;
				*pixY += dy;
			}
			error += 2 * ey;
			*pixX += dx;
			pix++;
			if (*pixX == endX)
				return True;
		}
		return False;
	} else {
		error = 2 * ex - ey;
		while (pix != *pixNo) {
			if (error >= 0) {
				error -= 2 * ey;
				*pixX += dx;
			}
			error += 2 * ex;
			*pixY += dy;
			pix++;
			if (*pixY == endY)
				return True; 
		}
		return False;
	}
}

static int
int2String(HexagonsWidget w, char *buf, int number, int base, Boolean capital)
{
	int digit, mult = base, last, position;
	int a, i, j, s, r;

	if (capital) {
		a = 'A', i = 'I', j = 'J', s = 'S', r = 'R';
	} else {
		a = 'a', i = 'i', j = 'j', s = 's', r = 'r';
	}
	if (number < 0) {
		char *buff;

		intCat(&buff, "int2String: 0 > ", number);
		DISPLAY_WARNING(buff);
		free(buff);
		return 0;
	}
	last = 1;
	while (number >= mult) {
		last++;
		mult *= base;
	}
	for (position = 0; position < last; position++) {
		mult /= base;
		digit = number / mult;
		number -= digit * mult;
		buf[position] = digit + '0';
		if (buf[position] > '9') {	/* ASCII */
			buf[position] += (a - '9' - 1);
		} else if (buf[position] < '0') {	/* EBCDIC */
			buf[position] += (a - '9' - 1);
			if (buf[position] > i)
				buf[position] += (j - i - 1);
			if (buf[position] > r)
				buf[position] += (s - r - 1);
		}
	}
	buf[last] = '\0';
	return last;
}

static void
fill3DHexagon(HexagonsWidget w, Pixmap dr, GC gc, GC darkerGC, GC brighterGC,
	Point * list, Boolean raised, Boolean corners)
{
	GC currentGC = (raised) ? gc : darkerGC;
	Point tempList[7];
	int i;

	for (i = 0; i < 7; i++) {
		tempList[i].x = ((i == 0) ? 0 : tempList[i - 1].x) + list[i].x;
		tempList[i].y = ((i == 0) ? 0 : tempList[i - 1].y) + list[i].y;
	}

	POLYGON(w, dr, currentGC, currentGC, list, 6, True, False);
	if (!corners) {
		DRAWLINE(w, dr, currentGC,
			tempList[4].x, tempList[4].y,
			tempList[5].x, tempList[5].y);
		DRAWLINE(w, dr, currentGC,
			tempList[1].x, tempList[1].y,
			tempList[2].x, tempList[2].y);
	}
	if (!raised && corners) {
		currentGC = darkerGC;
		DRAWLINE(w, dr, currentGC,
			tempList[4].x, tempList[4].y,
			tempList[5].x, tempList[5].y);
		DRAWLINE(w, dr, currentGC,
			tempList[4].x + 1, tempList[4].y,
			tempList[5].x + 1, tempList[5].y);
	}
	currentGC = (raised) ? brighterGC : darkerGC;
	DRAWLINE(w, dr, currentGC,
		tempList[5].x, tempList[5].y,
		tempList[0].x, tempList[0].y);
	DRAWLINE(w, dr, currentGC,
		tempList[0].x, tempList[0].y,
		tempList[1].x, tempList[1].y);
	DRAWLINE(w, dr, currentGC,
		tempList[0].x, tempList[0].y + 1,
		tempList[1].x, tempList[1].y + 1);
	if (corners) {
		DRAWLINE(w, dr, currentGC,
			tempList[5].x, tempList[5].y + 1,
			tempList[0].x, tempList[0].y + 1);
		DRAWLINE(w, dr, currentGC,
			tempList[5].x, tempList[5].y + 2,
			tempList[0].x, tempList[0].y + 2);
	} else {
		DRAWLINE(w, dr, currentGC,
			tempList[5].x + 1, tempList[5].y,
			tempList[0].x + 1, tempList[0].y);
		DRAWLINE(w, dr, currentGC,
			tempList[5].x + 2, tempList[5].y,
			tempList[0].x + 2, tempList[0].y);
	}
	currentGC = (raised) ? darkerGC : gc;
	DRAWLINE(w, dr, currentGC,
		tempList[2].x, tempList[2].y, tempList[3].x, tempList[3].y);
	DRAWLINE(w, dr, currentGC,
		tempList[3].x, tempList[3].y, tempList[4].x, tempList[4].y);
	DRAWLINE(w, dr, currentGC,
		tempList[3].x, tempList[3].y - 1,
		tempList[4].x, tempList[4].y - 1);
	if (corners) {
		DRAWLINE(w, dr, currentGC,
			tempList[1].x, tempList[1].y,
			tempList[2].x, tempList[2].y);
		DRAWLINE(w, dr, currentGC,
			tempList[1].x - 1, tempList[1].y,
			tempList[2].x - 1, tempList[2].y);
		DRAWLINE(w, dr, currentGC,
			tempList[2].x, tempList[2].y - 1,
			tempList[3].x, tempList[3].y - 1);
		DRAWLINE(w, dr, currentGC,
			tempList[2].x, tempList[2].y - 2,
			tempList[3].x, tempList[3].y - 2);
		if (raised) {
			currentGC = brighterGC;
			DRAWLINE(w, dr, currentGC,
				tempList[4].x, tempList[4].y,
				tempList[5].x, tempList[5].y);
			DRAWLINE(w, dr, currentGC,
				tempList[4].x + 1, tempList[4].y,
				tempList[5].x + 1, tempList[5].y);
		}
	} else {
		DRAWLINE(w, dr, currentGC,
			tempList[2].x - 1, tempList[2].y,
			tempList[3].x - 1, tempList[3].y);
		DRAWLINE(w, dr, currentGC,
			tempList[2].x - 2, tempList[2].y,
			tempList[3].x - 2, tempList[3].y);
	}
}

static void
drawShadow(HexagonsWidget w, Pixmap dr, GC gc, int startX, int startY,
	Point * list, Boolean corners)
{
	Point tempList[6];
	int i;

	for (i = 0; i < 6; i++) {
		tempList[i].x = ((i == 0) ? startX : tempList[i - 1].x) +
			list[i].x;
		tempList[i].y = ((i == 0) ? startY : tempList[i - 1].y) +
			list[i].y;
	}
	DRAWLINE(w, dr, gc,
		tempList[4].x, tempList[4].y,
		tempList[5].x, tempList[5].y);
	DRAWLINE(w, dr, gc,
		tempList[5].x, tempList[5].y,
		tempList[0].x, tempList[0].y);
	DRAWLINE(w, dr, gc,
		tempList[0].x, tempList[0].y,
		tempList[1].x, tempList[1].y);
	if (corners) {
		DRAWLINE(w, dr, gc,
			tempList[5].x, tempList[5].y + 1,
			tempList[0].x, tempList[0].y + 1);
		DRAWLINE(w, dr, gc,
			tempList[0].x, tempList[0].y,
			tempList[0].x, tempList[0].y + 2);
	} else {
		DRAWLINE(w, dr, gc,
			tempList[5].x + 1, tempList[5].y,
			tempList[0].x + 1, tempList[0].y);
		DRAWLINE(w, dr, gc,
			tempList[1].x, tempList[1].y,
			tempList[1].x, tempList[1].y + 2);
	}
}

static void
DrawTile(HexagonsWidget w, int pos, Boolean blank, Boolean erase,
		int pressedOffset, int offsetX, int offsetY)
{
	Pixmap *dr;
	Pixmap adr = 0;
	int dx, dy, row;
	int corners = (w->hexagons.corners) ? 1 : 0;
	GC tileGC, borderGC;

	/*dr = &(w->hexagons.bufferTiles[pressedOffset]);*/
	dr = &adr;
	if (erase) {
		tileGC = w->hexagons.inverseGC;
		borderGC = w->hexagons.inverseGC;
	} else {
		tileGC = w->hexagons.tileGC;
		borderGC = w->hexagons.borderGC;
	}
	
	row = Row(w, pos);
	dx = CartesianX(w, pos, row) + offsetX + pressedOffset;
	dy = CartesianY(w, row) + offsetY + pressedOffset;
	if (corners) {
		hexagonList[corners][0].x = dx;
		hexagonList[corners][0].y = dy;
	} else {
		hexagonList[corners][0].x = dx - w->hexagons.offset.x / 4;
		hexagonList[corners][0].y = dy +
			(w->hexagons.tileSize.y + w->hexagons.delta.y) / 4;
	}
	if (blank) {
		POLYGON(w, *dr, tileGC, borderGC,
			hexagonList[corners], 6, True, False);
	} else {
		if (pressedOffset != 0) {
			drawShadow(w, *dr, w->hexagons.tileDarkerGC,
				-pressedOffset, -pressedOffset,
				hexagonList[corners], w->hexagons.corners);
		}
		fill3DHexagon(w, *dr, tileGC, w->hexagons.tileDarkerGC,
			w->hexagons.tileBrighterGC,
			hexagonList[corners],
			pressedOffset == 0, w->hexagons.corners);
	}
	if (!blank) {
		int i = 0, digitOffsetX = 0;
		int tile = w->hexagons.tileOfPosition[pos];
		char buf[65];

		(void) int2String(w, buf, tile, w->hexagons.base, True);
		while (tile >= 1) {
			tile /= w->hexagons.base;
			digitOffsetX += w->hexagons.digitOffset.x;
			i++;
		}
		DRAWTEXT(w, *dr, borderGC,
			dx - digitOffsetX,
			dy + w->hexagons.tileSize.y + w->hexagons.digitOffset.y,
			buf, i);
	}
}

static void
DrawAllBufferedTiles(const HexagonsWidget w)
{
	int k;

	for (k = 0; k < w->hexagons.sizeSize; k++)
		DrawTile(w, k, (w->hexagons.tileOfPosition[k] <= 0),
			(w->hexagons.tileOfPosition[k] <= 0), False, 0, 0);
}

void
DrawAllTiles(const HexagonsWidget w)
{
	int k;

	for (k = 0; k < w->hexagons.sizeSize; k++)
		DrawTile(w, k, (w->hexagons.tileOfPosition[k] <= 0),
			(w->hexagons.tileOfPosition[k] <= 0), False, 0, 0);
}

static int
FindSpaceType(HexagonsWidget w, int pos1, int pos2, int row1, int row2)
{
	if (row1 == row2 && (pos1 == pos2 + 1 || pos1 == pos2 - 1))
		return (ROW);
	else if (row1 == row2 - 1) {
		swap(&row1, &row2);
		swap(&pos1, &pos2);
	}
	if (row1 == row2 + 1) {
		if (row1 <= w->hexagons.sizeY - 1) {
			if (pos2 == pos1 - w->hexagons.sizeX - row1)
				return (TLBR);
			else if (pos2 == pos1 - w->hexagons.sizeX - row1 + 1)
				return (TRBL);
		} else {	/* row1 > w->hexagons.sizeY - 1 */
			if (pos2 == pos1 - 2 * w->hexagons.sizeY -
					w->hexagons.sizeX + row1 + 1)
				return (TLBR);
			else if (pos2 == pos1 - 2 * w->hexagons.sizeY -
					w->hexagons.sizeX + row1 + 2)
				return (TRBL);
		}
	}
	return (-1);
}

static void
FindMovableTile(HexagonsWidget w, int pos, int posRow, int spaceType, int side,
	int *tilePos, int *tileRow)
{
	switch (spaceType) {
		case TRBL:
			if (side == HIGH) {
				*tileRow = posRow;
				*tilePos = pos + 1;
			} else {	/* side == LOW */
				*tileRow = posRow - 1;
				*tilePos = (posRow <= w->hexagons.sizeY - 1) ?
					pos - w->hexagons.sizeX - posRow :
					pos - 2 * w->hexagons.sizeY -
					w->hexagons.sizeX + posRow + 1;
			}
			break;
		case TLBR:
			if (side == HIGH) {
				*tileRow = posRow;
				*tilePos = pos - 1;
			} else {	/* side == LOW */
				*tileRow = posRow - 1;
				*tilePos = (posRow <= w->hexagons.sizeY - 1) ?
					pos - w->hexagons.sizeX - posRow + 1 :
					pos - 2 * w->hexagons.sizeY -
					w->hexagons.sizeX + posRow + 2;
			}
			break;
		case ROW:
			if (side == HIGH) {
				*tileRow = posRow + 1;
				*tilePos = (posRow < w->hexagons.sizeY - 1) ?
					pos + w->hexagons.sizeX + posRow :
					pos + 2 * w->hexagons.sizeY +
					w->hexagons.sizeX - posRow - 3;
			} else {	/* side == LOW */
				*tileRow = posRow - 1;
				*tilePos = (posRow <= w->hexagons.sizeY - 1) ?
					pos - w->hexagons.sizeX - posRow :
					pos - 2 * w->hexagons.sizeY -
					w->hexagons.sizeX + posRow + 1;
			}
			break;
		default:
			{
				char *buf;

				intCat(&buf,
					"FindMovableTile: spaceType \n",
					spaceType);
				DISPLAY_WARNING(buf);
				free(buf);
			}
	}
}

static int
NextToWall(HexagonsWidget w, int pos, int posRow, int spaceType)
{
	switch (spaceType) {
		case TRBL:
			if (posRow < w->hexagons.sizeY &&
					pos ==w->hexagons.sizeX * posRow +
					posRow * (posRow - 1) / 2)
				return (HIGH);
			else if (posRow >= w->hexagons.sizeY &&
					pos == w->hexagons.sizeY *
					(2 * posRow - w->hexagons.sizeY + 3) -
					w->hexagons.sizeX * posRow - 2 -
					posRow + 4 * w->hexagons.sizeX *
					(w->hexagons.sizeX - 1) / 2 -
					(2 * w->hexagons.sizeX - posRow - 2) *
					(2 * w->hexagons.sizeX - posRow - 1) / 2)
				return (LOW);
			else
				return (-1);
		case TLBR:
			if (posRow < w->hexagons.sizeY && pos ==
					w->hexagons.sizeX * (posRow + 1) +
					posRow * (posRow + 1) / 2 - 1)
				return (HIGH);
			else if (posRow >= w->hexagons.sizeY &&
					pos == w->hexagons.sizeY *
					(2 * posRow - w->hexagons.sizeY + 1) -
					 w->hexagons.sizeX * (posRow + 1) + 1 +
					 4 * w->hexagons.sizeX *
					(w->hexagons.sizeX - 1) / 2 -
					(2 * w->hexagons.sizeX - posRow - 2) *
					(2 * w->hexagons.sizeX - posRow - 1) / 2)
				return (LOW);
			else
				return (-1);
		case ROW:
			if (posRow == 0)
				return (HIGH);
			else if (posRow == 2 * (w->hexagons.sizeY - 1))
				return (LOW);
			else
				return (-1);
	}
	return (-2);		/*Unknown space formation. */
}

static int
MovableCornerTiles(HexagonsWidget w, int direction,
		int *position, int *posRow, int *space)
{
	int spaceType, highest, side = -1;

	highest = (w->hexagons.spacePosition[HIGH] >
		w->hexagons.spacePosition[LOW]) ? HIGH : LOW;
	spaceType = FindSpaceType(w,
		w->hexagons.spacePosition[HIGH], w->hexagons.spacePosition[LOW],
		w->hexagons.spaceRow[HIGH], w->hexagons.spaceRow[LOW]);
	switch (spaceType) {
		case TRBL:
			if (direction == TR || direction == BL)
				return FALSE;
			side = NextToWall(w, w->hexagons.spacePosition[highest],
				w->hexagons.spaceRow[highest], spaceType);
			if (side != -1) {
				if ((side == HIGH && direction == RIGHT) ||
				    (side == HIGH && direction == BR) ||
				    (side == LOW && direction == LEFT) ||
				    (side == LOW && direction == TL))
					return FALSE;
			} else
				side = (direction == TL || direction == LEFT);
			*space = (direction == BR || direction == LEFT);
			break;
		case TLBR:
			if (direction == TL || direction == BR)
				return FALSE;
			side = NextToWall(w, w->hexagons.spacePosition[highest],
				w->hexagons.spaceRow[highest], spaceType);
			if (side != -1) {
				if ((side == LOW && direction == TR) ||
				    (side == LOW && direction == RIGHT) ||
				    (side == HIGH && direction == BL) ||
				    (side == HIGH && direction == LEFT))
					return FALSE;
			} else
				side = (direction == TR || direction == RIGHT);
			*space = (direction == RIGHT || direction == BL);
			break;
		case ROW:
			if (direction == LEFT || direction == RIGHT)
				return FALSE;
			side = NextToWall(w, w->hexagons.spacePosition[highest],
				   w->hexagons.spaceRow[highest], spaceType);
			if (side != -1) {
				if ((side == LOW && direction == TR) ||
				    (side == HIGH && direction == BR) ||
				    (side == HIGH && direction == BL) ||
				    (side == LOW && direction == TL))
					return FALSE;
			} else
				side = (direction == TR || direction == TL);
			*space = (direction == TR || direction == BR);
			break;
	}
	FindMovableTile(w, w->hexagons.spacePosition[highest],
	  w->hexagons.spaceRow[highest], spaceType, side, position, posRow);
	return TRUE;
}

static int
MovableNoCornTile(HexagonsWidget w)
{
	int rowType = HEXAGONS_BLOCKED, l;

	/* Are the spaces in a "row" with the mouse click?
	   (If two, then one clicked on a space). */
	for (l = 0; l < ROWTYPES; l++) {
		if (w->hexagons.currentRow[l] == w->hexagons.spaceRow[l]) {
			if (rowType == HEXAGONS_BLOCKED) {
				rowType = l;
			} else {
				return HEXAGONS_SPACE;
			}
		}
	}
	return rowType;
}

#ifdef ANIMATE
static int
CountTiles(HexagonsWidget w, int rowType)
{
	switch (rowType) {
	case TLBR:
		return ABS(w->hexagons.spaceRow[ROW] -
		 	w->hexagons.currentRow[ROW]);
	case TRBL:
		return ABS(w->hexagons.spaceRow[ROW] -
			w->hexagons.currentRow[ROW]);
	case ROW:
		return ABS(w->hexagons.spaceRow[TRBL] -
			w->hexagons.currentRow[TRBL]);
	default:
		{
			char * buf;

			intCat(&buf, "CountTiles: rowType ", rowType);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
	return 0;
}

static int
DirTiles(HexagonsWidget w, int dir, int rowType)
{
	switch (rowType) {
	case TLBR:
		return (dir == HIGH) ? BR : TL;
	case TRBL:
		return (dir == HIGH) ? BL : TR;
	case ROW:
		return (dir == HIGH) ? RIGHT : LEFT;
	default:
		{
			char *buf;

			intCat(&buf, "DirTiles: rowType ", rowType);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
	return dir;
}

static void
AnimateSlide(HexagonsWidget w, int dir, int fast, Boolean logMoves)
{
	int aTile, numTiles, direction = dir, rowType;
	int fillBeginPos;
	int pos, posNext;

	rowType = MovableNoCornTile(w);
	if (rowType < 0) {
		char *buf;

		intCat(&buf, "AnimateTiles: rowType ", rowType);
		DISPLAY_WARNING(buf);
		free(buf);
		numTiles = 0;
	} else {
		numTiles = CountTiles(w, rowType);
		direction = DirTiles(w, dir, rowType);
	}
	if (numTiles < 0)
		numTiles = -numTiles;
	fillBeginPos = TileNFromSpace(w, numTiles, rowType, dir);
	{
		int pixNo = 0;
		int pixX = 0, pixY = 0;
		int fromdx = 0, fromdy = 0;
		int todx = 0, tody = 0;
		int dx, dy, row;
		int *rowPos;
		int i, k;
		Boolean slideDone = False;

		if (!(rowPos = (int *)
				malloc(sizeof (int) * (numTiles + 1)))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		rowPos[1] = TileNFromSpace(w, 1, rowType, dir);
		rowPos[0] = w->hexagons.spacePosition[HIGH];
		for (i = 1; i < numTiles; i++) {
			rowPos[i + 1] = TileNFromSpace(w,
				i + 1, rowType, dir);
		}
		row = Row(w, rowPos[1]);
		fromdx = CartesianX(w, rowPos[1], row);
		fromdy = CartesianY(w, row);
		row = Row(w, rowPos[0]);
		todx = CartesianX(w, rowPos[0], row);
		tody = CartesianY(w, row);
		dx = todx - fromdx;
		dy = tody - fromdy;
		i = 0;
		while (!slideDone) {
			for (k = 1; k < numTiles + 1; k++)
				DrawTile(w, rowPos[k],
					True, True, FALSE, pixX, pixY);
			slideDone = bresenhamLine(0, 0, dx, dy,
				&pixX, &pixY, &pixNo);
			for (k = 1; k < numTiles + 1; k++)
				DrawTile(w, rowPos[k],
					False, False, FALSE, pixX, pixY);
			FLUSH(w);
			if (i % 8 == 0)
				Sleep((unsigned int) (w->hexagons.delay /
					fast));
			i++;
		}
		free(rowPos);
	}
	for (aTile = 0; aTile < numTiles; aTile++) {
		if (logMoves) {
			SetHexagons(w, HEXAGONS_MOVED);
			PutMove(direction);
		}
		posNext = TileNFromSpace(w, aTile + 1, rowType, !dir);
	}
	pos = w->hexagons.spacePosition[HIGH];
	for (aTile = 0; aTile < numTiles; aTile++) {
		posNext = TileNFromSpace(w, aTile + 1, rowType, dir);
		w->hexagons.tileOfPosition[pos] =
			w->hexagons.tileOfPosition[posNext];
		pos = posNext;
	}
	w->hexagons.tileOfPosition[fillBeginPos] =
		w->hexagons.spacePosition[HIGH];
	w->hexagons.spacePosition[HIGH] = fillBeginPos;
	w->hexagons.tileOfPosition[fillBeginPos] = 0;
	w->hexagons.spaceRow[ROW] = Row(w, fillBeginPos);
	w->hexagons.spaceRow[TRBL] =
		TrBl(w, fillBeginPos, w->hexagons.spaceRow[ROW]);
	w->hexagons.spaceRow[TLBR] =
		TlBr(w, fillBeginPos, w->hexagons.spaceRow[ROW]);
}
#endif

static void
ResetTiles(HexagonsWidget w)
{
	int i;

	w->hexagons.sizeSize = w->hexagons.sizeY * w->hexagons.sizeY +
		(w->hexagons.sizeX - 1) * (w->hexagons.sizeY * 2 - 1);
	w->hexagons.sizeCenter = (w->hexagons.sizeSize - 1) / 2;
	if (w->hexagons.tileOfPosition)
		free(w->hexagons.tileOfPosition);
	if (!(w->hexagons.tileOfPosition = (int *)
			malloc(sizeof (int) * w->hexagons.sizeSize))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}

	if (startPosition)
		free(startPosition);
	if (!(startPosition = (int *)
			malloc(sizeof (int) * w->hexagons.sizeSize))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}

	w->hexagons.spacePosition[HIGH] = w->hexagons.sizeSize - 1;
	if (w->hexagons.corners) {
		w->hexagons.spaceRow[HIGH] = 2 * w->hexagons.sizeY - 2;
		if (w->hexagons.sizeX > 1 || w->hexagons.sizeY > 1) {
			w->hexagons.spacePosition[LOW] = w->hexagons.sizeSize - 2;
			w->hexagons.spaceRow[LOW] = 2 * w->hexagons.sizeY -
				((w->hexagons.sizeX == 1) ? 3 : 2);
			w->hexagons.tileOfPosition[w->hexagons.sizeSize - 2] = -1;
		}
	} else {
		w->hexagons.spaceRow[ROW] = w->hexagons.spaceRow[TRBL] =
			2 * w->hexagons.sizeY - 2;
		w->hexagons.spaceRow[TLBR] = w->hexagons.sizeY - 1;
	}
	w->hexagons.tileOfPosition[w->hexagons.sizeSize - 1] = 0;
	for (i = 1; i < w->hexagons.sizeSize - ((w->hexagons.corners) ? 1 : 0);
			i++)
		w->hexagons.tileOfPosition[i - 1] = i;
	FlushMoves(w);
	w->hexagons.currentPosition = -1;
	w->hexagons.started = FALSE;
}

static void
EraseFrame(HexagonsWidget w, Pixmap dr)
{
	FILLRECTANGLE(w, dr, w->hexagons.inverseGC,
		0, 0, w->core.width, w->core.height);
}

static void
DrawFrame(HexagonsWidget w, Pixmap dr, Boolean all, Boolean focus)
{
	int sumX, sumY, sumX4, sum3X4, sumY2, offsetX, offsetY;
	Point tempList[8];
	GC gc = (focus) ? w->hexagons.frameGC : w->hexagons.borderGC;

	sumX = (w->hexagons.sizeX + w->hexagons.sizeY) * w->hexagons.offset.x -
		w->hexagons.tileSize.x / 2 - 2 * w->hexagons.delta.x - 1;
	sumY = w->hexagons.sizeY * (3 * w->hexagons.tileSize.y + 2 *
		w->hexagons.delta.y) - w->hexagons.tileSize.y - 1;
	offsetX = w->hexagons.puzzleOffset.x;
	offsetY = w->hexagons.puzzleOffset.y;
	sumX4 = w->hexagons.sizeY * w->hexagons.offset.x / 2 + offsetX -
		w->hexagons.tileSize.x / 4 + 1;
	sum3X4 = w->hexagons.sizeX * w->hexagons.offset.x + sumX4 - 12;
	sumY2 = sumY / 2 + offsetY;
	sumX += offsetX + w->hexagons.sizeX / 2;
	sumY += offsetY;
	offsetX += -1 - w->hexagons.sizeX / 2;
	if (all) {
		if (offsetY > 0) {
			FILLRECTANGLE(w, dr, w->hexagons.borderGC,
				0, 0, w->core.width, offsetY);
		}
		if (sumY < w->core.height) {
			FILLRECTANGLE(w, dr, w->hexagons.borderGC,
				0, sumY + 1, w->core.width, w->core.height - sumY);
		}
		if (offsetX > 0) {
			FILLRECTANGLE(w, dr, w->hexagons.borderGC,
				0, 0, offsetX, w->core.height);
		}
		if (sumX < w->core.width) {
			FILLRECTANGLE(w, dr, w->hexagons.borderGC,
				sumX + 1, 0, w->core.width - sumX - 1, w->core.height);
		}
		tempList[0].x = offsetX;
		tempList[0].y = offsetY;
		tempList[1].x = sumX4;
		tempList[1].y = offsetY;
		tempList[2].x = offsetX;
		tempList[2].y = sumY2;
		tempList[3].x = tempList[0].x;
		tempList[3].y = tempList[0].y;
		POLYGON(w, dr, w->hexagons.borderGC, w->hexagons.borderGC,
			 tempList, 3, True, True);
		tempList[0].x = sumX + 1;
		tempList[0].y = offsetY;
		tempList[1].x = sumX + 1;
		tempList[1].y = sumY2;
		tempList[2].x = sumX;
		tempList[2].y = sumY2;
		tempList[3].x = sum3X4;
		tempList[3].y = offsetY;
		tempList[4].x = tempList[0].x;
		tempList[4].y = tempList[0].y;
		POLYGON(w, dr, w->hexagons.borderGC, w->hexagons.borderGC,
			 tempList, 4, True, True);
		tempList[0].x = offsetX;
		tempList[0].y = sumY + 1;
		tempList[1].x = offsetX;
		tempList[1].y = sumY2;
		tempList[2].x = sumX4;
		tempList[2].y = sumY;
		tempList[3].x = sumX4;
		tempList[3].y = sumY + 1;
		tempList[4].x = tempList[0].x;
		tempList[4].y = tempList[0].y;
		POLYGON(w, dr, w->hexagons.borderGC, w->hexagons.borderGC,
			 tempList, 4, True, True);
		tempList[0].x = sumX + 1;
		tempList[0].y = sumY + 1;
		tempList[1].x = sum3X4;
		tempList[1].y = sumY + 1;
		tempList[2].x = sum3X4;
		tempList[2].y = sumY;
		tempList[3].x = sumX;
		tempList[3].y = sumY2;
		tempList[4].x = sumX + 1;
		tempList[4].y = sumY2;
		tempList[5].x = tempList[0].x;
		tempList[5].y = tempList[0].y;
		POLYGON(w, dr, w->hexagons.borderGC, w->hexagons.borderGC,
			 tempList, 5, True, True);
	}
	tempList[0].x = sumX4;
	tempList[0].y = offsetY;
	tempList[1].x = sum3X4;
	tempList[1].y = offsetY;
	tempList[2].x = sumX;
	tempList[2].y = sumY2;
	tempList[3].x = sum3X4;
	tempList[3].y = sumY;
	tempList[4].x = sumX4;
	tempList[4].y = sumY;
	tempList[5].x = offsetX;
	tempList[5].y = sumY2;
	tempList[6].x = sumX4;
	tempList[6].y = offsetY;
	tempList[7].x = tempList[0].x;
	tempList[7].y = tempList[0].y;
	if (all) {
		POLYGON(w, dr, w->hexagons.inverseGC, gc,
			tempList, 6, True, True);
	} else {
		POLYLINE(w, dr, gc, tempList, 7, True);
	}
}

static void
MoveNoTiles(HexagonsWidget w)
{
	SetHexagons(w, HEXAGONS_IGNORE);
}

static void
MoveCornerTiles(HexagonsWidget w, int from, int posRow, int space, int fast)
{
	int fromTile, nextPos, row;
	int dx, dy, s0x, s0y, s1x, s1y, cx, cy;

	fromTile = w->hexagons.tileOfPosition[from];
	nextPos = w->hexagons.spacePosition[space];
	row = Row(w, from);
	dx = CartesianX(w, from, row);
	dy = CartesianY(w, row);
	row = Row(w, w->hexagons.spacePosition[space]);
	s0x = CartesianX(w, w->hexagons.spacePosition[space], row);
	s0y = CartesianY(w, row);
	row = Row(w, w->hexagons.spacePosition[!space]);
	s1x = CartesianX(w, w->hexagons.spacePosition[!space], row);
	s1y = CartesianY(w, row);
	cx = (s0x + s1x - 2 * dx) / 3;
	cy = (s0y + s1y - 2 * dy) / 3;
	if (fast == INSTANT) {
		DrawTile(w, from, True, True, FALSE, 0, 0);
		DrawTile(w, from, False, False, FALSE, cx, cy);
		FLUSH(w);
#ifdef USE_SOUND
		if (w->hexagons.sound) {
			playSound((char *) BUMPSOUND);
		}
#endif
		DrawTile(w, from, True, True, FALSE, cx, cy);
		DrawTile(w, from, False, False, FALSE, s0x - dx, s0y - dy);
	} else {
		Boolean slideDone = False;
		int pixX = 0, pixY = 0, pixNo = 0, i = 0;

		while (!slideDone) {
			DrawTile(w, from, True, True, FALSE, pixX, pixY);
			slideDone = bresenhamLine(0, 0, cx, cy,
				&pixX, &pixY, &pixNo);
			DrawTile(w, from, False, False, FALSE, pixX, pixY);
			FLUSH(w);
			if (i % 8 == 0)
				Sleep((unsigned int) (w->hexagons.delay /
					fast));
			i++;
		}
#ifdef USE_SOUND
		if (w->hexagons.sound) {
			playSound((char *) BUMPSOUND);
		}
#endif
		slideDone = False;
		i = 0;
		pixNo = 0;
		while (!slideDone) {
			DrawTile(w, from, True, True, FALSE, pixX, pixY);
			slideDone = bresenhamLine(cx, cy, s0x - dx, s0y - dy,
				&pixX, &pixY, &pixNo);
			DrawTile(w, from, False, False, FALSE, pixX, pixY);
			FLUSH(w);
			if (i % 8 == 0)
				Sleep((unsigned int) (w->hexagons.delay /
					fast));
			i++;
		}
	}
	w->hexagons.tileOfPosition[from] =
		w->hexagons.tileOfPosition[w->hexagons.spacePosition[space]];
	w->hexagons.spacePosition[space] = from;
	w->hexagons.spaceRow[space] = posRow;
	w->hexagons.tileOfPosition[nextPos] = fromTile;
	FLUSH(w);
#ifdef USE_SOUND
	if (w->hexagons.sound) {
		playSound((char *) BUMPSOUND);
	}
#endif
	if (fast != INSTANT) {
		Sleep((unsigned int) 10 * w->hexagons.delay / fast);
	}
}

static void
MoveNoCornTiles(HexagonsWidget w, int from, int posRow, int fast)
{
#ifdef ANIMATE
	if (fast != INSTANT && w->hexagons.delay > 0) {
		w->hexagons.currentPosition = from; 
		w->hexagons.currentRow[ROW] = Row(w, from);
		w->hexagons.currentRow[TRBL] =
			TrBl(w, from, w->hexagons.currentRow[ROW]);
		w->hexagons.currentRow[TLBR] =
			TlBr(w, from, w->hexagons.currentRow[ROW]);
		if (w->hexagons.currentPosition <
				w->hexagons.spacePosition[HIGH]) {
			AnimateSlide(w, HIGH, fast, False);
		} else if (w->hexagons.currentPosition >
				w->hexagons.spacePosition[HIGH]) {
			AnimateSlide(w, LOW, fast, False);
		} else
			return;
		w->hexagons.currentPosition = -1;
	} else
#endif
	{
		int tempTile;

		tempTile = w->hexagons.tileOfPosition[from];
		w->hexagons.tileOfPosition[from] =
			w->hexagons.tileOfPosition[w->hexagons.spacePosition[HIGH]];
		w->hexagons.tileOfPosition[w->hexagons.spacePosition[HIGH]] = tempTile;
		DrawTile(w, w->hexagons.spacePosition[HIGH],
			False, False, FALSE, 0, 0);
		w->hexagons.spacePosition[HIGH] = from;
		w->hexagons.spaceRow[ROW] = posRow;
		w->hexagons.spaceRow[TRBL] = TrBl(w, from, posRow);
		w->hexagons.spaceRow[TLBR] = TlBr(w, from, posRow);
		DrawTile(w, w->hexagons.spacePosition[HIGH],
			True, True, FALSE, 0, 0);
	}
#ifdef USE_SOUND
	if (w->hexagons.sound) {
		playSound((char *) BUMPSOUND);
	}
#endif
}

static int
FindTileTriangle(HexagonsWidget w, int pI, int pJ, int pK,
		int rI, int rJ, int rK)
{
	int found = TRUE, temp = 0, k = 0, row1 = 0, row2 = 0, pos;

	if (rI == rJ) {
		if (pI == pJ - 1)
			temp = pJ;
		else if (pI == pJ + 1)
			temp = pI;
		else
			found = FALSE;
		k = pK;
		row1 = rI;
		row2 = rK;
	} else if (rJ == rK) {
		if (pJ == pK - 1)
			temp = pK;
		else if (pJ == pK + 1)
			temp = pJ;
		else
			found = FALSE;
		k = pI;
		row1 = rJ;
		row2 = rI;
	} else if (rK == rI) {
		if (pK == pI - 1)
			temp = pI;
		else if (pK == pI + 1)
			temp = pK;
		else
			found = FALSE;
		k = pJ;
		row1 = rK;
		row2 = rJ;
	}
	if (found == FALSE)
		return (0);
	pos = -1;
	if (row1 == row2 + 1) {
		if (row1 <= w->hexagons.sizeY - 1)
			pos = temp - w->hexagons.sizeX - row1;
		else		/* row1 > w->hexagons.sizeY - 1 */
			pos = temp - 2 * w->hexagons.sizeY - w->hexagons.sizeX +
				row1 + 1;
	} else if (row1 == row2 - 1) {
		if (row1 < w->hexagons.sizeY - 1)
			pos = temp + w->hexagons.sizeX + row1;
		else		/* row1 >= w->hexagons.sizeY - 1 */
			pos = temp + 2 * (w->hexagons.sizeY - 1) +
				(w->hexagons.sizeX - 1) - row1;
	}
	if (k == pos)
		return (1);
	return (0);
}

static int
FindDir(HexagonsWidget w, int posTile, int posSpace, int rowTile, int rowSpace)
{
	if (rowTile == rowSpace) {
		if (posTile > posSpace)
			return LEFT;
		else
			return RIGHT;
	} else if (TrBl(w, posTile, rowTile) == TrBl(w, posSpace, rowSpace)) {
		if (posTile > posSpace)
			return TR;
		else
			return BL;
	} else {
		/* if (TlBr(w, posTile, rowTile) == TlBr(w, posSpace, rowSpace)) */
		if (posTile > posSpace)
			return TL;
		else
			return BR;
	}
}

#ifdef DEBUG
static int
WithinFrame(HexagonsWidget w, int x, int y, int dx, int dy)
{
	return
		(x < dx + w->hexagons.tileSize.x / 2 &&
		 x > dx - w->hexagons.tileSize.x / 2 &&
		 w->hexagons.tileSize.y * (x - dx) < w->hexagons.tileSize.x * (y - dy) &&
		 w->hexagons.tileSize.y * (dx - x) < w->hexagons.tileSize.x * (y - dy) &&
		 w->hexagons.tileSize.y * (x - dx + 2 * w->hexagons.tileSize.x) >
		 w->hexagons.tileSize.x * (y - dy) &&
		 w->hexagons.tileSize.y * (dx - x + 2 * w->hexagons.tileSize.x) >
		 w->hexagons.tileSize.x * (y - dy));
}
#endif

static int
MoveCornerTilesDir(HexagonsWidget w, int direction, int fast)
{
	int position, posRow, space;

	if (MovableCornerTiles(w, direction, &position, &posRow, &space)) {
		MoveCornerTiles(w, position, posRow, (w->hexagons.spacePosition[HIGH] <
			w->hexagons.spacePosition[LOW]) ? !space : space, fast);
		return TRUE;
	}
	return FALSE;
}

static int
MoveNoCornTilesDir(HexagonsWidget w, int direction, int fast)
{
	switch (direction) {
		case TR:
			if (w->hexagons.spaceRow[ROW] != 2 * w->hexagons.sizeY - 2 &&
			    w->hexagons.spaceRow[TLBR] != w->hexagons.sizeX + w->hexagons.sizeY - 2) {
				MoveNoCornTiles(w, TileNFromSpace(w, 1, TRBL, LOW),
					w->hexagons.spaceRow[ROW] + 1, fast);
				return TRUE;
			}
			break;
		case RIGHT:
			if (w->hexagons.spaceRow[TRBL] != 0 &&
			    w->hexagons.spaceRow[TLBR] != w->hexagons.sizeX + w->hexagons.sizeY - 2) {
				MoveNoCornTiles(w, TileNFromSpace(w, 1, ROW, HIGH),
						w->hexagons.spaceRow[ROW], fast);
				return TRUE;
			}
			break;
		case BR:
			if (w->hexagons.spaceRow[ROW] != 0 &&
			    w->hexagons.spaceRow[TRBL] != 0) {
				MoveNoCornTiles(w, TileNFromSpace(w, 1, TLBR, HIGH),
					w->hexagons.spaceRow[ROW] - 1, fast);
				return TRUE;
			}
			break;
		case BL:
			if (w->hexagons.spaceRow[ROW] != 0 &&
			    w->hexagons.spaceRow[TLBR] != 0) {
				MoveNoCornTiles(w, TileNFromSpace(w, 1, TRBL, HIGH),
					w->hexagons.spaceRow[ROW] - 1, fast);
				return TRUE;
			}
			break;
		case LEFT:
			if (w->hexagons.spaceRow[TLBR] != 0 &&
			    w->hexagons.spaceRow[TRBL] != w->hexagons.sizeX + w->hexagons.sizeY - 2) {
				MoveNoCornTiles(w, TileNFromSpace(w, 1, ROW, LOW),
						w->hexagons.spaceRow[ROW], fast);
				return TRUE;
			}
			break;
		case TL:
			if (w->hexagons.spaceRow[ROW] != 2 * w->hexagons.sizeY - 2 &&
			    w->hexagons.spaceRow[TRBL] != w->hexagons.sizeX + w->hexagons.sizeY - 2) {
				MoveNoCornTiles(w, TileNFromSpace(w, 1, TLBR, LOW),
					w->hexagons.spaceRow[ROW] + 1, fast);
				return TRUE;
			}
			break;
		default:
			{
				char *buf;

				intCat(&buf, "MoveNoCornTilesDir: direction ",
					direction);
				DISPLAY_WARNING(buf);
				free(buf);
			}
	}
	return FALSE;
}

static int
MoveTilesDir(HexagonsWidget w, int direction, int fast)
{
	if (w->hexagons.corners)
		return MoveCornerTilesDir(w, direction, fast);
	else
		return MoveNoCornTilesDir(w, direction, fast);
}

int
MoveHexagonsDir(HexagonsWidget w, const int direction, const int fast)
{
	if (MoveTilesDir(w, direction, fast)) {
		SetHexagons(w, HEXAGONS_MOVED);
		PutMove(direction);
		return TRUE;
	}
	return FALSE;
}

#ifndef WINVER
static
#endif
int
MoveHexagons(HexagonsWidget w, const int direction, const int control)
{
	int reason;

	if (control) {
		reason = HEXAGONS_IGNORE;
		switch (direction) {
			case TL:
			case TOP:
			case TR:
				if (w->hexagons.sizeY <= MINTILES +
						w->hexagons.corners)
					return FALSE;
				reason = HEXAGONS_DEC_Y;
				break;
			case RIGHT:
				reason = HEXAGONS_INC_X;
				break;
			case BL:
			case BOTTOM:
			case BR:
				reason = HEXAGONS_INC_Y;
				break;
			case LEFT:
				if (w->hexagons.sizeX <= MINTILES)
					return FALSE;
				reason = HEXAGONS_DEC_X;
				break;
			default:
				{
					char *buf;

					intCat(&buf,
						"MoveHexagons: direction ",
						direction);
					DISPLAY_WARNING(buf);
					free(buf);
				}
		}
		SetHexagons(w, reason);
		return TRUE;
	}
	if (CheckSolved(w)) {
		MoveNoTiles(w);
		return FALSE;
	}
	if (direction == TOP || direction == BOTTOM) {
		return FALSE;
	}
	if (!MoveHexagonsDir(w, direction, NORMAL)) {
		SetHexagons(w, HEXAGONS_BLOCKED);
		return FALSE;
	}
	if (CheckSolved(w)) {
		SetHexagons(w, HEXAGONS_SOLVED);
	}
	return TRUE;
}

static int
ExchangeTiles(HexagonsWidget w, int pos1, int pos2)
{
	int tempTile;

	if (w->hexagons.tileOfPosition[pos1] <= 0 ||
			w->hexagons.tileOfPosition[pos2] <= 0)
		return FALSE;
	tempTile = w->hexagons.tileOfPosition[pos1];
	w->hexagons.tileOfPosition[pos1] = w->hexagons.tileOfPosition[pos2];
	w->hexagons.tileOfPosition[pos2] = tempTile;
	return TRUE;
}

static void
DiscreteNoCornMoves(HexagonsWidget w, int dir, int fast)
{
	int rowType, orient;
	char *buf;

	rowType = MovableNoCornTile(w);
	if (rowType < 0) {
		intCat(&buf, "DiscreteMoves: rowType ", rowType);
			DISPLAY_WARNING(buf);
		free(buf);
		DISPLAY_WARNING(buf);
	}
	if (dir == HIGH) {
		 orient = (rowType == ROW) ? w->hexagons.spaceRow[ROW] :
			 w->hexagons.spaceRow[ROW] - 1;
		MoveNoCornTiles(w, TileNFromSpace(w, 1, rowType, HIGH),
			orient, fast);
		SetHexagons(w, HEXAGONS_MOVED);
		switch (rowType) {
		case TLBR:
			PutMove(BR);
			return;
		case TRBL:
			PutMove(BL);
			return;
		case ROW:
			PutMove(RIGHT);
			return;
		default:
			{
				intCat(&buf,
					"DiscreteNoCornTiles: rowType ",
						rowType);
				DISPLAY_WARNING(buf);
				free(buf);
				return;
			}
		}
	} else { /* w->hexagons.currentPosition > w->hexagons.spacePosition[HIGH] */
		orient = (rowType == ROW) ? w->hexagons.spaceRow[ROW] :
			w->hexagons.spaceRow[ROW] + 1;
		MoveNoCornTiles(w, TileNFromSpace(w, 1, rowType, LOW),
			orient, fast);
		SetHexagons(w, HEXAGONS_MOVED);
		switch (rowType) {
		case TLBR:
			PutMove(TL);
			return;
		case TRBL:
			PutMove(TR);
			return;
		case ROW:
			PutMove(LEFT);
			return;
		default:
			{
				intCat(&buf,
					"DiscreteNoCornTiles: rowType ",
					rowType);
				DISPLAY_WARNING(buf);
			 	free(buf);
				return;
			}
		}
	}
}

static int
PositionToTile(HexagonsWidget w, int x, int y, int *row)
{
	int i, j, k, modI, modJ, rowPos;

	x -= w->hexagons.puzzleOffset.x;
	y -= w->hexagons.puzzleOffset.y;
	/* First convert x and y coordinates to hexagon grid.
	   Keep in mind that the starting hexagon x position
	   changes with "w->hexagons.sizeX % 2". */
	if (x < w->hexagons.tileSize.x / 4)
		return -1;
	i = 2 * (x - w->hexagons.tileSize.x / 4) / w->hexagons.offset.x;
	j = 3 * (y - w->hexagons.delta.y) /
		(3 * w->hexagons.tileSize.y / 2 + w->hexagons.delta.y);
	modI = 2 * (x - w->hexagons.tileSize.x / 4) % w->hexagons.offset.x;
	modJ = 3 * (y - w->hexagons.delta.y) %
		(3 * w->hexagons.tileSize.y / 2 + w->hexagons.delta.y);
	*row = j / 3;		/* Approximate to a rectangle just for now */
	if (j % 3 == 0) {
		/* Then it is the hexagon near bottom or top point */
		if ((w->hexagons.sizeY - 1 + *row + i) % 2)	/* \ */
			*row -= (modJ * w->hexagons.offset.x < modI *
				 (3 * w->hexagons.tileSize.y / 2 + w->hexagons.delta.y));
		else		/* / */
			*row -= (modJ * w->hexagons.offset.x < (w->hexagons.offset.x - modI) *
				 (3 * w->hexagons.tileSize.y / 2 + w->hexagons.delta.y));
	}
	if (i < (w->hexagons.sizeY - 1 + *row) % 2 || *row < 0 ||
			*row > 2 * (w->hexagons.sizeY - 1))
		return -1;
	k = (i - ((w->hexagons.sizeY - 1 + *row) % 2)) / 2;
	/* Map the hexagon grid to hexagon position in puzzle. */
	rowPos = (*row < w->hexagons.sizeY) ?
		k - (w->hexagons.sizeY - 1 - *row) / 2 :
		k + (w->hexagons.sizeY - 1 - *row) / 2;
	j = (*row < w->hexagons.sizeY) ?
		w->hexagons.sizeX - 1 + *row : 2 * (w->hexagons.sizeY - 1) +
		w->hexagons.sizeX - 1 - *row;
	if (rowPos < 0 || rowPos > j)
		return -1;
	return PositionFromRow(w, rowPos, *row);
}

static int
MovableCornerTile(HexagonsWidget w)
{
	/* (Clicked on a space?) */
	if (w->hexagons.currentPosition == w->hexagons.spacePosition[LOW] ||
	    w->hexagons.currentPosition == w->hexagons.spacePosition[HIGH]) {
		return HEXAGONS_SPACE;
	}
	if (FindTileTriangle(w,
		w->hexagons.currentPosition, w->hexagons.spacePosition[HIGH],
		 w->hexagons.spacePosition[LOW], w->hexagons.currentRow[ROW],
		    w->hexagons.spaceRow[HIGH], w->hexagons.spaceRow[LOW])) {
		return 0;
	}
	return HEXAGONS_BLOCKED;
}

static void
SelectCornerTiles(HexagonsWidget w, int space)
{
	int rowType, orient;

	rowType = MovableCornerTile(w);
	if (rowType < 0) {
		char *buf;

		intCat(&buf, "SelectCornerTiles: rowType ", rowType);
		DISPLAY_WARNING(buf);
		free(buf);
		return;
	}
	orient = (w->hexagons.spacePosition[HIGH] <
		w->hexagons.spacePosition[LOW]) ? !space : space;
	PutMove(FindDir(w,
		w->hexagons.currentPosition, w->hexagons.spacePosition[orient],
		w->hexagons.currentRow[ROW], w->hexagons.spaceRow[orient]));
	MoveCornerTiles(w, w->hexagons.currentPosition,
			w->hexagons.currentRow[ROW], orient, NORMAL);
	SetHexagons(w, HEXAGONS_MOVED);
	if (CheckSolved(w)) {
		SetHexagons(w, HEXAGONS_SOLVED);
	}
}

static void
SelectNoCornTiles(HexagonsWidget w)
{
	if (w->hexagons.currentPosition < w->hexagons.spacePosition[HIGH]) {
#ifdef ANIMATE
		if (w->hexagons.delay > 0) {
			AnimateSlide(w, HIGH, NORMAL, True);
#ifdef USE_SOUND
			if (w->hexagons.sound) {
				playSound((char *) BUMPSOUND);
			}
#endif
		} else
#endif
		{
			while (w->hexagons.currentPosition <
					w->hexagons.spacePosition[HIGH]) {
				DiscreteNoCornMoves(w, HIGH, NORMAL);
			}
		}
	} else { /* w->hexagons.currentPosition > w->hexagons.spacePosition[HIGH] */
#ifdef ANIMATE
		if (w->hexagons.delay > 0) {
			AnimateSlide(w, LOW, NORMAL, True);
#ifdef USE_SOUND
			if (w->hexagons.sound) {
				playSound((char *) BUMPSOUND);
			}
#endif
		} else
#endif
		{
			while (w->hexagons.currentPosition > w->hexagons.spacePosition[HIGH]) {
				DiscreteNoCornMoves(w, LOW, NORMAL);
			}
		}
	}
	if (CheckSolved(w)) {
		SetHexagons(w, HEXAGONS_SOLVED);
	}
}

static void
RandomizeTiles(HexagonsWidget w)
{
	if (w->hexagons.currentPosition >= 0)
		return;
	w->hexagons.cheat = False;
	/* First interchange tiles an even number of times */
	if ((w->hexagons.sizeY > 1 && !w->hexagons.corners) ||
			(w->hexagons.sizeX > 2 && w->hexagons.sizeY > 1) ||
			(w->hexagons.sizeX > 1 && w->hexagons.sizeY > 2) ||
			(w->hexagons.sizeX == 1 && w->hexagons.sizeY > 3)) {
		int currentPos, randomPos;
		int count = 0;

		for (currentPos = 0; currentPos < w->hexagons.sizeSize; currentPos++) {
			randomPos = currentPos;
			while (currentPos == randomPos)
				randomPos = NRAND(w->hexagons.sizeSize);
			count += ExchangeTiles(w, currentPos, randomPos);
		}
		if (count % 2 && w->hexagons.corners) {
			if (w->hexagons.sizeX == 1) {
				if (!ExchangeTiles(w, 3, 4) &&
				    !ExchangeTiles(w,
				      w->hexagons.sizeSize - 4,
				      w->hexagons.sizeSize - 5)) {
					DISPLAY_WARNING("RandomizeTiles: should not get here");
				}
			} else if (!ExchangeTiles(w, 0, 1) &&
				   !ExchangeTiles(w, w->hexagons.sizeSize - 2,
						w->hexagons.sizeSize - 1)) {
					DISPLAY_WARNING("RandomizeTiles: should not get here");
			}
		}
		DrawAllTiles(w);
	}
	/* Now move the spaces around randomly */
	if (w->hexagons.sizeY > 1 ||
			(!w->hexagons.corners && w->hexagons.sizeX > 1)) {
		int big = w->hexagons.sizeSize + NRAND(2);
		int lastDirection = -1;
		int randomDirection;

		SetHexagons(w, HEXAGONS_RESET);
#ifdef DEBUG
		big = 3;
#endif
		if (big > 1000)
			big = 1000;
		while (big--) {
			randomDirection = NRAND(COORD);
#ifdef DEBUG
			Sleep(1);
#endif
			if ((randomDirection + COORD / 2) % COORD != lastDirection) {
				if (MoveHexagonsDir(w, randomDirection, INSTANT))
					lastDirection = randomDirection;
				else
					big++;
			}
		}
		FlushMoves(w);
		SetHexagons(w, HEXAGONS_RANDOMIZE);
	}
	if (CheckSolved(w)) {
		SetHexagons(w, HEXAGONS_SOLVED);
	}
}

static void
GetTiles(HexagonsWidget w)
{
	FILE *fp;
	int c, i, sizeX, sizeY, corners, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	FlushMoves(w);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &corners);
	if (w->hexagons.corners != (Boolean) corners) {
		SetHexagons(w, HEXAGONS_CORNERS);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &sizeX);
	if (sizeX >= MINTILES) {
		for (i = w->hexagons.sizeX; i < sizeX; i++) {
			SetHexagons(w, HEXAGONS_INC_X);
		}
		for (i = w->hexagons.sizeX; i > sizeX; i--) {
			SetHexagons(w, HEXAGONS_DEC_X);
		}
	} else {
		stringCat(&buf1, name, " corrupted: sizeX ");
		intCat(&buf2, buf1, sizeX);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINTILES);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}

	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &sizeY);
	if (sizeY >= MINTILES) {
		for (i = w->hexagons.sizeY; i < sizeY; i++) {
			SetHexagons(w, HEXAGONS_INC_Y);
		}
		for (i = w->hexagons.sizeY; i > sizeY; i--) {
			SetHexagons(w, HEXAGONS_DEC_Y);
		}
	} else {
		stringCat(&buf1, name, " corrupted: sizeY ");
		intCat(&buf2, buf1, sizeY);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINTILES);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
#ifdef WINVER
	ResetTiles(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	ScanStartPosition(fp, w);
	SetHexagons(w, HEXAGONS_RESTORE);
	SetStartPosition(w);
	ScanMoves(fp, w, moves);
	(void) fclose(fp);
	(void) printf("%s: corners %d, sizeX %d, sizeY %d, moves %d.\n",
		name, corners, sizeX, sizeY, moves);
	free(lname);
	free(fname);
	w->hexagons.cheat = True; /* Assume the worst. */
}

static void
WriteTiles(HexagonsWidget w)
{
	FILE *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(lname, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(fname, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "corners%c %d\n", SYMBOL,
		(w->hexagons.corners) ? 1 : 0);
	(void) fprintf(fp, "sizeX%c %d\n", SYMBOL, w->hexagons.sizeX);
	(void) fprintf(fp, "sizeY%c %d\n", SYMBOL, w->hexagons.sizeY);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL, NumMoves());
	PrintStartPosition(fp, w);
	PrintMoves(fp);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
ClearTiles(HexagonsWidget w)
{
	if (w->hexagons.currentPosition >= 0)
		return;
	ResetTiles(w);
	DrawAllTiles(w);
	SetHexagons(w, HEXAGONS_RESET);
}

static void
UndoTiles(HexagonsWidget w)
{
	if (MadeMoves() && w->hexagons.currentPosition < 0) {
		int direction;

		GetMove(&direction);
		direction = (direction + (COORD / 2)) % COORD;
		if (MoveTilesDir(w, direction, DOUBLE)) {
			SetHexagons(w, HEXAGONS_UNDO);
		} else {
			char *buf1, *buf2;

			intCat(&buf1, "Move ", direction);
			stringCat(&buf2, buf1, " can not be made");
			free(buf1);
			DISPLAY_WARNING(buf2);
			free(buf2);
		}
	}
}

static void
SolveTiles(HexagonsWidget w)
{
	if (CheckSolved(w) || w->hexagons.currentPosition >= 0)
		return;
	{
		SetHexagons(w, HEXAGONS_SOLVE_MESSAGE);
	}
}

static void
ModeTiles(HexagonsWidget w)
{
	if (w->hexagons.corners || w->hexagons.sizeY > 1) {
		SetHexagons(w, HEXAGONS_CORNERS);
	}
}

static void
SpeedTiles(HexagonsWidget w)
{
	w->hexagons.delay -= 5;
	if (w->hexagons.delay < 0)
		w->hexagons.delay = 0;
#ifdef HAVE_MOTIF
	SetHexagons(w, HEXAGONS_SPEED);
#endif
}

static void
SlowTiles(HexagonsWidget w)
{
	w->hexagons.delay += 5;
#ifdef HAVE_MOTIF
	SetHexagons(w, HEXAGONS_SPEED);
#endif
}

static void
SoundTiles(HexagonsWidget w)
{
	w->hexagons.sound = !w->hexagons.sound;
}

#define FACTOR 0.7

#ifdef WINVER
#define MAXINTENSITY 0xFF
static int
brighter(const int light)
{
	int i = (int) ((1 - FACTOR) * MAXINTENSITY);
	int temp = light;

	if (temp < i)
		temp = i;
	return MIN(temp / FACTOR, MAXINTENSITY);
}

static int
darker(const int light)
{
 return (int) (light * FACTOR);
}

static void
SetValuesHexagons(HexagonsWidget w)
{
	struct tagColor {
		int red, green, blue;
	} color;
	char szBuf[80];

	w->hexagons.sizeX = GetPrivateProfileInt(SECTION, "sizeX",
		DEFAULTTILESX, INIFILE);
	w->hexagons.sizeY = GetPrivateProfileInt(SECTION, "sizeY",
		DEFAULTTILESY, INIFILE);
	w->hexagons.base = GetPrivateProfileInt(SECTION, "base",
		DEFAULTBASE, INIFILE);
	w->hexagons.corners = (BOOL) GetPrivateProfileInt(SECTION, "corners",
		DEFAULTCORNERS, INIFILE);
	w->hexagons.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULTMONO, INIFILE);
	w->hexagons.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverse",
		DEFAULTREVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 255",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->hexagons.frameGC = RGB(color.red, color.green, color.blue);
	/* gray75 */
	(void) GetPrivateProfileString(SECTION, "tileColor", "191 191 191",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->hexagons.tileGC = RGB(color.red, color.green, color.blue);
	w->hexagons.tileBrighterGC = RGB(brighter(color.red),
		brighter(color.green), brighter(color.blue));
	w->hexagons.tileDarkerGC = RGB(darker(color.red),
		darker(color.green), darker(color.blue));
	/* gray25 */
	(void) GetPrivateProfileString(SECTION, "tileBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->hexagons.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->hexagons.inverseGC = RGB(color.red, color.green, color.blue);
	(void) GetPrivateProfileString(SECTION, "userName", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->hexagons.userName, szBuf);
	w->hexagons.userName[80] = 0;
	(void) GetPrivateProfileString(SECTION, "scoreFile", "",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->hexagons.scoreFile, szBuf);
	w->hexagons.scoreFile[80] = 0;
}

void
DestroyHexagons(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

#else
#define MAXINTENSITY 0xFFFF

static Pixel
brighter(HexagonsWidget w, Pixel pixel)
{
	XColor color;
	int i = (int) ((1 - FACTOR) * MAXINTENSITY);

	color.pixel = pixel;
	XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
		&color);
	if (color.red < i)
		color.red = i;
	if (color.green < i)
		color.green = i;
	if (color.blue < i)
		color.blue = i;
	color.red = (unsigned short) MIN(color.red / FACTOR, MAXINTENSITY);
	color.green = (unsigned short) MIN(color.green / FACTOR, MAXINTENSITY);
	color.blue = (unsigned short) MIN(color.blue / FACTOR, MAXINTENSITY);
	if (XAllocColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color))
		return color.pixel;
	return pixel;
}

static Pixel
darker(HexagonsWidget w, Pixel pixel)
{
	XColor color;

	color.pixel = pixel;
	XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
		&color);
	color.red = (unsigned short) (color.red * FACTOR);
	color.green = (unsigned short) (color.green * FACTOR);
	color.blue = (unsigned short) (color.blue * FACTOR);
	if (XAllocColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color))
		return color.pixel;
	return pixel;
}

static void
SetAllColors(HexagonsWidget w)
{
	XGCValues values;
	XtGCMask valueMask;

	valueMask = GCForeground | GCBackground;
	if (w->hexagons.reverse) {
		values.foreground = w->hexagons.foreground;
		values.background = w->hexagons.background;
	} else {
		values.foreground = w->hexagons.background;
		values.background = w->hexagons.foreground;
	}
	if (w->hexagons.inverseGC)
		XtReleaseGC((Widget) w, w->hexagons.inverseGC);
	w->hexagons.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->hexagons.mono) {
		if (w->hexagons.reverse) {
			values.foreground = w->hexagons.background;
			values.background = w->hexagons.foreground;
		} else {
			values.foreground = w->hexagons.foreground;
			values.background = w->hexagons.background;
		}
	} else {
		values.foreground = w->hexagons.frameColor;
		values.background = w->hexagons.borderColor;
	}
	if (w->hexagons.frameGC)
		XtReleaseGC((Widget) w, w->hexagons.frameGC);
	w->hexagons.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->hexagons.mono) {
		if (w->hexagons.reverse) {
			values.foreground = w->hexagons.background;
			values.background = w->hexagons.foreground;
		} else {
			values.foreground = w->hexagons.foreground;
			values.background = w->hexagons.background;
		}
	} else {
		values.foreground = w->hexagons.tileColor;
		values.background = w->hexagons.borderColor;
	}
	if (w->hexagons.tileGC)
		XtReleaseGC((Widget) w, w->hexagons.tileGC);
	w->hexagons.tileGC = XtGetGC((Widget) w, valueMask, &values);
	if (!w->hexagons.mono) {
		values.foreground = brighter(w, w->hexagons.tileColor);
	}
	if (w->hexagons.tileBrighterGC)
		XtReleaseGC((Widget) w, w->hexagons.tileBrighterGC);
		w->hexagons.tileBrighterGC = XtGetGC((Widget) w, valueMask, &values);
		if (!w->hexagons.mono) {
			values.foreground = darker(w, w->hexagons.tileColor);
		}
	if (w->hexagons.tileDarkerGC)
		XtReleaseGC((Widget) w, w->hexagons.tileDarkerGC);
	w->hexagons.tileDarkerGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->hexagons.mono) {
		if (w->hexagons.reverse) {
			values.foreground = w->hexagons.foreground;
			values.background = w->hexagons.background;
		} else {
			values.foreground = w->hexagons.background;
			values.background = w->hexagons.foreground;
		}
	} else {
		values.foreground = w->hexagons.borderColor;
		values.background = w->hexagons.tileColor;
	}
	if (w->hexagons.borderGC)
		XtReleaseGC((Widget) w, w->hexagons.borderGC);
	w->hexagons.borderGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->hexagons.fontInfo)
		XSetFont(XtDisplay(w), w->hexagons.borderGC,
			w->hexagons.fontInfo->fid);
}

static Boolean
SetValuesHexagons(Widget current, Widget request, Widget renew)
{
	HexagonsWidget c = (HexagonsWidget) current, w = (HexagonsWidget) renew;
	Boolean redraw = FALSE;
	Boolean redrawTiles = FALSE;

	CheckTiles(w);
	if (w->hexagons.font != c->hexagons.font ||
			w->hexagons.borderColor != c->hexagons.borderColor ||
			w->hexagons.reverse != c->hexagons.reverse ||
			w->hexagons.mono != c->hexagons.mono) {
		loadFont(w);
		SetAllColors(w);
		redrawTiles = True;
	} else if (w->hexagons.background != c->hexagons.background ||
			w->hexagons.foreground != c->hexagons.foreground ||
			w->hexagons.tileColor != c->hexagons.tileColor) {
		SetAllColors(w);
		redrawTiles = TRUE;
	}
	if (w->hexagons.sizeX != c->hexagons.sizeX ||
			w->hexagons.sizeY != c->hexagons.sizeY ||
			w->hexagons.corners != c->hexagons.corners ||
			w->hexagons.base != c->hexagons.base) {
		SizeHexagons(w);
		redraw = TRUE;
	} else if (w->hexagons.offset.x != c->hexagons.offset.x ||
			w->hexagons.offset.y != c->hexagons.offset.y) {
		ResizeHexagons(w);
		redraw = TRUE;
	}
	if (w->hexagons.menu != -1) {
		int menu = w->hexagons.menu;

		w->hexagons.menu = -1;
		switch (menu) {
		case 0:
			GetTiles(w);
			break;
		case 1:
			WriteTiles(w);
			break;
		case 3:
			ClearTiles(w);
			break;
		case 4:
			UndoTiles(w);
			break;
		case 5:
			RandomizeTiles(w);
			break;
		case 6:
			SolveTiles(w);
			break;
		case 7:
			ModeTiles(w);
			break;
		case 8:
			SpeedTiles(w);
			break;
		case 9:
			SlowTiles(w);
			break;
		case 10:
			SoundTiles(w);
			break;
		default:
			break;
		}
	}
	if (redrawTiles && !redraw && XtIsRealized(renew) && renew->core.visible) {
		EraseFrame(c, 0);
		DrawFrame(w, 0, True, w->hexagons.focus);
		DrawAllTiles(w);
	}
	return (redraw);
}

static void
QuitHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
#ifdef WINVER
	if (w->hexagons.freeDC) {
		if (w->hexagons.bufferTiles != NULL) {
			DeleteObject(w->hexagons.bufferTiles);
		}
		DeleteDC(w->core.memDC);
		w->core.memDC = NULL;
	}
#else
	Display *display = XtDisplay(w);
	int sel;

	if (w->hexagons.colormap != None) {
		XInstallColormap(display, w->hexagons.oldColormap);
		XFreeColormap(display, w->hexagons.colormap);
	}
	for (sel = 0; sel < 2; sel++)
		if (w->hexagons.bufferTiles[sel] != None)
			XFreePixmap(display, w->hexagons.bufferTiles[sel]);
	if (w->hexagons.fontInfo) {
		XUnloadFont(display, w->hexagons.fontInfo->fid);
		XFreeFont(display, w->hexagons.fontInfo);
	}
	XtCloseDisplay(display);
#endif
	exit(0);
}

static void
DestroyHexagons(Widget old)
{
	HexagonsWidget w = (HexagonsWidget) old;

	XtReleaseGC(old, w->hexagons.tileGC);
	XtReleaseGC(old, w->hexagons.borderGC);
	XtReleaseGC(old, w->hexagons.tileBrighterGC);
	XtReleaseGC(old, w->hexagons.tileDarkerGC);
	XtReleaseGC(old, w->hexagons.frameGC);
	XtReleaseGC(old, w->hexagons.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->hexagons.select);
}
#endif

static void
ResizeTiles(HexagonsWidget w)
{
	int i;
	int sel;

#ifdef WINVER
	if (w->core.memDC == NULL) {
		w->core.memDC = CreateCompatibleDC(w->core.hDC);
		if (w->core.memDC == NULL) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (sel = 0; sel < 2; sel++) {
		if (w->hexagons.bufferTiles[sel] != NULL) {
			DeleteObject(w->hexagons.bufferTiles[sel]);
				w->hexagons.bufferTiles[sel] = NULL;
		}
		if (!(w->hexagons.picture && *(w->hexagons.picture))) {
			if ((w->hexagons.bufferTiles[sel] =
			    CreateCompatibleBitmap(w->core.hDC,
			    w->core.width, w->core.height)) == NULL) {
				DISPLAY_ERROR("Not enough memory, exiting.");
			}
		}
	}
#else
	Display *display = XtDisplay(w);
	Window window = XtWindow(w);
	XWindowAttributes xgwa;

	(void) XGetWindowAttributes(display, window, &xgwa);
	if (w->hexagons.oldColormap == None) {
		w->hexagons.mono = (xgwa.depth < 2 || w->hexagons.mono);
		w->hexagons.oldColormap = xgwa.colormap;
	}
	for (sel = 0; sel < 2; sel++) {
		if (w->hexagons.bufferTiles[sel] != None) {
			XFreePixmap(display, w->hexagons.bufferTiles[sel]);
			w->hexagons.bufferTiles[sel] = None;
		}
		if ((w->hexagons.bufferTiles[sel] = XCreatePixmap(display,
			window, w->core.width, w->core.height,
			xgwa.depth)) == None) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
#endif
	if (w->hexagons.picture && *(w->hexagons.picture)) {
#ifdef WINVER
		for (sel = 0; sel < 2; sel++) {
			w->hexagons.bufferTiles[sel] =
				LoadBitmap(w->core.hInstance,
				w->hexagons.picture);
		}
#else
		if (w->hexagons.image != NULL) {
			destroyImage(&(w->hexagons.image),
				&(w->hexagons.graphicsFormat));
		}
		if (!getImage(display, window,
				xgwa.visual, w->hexagons.oldColormap, xgwa.depth,
				&(w->hexagons.image), w->hexagons.picture,
				w->hexagons.install, &(w->hexagons.graphicsFormat),
				&(w->hexagons.colormap))) {
			w->hexagons.picture = NULL;
		} else if (w->hexagons.image == NULL) {
			w->hexagons.picture = NULL;
		}
#endif
	}
#ifndef WINVER
	if (!(w->hexagons.picture && *(w->hexagons.picture)) &&
	    !fixedColors(xgwa.visual, xgwa.depth, w->hexagons.install) &&
	    w->hexagons.colormap == None) {
		w->hexagons.colormap = XCreateColormap(display, window,
			xgwa.visual, AllocNone);
	}
	SetAllColors(w);
	for (sel = 0; sel < 2; sel++) {
		FILLRECTANGLE(w, w->hexagons.bufferTiles[sel],
			w->hexagons.inverseGC,
			0, 0, w->core.width, w->core.height);
		if ((w->hexagons.picture && *(w->hexagons.picture))) {

			(void) XPutImage(display, w->hexagons.bufferTiles[sel],
				w->hexagons.inverseGC, w->hexagons.image, 0, 0,
				0, 0,
				MIN(w->hexagons.image->width, w->core.width),
				MIN(w->hexagons.image->height, w->core.height));
		}
	}
#endif
	for (i = 0; i <= 6; i++) {
		hexagonList[NOCORN][i].x = w->hexagons.tileSize.x *
			hexagonUnit[NOCORN][i].x / 4;
		hexagonList[NOCORN][i].y = 3 * w->hexagons.tileSize.y *
			hexagonUnit[NOCORN][i].y / 4;
		hexagonList[CORNERS][i].x = w->hexagons.tileSize.x *
			hexagonUnit[CORNERS][i].x / 2;
		hexagonList[CORNERS][i].y = w->hexagons.tileSize.y *
			hexagonUnit[CORNERS][i].y / 2;
	}
	if (!(w->hexagons.picture && *(w->hexagons.picture))) {
		DrawAllBufferedTiles(w);
	}
}

#ifndef WINVER
static
#endif
void
ResizeHexagons(HexagonsWidget w)
{
	const double sqrt_3 = 1.73205080756887729352744634150587237;
	Point tempSize;
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif

	w->hexagons.delta.x = 2;
	w->hexagons.delta.y = 2;
	w->hexagons.tileSize.x = MAX((2 * ((int) w->core.width +
			2 * w->hexagons.delta.x - 1) - 2 * (w->hexagons.sizeX +
			w->hexagons.sizeY) * w->hexagons.delta.x) / (2 *
			(w->hexagons.sizeX + w->hexagons.sizeY) - 1), 0);
	w->hexagons.tileSize.y = MAX(((int) w->core.height - 1 -
				2 * w->hexagons.sizeY * w->hexagons.delta.y) /
				(3 * w->hexagons.sizeY - 1), 0);
	w->hexagons.offset.x = w->hexagons.tileSize.x + w->hexagons.delta.x;
	w->hexagons.offset.y = w->hexagons.tileSize.y + 2 * w->hexagons.delta.y;
	tempSize.y = (int) ((double) w->hexagons.offset.x / sqrt_3);
	tempSize.x = (int) ((double) w->hexagons.offset.y * sqrt_3);
	if (tempSize.y < w->hexagons.offset.y) {
		w->hexagons.offset.x = w->hexagons.tileSize.x + w->hexagons.delta.x;
		w->hexagons.offset.y = tempSize.y;
	} else {		/* tempSize.x <= w->hexagons.offset.x */
		w->hexagons.offset.x = tempSize.x;
		w->hexagons.offset.y = w->hexagons.tileSize.y + 2 * w->hexagons.delta.y;
	}
	w->hexagons.tileSize.x = MAX(w->hexagons.offset.x - w->hexagons.delta.x + 1, 0);
	w->hexagons.tileSize.y = MAX(w->hexagons.offset.y - 2 * w->hexagons.delta.y - 1,
		0);
	w->hexagons.puzzleSize.x = (w->hexagons.sizeX + w->hexagons.sizeY) *
		w->hexagons.offset.x - w->hexagons.tileSize.x / 2 -
		2 * w->hexagons.delta.x + 1;
	w->hexagons.puzzleSize.y = w->hexagons.sizeY * (3 *
		w->hexagons.tileSize.y + 2 * w->hexagons.delta.y) -
		w->hexagons.tileSize.y + 2;
	w->hexagons.puzzleOffset.x =
		((int) w->core.width - w->hexagons.puzzleSize.x + 2) / 2;
	w->hexagons.puzzleOffset.y =
		((int) w->core.height - w->hexagons.puzzleSize.y + 4) / 2;
}

#ifndef WINVER
static
#endif
void
SizeHexagons(HexagonsWidget w)
{
	ResetTiles(w);
	ResizeHexagons(w);
}

#ifndef WINVER
static
#endif
void
InitializeHexagons(
#ifdef WINVER
HexagonsWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
#ifdef WINVER
	SetValuesHexagons(w);
	brush = CreateSolidBrush(w->hexagons.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
	w->hexagons.bufferTiles[0] = NULL;
	w->hexagons.bufferTiles[1] = NULL;
#else
	HexagonsWidget w = (HexagonsWidget) renew;

#ifdef DEBUG
XSynchronize(XtDisplay(w), True);
#endif
	(void) SRAND(getpid());
	w->hexagons.bufferTiles[0] = None;
	w->hexagons.bufferTiles[1] = None;
	w->hexagons.colormap = None;
	w->hexagons.oldColormap = None;
	w->hexagons.fontInfo = NULL;
	w->hexagons.tileGC = NULL;
	w->hexagons.borderGC = NULL;
	w->hexagons.tileBrighterGC = NULL;
	w->hexagons.tileDarkerGC = NULL;
	w->hexagons.frameGC = NULL;
	w->hexagons.inverseGC = NULL;
#endif
	w->hexagons.focus = False;
	loadFont(w);
	w->hexagons.tileOfPosition = NULL;
	CheckTiles(w);
	InitMoves();
	w->hexagons.numSlices = ((w->hexagons.delay < MAXSLICES) ?
		w->hexagons.delay + 1 : MAXSLICES);
	w->hexagons.cheat = False;
	SizeHexagons(w);
}

#ifndef WINVER
static
#endif
void
ExposeHexagons(
#ifdef WINVER
HexagonsWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
#ifndef WINVER
	HexagonsWidget w = (HexagonsWidget) renew;

	if (!w->core.visible)
		return;
#endif
	ResizeTiles(w);
	EraseFrame(w, 0);
	DrawFrame(w, 0, True, w->hexagons.focus);
	DrawAllTiles(w);
}

#ifndef WINVER
static
#endif
void
HideHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SetHexagons(w, HEXAGONS_HIDE);
}

#ifndef WINVER
static
#endif
void
SelectHexagons(HexagonsWidget w
#ifdef WINVER
, const int x, const int y
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int row, rowType;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
#endif

	w->hexagons.currentPosition = PositionToTile(w, x, y, &row);
	if (w->hexagons.currentPosition >= 0) {
		if (CheckSolved(w)) {
			MoveNoTiles(w);
			w->hexagons.currentPosition = -1;
			return;
		}
		w->hexagons.currentRow[ROW] = row;
		w->hexagons.currentRow[TRBL] = TrBl(w, w->hexagons.currentPosition, row);
		w->hexagons.currentRow[TLBR] = TlBr(w, w->hexagons.currentPosition, row);
		if (w->hexagons.corners)
			rowType = MovableCornerTile(w);
		else
			rowType = MovableNoCornTile(w);
		if (rowType < 0) {
			DrawTile(w, w->hexagons.currentPosition,
				rowType == HEXAGONS_SPACE, False, TRUE, 0, 0);
			FLUSH(w);
			Sleep(100);
			DrawTile(w, w->hexagons.currentPosition,
				True, True, TRUE, 0, 0);
			if (rowType != HEXAGONS_SPACE)
				DrawTile(w, w->hexagons.currentPosition,
					False, False, FALSE, 0, 0);
			SetHexagons(w, rowType);
			w->hexagons.currentPosition = -1;
			return;
		}
		DrawTile(w, w->hexagons.currentPosition,
			False, False, TRUE, 0, 0);
	} else
		w->hexagons.currentPosition = -1;
}

#ifndef WINVER
static
#endif
void
ReleaseHexagons(HexagonsWidget w
#ifdef WINVER
, const int x, const int y
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int pos, row, space;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
#endif

	if (w->hexagons.currentPosition < 0)
		return;
	DrawTile(w, w->hexagons.currentPosition, True, True, TRUE, 0, 0);
	DrawTile(w, w->hexagons.currentPosition, False, False, FALSE, 0, 0);
	if (!w->hexagons.corners) {
		SelectNoCornTiles(w);
		w->hexagons.currentPosition = -1;
		return;
	}
	pos = PositionToTile(w, x, y, &row);
	if (pos >= 0) {
		if (w->hexagons.spacePosition[HIGH] == pos)
			space = (w->hexagons.spacePosition[HIGH] >
				 w->hexagons.spacePosition[LOW]);
		else if (w->hexagons.spacePosition[LOW] == pos)
			space = (w->hexagons.spacePosition[HIGH] <
				 w->hexagons.spacePosition[LOW]);
		else {
			w->hexagons.currentPosition = -1;
			return;
		}
		SelectCornerTiles(w, space);
	}
	w->hexagons.currentPosition = -1;
}

#ifndef WINVER
static
#endif
void
RandomizeHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	RandomizeTiles(w);
}

#ifndef WINVER
static void
RandomizeHexagonsMaybe(HexagonsWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->hexagons.started)
		RandomizeTiles(w);
#ifdef HAVE_MOTIF
	else {
		SetHexagons(w, HEXAGONS_RANDOMIZE_QUERY);
	}
#endif
}

static void
RandomizeHexagons2(HexagonsWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->hexagons.started)
#endif
		RandomizeTiles(w);
}
#endif

#ifndef WINVER
static
#endif
void
GetHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	GetTiles(w);
}

#ifndef WINVER
static
#endif
void
WriteHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	WriteTiles(w);
}

#ifndef WINVER
static
#endif
void
ClearHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ClearTiles(w);
}

#ifndef WINVER
static
#endif
void
UndoHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	UndoTiles(w);
}

#ifndef WINVER
static
#endif
void
SolveHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SolveTiles(w);
}

#ifndef WINVER
static
#endif
void
ModeHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ModeTiles(w);
}

#ifndef WINVER
static
#endif
void
SpeedHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SpeedTiles(w);
} 

#ifndef WINVER
static
#endif
void
SlowHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SlowTiles(w);
} 

#ifndef WINVER
static
#endif
void
SoundHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SoundTiles(w);
}

#ifndef WINVER
static
#endif
void
EnterHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->hexagons.focus = True;
	DrawFrame(w, 0, False, w->hexagons.focus);
}

#ifndef WINVER
static
#endif
void
LeaveHexagons(HexagonsWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->hexagons.focus = False;
	DrawFrame(w, 0, False, w->hexagons.focus);
}

#ifndef WINVER
static void
MoveHexagonsTl(HexagonsWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveHexagons(w, TL, (int) (event->xkey.state & ControlMask));
}

static void
MoveHexagonsTop(HexagonsWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveHexagons(w, TOP, (int) (event->xkey.state & ControlMask));
}

static void
MoveHexagonsTr(HexagonsWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveHexagons(w, TR, (int) (event->xkey.state & ControlMask));
}

static void
MoveHexagonsLeft(HexagonsWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveHexagons(w, LEFT, (int) (event->xkey.state & ControlMask));
}

static void
MoveHexagonsRight(HexagonsWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveHexagons(w, RIGHT, (int) (event->xkey.state & ControlMask));
}

static void
MoveHexagonsBl(HexagonsWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveHexagons(w, BL, (int) (event->xkey.state & ControlMask));
}

static void
MoveHexagonsBottom(HexagonsWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveHexagons(w, BOTTOM, (int) (event->xkey.state & ControlMask));
}

static void
MoveHexagonsBr(HexagonsWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveHexagons(w, BR, (int) (event->xkey.state & ControlMask));
}
#endif
