/*-
# MOTIF/X-BASED BARREL(tm)
#
#  xbarrel.c
#
###
#
#  Copyright (c) 2003 - 2006	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/11/15 X/Windows
*/

#include "file.h"
#ifdef WINVER
#include "BarrelP.h"
#include "wbarrel.h"
#define TITLE "wbarrel"

static BarrelRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->barrel.started = b
#else
#include "xwin.h"
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Barrel.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/barrel.xpm"
#endif
#include "icons/barrel.xbm"
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef WINVER
static const char aboutHelp[] = {
"Barrel Version 7.2.1\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd@tux.org>\n"
"The latest version is at: "
"http://www.tux.org/~bagleyd/puzzles.html\n"
};

static const char optionsHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-tile {color}] [-face{0|1|2|3|4|5} {color}]\n"
"[-{border|bd} {color}] [-[no]install]\n"
"[-picture {filename}] [-delay msecs] [-[no]sound]\n"
"[-bumpSound {filename}] [-moveSound {filename}]\n"
"[-{font|fn} {fontname}] [-[no]orient] [-[no]pairs]\n"
"[-base {int}] [-userName {string}] [-scoreFile {filename}]\n"
"[-scores] [-version]\n"
};
#endif

#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
"The Barrel, the Billion Barrel, or Wonderful Barrel also "
"seen as the Cylinder Puzzle of Japanese origin.  The puzzle\n"
"contains 23 beads of 6 different colors.  4 each of 5 "
"different colors arranged in 5 rows and 3 black beads making\n"
"3 rows one unit longer.  A system of plungers slide all "
"three of the long rows left and right.  Two of the plungers\n"
"sit together and the other (the middle plunger) is "
"separate from the other two.  There are 2 separate wheels or\n"
"disks that rotate about the center axis.\n"
};

static const char featuresHelp[] = {
"Press \"mouse-left\" button to move a tile. Release "
"\"mouse-left\" button in the same row would move the tile towards\n"
"the space (unless there is no space in that row).  "
"Release on a different row will move the tiles up or down.\n"
"\n"
"Click \"mouse-center\" button, or press \"P\" or \"p\" "
"keys, to toggle the practice mode (in practice mode the\n"
"record should say \"practice\").  This is good for learning "
"moves and experimenting.\n"
"\n"
"Click \"mouse-right\" button, or press \"R\" or \"r\" "
"keys, to randomize the puzzle.\n"
"\n"
"Press \"O\" or \"o\" keys to toggle orient mode.  Slightly "
"more challenging on (default is off).\n"
"\n"
"Press \"I\" or \"i\" keys to toggle the pairs mode.  All "
"the tiles rotate in pairs.  This is considerably\n"
"more challenging on (default is on).\n"
"\n"
"\"S\" or \"s\" keys reserved for the auto-solver "
"(not implemented).\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save a puzzle.\n"
"\n"
"Press \"C\" or \"c\" keys to clear a puzzle.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of tiles.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of tiles.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Barrel as:\n"
"  8    Up\n"
"  ^\n"
"4< >6  Left, Right\n"
"  v\n"
"  2    Down\n"
"\n"
"Use the shift key and the up or down arrow keys to rotate "
"all the tiles.\n"
};

static const char referencesHelp[] = {
"L. E. Horden, Sliding Piece Puzzles (Recreations in "
"Mathematics Series), Oxford University\n"
"Press 1986, Plate X.\n"
"James G Nourse, The Simple Solutions to Cubic, Puzzles, "
"Bantam Books, New York, November 1981, pp 25-42.\n"
"John Ewing & Czes Kosniowski, Puzzle it Out: "
"Cubes Groups and Puzzles, Cambridge University Press,\n"
"New York, 1982, pp 50-54?\n"
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, not implemented.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "barrel.scores"
#endif

#define NEVER (-1)
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord barrelRecord[2][2];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
static Widget moves, record, message;
static Widget orientSwitch, pairsSwitch, practiceSwitch;
#if 0
static Widget tile, face;
#endif
static Widget speed;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
static Arg arg[3];
#else
static char titleDsp[TITLELEN] = "";
#endif
static char scoreFileName[FILENAMELEN] = SCOREFILE;
static char fileName[FILENAMELEN];
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char userNameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap barrelIcon = None;
static Widget topLevel, barrel;
static char *progDsp;
static char userNameDsp[USERNAMELEN] = "";

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w, XmNlabelString, xmstr, NULL);
}
#endif

static void
PrintRecords(void)
{
	int k, l;

	(void) printf("        BARREL  HALL OF FAME\n\n");
	(void) printf("PAIR ORIENT USER            MOVES\n");
	for (l = 0; l < 2; l++)
		for (k = 0; k < 2; k++) {
			if (barrelRecord[l][k].score > 0)
				(void) printf("%4d%7d %-12s%9d\n",
					l, k,
					barrelRecord[l][k].name,
					barrelRecord[l][k].score);
		}
}
#endif

static void
InitRecords(void)
{
	int k, l;

	for (l = 0; l < 2; l++)
		for (k = 0; k < 2; k++) {
			barrelRecord[l][k].score = NEVER;
			(void) strncpy(barrelRecord[l][k].name,
				NOACCESS, USERNAMELEN);
		}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int k, l, n;
	char userName[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USERNAMELEN);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USERNAMELEN);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (l = 0; l < 2; l++)
		for (k = 0; k < 2; k++) {
			(void) fscanf(fp, "%d %s\n", &n, userName);
			if (n <= barrelRecord[l][k].score ||
					barrelRecord[l][k].score <= NEVER) {
				barrelRecord[l][k].score = n;
				(void) strncpy(barrelRecord[l][k].name,
					userName, USERNAMELEN);
			}
		}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int k, l;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Can not write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILENAMELEN];

		(void) strncpy(lockFile, fileName, FILENAMELEN - 6);
		(void) strcat(lockFile, ".lock");
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (l = 0; l < 2; l++) {
			for (k = 0; k < 2; k++) {
				(void) fprintf(fp, "%d %s\n",
					barrelRecord[l][k].score,
					barrelRecord[l][k].name);
			}
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
PrintRecord(Boolean orient, Boolean pairs, Boolean practice)
{
	int k = (orient) ? 1 : 0, l = (pairs) ? 1 : 0;

	if (practice) {
		(void) strncpy(recordDsp, "practice", MESSAGELEN);
#if 0
	} else if (tiles > MAXTILES) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
#endif
	} else if (barrelRecord[l][k].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			barrelRecord[l][k].score,
			barrelRecord[l][k].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static void
PrintStatus(char *msg, int nMoves
#ifndef HAVE_MOTIF
		, Boolean pairs
#endif
		)
{
#ifdef HAVE_MOTIF
	PrintState(message, msg);
	(void) sprintf(buff, "%d", nMoves);
	PrintState(moves, buff);
#else
	if (pairs)
		(void) sprintf(titleDsp, "%s: norm@ (%d/%s) - %s",
			progDsp, nMoves, recordDsp, msg);
	else
		(void) sprintf(titleDsp, "%s: ind@ (%d/%s) - %s",
			progDsp, nMoves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent(barrel), XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
HandleSolved(int counter, Boolean orient, Boolean pairs)
{
	int k = (orient) ? 1 : 0, l = (pairs) ? 1 : 0;

	if ((counter < barrelRecord[l][k].score ||
			barrelRecord[l][k].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		barrelRecord[l][k].score = counter;
		(void) strncpy(barrelRecord[l][k].name, userNameDsp,
			USERNAMELEN);
		WriteRecords();
		PrintRecord(orient, pairs, False);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
BarrelWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	Boolean orient, pairs, practice;
	char *userName, *scoreFile;

#ifdef WINVER
	InitializeBarrel(w, brush);

	w->barrel.started = FALSE;
	orient = w->barrel.orient;
	pairs = w->barrel.pairs;
	practice = w->barrel.practice;
	userName = w->barrel.userName;
	scoreFile = w->barrel.scoreFile;
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
#else
	Boolean scoreOnly, versionOnly;

	XtVaGetValues(w,
		XtNuserName, &userName,
		XtNscoreFile, &scoreFile,
		XtNorient, &orient,
		XtNpairs, &pairs,
		XtNpractice, &practice,
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf(aboutHelp);
		exit(0);
	}
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
	if (scoreOnly) {
		InitRecords();
		ReadRecords();
		PrintRecords();
		exit(0);
	}
#ifdef HAVE_MOTIF
#if 0
	if (tiles > MAXTILES)
		XtVaSetValues(tile, XmNmaximum, tiles, NULL);
	XmScaleSetValue(tile, tiles);
	XmScaleSetValue(face, faces);
#endif
	XmToggleButtonSetState(orientSwitch, orient, True);
	XmToggleButtonSetState(pairsSwitch, pairs, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
	{
		int delay;

		XtVaGetValues(barrel, XtNdelay, &delay, NULL);
		XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay - 1);
	}
#endif
#endif
	SET_STARTED(w, False);
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USERNAMELEN);
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USERNAMELEN);
#else
		char * login = getlogin();

		if (login == NULL) {
			(void) strcpy(userNameDsp, "");
		} else {
			(void) sprintf(userNameDsp, "%s", login);
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(userNameDsp, "%s", "guest");
#endif
	}
	PrintRecord(orient, pairs, practice);
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, pairs
#endif
		);
}

#ifdef WINVER
void
SetBarrel(BarrelWidget w, int reason)
#else
static void
CallbackBarrel(Widget w, caddr_t clientData, barrelCallbackStruct * callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	Boolean orient, pairs, practice, cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	orient = w->barrel.orient;
	pairs = w->barrel.pairs;
	practice = w->barrel.practice;
	cheat = w->barrel.cheat;
#else
	XtVaGetValues(w,
		XtNorient, &orient,
		XtNpairs, &pairs,
		XtNpractice, &practice,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case BARREL_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
			break;
#ifndef WINVER
		case BARREL_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(practiceDialog);
#else
			XtVaSetValues(barrel,
				XtNmenu, 9, NULL); /* menu choice */
#endif
			break;
		case BARREL_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtVaSetValues(barrel,
				XtNmenu, 5, NULL); /* menu choice */
#endif
			break;
#endif
		case BARREL_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case BARREL_RESTORE:
			if (practice) {
				(void) strncpy(recordDsp, "practice", MESSAGELEN);
#ifdef HAVE_MOTIF
				PrintState(record, recordDsp);
#endif
			}
			movesDsp = 0;
			break;
		case BARREL_RESET:
			movesDsp = 0;
			break;
		case BARREL_BLOCKED:
			(void) strncpy(messageDsp, "Blocked", MESSAGELEN);
			break;
		case BARREL_SPACE:
#if 0
			/* Too annoying */
			(void) strncpy(messageDsp, "A space can not slide",
				MESSAGELEN);
#endif
			break;
		case BARREL_IGNORE:
			(void) strncpy(messageDsp, "Randomize to start",
				MESSAGELEN);
			break;
		case BARREL_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case BARREL_CONTROL:
			return;
		case BARREL_SOLVED:
			if (practice)
				movesDsp = 0;
			else if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", userNameDsp);
			else if (HandleSolved(movesDsp, orient, pairs))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", userNameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
			SET_STARTED(w, False);
			break;
		case BARREL_PRACTICE:
			movesDsp = 0;
			practice = !practice;
			if (!practice)
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			PrintRecord(orient, pairs, practice);
#ifdef WINVER
			w->barrel.practice = practice;
			w->barrel.started = False;
#else
			XtVaSetValues(w,
				XtNpractice, practice,
				XtNstart, False, NULL);
#endif
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
			break;
		case BARREL_RANDOMIZE:
			movesDsp = 0;
#ifdef WINVER
			w->barrel.practice = False;
			w->barrel.started = False;
#else
			XtVaSetValues(w,
				XtNpractice, False,
				XtNstart, False, NULL);
#endif
			break;
		case BARREL_ORIENT:
			movesDsp = 0;
			orient = !orient;
			PrintRecord(orient, pairs, practice);
#ifdef WINVER
			w->barrel.orient = orient;
#else
			XtVaSetValues(w, XtNorient, orient, NULL);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(orientSwitch, orient, True);
#endif
#endif
			break;
		case BARREL_PAIRS:
			movesDsp = 0;
			pairs = !pairs;
			PrintRecord(orient, pairs, practice);
#ifdef WINVER
			w->barrel.pairs = pairs;
#else
			XtVaSetValues(w, XtNpairs, pairs, NULL);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(pairsSwitch, pairs, True);
#endif
#endif
			break;
#if 0
		case BARREL_DEC_X:
			movesDsp = 0;
			tiles--;
			PrintRecord(tiles, faces, orient, pairs, practice);
#ifdef WINVER
			w->barrel.tiles = tiles;
#else
			XtVaSetValues(w, XtNtiles, tiles, NULL);
#ifdef HAVE_MOTIF
			XmScaleSetValue(tile, tiles);
			if (tiles >= MAXTILES)
				XtVaSetValues(tile, XmNmaximum, tiles, NULL);
#endif
#endif
			break;
		case BARREL_INC_X:
			movesDsp = 0;
			tiles++;
			PrintRecord(tiles, faces, orient, pairs, practice);
#ifdef WINVER
			w->barrel.tiles = tiles;
#else
			XtVaSetValues(w, XtNtiles, tiles, NULL);
#ifdef HAVE_MOTIF
			if (tiles > MAXTILES)
				XtVaSetValues(tile, XmNmaximum, tiles, NULL);
			XmScaleSetValue(tile, tiles);
#endif
#endif
			break;
		case BARREL_DEC_Y:
			movesDsp = 0;
			faces--;
			PrintRecord(tiles, faces, orient, pairs, practice);
#ifdef WINVER
			w->barrel.faces = faces;
#else
			XtVaSetValues(w, XtNfaces, faces, NULL);
#ifdef HAVE_MOTIF
			XmScaleSetValue(face, faces);
#endif
#endif
			break;
		case BARREL_INC_Y:
			movesDsp = 0;
			faces++;
			PrintRecord(tiles, faces, orient, pairs, practice);
#ifdef WINVER
			w->barrel.faces = faces;
#else
			XtVaSetValues(w, XtNfaces, faces, NULL);
#ifdef HAVE_MOTIF
			XmScaleSetValue(face, faces);
#endif
			break;
#endif
#endif
		case BARREL_COMPUTED:
			SET_STARTED(w, False);
			break;
		case BARREL_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
#ifdef HAVE_MOTIF
		case BARREL_SPEED:
			{
				int oldDelay, delay;

				XtVaGetValues(barrel, XtNdelay, &delay, NULL);
				oldDelay = delay;
				if (delay > MAXSPEED - MINSPEED)
					delay = MAXSPEED - MINSPEED;
				if (delay < 0)
					delay = 0;
				if (delay != oldDelay) {
					XtVaSetValues(w,
						XtNdelay, delay, NULL);
				}
				XmScaleSetValue(speed,
					MAXSPEED + MINSPEED - delay - 1);
			}
			return;
#endif
	}
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, pairs
#endif
		);
}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;
	static Boolean mousePressed = False;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.barrel.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterBarrel(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.barrel.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeaveBarrel(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyBarrel(brush);
			break;
		case WM_SIZE:
			ResizeBarrel(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposeBarrel(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			RandomizeBarrel(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			mousePressed = True;
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectBarrel(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_MOUSEMOVE:
			if (!(wParam & MK_LBUTTON)) {
				if (mousePressed) {
					mousePressed = False;
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ReleaseBarrel(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
					(void) ReleaseDC(hWnd, widget.core.hDC);
				}
				break;
			}
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			MotionBarrel(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			mousePressed = False;
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleaseBarrel(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
		case WM_MOUSEWHEEL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			{
				int zDelta = ((short) HIWORD(wParam));
				POINT cursor, origin;

				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				if (zDelta > (WHEEL_DELTA >> 1)) {
					MoveBarrelInput(&widget, cursor.x - origin.x, TOP,
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
				} else if (zDelta < -(WHEEL_DELTA >> 1)) {
					MoveBarrelInput(&widget, cursor.x - origin.x, BOTTOM,
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)));
				}
			}
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#endif
		case WM_COMMAND:
			switch LOWORD((wParam)) {
				case IDM_GET:
					GetBarrel(&widget);
					ResizeBarrel(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WriteBarrel(&widget);
					break;
				case IDM_EXIT:
					DestroyBarrel(brush);
					break;
				case IDM_HIDE:
					HideBarrel(&widget);
					break;
				case IDM_CLEAR:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ClearBarrel(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoBarrel(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					RandomizeBarrel(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_PRACTICE:
					PracticeBarrel(&widget);
					SizeBarrel(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolveBarrel(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_ORIENT:
					OrientizeBarrel(&widget);
					SizeBarrel(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_PAIRS:
					(void) PairsBarrel(&widget);
					SizeBarrel(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SPEED:
					SpeedBarrel(&widget);
					break;
				case IDM_SLOW:
					SlowBarrel(&widget);
					break;
				case IDM_SOUND:
					SoundBarrel(&widget);
					break;
				case IDM_LEFT:
				case IDM_RIGHT:
				case IDM_SHIFT_UP:
				case IDM_SHIFT_DOWN:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					(void) MoveBarrelInput(&widget, 0,
							(int) LOWORD(wParam) - IDM_SHIFT_UP, !((int) wParam % 2));
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UP:
				case IDM_DOWN:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveBarrelInput(&widget, cursor.x - origin.x,
								(int) LOWORD(wParam) - IDM_UP, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
						"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
						"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
						"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#if 0
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#else
		512 + 10, 512 + 10 + 34,
#endif
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char * programName)
{
	unsigned int i;

	(void) fprintf(stderr, "usage: %s\n", programName);
	for (i = 0; i < strlen(optionsHelp); i++) {
		if (i == 0 || optionsHelp[i - 1] == '\n')
			(void) fprintf(stderr, "\t");
		(void) fprintf(stderr, "%c", (optionsHelp[i]));
	}
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*barrel.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*barrel.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*barrel.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*barrel.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*barrel.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*barrel.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*barrel.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*barrel.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*barrel.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*barrel.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-tile", (char *) "*barrel.tileColor", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*barrel.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*barrel.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*barrel.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*barrel.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*barrel.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*barrel.faceColor5", XrmoptionSepArg, NULL},
#if 0
	{(char *) "-tiles", (char *) "*barrel.tiles", XrmoptionSepArg, NULL},
	{(char *) "-faces", (char *) "*barrel.faces", XrmoptionSepArg, NULL},
#endif
	{(char *) "-orient", (char *) "*barrel.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*barrel.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-pairs", (char *) "*barrel.pairs", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopairs", (char *) "*barrel.pairs", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-install", (char *) "*barrel.install", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noinstall", (char *) "*barrel.install", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-picture", (char *) "*barrel.picture", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*barrel.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*barrel.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*barrel.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bumpSound", (char *) "*barrel.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-moveSound", (char *) "*barrel.moveSound", XrmoptionSepArg, NULL},
	{(char *) "-base", (char *) "*barrel.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*barrel.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*barrel.font", XrmoptionSepArg, NULL},
	{(char *) "-userName", (char *) "*barrel.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*barrel.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*barrel.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*barrel.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
CallbackBarrelPractice(Widget w, XtPointer clientData,
		XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(barrel, XtNmenu, 9, NULL); /* menu choice */
	}
}

static void
CallbackBarrelRandomize(Widget w, XtPointer clientData,
		XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(barrel, XtNmenu, 5, NULL); /* menu choice */
	}
}

#if 0
static void
TileSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int tiles = cbs->value, faces, old;
	Boolean orient, pairs, practice;

	XtVaGetValues(barrel,
		XtNtiles, &old,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNpairs, &pairs,
		XtNpractice, &practice, NULL);
	if (old == tiles)
		return;
	XtVaSetValues(barrel,
		XtNtiles, tiles, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, pairs, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
FaceSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int tiles, faces = cbs->value, old;
	Boolean orient, pairs, practice;

	XtVaGetValues(barrel,
		XtNtiles, &tiles,
		XtNfaces, &old,
		XtNorient, &orient,
		XtNpairs, &pairs,
		XtNpractice, &practice, NULL);
	if (old == faces)
		return;
	XtVaSetValues(barrel,
		XtNfaces, faces, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, pairs, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}
#endif

static void
OrientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	Boolean orient = cbs->set, pairs, practice, oldOrient;
	char *picture;

	XtVaGetValues(barrel,
		XtNorient, &oldOrient,
		XtNpairs, &pairs,
		XtNpicture, &picture,
		XtNpractice, &practice, NULL);
	if (oldOrient == orient)
		return;
	if (!orient && picture && *picture) {
		XmToggleButtonSetState(orientSwitch, oldOrient, True);
		return;
	}
	XtVaSetValues(barrel,
		XtNorient, orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(orient, pairs, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PairsToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	Boolean orient, pairs = cbs->set, practice, oldPairs;

	XtVaGetValues(barrel,
		XtNorient, &orient,
		XtNpairs, &oldPairs,
		XtNpractice, &practice, NULL);
	if (oldPairs == pairs)
		return;
	XtVaSetValues(barrel,
		XtNpairs, pairs, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(orient, pairs, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PracticeToggle(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct * cbs)
{
	Boolean orient, pairs, practice = cbs->set, oldPract;

	XtVaGetValues(barrel,
		XtNorient, &orient,
		XtNpairs, &pairs,
		XtNpractice, &oldPract, NULL);
	if (oldPract == practice)
		return;
	XtVaSetValues(barrel,
		XtNpractice, practice,
		XtNstart, False, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(orient, pairs, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGELEN);
	PrintState(message, messageDsp);
}

static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int delay = MAXSPEED + MINSPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(barrel,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(barrel, XtNdelay, delay, NULL);
	}
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtVaSetValues(barrel, XtNmenu, val, NULL);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	XtVaSetValues(barrel, XtNmenu, val + 3, NULL); /* GWQ */
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, speedRowCol, switchRowCol;
	Widget messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, randomizeString, solveString;
	XmString orientString, pairsString, practiceString;
	XmString speedString, slowString, soundString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Barrel",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) barrel_xpm, &barrelIcon, NULL,
			&xpmAttributes);
	}
	if (barrelIcon == (Pixmap) NULL)
#endif
		barrelIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) barrel_bits,
			barrel_width, barrel_height);
#ifdef HAVE_MOTIF
	XtVaSetValues(topLevel,
		XtNiconPixmap, barrelIcon,
		XmNkeyboardFocusPolicy, XmPOINTER, NULL); /* not XmEXPLICIT */
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "Solve");
	orientString = XmStringCreateSimple((char *) "Orientize");
	pairsString = XmStringCreateSimple((char *) "PAirs");
	practiceString = XmStringCreateSimple((char *) "Practice");
	speedString = XmStringCreateSimple((char *) ">Speed");
	slowString = XmStringCreateSimple((char *) "<Slow");
	soundString = XmStringCreateSimple((char *) "@Sound");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, pairsString, 'A', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(orientString);
	XmStringFree(pairsString);
	XmStringFree(practiceString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) CallbackBarrelPractice);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackBarrelRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move tile", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

#if 0
	tilesRowCol = XtVaCreateManagedWidget("tilesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	tile = XtVaCreateManagedWidget("tile",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Tiles", 6,
		XmNminimum, MINTILES,
		XmNmaximum, MAXTILES,
		XmNvalue, DEFAULTTILES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(tile, XmNvalueChangedCallback,
		(XtCallbackProc) TileSlider, (XtPointer) NULL);
	face = XtVaCreateManagedWidget("face",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Faces", 6,
		XmNminimum, MINFACES,
		XmNmaximum, MAXFACES,
		XmNvalue, DEFAULTFACES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(face, XmNvalueChangedCallback,
		(XtCallbackProc) FaceSlider, (XtPointer) NULL);
#endif
	speedRowCol = XtVaCreateManagedWidget("speedRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	speed = XtVaCreateManagedWidget("Animation Speed",
		xmScaleWidgetClass, speedRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - DEFAULTDELAY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTORIENT, NULL);
	XtAddCallback(orientSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) OrientToggle, (XtPointer) NULL);
	pairsSwitch = XtVaCreateManagedWidget("Pairs rigid",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTPAIRS, NULL);
	XtAddCallback(pairsSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PairsToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTPRACTICE, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PracticeToggle, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Barrel! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);

	barrel = XtCreateManagedWidget("barrel",
		barrelWidgetClass, mainPanel, NULL, 0);
#else
	XtVaSetValues(topLevel,
		XtNiconPixmap, barrelIcon,
		XtNinput, True, NULL);
	barrel = XtCreateManagedWidget("barrel",
		barrelWidgetClass, topLevel, NULL, 0);
#endif
	XtAddCallback(barrel, XtNselectCallback,
		(XtCallbackProc) CallbackBarrel, (XtPointer) NULL);
	Initialize(barrel);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(barrel), (unsigned int) AnyButton, AnyModifier,
		XtWindow(barrel), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(barrel),
		XCreateFontCursor(XtDisplay(barrel), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
