/*-
# MOTIF/X-BASED HEXAGONS
#
#  xhexagons.c
#
###
#
#  Copyright (c) 1993 - 2006	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/11/15 X/Windows
  Version 5: 95/10/01 Xt/Motif
  Version 4: 94/06/07 Xt
  Version 3: 93/04/01 Motif
  Version 2: 92/01/07 XView
  Version 1: 91/09/05 SunView
*/

#include "file.h"
#ifdef WINVER
#include "HexagonsP.h"
#include "whexagons.h"
#define TITLE "whexagons"

static HexagonsRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->hexagons.started = b
#else
#include "xwin.h"
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Hexagons.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/hexagons.xpm"
#endif
#include "icons/hexagons.xbm"
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef WINVER
static const char aboutHelp[] = {
"Hexagons Version 7.2\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd@tux.org>\n"
"The latest version is at: "
"http://www.tux.org/~bagleyd/puzzles.html\n"
};

static const char optionsHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-tile {color}] [-{border|bd} {color}] [-delay msecs]\n"
"[-[no]sound] [-bumpSound {filename}]\n"
"[-{font|fn} {fontname}] [-size{x|y} {int}] [-[no]corners]\n"
"[-base {int}] [-userName {string}] [-scoreFile {filename}]\n"
"[-scores] [-version]\n"
};
#endif

#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
"A sliding tile puzzle.  A hexagonal version of the "
"15 puzzle.  The 17 puzzle was conceived of by Alwin Vietor\n"
"and marketed in 1885 in Germany as \"Trilemma\" with "
"round pieces.  Sam Cornwell rediscovered this in 1989\n"
"with \"Clockblocks\" of which 12 were numbered using "
"hexagonal pieces.  Also he produced a smaller 12 block\n"
"puzzle with 9 that were numbered called \"Ninefield\".  Also "
"Bernhard Wiezorke came up with the similar 17 \n"
"puzzle, \"Hexafax, the New Fifteen Puzzle\", in 1991 and a "
"smaller 11 piece \"Counterfax\" of which 8 were\n"
"numbered.\n"
};

static const char featuresHelp[] = {
"If in corners mode: Press \"mouse-left\" button to move "
"a tile, then release \"mouse-left\" button on a space.\n"
"It will not move if blocked.  One can only move one "
"tile at a time. The only tiles that can be moved are\n"
"the tile(s) next to both spaces.\n"
"If in nocorners mode: Click \"mouse-left\" button to "
"move a tile.  One can move more than one tile at a time\n"
"and one is not constrained by parity.\n"
"\n"
"Click \"mouse-right\" button, or press \"R\" or \"r\" "
"keys, to randomize the puzzle.\n"
"\n"
"Press \"O\" or \"o\" keys to toggle corners mode.  "
"(Default is on).\n"
"\n"
"\"S\" or \"s\" keys reserved for the auto-solver "
"(not implemented).\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save a puzzle.\n"
"\n"
"Press \"C\" or \"c\" keys to clear a puzzle.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of tiles.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of tiles.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Hexagons as:\n"
"7   9  Upper Left, Upper Right\n"
"  ^\n"
"4< >6  Left, Right\n"
"  v\n"
"1   3  Bottom Left, Bottom Right\n"
"\n"
"The control key allows you to change the complexity of "
"xhexagons.  Hold down control key and use the keypad\n"
"or arrow keys to reduce or enlarge puzzle complexity.\n"
};

static const char referencesHelp[] = {
"Dieter Gebhardt, \"From Trilemma to Counterfax\", CFF 53, "
"October 2000, pp 8-10.\n"
"Sam Cornwell, \"Ninefield: a Sliding Block Puzzle\", "
"CFF 57, March 2002, pp 4-5.\n"
"Dieter Gebhardt, \"Ninefield Revisited\", CFF 58, "
"July 2002, p 21.\n"
"Bernhard Wiezorke, \"Hexafax, the New Fifteen Puzzle\" "
"brochure.\n"
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, not implemented.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "hexagons.scores"
#endif

#define MAXTILES 10
#define NEVER (-1)
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord hexagonsRecord[MAXORIENT][MAXTILES - MINTILES + 1][MAXTILES - MINTILES + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
#ifdef ANIMATE
static Widget speed;
#endif
static Widget moves, record, message, cornersSwitch, tileX, tileY;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, randomizeDialog;
static Arg arg[3];
#else
static char titleDsp[TITLELEN] = "";
#endif
static char scoreFileName[FILENAMELEN] = SCOREFILE;
static char fileName[FILENAMELEN];
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char userNameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap hexagonsIcon = None;
static Widget topLevel, hexagons;
static char *progDsp;
static char userNameDsp[USERNAMELEN] = "";

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w, XmNlabelString, xmstr, NULL);
}
#endif

static void
PrintRecords(void)
{
	int i, j, k;

	(void) printf("         HEXAGONS  HALL OF FAME\n\n");
	(void) printf("CORNER   X   Y USER            MOVES\n");
	for (i = 0; i < MAXORIENT; i++)
		for (j = 0; j < MAXTILES - MINTILES + 1; j++)
			for (k = 0; k < MAXTILES - MINTILES + 1; k++) {
				if (hexagonsRecord[i][j][k].score > 0)
					(void) printf("%6d%4d%4d %-12s%9d\n",
						i, j + 1, k + 1,
						hexagonsRecord[i][j][k].name,
						hexagonsRecord[i][j][k].score);
			}
}
#endif

static void
InitRecords(void)
{
	int i, j, k;

	for (i = 0; i < MAXORIENT; i++)
		for (j = 0; j < MAXTILES - MINTILES + 1; j++)
			for (k = 0; k < MAXTILES - MINTILES + 1; k++) {
				hexagonsRecord[i][j][k].score = NEVER;
				(void) strncpy(hexagonsRecord[i][j][k].name,
					NOACCESS, USERNAMELEN);
			}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int i, j, k, n;
	char userName[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USERNAMELEN);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USERNAMELEN);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (i = 0; i < MAXORIENT; i++)
		for (j = 0; j < MAXTILES - MINTILES + 1; j++)
		  for (k = 0; k < MAXTILES - MINTILES + 1; k++) {
		    (void) fscanf(fp, "%d %s\n", &n, userName);
		    if (n <= hexagonsRecord[i][j][k].score ||
		        hexagonsRecord[i][j][k].score <= NEVER) {
		      hexagonsRecord[i][j][k].score = n;
		      (void) strncpy(hexagonsRecord[i][j][k].name,
		        userName, USERNAMELEN);
		    }
		}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i, j, k;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Can not write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILENAMELEN];

		(void) strncpy(lockFile, fileName, FILENAMELEN - 6);
		(void) strcat(lockFile, ".lock");
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (i = 0; i < MAXORIENT; i++) {
			for (j = 0; j < MAXTILES - MINTILES + 1; j++)
			  for (k = 0; k < MAXTILES - MINTILES + 1; k++)
			    (void) fprintf(fp, "%d %s\n",
			    hexagonsRecord[i][j][k].score,
			    hexagonsRecord[i][j][k].name);
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
PrintRecord(int sizeX, int sizeY, Boolean corners)
{
	int i = (corners) ? 1 : 0;
	int j = sizeX - MINTILES, k = sizeY - MINTILES;

	if (sizeX > MAXTILES || sizeY > MAXTILES) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	} else if (hexagonsRecord[i][j][k].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			hexagonsRecord[i][j][k].score,
			hexagonsRecord[i][j][k].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static void
PrintStatus(char *msg, int nMoves
#ifndef HAVE_MOTIF
		, int sizeX, int sizeY
#endif
		)
{
#ifdef HAVE_MOTIF
	PrintState(message, msg);
	(void) sprintf(buff, "%d", nMoves);
	PrintState(moves, buff);
#else
	(void) sprintf(titleDsp, "%s: %dx%d@ (%d/%s) - %s",
		progDsp, sizeX, sizeY, nMoves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent(hexagons), XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
HandleSolved(int counter, int sizeX, int sizeY, int corners)
{
	int i = (corners) ? 1 : 0;
	int j = sizeX - MINTILES, k = sizeY - MINTILES;

	if (sizeX <= MAXTILES && sizeY <= MAXTILES &&
			(counter < hexagonsRecord[i][j][k].score ||
			hexagonsRecord[i][j][k].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		hexagonsRecord[i][j][k].score = counter;
		(void) strncpy(hexagonsRecord[i][j][k].name, userNameDsp,
			USERNAMELEN);
		WriteRecords();
		PrintRecord(sizeX, sizeY, corners);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
HexagonsWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int sizeX, sizeY;
	Boolean corners;
	char *userName, *scoreFile;

#ifdef WINVER
	InitializeHexagons(w, brush);

	sizeX = w->hexagons.sizeX;
	sizeY = w->hexagons.sizeY;
	corners = w->hexagons.corners;
	userName = w->hexagons.userName;
	scoreFile = w->hexagons.scoreFile;
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
#else
	Boolean scoreOnly, versionOnly;

	XtVaGetValues(w,
		XtNuserName, &userName,
		XtNscoreFile, &scoreFile,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNcorners, &corners,
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf(aboutHelp);
		exit(0);
	}
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
	if (scoreOnly) {
		InitRecords();
		ReadRecords();
		PrintRecords();
		exit(0);
	}
#ifdef HAVE_MOTIF
	if (sizeX > MAXTILES)
		XtVaSetValues(tileX, XmNmaximum, sizeX, NULL);
	XmScaleSetValue(tileX, sizeX);
	if (sizeY > MAXTILES)
		XtVaSetValues(tileY, XmNmaximum, sizeY, NULL);
	XmScaleSetValue(tileY, sizeY);
	XmToggleButtonSetState(cornersSwitch, corners, True);
#ifdef ANIMATE
	{
		int delay;

		XtVaGetValues(w, XtNdelay, &delay, NULL);
		XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay - 1);
	}
#endif
#endif
#endif
	SET_STARTED(w, False);
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USERNAMELEN);
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USERNAMELEN);
#else
		char * login = getlogin();

		if (login == NULL) {
			(void) strcpy(userNameDsp, "");
		} else {
			(void) sprintf(userNameDsp, "%s", login);
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(userNameDsp, "%s", "guest");
#endif
	}
	PrintRecord(sizeX, sizeY, corners);
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, sizeX, sizeY
#endif
		);
}

#ifdef WINVER
void
SetHexagons(HexagonsWidget w, int reason)
#else
static void
CallbackHexagons(Widget w, caddr_t clientData,
		hexagonsCallbackStruct * callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int sizeX, sizeY;
	Boolean corners, cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	sizeX = w->hexagons.sizeX;
	sizeY = w->hexagons.sizeY;
	corners = w->hexagons.corners;
	cheat = w->hexagons.cheat;
#else
	XtVaGetValues(w,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNcorners, &corners,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case HEXAGONS_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
			break;
#ifndef WINVER
		case HEXAGONS_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtVaSetValues(hexagons,
				XtNmenu, 5, NULL); /* menu choice */
#endif
			break;
#endif
		case HEXAGONS_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case HEXAGONS_RESTORE:
		case HEXAGONS_RESET:
			movesDsp = 0;
			break;
		case HEXAGONS_BLOCKED:
			(void) strncpy(messageDsp, "Blocked", MESSAGELEN);
			break;
		case HEXAGONS_SPACE:
#if 0
			/* Too annoying */
			(void) sprintf(messageDsp, "%s can not slide",
				(corners) ? "Spaces" : "A space");
#endif
			break;
		case HEXAGONS_IGNORE:
			(void) strncpy(messageDsp, "Randomize to start",
				MESSAGELEN);
			break;
		case HEXAGONS_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case HEXAGONS_SOLVED:
			if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", userNameDsp);
			else if (HandleSolved(movesDsp, sizeX, sizeY, corners))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", userNameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
			SET_STARTED(w, False);
			break;
		case HEXAGONS_RANDOMIZE:
			movesDsp = 0;
			SET_STARTED(w, False);
			break;
		case HEXAGONS_DEC_X:
			movesDsp = 0;
			sizeX--;
			PrintRecord(sizeX, sizeY, corners);
#ifdef WINVER
			w->hexagons.sizeX = sizeX;
#else
			XtVaSetValues(w, XtNsizeX, sizeX, NULL);
#ifdef HAVE_MOTIF
			XmScaleSetValue(tileX, sizeX);
			if (sizeX >= MAXTILES)
				XtVaSetValues(tileX, XmNmaximum, sizeX, NULL);
#endif
#endif
			break;
		case HEXAGONS_INC_X:
			movesDsp = 0;
			sizeX++;
			PrintRecord(sizeX, sizeY, corners);
#ifdef WINVER
			w->hexagons.sizeX = sizeX;
#else
			XtVaSetValues(w, XtNsizeX, sizeX, NULL);
#ifdef HAVE_MOTIF
			if (sizeX > MAXTILES)
				XtVaSetValues(tileX, XmNmaximum, sizeX, NULL);
			XmScaleSetValue(tileX, sizeX);
#endif
#endif
			break;
		case HEXAGONS_DEC_Y:
			movesDsp = 0;
			sizeY--;
			PrintRecord(sizeX, sizeY, corners);
#ifdef WINVER
			w->hexagons.sizeY = sizeY;
#else
			XtVaSetValues(w, XtNsizeY, sizeY, NULL);
#ifdef HAVE_MOTIF
			XmScaleSetValue(tileY, sizeY);
			if (sizeY >= MAXTILES)
				XtVaSetValues(tileY, XmNmaximum, sizeY, NULL);
#endif
#endif
			break;
		case HEXAGONS_INC_Y:
			movesDsp = 0;
			sizeY++;
			PrintRecord(sizeX, sizeY, corners);
#ifdef WINVER
			w->hexagons.sizeY = sizeY;
#else
			XtVaSetValues(w, XtNsizeY, sizeY, NULL);
#ifdef HAVE_MOTIF
			if (sizeY > MAXTILES)
				XtVaSetValues(tileY, XmNmaximum, sizeY, NULL);
			XmScaleSetValue(tileY, sizeY);
#endif
#endif
			break;
		case HEXAGONS_CORNERS:
			movesDsp = 0;
			corners = !corners;
			PrintRecord(sizeX, sizeY, corners);
#ifdef WINVER
			w->hexagons.corners = corners;
#else
			XtVaSetValues(w, XtNcorners, corners, NULL);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(cornersSwitch, corners, True);
#endif
#endif
			break;
		case HEXAGONS_COMPUTED:
			SET_STARTED(w, False);
			break;
		case HEXAGONS_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
#if defined(HAVE_MOTIF) && defined(ANIMATE)
		case HEXAGONS_SPEED:
			{
				int oldDelay, delay;

				XtVaGetValues(hexagons,
					XtNdelay, &delay, NULL);
				oldDelay = delay;
				if (delay > MAXSPEED - MINSPEED)
					delay = MAXSPEED - MINSPEED;
				if (delay < 0)
					delay = 0;
				if (delay != oldDelay) {
					XtVaSetValues(w,
						XtNdelay, delay, NULL);
				}
				XmScaleSetValue(speed,
					MAXSPEED + MINSPEED - delay - 1);
			}
			return;
#endif
	}
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, sizeX, sizeY
#endif
		);
}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.hexagons.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterHexagons(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.hexagons.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeaveHexagons(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyHexagons(brush);
			break;
		case WM_SIZE:
			ResizeHexagons(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposeHexagons(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			RandomizeHexagons(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectHexagons(&widget, LOWORD(lParam), HIWORD(lParam));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleaseHexagons(&widget, LOWORD(lParam), HIWORD(lParam));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
		case WM_MOUSEWHEEL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			{
				int zDelta = ((short) HIWORD(wParam));
				POINT cursor, origin;

				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				if (zDelta > (WHEEL_DELTA >> 1)) {
					MoveHexagons(&widget, TOP,
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
					if (GetKeyState(VK_CONTROL) >> 1) {
						SizeHexagons(&widget);
						(void) InvalidateRect(hWnd, NULL, TRUE);
					}
				} else if (zDelta < -(WHEEL_DELTA >> 1)) {
					MoveHexagons(&widget, BOTTOM,
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
					if (GetKeyState(VK_CONTROL) >> 1) {
						SizeHexagons(&widget);
						(void) InvalidateRect(hWnd, NULL, TRUE);
					}
				}
			}
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#endif
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetHexagons(&widget);
					ResizeHexagons(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WriteHexagons(&widget);
					break;
				case IDM_EXIT:
					DestroyHexagons(brush);
					break;
				case IDM_HIDE:
					HideHexagons(&widget);
					break;
				case IDM_CLEAR:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ClearHexagons(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoHexagons(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					RandomizeHexagons(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolveHexagons(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_CORNERS:
					ModeHexagons(&widget);
					SizeHexagons(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SPEED:
					SpeedHexagons(&widget);
					break;
				case IDM_SLOW:
					SlowHexagons(&widget);
					break;
				case IDM_SOUND:
					SoundHexagons(&widget);
					break;
				case IDM_TR:
				case IDM_RIGHT:
				case IDM_BR:
				case IDM_BL:
				case IDM_LEFT:
				case IDM_TL:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					(void) MoveHexagons(&widget,
						(int) LOWORD(wParam) - IDM_TR, FALSE);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_DECY:
				case IDM_INCX:
				case IDM_INCY:
				case IDM_DECX:
					if (MoveHexagons(&widget, (int) LOWORD(wParam) - IDM_DECY, TRUE)) {
						SizeHexagons(&widget);
						(void) InvalidateRect(hWnd, NULL, TRUE);
					}
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
						"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
						"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
						"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char * programName)
{
	unsigned int i;

	(void) fprintf(stderr, "usage: %s\n", programName);
	for (i = 0; i < strlen(optionsHelp); i++) {
		if (i == 0 || optionsHelp[i - 1] == '\n')
			(void) fprintf(stderr, "\t");
		(void) fprintf(stderr, "%c", (optionsHelp[i]));
	}
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*hexagons.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*hexagons.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*hexagons.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*hexagons.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*hexagons.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*hexagons.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*hexagons.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*hexagons.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*hexagons.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*hexagons.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-tile", (char *) "*hexagons.tileColor", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*hexagons.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-sizex", (char *) "*hexagons.sizeX", XrmoptionSepArg, NULL},
	{(char *) "-sizey", (char *) "*hexagons.sizeY", XrmoptionSepArg, NULL},
	{(char *) "-corners", (char *) "*hexagons.corners", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nocorners", (char *) "*hexagons.corners", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-install", (char *) "*hexagons.install", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noinstall", (char *) "*hexagons.install", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-picture", (char *) "*hexagons.picture", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*hexagons.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*hexagons.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*hexagons.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bumpSound", (char *) "*hexagons.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-base", (char *) "*hexagons.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*hexagons.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*hexagons.font", XrmoptionSepArg, NULL},
	{(char *) "-userName", (char *) "*hexagons.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*hexagons.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*hexagons.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*hexagons.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
CallbackHexagonsRandomize(Widget w, XtPointer clientData,
		XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(hexagons, XtNmenu, 5, NULL); /* menu choice */
	}
}

static void
TileXSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int sizeX = cbs->value, sizeY, oldX;
	Boolean corners;

	XtVaGetValues(hexagons,
		XtNcorners, &corners,
		XtNsizeX, &oldX,
		XtNsizeY, &sizeY, NULL);
	if (oldX == sizeX)
		return;
	XtVaSetValues(hexagons,
		XtNsizeX, sizeX, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(sizeX, sizeY, corners);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
TileYSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int sizeX, sizeY = cbs->value, oldY;
	Boolean corners;

	XtVaGetValues(hexagons,
		XtNcorners, &corners,
		XtNsizeX, &sizeX,
		XtNsizeY, &oldY, NULL);
	if (oldY == sizeY)
		return;
	if (corners && sizeY == 1) {
		XmScaleSetValue(tileY, oldY);
		return;
	}
	XtVaSetValues(hexagons,
		XtNsizeY, sizeY, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(sizeX, sizeY, corners);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
CornersToggle(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct * cbs)
{
	int sizeX, sizeY;
	Boolean corners = cbs->set, oldCorn;

	XtVaGetValues(hexagons,
		XtNcorners, &oldCorn,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY, NULL);
	if (oldCorn == corners)
		return;
	if (corners && sizeY == 1) {
		XmToggleButtonSetState(cornersSwitch, !corners, True);
		return;
	}
	XtVaSetValues(hexagons,
		XtNcorners, corners, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(sizeX, sizeY, corners);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

#ifdef ANIMATE
static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int delay = MAXSPEED + MINSPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(hexagons,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(hexagons, XtNdelay, delay, NULL);
	}
}
#endif

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtVaSetValues(hexagons, XtNmenu, val, NULL);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;
	XtVaSetValues(hexagons, XtNmenu, val + 3, NULL); /* GWQ */
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, tilesRowCol, speedRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, randomizeString, solveString;
	XmString modeString;
	XmString speedString, slowString, soundString;
#endif

	progDsp = argv[0];

	topLevel = XtInitialize(argv[0], "Hexagons",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) hexagons_xpm, &hexagonsIcon, NULL,
			&xpmAttributes);
	}
	if (hexagonsIcon == (Pixmap) NULL)
#endif
		hexagonsIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) hexagons_bits,
			hexagons_width, hexagons_height);
#ifdef HAVE_MOTIF
	XtVaSetValues(topLevel,
		XtNiconPixmap, hexagonsIcon,
		XmNkeyboardFocusPolicy, XmPOINTER, NULL); /* not XmEXPLICIT */
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "(Solve)");
	modeString = XmStringCreateSimple((char *) "Mode");
	speedString = XmStringCreateSimple((char *) ">Speed");
	slowString = XmStringCreateSimple((char *) "<Slow");
	soundString = XmStringCreateSimple((char *) "@Sound");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, modeString, 'M', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(modeString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackHexagonsRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move tile", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	tilesRowCol = XtVaCreateManagedWidget("tilesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	tileX = XtVaCreateManagedWidget("tileX",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size X", 7,
		XmNminimum, MINTILES,
		XmNmaximum, MAXTILES,
		XmNvalue, DEFAULTTILESX,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(tileX, XmNvalueChangedCallback,
		(XtCallbackProc) TileXSlider, (XtPointer) NULL);
	tileY = XtVaCreateManagedWidget("tileY",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size Y", 7,
		XmNminimum, MINTILES,
		XmNmaximum, MAXTILES,
		XmNvalue, DEFAULTTILESY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(tileY, XmNvalueChangedCallback,
		(XtCallbackProc) TileYSlider, (XtPointer) NULL);
	cornersSwitch = XtVaCreateManagedWidget("Corners",
		xmToggleButtonWidgetClass, tilesRowCol,
		XmNset, DEFAULTCORNERS, NULL);
	XtAddCallback(cornersSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) CornersToggle, (XtPointer) NULL);
#ifdef ANIMATE
	speedRowCol = XtVaCreateManagedWidget("speedRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	speed = XtVaCreateManagedWidget("Animation Speed",
		xmScaleWidgetClass, speedRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - DEFAULTDELAY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
#endif
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Hexagons! (use mouse or keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	hexagons = XtCreateManagedWidget("hexagons",
		hexagonsWidgetClass, mainPanel, NULL, 0);
#else
	XtVaSetValues(topLevel,
		XtNiconPixmap, hexagonsIcon,
		XtNinput, True, NULL);
	hexagons = XtCreateManagedWidget("hexagons",
		hexagonsWidgetClass, topLevel, NULL, 0);
#endif
	XtAddCallback(hexagons, XtNselectCallback,
		(XtCallbackProc) CallbackHexagons, (XtPointer) NULL);
	Initialize(hexagons);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(hexagons), (unsigned int) AnyButton, AnyModifier,
		XtWindow(hexagons), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(hexagons),
		XCreateFontCursor(XtDisplay(hexagons), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
