// file      : tests/cxx/parser/validation/built-in/string/driver.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2006-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

// Test the built-in string & friends types validation.
//
#include <string>
#include <cassert>

#include <xsde/config.h>

// Let the runtime header sort out which version (stl/no-stl) to
// include.
//
#include <xsde/cxx/parser/validating/xml-schema-pimpl.hxx>

using namespace xsde::cxx;
using namespace xsde::cxx::parser::validating;

#ifdef XSDE_STL
bool
compare (const string_sequence& x, const string_sequence& y)
{
  return x == y;
}
#else
bool
compare (const string_sequence* x, const string_sequence& y)
{
  bool r = *x == y;
  delete x;
  return r;
}
#endif

int
main (int argc, char* argv[])
{
  // We are going to leak a bit of memory in the no-STL case.
  //
  using std::string;

  // Good.
  //

  // string
  //
  {
    string_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" aaa ");
    p._characters ("bbb");
    p._characters (" ");
    p._post ();
    assert (!p._error_type () &&
            p.post_string () == string (" \n\t aaa bbb "));
  }

  // normalized_string
  //
  {
    normalized_string_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" aaa \n\t ");
    p._characters (" bbb");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () &&
            p.post_normalized_string () == string ("    aaa     bbb  "));
  }

  // token
  //
  {
    token_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" aaa \n\t ");
    p._characters (" bbb \n\t");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () &&
            p.post_token () == string ("aaa bbb"));
  }

  // name
  //
  {
    name_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" a:b-c_d123 ");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () &&
            p.post_name () == string ("a:b-c_d123"));
  }

  {
    name_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" _12 ");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () &&
            p.post_name () == string ("_12"));
  }

  {
    name_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" :12 ");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () &&
            p.post_name () == string (":12"));
  }

  // nmtoken
  //
  {
    nmtoken_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" 123a:b-c_d123 ");
    p._characters (" \n\t");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () &&
            p.post_nmtoken () == string ("123a:b-c_d123"));
  }

  // nmtokens
  //
  {
    string_sequence s;
    s.push_back ("123");
    s.push_back ("abc");

    nmtokens_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" 123 ");
    p._characters (" \n\t abc ");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () && compare (p.post_nmtokens (), s));
  }

  // ncname
  //
  {
    ncname_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" a.b-c_d123 ");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () &&
            p.post_ncname () == string ("a.b-c_d123"));
  }

  // id
  //
  {
    id_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" a.b-c_d123 ");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () &&
            p.post_id () == string ("a.b-c_d123"));
  }

  // idref
  //
  {
    idref_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" a.b-c_d123 ");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () &&
            p.post_idref () == string ("a.b-c_d123"));
  }

  // idrefs
  //
  {
    string_sequence s;
    s.push_back ("a123");
    s.push_back ("abc");

    idrefs_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" \n\t");
    p._characters (" a123 ");
    p._characters (" \n\t abc ");
    p._characters ("  ");
    p._post ();
    assert (!p._error_type () && compare (p.post_idrefs (), s));
  }

  // language
  //
  {
    language_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" x ");
    p._post ();
    assert (!p._error_type () && p.post_language () == string ("x"));
  }

  {
    language_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" en ");
    p._post ();
    assert (!p._error_type () && p.post_language () == string ("en"));
  }

  {
    language_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" en");
    p._characters ("-us ");
    p._post ();
    assert (!p._error_type () && p.post_language () == string ("en-us"));
  }

  {
    language_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("one-two-three-four44-seven77-eight888");
    p._post ();
    assert (!p._error_type () &&
            p.post_language () ==
            string ("one-two-three-four44-seven77-eight888"));
  }

  // Bad
  //

  // name
  //
  {
    name_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("");
    p._post ();
    assert (p._schema_error ());
  }

  {
    name_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (".a");
    p._post ();
    assert (p._schema_error ());
  }

  {
    name_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("-a");
    p._post ();
    assert (p._schema_error ());
  }

  {
    name_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("1a");
    p._post ();
    assert (p._schema_error ());
  }

  {
    name_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a,b");
    p._post ();
    assert (p._schema_error ());
  }

  {
    name_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a b");
    p._post ();
    assert (p._schema_error ());
  }

  {
    name_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a<b");
    p._post ();
    assert (p._schema_error ());
  }

  // nmtoken
  //
  {
    nmtoken_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("");
    p._post ();
    assert (p._schema_error ());
  }

  {
    nmtoken_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a,b");
    p._post ();
    assert (p._schema_error ());
  }

  {
    nmtoken_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a b");
    p._post ();
    assert (p._schema_error ());
  }

  {
    nmtoken_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a<b");
    p._post ();
    assert (p._schema_error ());
  }

  // nmtokens
  //
  {
    nmtokens_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" ");
    p._characters (" \t\n  ");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_nmtokens_value);
  }

  {
    nmtokens_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("ab a,b");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_nmtoken_value);
  }

  // ncname
  //
  {
    ncname_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("");
    p._post ();
    assert (p._schema_error ());
  }

  {
    ncname_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (".a");
    p._post ();
    assert (p._schema_error ());
  }

  {
    ncname_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("-a");
    p._post ();
    assert (p._schema_error ());
  }

  {
    ncname_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (":a");
    p._post ();
    assert (p._schema_error ());
  }

  {
    ncname_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("1a");
    p._post ();
    assert (p._schema_error ());
  }

  {
    ncname_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a:b");
    p._post ();
    assert (p._schema_error ());
  }

  {
    ncname_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a,b");
    p._post ();
    assert (p._schema_error ());
  }

  {
    ncname_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a b");
    p._post ();
    assert (p._schema_error ());
  }

  {
    ncname_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a<b");
    p._post ();
    assert (p._schema_error ());
  }

  // id
  //
  {
    id_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a b");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_id_value);
  }

  // idref
  //
  {
    idref_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a b");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_idref_value);
  }

  // idrefs
  //
  {
    idrefs_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("  ");
    p._characters (" \t\n ");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_idrefs_value);
  }

  {
    idrefs_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("ab a<b");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_idref_value);
  }

  // language
  //
  {
    language_pimpl p;
    p.pre ();
    p._pre ();
    p._characters (" ");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_language_value);
  }

  {
    language_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("en-");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_language_value);
  }

  {
    language_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("a1");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_language_value);
  }

  {
    language_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("en+us");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_language_value);
  }

  {
    language_pimpl p;
    p.pre ();
    p._pre ();
    p._characters ("en-nine99999");
    p._post ();
    assert (p._schema_error () == schema_error::invalid_language_value);
  }
}
