/*
 * Copyright 2007  Luc Verhaegen <lverhaegen@novell.com>
 * Copyright 2007  Matthias Hopf <mhopf@novell.com>
 * Copyright 2007  Egbert Eich   <eich@novell.com>
 * Copyright 2007  Advanced Micro Devices, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER(S) OR AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

/*
 * Cursor handling.
 *
 * Only supports ARGB cursors.
 * Bitmap cursors are converted to ARGB internally.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* All drivers should typically include these */
#include "xf86.h"

#include "xf86Cursor.h"
#include "cursorstr.h"
#include "servermd.h"

/* Driver specific headers */
#include "rhd.h"
#include "rhd_cursor.h"
#include "rhd_crtc.h"
#include "rhd_regs.h"

/* System headers */
#include <assert.h>

/* Internal interface to RealizeCursor - we need width/height */
struct rhd_Cursor_Bits {
    int width, height;
    /* Cursor source bitmap follows */
    /* Cursor mask bitmap follows */
} ;

/*
 * Bit-banging ONLY
 */

/* RadeonHD registers are double buffered, exchange only during vertical blank.
 * By locking registers, a set of registers is updated atomically.
 * Probably not necessary for cursors, but trivial and fast. */
static void
lockCursor(struct rhdCursor *Cursor, Bool Lock)
{
    /* Double Buffering: Set _UPDATE_LOCK bit */
    if (Lock)
	RHDRegMask(Cursor, Cursor->RegOffset + D1CUR_UPDATE,
		   0x00010000, 0x00010000);
    else
	RHDRegMask(Cursor, Cursor->RegOffset + D1CUR_UPDATE,
		   0x00000000, 0x00010000);
}

/* RadeonHD has hardware support for hotspots, but doesn't allow negative
 * cursor coordinates. Emulated in rhdShowCursor.
 * Coordinates are absolute, not relative to visible fb portion. */
static void
setCursorPos(struct rhdCursor *Cursor, CARD32 x, CARD32 y,
	     CARD32 hotx, CARD32 hoty)
{
    /* R600 only has 13 bits, but well... */
    assert (x >= 0 && x < 0x10000);
    assert (y >= 0 && y < 0x10000);
    RHDRegWrite(Cursor, Cursor->RegOffset + D1CUR_POSITION, x << 16 | y);
    /* Note: unknown whether hotspot may be ouside width/height */
    assert (hotx >= 0 && hotx < MAX_CURSOR_WIDTH);
    assert (hoty >= 0 && hoty < MAX_CURSOR_HEIGHT);
    RHDRegWrite(Cursor, Cursor->RegOffset + D1CUR_HOT_SPOT, hotx << 16 | hoty);
}

static void
enableCursor(struct rhdCursor *Cursor, Bool Enable)
{
    if (Enable)
	/* pre-multiplied ARGB, Enable */
	RHDRegWrite(Cursor, Cursor->RegOffset + D1CUR_CONTROL, 0x00000201);
    else
	RHDRegWrite(Cursor, Cursor->RegOffset + D1CUR_CONTROL, 0);
}

/* Activate already uploaded cursor image. */
static void
setCursorImage(struct rhdCursor *Cursor)
{
    RHDPtr rhdPtr = RHDPTR(xf86Screens[Cursor->scrnIndex]);

    RHDRegWrite(Cursor, Cursor->RegOffset + D1CUR_SURFACE_ADDRESS,
		rhdPtr->FbIntAddress + Cursor->Base);
    assert ((Cursor->Width > 0) && (Cursor->Width  <= MAX_CURSOR_WIDTH));
    assert ((Cursor->Height > 0) && (Cursor->Height <= MAX_CURSOR_HEIGHT));
    RHDRegWrite(Cursor, Cursor->RegOffset + D1CUR_SIZE,
		(Cursor->Width - 1) << 16 | (Cursor->Height - 1));
}

/* Upload image.
 * Hardware only supports 64-wide cursor images.
 * img: (MAX_CURSOR_WIDTH * height) ARGB tuples */
static void
uploadCursorImage(struct rhdCursor *Cursor, CARD32 *img)
{
    RHDPtr rhdPtr = RHDPTR(xf86Screens[Cursor->scrnIndex]);

    memcpy(((char *) rhdPtr->FbBase + Cursor->Base), img,
	   MAX_CURSOR_WIDTH * Cursor->Height * 4);
}

/*
 *
 */
static void
rhdCursorSave(struct rhdCursor *Cursor)
{
    Cursor->StorePosition = RHDRegRead(Cursor, Cursor->RegOffset + D1CUR_POSITION);
    Cursor->StoreHotSpot = RHDRegRead(Cursor, Cursor->RegOffset + D1CUR_HOT_SPOT);
    Cursor->StoreAddress = RHDRegRead(Cursor, Cursor->RegOffset + D1CUR_SURFACE_ADDRESS);
    Cursor->StoreSize = RHDRegRead(Cursor, Cursor->RegOffset + D1CUR_SIZE);

    /* TODO: Store pixmap */
    Cursor->Stored = FALSE;
}

/*
 *
 */
static void
rhdCursorRestore(struct rhdCursor *Cursor)
{
    Cursor->Lock(Cursor, TRUE);

    RHDRegWrite(Cursor, Cursor->RegOffset + D1CUR_POSITION, Cursor->StorePosition);
    RHDRegWrite(Cursor, Cursor->RegOffset + D1CUR_HOT_SPOT, Cursor->StoreHotSpot);
    RHDRegWrite(Cursor, Cursor->RegOffset + D1CUR_SURFACE_ADDRESS, Cursor->StoreAddress);
    RHDRegWrite(Cursor, Cursor->RegOffset + D1CUR_SIZE, Cursor->StoreSize);

    /* TODO: Restore pixmap */

    Cursor->Lock(Cursor, FALSE);
}

/*
 *
 */
void
RHDCursorsInit(RHDPtr rhdPtr)
{
    struct rhdCursor *Cursor;
    int size = RHD_FB_CHUNK(MAX_CURSOR_WIDTH * MAX_CURSOR_HEIGHT * 4);

    RHDFUNC(rhdPtr);

    /* First Cursor */
    Cursor = xnfcalloc(sizeof(struct rhdCursor), 1);

    Cursor->scrnIndex = rhdPtr->scrnIndex;

    Cursor->RegOffset = 0;

    /* Don't set width and height yet */

    /* grab our cursor FB */
    /* I love a bit of a challenge, so move start instead of end */
    Cursor->Base = rhdPtr->FbFreeStart;
    rhdPtr->FbFreeStart += size;
    rhdPtr->FbFreeSize -= size;

    Cursor->Lock = lockCursor;
    Cursor->Enable = enableCursor;
    Cursor->Position = setCursorPos;
    Cursor->Set = setCursorImage;
    Cursor->Load = uploadCursorImage;
    Cursor->Save = rhdCursorSave;
    Cursor->Restore = rhdCursorRestore;

    rhdPtr->Crtc[0]->Cursor = Cursor; /* HW is fixed anyway */

    /* Second Cursor */
    Cursor = xnfcalloc(sizeof(struct rhdCursor), 1);

    Cursor->scrnIndex = rhdPtr->scrnIndex;

    Cursor->RegOffset = 0x800;

    Cursor->Base = rhdPtr->FbFreeStart;
    rhdPtr->FbFreeStart += size;
    rhdPtr->FbFreeSize -= size;

    Cursor->Lock = lockCursor;
    Cursor->Enable = enableCursor;
    Cursor->Position = setCursorPos;
    Cursor->Set = setCursorImage;
    Cursor->Load = uploadCursorImage;
    Cursor->Save = rhdCursorSave;
    Cursor->Restore = rhdCursorRestore;

    rhdPtr->Crtc[1]->Cursor = Cursor;
}

/*
 *
 */
void
RHDCursorsDestroy(RHDPtr rhdPtr)
{
    RHDFUNC(rhdPtr);

    if (rhdPtr->Crtc[0])
	    xfree(rhdPtr->Crtc[0]->Cursor);
    if (rhdPtr->Crtc[1])
	    xfree(rhdPtr->Crtc[1]->Cursor);
}

/*
 * Helper functions
 */

/* Convert bitmaps as defined in rhd_CursorBits to ARGB tupels */
static void
convertBitsToARGB(struct rhd_Cursor_Bits *bits, CARD32 *dest,
		  CARD32 color0, CARD32 color1)
{
    CARD8 *src      = (CARD8 *) &bits[1];
    int    srcPitch = BitmapBytePad(bits->width);
    CARD8 *mask     = src + srcPitch * bits->height;
    int x, y;

    for (y = 0; y < bits->height; y++) {
	CARD8  *s = src, *m = mask;
	CARD32 *d = dest;
	for (x = 0; x < bits->width; x++) {
	    if (m[x/8] & (1<<(x&7))) {
		if (s[x/8] & (1<<(x&7)))
		    *d++ = color1;
		else
		    *d++ = color0;
	    } else
		*d++ = 0;
	}
	src  += srcPitch;
	mask += srcPitch;
	dest += MAX_CURSOR_WIDTH;
    }
}


/*
 *
 */
static void
rhdCursorSet(struct rhdCrtc *Crtc)
{
    struct rhdCursor *Cursor = Crtc->Cursor;

    /* Hardware doesn't allow negative cursor pos. Use hardware
     * hotspot support for that. Cannot exceed width, but cursor is
     * not visible in this case. */

    if (Cursor->X >= Crtc->X - Cursor->Width  &&
	Cursor->X <  Crtc->X + Crtc->Width    &&
	Cursor->Y >= Crtc->Y - Cursor->Height &&
	Cursor->Y <  Crtc->Y + Crtc->Height) {
	int X, Y, HotX, HotY;

	X = Cursor->X >= 0 ? Cursor->X : 0;
	Y = Cursor->Y >= 0 ? Cursor->Y : 0;
	HotX = Cursor->X >= 0 ? 0 : -Cursor->X;
	HotY = Cursor->Y >= 0 ? 0 : -Cursor->Y;

	Cursor->Lock(Cursor, TRUE);
	Cursor->Enable(Cursor, TRUE);
	Cursor->Position(Cursor, X, Y, HotX, HotY);
	Cursor->Lock(Cursor, FALSE);
    } else {
	Cursor->Lock(Cursor, TRUE);
	Cursor->Enable(Cursor, FALSE);
	Cursor->Lock(Cursor, FALSE);
    }
}

/*
 * Interface
 */

void
rhdShowCursor(ScrnInfoPtr pScrn)
{
    RHDPtr rhdPtr = RHDPTR(pScrn);
    int i;

    for (i = 0; i < 2; i++) {
	struct rhdCrtc *Crtc = rhdPtr->Crtc[i];

	if (Crtc->Active && Crtc->scrnIndex == pScrn->scrnIndex)
	    rhdCursorSet(Crtc);
    }
}

void
rhdHideCursor(ScrnInfoPtr pScrn)
{
    RHDPtr rhdPtr = RHDPTR(pScrn);
    int i;

    for (i = 0; i < 2; i++) {
	struct rhdCrtc *Crtc = rhdPtr->Crtc[i];

	if (Crtc->Active && Crtc->scrnIndex == pScrn->scrnIndex) {
	    struct rhdCursor *Cursor = Crtc->Cursor;

	    Cursor->Lock(Cursor, TRUE);
	    Cursor->Enable(Cursor, FALSE);
	    Cursor->Lock(Cursor, FALSE);
	}
    }
}

static void
rhdSetCursorPosition(ScrnInfoPtr pScrn, int x, int y)
{
    RHDPtr rhdPtr = RHDPTR(pScrn);
    int i;

    for (i = 0; i < 2; i++) {
	struct rhdCrtc *Crtc = rhdPtr->Crtc[i];

	if (Crtc->Active && Crtc->scrnIndex == pScrn->scrnIndex) {

	    /* Given cursor pos is always relative to frame - make absolute here */
	    Crtc->Cursor->X = x + pScrn->frameX0;
	    Crtc->Cursor->Y = y + pScrn->frameY0;

	    rhdCursorSet(Crtc);
	}
    }
}

static void
rhdSetCursorColors(ScrnInfoPtr pScrn, int bg, int fg)
{
    RHDPtr rhdPtr = RHDPTR(pScrn);
    int i;

    rhdPtr->CursorColor0 = bg | 0xff000000;
    rhdPtr->CursorColor1 = fg | 0xff000000;

    if (!rhdPtr->CursorBits)
	return;

    /* Re-convert cursor bits if color changed */
    convertBitsToARGB(rhdPtr->CursorBits,   rhdPtr->CursorImage,
		      rhdPtr->CursorColor0, rhdPtr->CursorColor1);

    for (i = 0; i < 2; i++) {
	struct rhdCrtc *Crtc = rhdPtr->Crtc[i];

	if (Crtc->Active && Crtc->scrnIndex == pScrn->scrnIndex) {
	    struct rhdCursor *Cursor = Crtc->Cursor;

	    Cursor->Lock(Cursor, TRUE);
	    Cursor->Load(Cursor, rhdPtr->CursorImage);
	    Cursor->Set(Cursor);
	    Cursor->Lock(Cursor, FALSE);
	}
    }
}


static void
rhdLoadCursorImage(ScrnInfoPtr pScrn, unsigned char *src)
{
    RHDPtr rhdPtr = RHDPTR(pScrn);
    struct rhd_Cursor_Bits *bits = (struct rhd_Cursor_Bits *) src;
    int i;

    rhdPtr->CursorBits   = bits;
    convertBitsToARGB(bits, rhdPtr->CursorImage,
		      rhdPtr->CursorColor0, rhdPtr->CursorColor1);

    for (i = 0; i < 2; i++) {
	struct rhdCrtc *Crtc = rhdPtr->Crtc[i];

	if (Crtc->Active && Crtc->scrnIndex == pScrn->scrnIndex) {
	    struct rhdCursor *Cursor = Crtc->Cursor;

	    Cursor->Width  = bits->width;
	    Cursor->Height = bits->height;

	    Cursor->Lock(Cursor, TRUE);
	    Cursor->Load(Cursor, rhdPtr->CursorImage);
	    Cursor->Set(Cursor);
	    Cursor->Lock(Cursor, FALSE);
	}
    }
}

static Bool
rhdUseHWCursorARGB(ScreenPtr pScreen, CursorPtr cur)
{
    /* Inconsistency in interface: UseHWCursor == NULL is trivial accept,
     * UseHWCursorARGB == NULL is trivial reject. */
    return TRUE;
}

static void
rhdLoadCursorARGB(ScrnInfoPtr pScrn, CursorPtr cur)
{
    RHDPtr rhdPtr = RHDPTR(pScrn);
    int i;

    rhdPtr->CursorBits   = NULL;

    /* Hardware only supports 64-wide cursor images. */
    for (i = 0; i < cur->bits->height; i++)
	memcpy(rhdPtr->CursorImage + MAX_CURSOR_WIDTH*i,
	       cur->bits->argb + cur->bits->width*i,
	       cur->bits->width*4);

    for (i = 0; i < 2; i++) {
	struct rhdCrtc *Crtc = rhdPtr->Crtc[i];

	if (Crtc->Active && Crtc->scrnIndex == pScrn->scrnIndex) {
	    struct rhdCursor *Cursor = Crtc->Cursor;

	    Cursor->Width = cur->bits->width;
	    Cursor->Height = cur->bits->height;

	    Cursor->Lock(Cursor, TRUE);
	    Cursor->Load(Cursor, rhdPtr->CursorImage);
	    Cursor->Set(Cursor);
	    Cursor->Lock(Cursor, FALSE);
	}
    }
}

/* Save cursor parameters for later re-use */
static unsigned char*
rhdRealizeCursor(xf86CursorInfoPtr infoPtr, CursorPtr cur)
{
    int    len = BitmapBytePad(cur->bits->width) * cur->bits->height;
    struct rhd_Cursor_Bits *bits = xalloc(sizeof(struct rhd_Cursor_Bits)
					  + 2*len);
    char  *bitmap = (char *) &bits[1];

    bits->width  = cur->bits->width;
    bits->height = cur->bits->height;
    memcpy (bitmap,     cur->bits->source, len);
    memcpy (bitmap+len, cur->bits->mask,   len);

    return (unsigned char *) bits;
}

/*
 * Init
 */

Bool
RHDxf86InitCursor(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    RHDPtr rhdPtr = RHDPTR(pScrn);
    xf86CursorInfoPtr infoPtr;

    infoPtr = xf86CreateCursorInfoRec();
    if (!infoPtr)
	return FALSE;

    infoPtr->MaxWidth  = MAX_CURSOR_WIDTH;
    infoPtr->MaxHeight = MAX_CURSOR_HEIGHT;
    infoPtr->Flags     = HARDWARE_CURSOR_TRUECOLOR_AT_8BPP |
			 HARDWARE_CURSOR_UPDATE_UNHIDDEN |
			 HARDWARE_CURSOR_AND_SOURCE_WITH_MASK
#if defined (ARGB_CURSOR) && defined (HARDWARE_CURSOR_ARGB)
			 | HARDWARE_CURSOR_ARGB
#endif
			 ;

    infoPtr->SetCursorColors   = rhdSetCursorColors;
    infoPtr->SetCursorPosition = rhdSetCursorPosition;
    infoPtr->LoadCursorImage   = rhdLoadCursorImage;
    infoPtr->HideCursor        = rhdHideCursor;
    infoPtr->ShowCursor        = rhdShowCursor;
    infoPtr->UseHWCursor       = NULL;
#ifdef ARGB_CURSOR
    infoPtr->UseHWCursorARGB   = rhdUseHWCursorARGB; /* may not be NULL */
    infoPtr->LoadCursorARGB    = rhdLoadCursorARGB;
#endif
    infoPtr->RealizeCursor     = rhdRealizeCursor;

    if (!xf86InitCursor(pScreen, infoPtr)) {
        xf86DestroyCursorInfoRec(infoPtr);
        return FALSE;
    }
    rhdPtr->CursorInfo   = infoPtr;
    rhdPtr->CursorImage  = xalloc(MAX_CURSOR_WIDTH * MAX_CURSOR_HEIGHT * 4);
    xf86DrvMsg(pScrn->scrnIndex,X_INFO,"Using HW cursor\n");

    return TRUE;
}

