/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>

#include <qevent.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qmessagebox.h>
#include <qpushbutton.h>

#include "GlobalIcon.h"
#include "HistoryDialog.h"

HistoryDialog::HistoryDialog(QWidget* parent, const char* name)
: HistoryDialogData(parent,name), file(0)
{
    setIcon(*mainIcon);
 
    list.setAutoDelete(true);
    list.clear();
    maxItems = 200;
    
    QString tmp;
    tmp.sprintf("%d",maxItems);
    sizeLineEdit->setText(tmp);
}

HistoryDialog::~HistoryDialog()
{
    if ( file!=0 )
        delete file;
}

bool HistoryDialog::load(const char* fileName)
{
    if ( file!=0 )
        delete file;
    file = new HistoryFile(fileName);
    bool ret = file->load(*this);
    QString tmp;
    tmp.sprintf("%d",maxItems);
    sizeLineEdit->setText(tmp);
    return ret;
}

bool HistoryDialog::save()
{
    if ( file!=0 )
        return file->save(*this);
    else
        return false;
}

// Store most recent entry at beginning of list.
void HistoryDialog::add(const char* fileName, const char* info)
{
    if (list.count() >= maxItems)
    {
        list.removeLast();
        // Remove oldest entry from bottom.
        historyListBox->removeItem( historyListBox->count()-1 );
    }
    list.prepend( new HistoryItem(fileName,info) );
    // Show most recently played entry at top.
    historyListBox->insertItem(info,0);
}

void HistoryDialog::historyItemHighlighted(int index)
{
    HistoryItem* pI = list.at( index );
    if ( !pI )
        return;
    emit playHistoryItem( pI->fileName() );
}

void HistoryDialog::prevButtonClicked()
{
    // Older => end of list.
    int count = historyListBox->count();
    if (count <= 0)
        return;
    int index = historyListBox->currentItem();
    index++;
    if (index <= (count-1))  // last is count-1 
        historyListBox->setCurrentItem(index);
}

void HistoryDialog::nextButtonClicked()
{
    // Newer => beginning of list.
    if (historyListBox->count() <= 0)
        return;
    int index = historyListBox->currentItem();
    index--;
    if (index >= 0)  // first is 0
        historyListBox->setCurrentItem(index);
}

void HistoryDialog::deleteButtonClicked()
{
    int index = historyListBox->currentItem();
    if (index < 0)
        return;
    list.remove(index);
    historyListBox->removeItem(index);
}

void HistoryDialog::clearButtonClicked()
{
    if ( list.isEmpty() )
        return;
    QMessageBox mb( "SIDPLAY",
                    "Do you really want to clear the history?",
                    QMessageBox::Information,
                    QMessageBox::Yes,
                    QMessageBox::No | QMessageBox::Default | QMessageBox::Escape,
                    0);
    mb.setButtonText( QMessageBox::Yes, "Yes" );
    mb.setButtonText( QMessageBox::No, "No" );
    switch( mb.exec() ) 
    {
     case QMessageBox::Yes:
        {
            historyListBox->clear();
            list.clear();
            break;
        }
     case QMessageBox::No:
        break;
    }
}

void HistoryDialog::historySizeChanged()
{
    maxItems = sizeLineEdit->text().toInt();
    // Select a good current item below the limit,
    // so highlighting doesn't trigger playback for each
    // entry to be deleted.
    int index = historyListBox->currentItem();
    if (index >= maxItems)
    {
        if ( (index=maxItems-1)<0 )
            index = 0;
        historyListBox->setCurrentItem(index);
    }
    // Remove all entries above maxItems.
    while (list.count() > maxItems)
    {
       list.removeLast();
       historyListBox->removeItem( historyListBox->count()-1 );
    }
}
