/**
 * A client-side 802.1x implementation 
 *
 * This code is released under both the GPL version 2 and BSD licenses.
 * Either license may be used.  The respective licenses are found below.
 *
 * Copyright (C) 2002 Bryan D. Payne & Nick L. Petroni Jr.
 * All Rights Reserved
 *
 * --- GPL Version 2 License ---
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * --- BSD License ---
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the University of
 *       Maryland at College Park and its contributors.
 *  - Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*******************************************************************
 * File: cardif_linux_rtnetlink.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * Comments :
 *     Large portions of the rtnetlink code is derived from the 
 *     wpa_supplicant code written by Jouni Malinen.  (In some cases,
 *     it is copied verbatim.)  wpa_supplicant can be found at 
 *     http://hostap.epitest.fi.
 *
 * $Id: cardif_linux_rtnetlink.c,v 1.11 2004/08/20 04:42:28 chessing Exp $
 * $Date: 2004/08/20 04:42:28 $
 * $Log: cardif_linux_rtnetlink.c,v $
 * Revision 1.11  2004/08/20 04:42:28  chessing
 * Set all of the new scanning and WPA code to be disabled by default, so that a 1.0.1 release can be made.  Added additional error message output to AKA code.  Fixed a few memory leaks in the AKA code.
 *
 * Revision 1.10  2004/08/19 02:28:07  chessing
 * First piece of WPA patch.  (The patch is growing fast, and this commit is to save what is done so far.)
 *
 * Revision 1.9  2004/08/15 04:29:35  chessing
 *
 * Completed support for scanning and joining wireless networks.  If your interface isn't configured, we will scan to gather information about all known wireless networks.  If a network in the list has a configuration, we will join it.  (Including turning on needed encryption.)  There are still a few rough edges that need to be hammered out.  But the overall user experience should be improved.
 *
 * Revision 1.8  2004/08/12 01:40:06  chessing
 *
 * Added "association" option to the global configuration file sections.  It can be set to either auto, or manual.  Setting "association" to auto means that XSupplicant will attempt to associate to a network with the lowest value in the "priority" setting.  The default for "association" is auto.  The other option added is "priority" which is a numeric value from 1..256.  A value of 1 is the highest priority, and will be the first network chosen.  (Assuming the network can be found in a scan.)  The default value for "priority" is 0, which means that network will be checked last.   --  Although the options are now in the config file, the actual functionality for auto association/priority is not yet complete.  (So, basically these options don't do anything yet. ;)
 *
 * Revision 1.7  2004/08/11 04:38:12  chessing
 *
 * Fixes to the new debug code.  (DEBUG_NORMAL stuff wasn't be displayed when it should have.)
 *
 * Revision 1.6  2004/08/11 04:01:18  chessing
 *
 * The -d option now supports letter designations for each debug level, in addition to 0..7.  (However, both letters and numbers can't be used at the same time!)  Also changed some formatting for the SNMP scoreboard.
 *
 * Revision 1.5  2004/08/10 01:59:26  chessing
 *
 * Added support for the SNMP supplicant counters defined in the IEEE 802.1X-2001 document.
 *
 * Revision 1.4  2004/08/05 23:56:21  chessing
 *
 * Added basic support for scanning for broadcast SSIDs.  This is another step closer to WPA/11i support. ;)
 *
 * Revision 1.3  2004/07/26 03:08:58  chessing
 *
 * Fixed a few bugs in the new hot plug code.  Hot unplugging (and replugging) now works.
 *
 * Revision 1.2  2004/07/25 19:36:41  chessing
 * Fixed a few more logical/bitwise nots.  Added rtnetlink support for hot-plugging interfaces.  (Hot unplugging is coming soon. ;)
 *
 * Revision 1.1  2004/07/23 04:05:50  chessing
 * Fixed a segfault problem.  Started to add rtnetlink support.
 *
 *
 *******************************************************************/

#include <unistd.h>
#include <strings.h>
#include <string.h>
#include <asm/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <linux/wireless.h>
#include <linux/if_packet.h>

#include "../../profile.h"
#include "../../config.h"
#include "../../xsup_debug.h"
#include "../../xsup_err.h"
#include "cardif_linux.h"
#include "cardif_linux_rtnetlink.h"
#include "cardif_linux_ssids.h"
#include "../cardif.h"
#include "../../wpa.h"
#include "../../eap_types/mschapv2/mschapv2.h"

// Define this, so the compiler doesn't complain.
extern char *if_indextoname(unsigned int, char *);

#define INT_DEL    0
#define INT_NEW    1

static int rtnl_sock=-1;
static struct sockaddr_nl rtnl_data;

/********************************************************
 *
 * Do whatever is needed to establish a netlink socket so that we can
 * catch events, and take action.
 *
 ********************************************************/
void cardif_linux_rtnetlink_init()
{
  debug_printf(DEBUG_INT, "Called cardif_linux_rtnetlink_init()!\n");

  rtnl_sock = socket(PF_NETLINK, SOCK_RAW, NETLINK_ROUTE);
  
  if (rtnl_sock < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't establish an rtnetlink socket!  Some functionality won't be available!\n");
      return;
    }

  bzero((char *)&rtnl_data, sizeof(rtnl_data));

  rtnl_data.nl_family = AF_NETLINK;
  rtnl_data.nl_groups = RTMGRP_LINK;

  if (bind(rtnl_sock, (struct sockaddr *)&rtnl_data, sizeof(rtnl_data)) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't bind netlink socket!  Some functionality won't be available!\n");
      close(rtnl_sock);
      return;
    }
}

/********************************************************
 *
 * Do whatever is needed to shutdown the netlink socket that we set up.
 *
 ********************************************************/
void cardif_linux_rtnetlink_cleanup()
{
  debug_printf(DEBUG_INT, "Called cardif_linux_rtnetlink_cleanup()!\n");

  // Close the rtnetlink socket.
  close(rtnl_sock);
}

/********************************************************
 *
 * Decode data, and populate it in our SSID structure.
 *
 ********************************************************/
void cardif_linux_rtnetlink_reap(struct interface_data *intdata,
				 char *bptr, char *bptrend)
{
#ifdef EXPERIMENTAL
  char *tptr, essid[IW_ESSID_MAX_SIZE+1], custom[256], *temp=NULL;
  unsigned char mac[6];
  unsigned char stale_mac[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  unsigned char zeromac[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  struct iw_event *iwe=NULL;
  unsigned char flags=0;

  tptr = bptr;
  bzero(mac, 6);
  bzero(stale_mac, 6);

  debug_printf(DEBUG_INT, "Reaped Data :\n");
  debug_hex_dump(DEBUG_INT, bptr, (bptrend - bptr));

  while (tptr < bptrend)
    {
      flags = 0;
      //      bzero(essid, IW_ESSID_MAX_SIZE+1);

      while ((tptr < bptrend) && (memcmp(stale_mac, mac, 6) == 0))
	{
	  iwe = (struct iw_event *)tptr;
	  //debug_printf(DEBUG_INT, "Event Lengh = %d\n",iwe->len);

	  //	  bzero(custom, 256);
      
	  switch (iwe->cmd)
	    {
	    case SIOCGIWAP:
	      memcpy(mac, iwe->u.ap_addr.sa_data, 6);
	      
	      tptr += iwe->len;
	      break;
	      
	    case SIOCGIWFREQ:
	      //printf("SIOCGIWFREQ\n");
	      
	      tptr += iwe->len;
	      break;
	      
	    case SIOCGIWMODE:
	      if (iwe->u.mode == 3) 
		{
		  SET_FLAG(flags, MANAGED);
		} else {
		  UNSET_FLAG(flags, MANAGED);
		}
	      
	      tptr += iwe->len;
	      break;
	      
	    case SIOCGIWESSID:
	      bzero((char *)&essid, IW_ESSID_MAX_SIZE+1);
	      printf("iwe->u.essid.length = %d\n", iwe->u.essid.length);
	      memcpy((char *)&essid, (char *)&tptr[12], iwe->u.essid.length);
	      essid[iwe->u.essid.length] = '\0';
	      
	      tptr += iwe->len;
	      break;
	      
	    case SIOCGIWENCODE:
	      if (iwe->u.data.flags & IW_ENCODE_DISABLED)
		{
		  UNSET_FLAG(flags, WEP);
		}
	      else
		{
		  SET_FLAG(flags, WEP);
		}
	      
	      tptr += iwe->len;
	      break;
	      
	    case SIOCGIWRATE:
	      /*
		printf("SIOCGIWRATE\n");
		rate = iwe->u.bitrate.value;
		
		printf("Rate (KILO : %d   %d) : ", MEGA, rate);
		if (rate > GIGA)
		{
		printf("%fGbs\n", (rate/GIGA));
		} else if (rate > MEGA)
		{
		printf("%fMbs\n", (rate/MEGA));
		} else if (rate > KILO)
		{
		printf("%fKbs\n", (rate/KILO));
		}
	      */
	      tptr += iwe->len;
	      break;
	      
	    case IWEVQUAL:
	      //debug_printf(DEBUG_INT, "IWEVQUAL\n");
	      
	      tptr += iwe->len;
	      break;
	      
	    case IWEVCUSTOM:
	      memcpy(&custom, &tptr[12], (iwe->len - 12));
	      debug_printf(DEBUG_INT, "(Custom) Value : %s\n", custom);

	      if (strncmp("wpa_ie=", custom, 7) == 0)
		{
		  SET_FLAG(flags, WPA_IE);
		  debug_printf(DEBUG_INT, "AP appears to support WPA!\n");
		  temp = (char *)malloc((iwe->len - 12)/2);
		  process_hex(&custom[7], (iwe->len - 12-7), temp);
		  wpa_parse_ie(temp);
		}
	      else
		{
		  UNSET_FLAG(flags, WPA_IE);
		}
	      
	      tptr += iwe->len;
	      break;

	    default:
	      debug_printf(DEBUG_INT, "Unknown event (%04X)\n", iwe->cmd);
	      tptr += iwe->len;
	    }
	}

      // Check and see if we have a new set.
      if (memcmp(stale_mac, mac, 6) != 0)
	{
	  // If the stale MAC is all 0s, then this is our first time, and
	  // we should ignore things.
	  if (memcmp(zeromac, stale_mac, 6) != 0)
	    {
	      // We have a new set.
	      debug_printf(DEBUG_INT, "ESSID : %s   Flags : %02X (", essid, flags);
	      if (flags & MANAGED) debug_printf_nl(DEBUG_INT, "Managed, ");
	      if (flags & WEP) debug_printf_nl(DEBUG_INT, "WEP, ");
	      if (flags & WPA_IE) debug_printf_nl(DEBUG_INT, "WPA IE, ");
	      if (flags & RSN_IE) debug_printf_nl(DEBUG_INT, "RSN IE, ");
	      debug_printf_nl(DEBUG_INT, ")\n");

	      cardif_linux_ssids_add(intdata, essid, flags, temp, 
				     (iwe->len - 12-7), NULL, 0);
	      intdata->flags |= SCAN_DONE;  // The scan has completed, we have data.
	      bzero(&custom, 256);
	      if (temp != NULL)
		{
		  free(temp);
		  temp = NULL;
		}
	    }
	  memcpy(stale_mac, mac, 6);
	}
    }

  // If the stale MAC is all 0s, then this is our first time, and
  // we should ignore things.
  if (memcmp(zeromac, stale_mac, 6) != 0)
    {
      // We have a new set.
      debug_printf(DEBUG_INT, "ESSID : %s   Flags : %02X (", essid, flags);
      if (flags & MANAGED) debug_printf_nl(DEBUG_INT, "Managed, ");
      if (flags & WEP) debug_printf_nl(DEBUG_INT, "WEP, ");
      if (flags & WPA_IE) debug_printf_nl(DEBUG_INT, "WPA IE, ");
      if (flags & RSN_IE) debug_printf_nl(DEBUG_INT, "RSN IE, ");
      debug_printf_nl(DEBUG_INT, ")\n");

      cardif_linux_ssids_add(intdata, essid, flags, temp, 
			     (iwe->len - 12-7), NULL, 0);
      intdata->flags |= SCAN_DONE;        // The scan has completed, we have data.
      
      if (temp != NULL)
	{
	  free(temp);
	  temp = NULL;
	}
    }
#endif
}



/*******************************************************
 *
 * Check to see if we have data in the returned scan buffer, even if 
 * we didn't get a scan complete event.  (Some cards may not send the
 * scan complete event.)
 *
 *******************************************************/
void cardif_linux_rtnetlink_check_nets(struct interface_data *idata)
{
  struct lin_sock_data *sockData;
  struct iwreq iwr;
  char buffer[8192];

  debug_printf(DEBUG_INT, "Checking for returned SSID information....\n");

  sockData = idata->sockData;

  iwr.u.data.pointer = buffer;
  iwr.u.data.flags = 0;
  iwr.u.data.length = sizeof(buffer);

  strcpy(iwr.ifr_name, idata->intName);
  
  if (ioctl(sockData->sockInt, SIOCGIWSCAN, &iwr) < 0)
    {
      debug_printf(DEBUG_INT, "No data available!\n");
      return;
    }

  // Otherwise, we have something.  Clear the scan flag.
  idata->flags &= (~SCANNING);

  // Then harvest the data.
  debug_printf(DEBUG_INT, "Reaping data. (Size : %d)\n", iwr.u.data.length);
  cardif_linux_rtnetlink_reap(idata, (char *)buffer, 
			      ((char *)buffer + iwr.u.data.length));

  // Then, dump what we have.
  cardif_linux_ssids_dump(idata);
}

void cardif_linux_rtnetlink_dump_ints(struct interface_data *);

/*******************************************************
 *
 * Check to see if we have any events, and act on them if we do.
 * intdata passed in should be the starting pointer in the list!
 *
 *******************************************************/
void cardif_linux_rtnetlink_check_event(struct interface_data **intdata)
{
  int rtnl_data_size, remain, length;
  char buf[8192];
  struct nlmsghdr *nlhead;
  struct interface_data *idata;

  idata = *intdata;

  debug_printf(DEBUG_EXCESSIVE, "Checking event interface.\n");

  // Grab the next message off the rtnetlink socket.
  rtnl_data_size = sizeof(rtnl_data);
  if (rtnl_sock < 0)
    {
      debug_printf(DEBUG_NORMAL, "RTnetlink socket not available!\n");
      return;
    }

  remain = recvfrom(rtnl_sock, buf, sizeof(buf), MSG_DONTWAIT,
			(struct sockaddr *)&rtnl_data, &rtnl_data_size);
  if (remain >= 0)
    {
      // We need a pointer to the buffer to work with.
      nlhead = (struct nlmsghdr *)buf;

      // There may be more than one message in the packet.  So, loop through!
      while (remain >= sizeof(struct nlmsghdr))
	{
	  // Make sure we have enough data for a real message.
	  if ((nlhead->nlmsg_len > remain) || 
	      ((nlhead->nlmsg_len - sizeof(struct nlmsghdr)) < 0))
	    {
	      debug_printf(DEBUG_NORMAL, "Invalid netlink message!\n");
	      break;
	    }
	  
	  // See what kind of message it is.  (New or deleted link?)
	  switch (nlhead->nlmsg_type)
	    {
	    case RTM_NEWLINK:
	      debug_printf(DEBUG_INT, "Got an RTM_NEWLINK!\n");
	      cardif_linux_rtnetlink_do_link(intdata, nlhead,
					     nlhead->nlmsg_len, INT_NEW);
	      break;
	      
	    case RTM_DELLINK:
	      debug_printf(DEBUG_INT, "Got an RTM_DELLINK!\n");
	      cardif_linux_rtnetlink_do_link(intdata, nlhead, 
					     nlhead->nlmsg_len, INT_DEL);
	      break;
	    }

	  // Find the aligned length of the message, so we can skip
	  // to the next message.
	  length = NLMSG_ALIGN(nlhead->nlmsg_len);

	  remain -= length;
	  
	  nlhead = (struct nlmsghdr *) ((char *)nlhead + length);
	}

      // If we have anything left, then there may be a problem.  So, report
      // the we may have a problem.
      if (remain > 0)
	{
	  debug_printf(DEBUG_NORMAL, "Extra bytes at the end of the netlink message.\n");
	}
    }

#ifdef EXPERIMENTAL
  // Do flags specific interface checks.
  while (idata != NULL)
    {
      if (idata->flags & SCANNING)
	{
	  // There may be situations where the wireless card driver can scan,
	  // but doesn't provide an event when the scan has completed.  So,
	  // we need to check if we have data, even if we don't have an event.
	  cardif_linux_rtnetlink_check_nets(idata);
	}
      
      idata = idata->next;
    }

  cardif_linux_rtnetlink_dump_ints(*intdata);
#endif
}

/************************************************************
 *
 *  We got an RTM_NEWLINK or RTM_DELLINK message, so process it, and 
 *  decide how to proceed.
 *
 ************************************************************/
void cardif_linux_rtnetlink_do_link(struct interface_data **intdata,
				    struct nlmsghdr *msg, int len, int type)
{
  struct ifinfomsg *ifinfo;
  int nlmsg_len, rtalen, rtlen;
  struct rtattr *rtattr;

  if (len < sizeof(struct ifinfomsg))
    {
      debug_printf(DEBUG_NORMAL, "Netlink message too short!\n");
      return;
    }

  // Get the actual message from the block.
  ifinfo = NLMSG_DATA(msg);

  // Find out how big the message is.
  nlmsg_len = NLMSG_ALIGN(sizeof(struct ifinfomsg));

  if ((msg->nlmsg_len - nlmsg_len) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Message inside newlink isn't valid!\n");
      return;
    }

  rtattr = (struct rtattr *)(((char *)ifinfo) + nlmsg_len);

  rtalen = RTA_ALIGN(sizeof(struct rtattr));
  
  // Validate the attribute we have, and determine if it is for wireless,
  // or wired.
  while (RTA_OK(rtattr, (msg->nlmsg_len - nlmsg_len)))
    {
      switch (rtattr->rta_type)
	{
	case IFLA_IFNAME:
	  // This is a non-wireless event.
	  cardif_linux_rtnetlink_ifla_ifname(intdata, ifinfo->ifi_index, type);
	  break;

	case IFLA_WIRELESS:
	  // This is a wireless event.
	  cardif_linux_rtnetlink_ifla_wireless(intdata, ifinfo->ifi_index,
					       ((char *) rtattr)+rtalen,
					       rtattr->rta_len - rtalen);
	  break;
	}

      rtlen = msg->nlmsg_len - nlmsg_len;

      // Get the next attribute
      rtattr = RTA_NEXT(rtattr, rtlen);
    }
}

void cardif_linux_rtnetlink_dump_ints(struct interface_data *intdata)
{
  struct interface_data *cur;

  cur = intdata;

  while (cur != NULL)
    {
      cur = cur->next;
    }
}

/***********************************************************
 *
 * Check to see if the interface that was removed is one we are working
 * with.  If it is, then take it out of the list.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_del_int(struct interface_data **intdata,
				    int ifindex)
{
  struct interface_data *cur, *prev;

  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Corrupt interface structure!?\n");
      return;
    }

  cur = *intdata;
  prev = cur;

  if (cur == NULL) 
    {
      debug_printf(DEBUG_INT, "Interface list is empty!\n");
      return;   // We don't have any interfaces, so just return.
    }

  if (cur->intIndex == ifindex)
    {
      debug_printf(DEBUG_INT, "Interface : %s  --  Index : %d\n", cur->intName,
		   cur->intIndex);

      // Clean up the interface.
      cardif_deinit(cur);
      cur = destroy_interface_struct(cur);
      *intdata = cur;
      return;
    }

  if (cur->next == NULL) 
    {
      debug_printf(DEBUG_INT, "Interface not found!  Bailing out!\n");
      return;
    }

  cur = cur->next;

  while (cur->next != NULL)
    {
      if (cur->intIndex == ifindex) 
	{
	  debug_printf(DEBUG_INT, "Found a match : %d == %d\n", cur->intIndex,
		       ifindex);
	  break;
	}
      cur = cur->next;
      prev = prev->next;
    }

  if (cur->intIndex != ifindex) 
    {
      debug_printf(DEBUG_INT, "Last attempt at matching : %d == %d\n",
		   cur->intIndex, ifindex);
      return;   // Nothing to do.
    }

  debug_printf(DEBUG_INT, "Interface : %s  --  Index : %d\n", cur->intName,
	       cur->intIndex);

  // Clean up the interface.
  cardif_deinit(cur);
  cur = destroy_interface_struct(cur);
  prev->next = cur;
}

/***********************************************************
 *
 * Check if the interface already exists.  If it does, do nothing.  If
 * it doesn't, then it is a new interface.  Verify that it is an allowed
 * interface, and add it.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_add_int(struct interface_data **intdata, 
				    int ifindex)
{
  struct interface_data *cur;
  char intname[16];
  struct config_string_list *denyints = NULL;

  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Corrupt interface structure!?\n");
      return;
    }

  cur = *intdata;

  if (cur == NULL)
    {
      debug_printf(DEBUG_INT, "There are not interfaces currently in the list!\n");

      if_indextoname(ifindex, (char *)&intname);
      debug_printf(DEBUG_INT, "Checking interface : %s\n", intname);

      denyints = config_denied_interfaces();

      if (denyints != NULL)
	{
	  if (config_int_in_list(intname, denyints) == TRUE)
	    {
	      debug_printf(DEBUG_INT, "Interface is not allowed!  Ignoring!\n");
	      return;
	    } else {
	      debug_printf(DEBUG_INT, "Attempting to add interface!\n");
	    }
	}

      if (cardif_validate(intname) == TRUE)
	{
	  // This interface is a real interface.
	  cur = (struct interface_data *)malloc(sizeof(struct interface_data));
	  if (cur == NULL)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for new interface!  (Interface : %s)\n", intname);
	      return;
	    }

	  // Zero out the new interface structure.
	  bzero(cur, sizeof(struct interface_data));

	  if (init_interface_struct(cur, (char *)&intname) != XENONE)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't initialize the interface struct!\n");
	      // Free up the memory we allocated.
	      free(cur);
	      return;
	    }
	  
	  cardif_init(cur);
	  cur->flags = (cur->flags & (~IS_WIRELESS));
	  cur->userdata = config_build(cur->cur_essid);
	  config_set_globals(cur);
	  cur->next = NULL;

	  *intdata = cur;
	} else {
	  debug_printf(DEBUG_INT, "Not a valid interface.  Ignoring.\n");
	} 
    } else {
      // Loop through all of the interfaces, to verify that we don't have
      // the interface.
      do
	{
	  // If the interface already exists, then don't do anything.
	  if (cur->intIndex == ifindex)
	    {
	      debug_printf(DEBUG_INT, "Found interface %s, with index of %d! (Ignored)\n",
			   cur->intName, ifindex);
	      return;
	    }

	  if (cur->next != NULL) cur = cur->next;
	} while (cur->next != NULL);

      if (cur->intIndex == ifindex) 
	{
	  debug_printf(DEBUG_INT, "This interface already exists!\n");
	  return;   // We have this interface.
	}

      if_indextoname(ifindex, (char *)&intname);
      debug_printf(DEBUG_INT, "Checking interface : %s\n", intname);

      denyints = config_denied_interfaces();

      if (denyints != NULL)
	{
	  if (config_int_in_list(intname, denyints) == TRUE)
	    {
	      debug_printf(DEBUG_INT, "Interface is not allowed!  Ignoring!\n");
	      return;
	    } else {
	      debug_printf(DEBUG_INT, "Attempting to add interface!\n");
	    }
	}

      if (cardif_validate(intname) == TRUE)
	{
	  // This interface is a real interface.
	  cur->next = (struct interface_data *)malloc(sizeof(struct interface_data));
	  if (cur == NULL)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for new interface!  (Interface : %s)\n", intname);
	      return;
	    }

	  cur = cur->next;

	  // Zero out the new interface structure.
	  bzero(cur, sizeof(struct interface_data));

	  if (init_interface_struct(cur, (char *)&intname) != XENONE)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't initialize the interface struct!\n");
	      // Free up the memory we allocated.
	      free(cur);
	      return;
	    }
	  
	  cardif_init(cur);
	  cur->flags = (cur->flags & (~IS_WIRELESS));
	  cur->userdata = config_build(cur->cur_essid);
	  config_set_globals(cur);
	  cur->next = NULL;
	  
	  *intdata = cur;
	} else {
	  debug_printf(DEBUG_INT, "Not a valid interface.  Ignoring.\n");
	}
    }
  cardif_linux_rtnetlink_dump_ints(*intdata);
}

/***********************************************************
 *
 * Process an event that isn't wireless.  (It may be a wired card, or
 * it may be something like a card insertion/removal.)
 *
 ***********************************************************/
void cardif_linux_rtnetlink_ifla_ifname(struct interface_data **intdata,
					int ifindex, int isdelete)
{
  debug_printf(DEBUG_INT, "Working with an interface with index of %d.\n", 
	       ifindex);

  switch (isdelete)
    {
    case INT_NEW:
      debug_printf(DEBUG_INT, "   -- Got a new interface request.\n");
      cardif_linux_rtnetlink_add_int(intdata, ifindex);
      break;

    case INT_DEL:
      debug_printf(DEBUG_INT, "   -- Got a deleted interface request.\n");
      cardif_linux_rtnetlink_del_int(intdata, ifindex);
      break;
    }
}

/***********************************************************
 *
 * Process wireless events.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_ifla_wireless(struct interface_data **intdata, 
					  int ifindex, char *data, int len)
{
  struct iw_event *iwe;
  struct interface_data *idata;
  char *pos, *end;

  idata = (struct interface_data *)*intdata;
  pos = data;
  end = data+len;

  while (pos + IW_EV_LCP_LEN <= end)
    {
      iwe = (struct iw_event *) pos;
      debug_printf(DEBUG_INT, "Wireless event: cmd=0x%x len=%d\n", iwe->cmd,
		   iwe->len);

      if (iwe->len <= IW_EV_LCP_LEN)
	{
	  debug_printf(DEBUG_NORMAL, "Not enough data in wireless event!\n");
	  return;
	}

      switch (iwe->cmd)
	{
	case SIOCGIWAP:
	  // We don't do anything with this event yet.
	  debug_printf(DEBUG_INT, "New AP found!\n");
	  break;

	case SIOCGIWSCAN:
	  // We have been told that the scan is complete.
	  debug_printf(DEBUG_INT, "Wireless scan complete!\n");
	  cardif_linux_rtnetlink_check_nets(idata);
	  break;
	}

      pos += iwe->len;
    }

}
