/**
 * A client-side 802.1x implementation 
 *
 * This code is released under both the GPL version 2 and BSD licenses.
 * Either license may be used.  The respective licenses are found below.
 *
 * Copyright (C) 2002 Bryan D. Payne & Nick L. Petroni Jr.
 * All Rights Reserved
 *
 * --- GPL Version 2 License ---
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * --- BSD License ---
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the University of
 *       Maryland at College Park and its contributors.
 *  - Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*******************************************************************
 * The core linux event loop.
 * File: linux_core.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: linux_core.c,v 1.5 2004/08/15 04:29:35 chessing Exp $
 * $Date: 2004/08/15 04:29:35 $
 * $Log: linux_core.c,v $
 * Revision 1.5  2004/08/15 04:29:35  chessing
 *
 * Completed support for scanning and joining wireless networks.  If your interface isn't configured, we will scan to gather information about all known wireless networks.  If a network in the list has a configuration, we will join it.  (Including turning on needed encryption.)  There are still a few rough edges that need to be hammered out.  But the overall user experience should be improved.
 *
 * Revision 1.4  2004/08/05 23:56:21  chessing
 *
 * Added basic support for scanning for broadcast SSIDs.  This is another step closer to WPA/11i support. ;)
 *
 * Revision 1.3  2004/07/25 19:36:41  chessing
 * Fixed a few more logical/bitwise nots.  Added rtnetlink support for hot-plugging interfaces.  (Hot unplugging is coming soon. ;)
 *
 * Revision 1.2  2004/07/23 04:05:50  chessing
 * Fixed a segfault problem.  Started to add rtnetlink support.
 *
 * Revision 1.1  2004/07/23 02:14:39  chessing
 * Moved main event loop from xsup_driver.c in to an OS specific file to allow for easier porting.
 *
 *
 *******************************************************************/

#include <sys/select.h>
#include <sys/socket.h>
#include <asm/types.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>

#include "profile.h"
#include "config.h"
#include "eap.h"
#include "statemachine.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "eapol.h"
#include "cardif/cardif.h"
#include "cardif/core.h"
#include "cardif/linux/cardif_linux_rtnetlink.h"

/*****************************************************************
 *
 * Do any setup that is needed for our event loop.  This is called only
 * once!  (And not once per interface.)  If you need to initalize something
 * per interface, is should either be included in cardif_init(), or using
 * some checks in event_core()!
 *
 *****************************************************************/
void event_core_setup()
{
  debug_printf(DEBUG_INT, "Called event_core_setup()!\n");
  cardif_linux_rtnetlink_init();
}

/*****************************************************************
 *
 * This will be called by our clock tick handler.  It should be used
 * for any internal counters.
 *
 *****************************************************************/
void event_core_clock_tick(struct interface_data *intdata)
{
  cardif_clock_tick(intdata);
}
  
/*****************************************************************
 *
 * Do any event processing that needs to happen.  This *MUST* include
 * executing the statemachine for each interface we know of, and sending
 * and receiving of frames.
 *
 *****************************************************************/
void event_core(struct interface_data **ppintcur)
{
  struct timeval tv;
  fd_set rfds;
  int maxfd = 0, ssid, mac, framewaiting = 0;
  struct interface_data *cur;
  
  cur = *ppintcur;

  cardif_linux_rtnetlink_check_event(ppintcur);

  FD_ZERO(&rfds);
  
  while (cur != NULL)
    {
      debug_printf(DEBUG_EVERYTHING, "Processing interface %s... (Flags : %02X)\n", cur->intName, cur->flags);

      // Check to see if we have hopped APs.
      mac = cardif_check_dest(cur);
      ssid = cardif_check_ssid(cur);

      if ((mac == TRUE) && (ssid != XNEWESSID))
	{
	  debug_printf(DEBUG_CONFIG, "MAC address changed!  Updating config!\n");
	  // DO NOT delete the current config struct that is hanging off of
	  // userdata.  It will cause strange results!
	  cur->userdata = config_build(cur->cur_essid);
	}

      if (ssid == XNEWESSID)
	{
	  // Our ESSID changed, we need to clear our config, so that
	  // we generate a new one on the way through the statemachine.
	  eap_clear_active_method(cur->userdata->activemethod);

	  // We also need to check to make sure we are pointing
	  // to the correct configuration information.
	  cur->userdata = config_build(cur->cur_essid);
	}

      if (cur->userdata == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Userdata is NULL!  We will probably have problems!\n");
	}

      eapol_execute(cur);
      FD_SET(cardif_get_socket(cur), &rfds);
      if (cardif_get_socket(cur) > maxfd)
	maxfd = cardif_get_socket(cur);
      
      // If we have a frame available on any interface, we won't
      // sleep.  Instead we will keep looping through to keep things
      // moving as fast as possible.
      if (frameavail(cur) == TRUE) framewaiting = 1;
      cur = cur->next;
    }

  if (framewaiting == 0)
    {
      tv.tv_sec = 0;
      tv.tv_usec = 500000;
      select(maxfd + 1, &rfds, NULL, NULL, &tv);
    }
  framewaiting = 0;
}

/******************************************************************
 *
 * Clean up anything that we initialized in event_core_setup().
 *
 ******************************************************************/
void event_core_cleanup()
{
  debug_printf(DEBUG_INT, "Called event_core_cleanup()!\n");
  cardif_linux_rtnetlink_cleanup();
}
