/*******************************************************************
 *
 * Licensed under a dual GPL/BSD license. (See LICENSE file for more info.)
 *
 * File: gui_interface.c
 *
 * Authors: Chris.Hessing@utah.edu, Terry.Simons@utah.edu
 *
 * $Id: gui_interface.c,v 1.20 2005/08/20 19:06:51 chessing Exp $
 * $Date: 2005/08/20 19:06:51 $
 * $Log: gui_interface.c,v $
 * Revision 1.20  2005/08/20 19:06:51  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.19  2005/08/09 01:39:13  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/un.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include "gui_interface.h"

#define XSUP_SOCKET "/tmp/xsupplicant.sock"
#define DEBUG  1

/*******************************************************************
 *
 * Establish a handler to talk to the supplicant.
 *
 *******************************************************************/
int gui_interface_connect(char *intname)
{
  int sockErr, skfd;
  struct sockaddr_un sa;

  skfd = socket(PF_UNIX, SOCK_STREAM, 0);
  if (skfd < 0)
    {
#if DEBUG
      printf("Error getting socket!\n");
#endif
      return ERR_SOCK;
    }

  strncpy(sa.sun_path, XSUP_SOCKET, sizeof(sa.sun_path)-1);
  strncat(sa.sun_path, ".", 1);
  strcat(sa.sun_path, intname);

  sa.sun_family = AF_LOCAL;
  
  sockErr = connect(skfd, (struct sockaddr *)&sa, sizeof(sa));
  if (sockErr < 0) 
    {
#if DEBUG
      printf("Socket Error : %d -- %s  (%s:%d)\n", errno, strerror(errno),
	     __FUNCTION__, __LINE__);
#endif
      return errno;
    }

  return skfd;
}

/******************************************************************
 *
 * Disconnect from the daemon.
 *
 ******************************************************************/
int gui_interface_disconnect(int skfd)
{
  close(skfd);
  return 0;
}

/******************************************************************
 *
 * See if we have any data.  If we aren't blocking, we will wait here
 * forever until we get something back!
 *
 ******************************************************************/
int gui_interface_get_packet(int skfd, char *buffer, int *bufsize, int block)
{
  int readStat = -1;
  fd_set tset;
  struct timeval tv;

  FD_ZERO(&tset);
  FD_SET(skfd, &tset);
  tv.tv_sec = 0;
  tv.tv_usec = 0;

  if (block != 1)
    {
      readStat = select(skfd+1, &tset, NULL, NULL, &tv);
      if (readStat <= 0) return ERR_NONE;
    }

  readStat = recv(skfd, buffer, *bufsize, 0);
 
  if (readStat < 0) 
    {
      printf("Socket error : %s\n",strerror(errno));
      *bufsize = 0;
      return ERR_SOCK;
    }

  *bufsize = readStat;

  return ERR_NONE;
}

/******************************************************************
 *
 * Validate, and send a packet.
 *
 ******************************************************************/
int gui_interface_send_packet(int skfd, char *buffer, int bufptr)
{
  int i;

  i=send(skfd, buffer, bufptr, 0);
  if (i < 0)
    {
      printf("Socket error! (%d : %d)\n",i, errno);
      return ERR_SOCK;
    }

  return ERR_NONE;
}

/******************************************************************
 *
 * bufptr should point to the next command record to be processed. We
 * will pull the record out, copy the data for the record in to the
 * *data, and return the value of the record we are using.
 *
 * The caller should check that *bufptr < buffer length.
 *
 ******************************************************************/
int gui_interface_parse_packet(char *buffer, int *bufptr, char *data, 
			       int *datalen, char *interface)
{
  struct ipc_cmd *cmd;

  if (buffer == NULL)
    {
#if DEBUG
      printf("buffer == NULL!\n");
#endif
      return ERR_PKT_BAD;
    }

  if (*bufptr < 0) 
    {
      *bufptr = 0;
    } 

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  if (cmd->version != IPC_VERSION_NUM) 
    {
#if DEBUG
      printf("IPC version mismatch!\n");
#endif
      return ERR_PKT_BAD;
    }

  if (if_indextoname(cmd->int_idx, interface) == NULL)
    {
#if DEBUG
      printf("Invalid interface requested!  (Index : %d)\n", cmd->int_idx);
#endif
      return ERR_PKT_BAD;
    }

  /* zero out the buffer, just to be safe. */
  bzero(data, cmd->len+1);

  *bufptr += sizeof(struct ipc_cmd);  /* Skip the command structure. */

  memcpy(data, (char *)&buffer[*bufptr], cmd->len);

  *bufptr += cmd->len;
  *datalen = cmd->len;

  return cmd->attribute;
}

/*****************************************************************
 *
 * Request a list of all known interfaces.
 *
 *****************************************************************/
int gui_interface_get_interfaces(char intidx, char *buffer, int *bufptr)
{
  printf("%s is deprecated, and will be removed in the next release!!\n",
	 __FUNCTION__);

  return ERR_NONE;
}

/*****************************************************************
 *
 * Request the authentication state of an interface.
 *
 *****************************************************************/
int gui_interface_get_state(char intidx, char *buffer, int *bufptr)
{
  struct ipc_cmd *cmd;

  cmd = (struct ipc_cmd *)&buffer[*bufptr];
  
  cmd->version = IPC_VERSION_NUM;
  cmd->int_idx = intidx;
  cmd->attribute = AUTH_STATE;
  cmd->getset = IPC_GET;
  cmd->len = 0;
  *bufptr += sizeof(struct ipc_cmd);

  return ERR_NONE;
}

/****************************************
 *
 * Send in a password set command.
 *
 ****************************************/
int gui_interface_set_password(char intidx, char *buffer, int *bufptr, 
			       char *password)
{
  struct ipc_cmd *cmd;

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->int_idx = intidx;
  cmd->attribute = TEMPPASSWORD;
  cmd->getset = IPC_SET;
  cmd->len = strlen(password);
  *bufptr += sizeof(struct ipc_cmd);

  strcpy((char *)&buffer[*bufptr], password);
  *bufptr += strlen(password);
  
  return ERR_NONE;
}

/****************************************
 *
 * Test for a PID file, and return an error if something seems to be running.
 *
 ****************************************/
int is_xsup_running(char *intname)
{
  char fullpidfilename[255];
  struct stat mystats;
  int err;

  strncpy(fullpidfilename, XSUP_SOCKET, 255);
  strncat(fullpidfilename, ".", 1);
  strncat(fullpidfilename, intname, 255);

  err = stat(fullpidfilename, &mystats);
  if (err == 0)
    {
      return TRUE;
    }
  return FALSE;
}
