/*****************************************************************************
 * File: backend_sm.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: backend_sm.c,v 1.12 2005/08/09 01:39:13 chessing Exp $
 * $Date: 2005/08/09 01:39:13 $
 * $Log: backend_sm.c,v $
 * Revision 1.12  2005/08/09 01:39:13  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 ****************************************************************************/

#include <stdio.h>
#include "xsup_debug.h"
#include "xsup_err.h"
#include "profile.h"
#include "config.h"
#include "statemachine.h"
#include "backend_sm.h"
#include "eap.h"
#include "snmp.h"
#include "cardif/cardif.h"

/****************************************************
 *
 * Init any variables that need to be set up, and do anything else that the
 * backend state machine needs.
 *
 ****************************************************/
void backend_sm_init(struct interface_data *thisint)
{
  backend_sm_reinit(thisint);

  eap_init(thisint);
}

/****************************************************
 *
 * Re-init any variables that need to be set up, and do anything else that the
 * backend state machine needs.
 *
 ****************************************************/
void backend_sm_reinit(struct interface_data *thisint)
{
  thisint->statemachine->eapNoResp = FALSE;
  thisint->statemachine->eapReq = FALSE;
  thisint->statemachine->eapResp = FALSE;
  thisint->statemachine->beCurState = INITIALIZE;
  thisint->statemachine->beLastState = 0xff;
}

/****************************************************
 *
 * Free any resources that were in use before telling the main state machine
 * that we have aborted this authentication.
 *
 ****************************************************/
void abortSupp()
{
  //XXX Nothing here yet.
}

/****************************************************
 *
 * Wait until the layers above have something to send.  (This is where we
 * get username/passwords.)
 *
 ****************************************************/
void getSuppRsp(struct interface_data *thisint)
{
  switch (eap_process_header(thisint)) 
    {
    case EAP_REQUEST_ID:
      snmp_dot1xSuppEapolReqIdFramesRx();
      if (eap_response_id(thisint) == XENONE)
	{
	  // We have something to send.
	  thisint->statemachine->eapResp = TRUE;
	  snmp_dot1xSuppEapolRespIdFramesTx();
	}
      break;

    case EAP_REQUEST_AUTH:
      snmp_dot1xSuppEapolReqFramesRx();
      if (eap_response_auth(thisint) == XENONE)
	{
	  // We have something to send.
	  thisint->statemachine->eapResp = TRUE;
	  snmp_dot1xSuppEapolRespFramesTx();
	}
      break;

    case EAP_REQUEST_NOTIFY:
      eap_do_notify(thisint);
      break;
    }
}

/****************************************************
 *
 * Transmit the response that should be in the queue.
 *
 ****************************************************/
void txSuppRsp(struct interface_data *thisint)
{
  // Make sure we know that we have sent the frame.
  thisint->statemachine->eapResp = FALSE;
  thisint->statemachine->eapolEap = FALSE;
  cardif_sendframe(thisint);
}

/****************************************************
 *
 * For some reason, the backend state machine timed out.  Depending on the
 * last packet we recieved, we may be able to provide the user with 
 * information about what went wrong.  For now, we are only going to report
 * on things that may be wrong when a Response ID message goes unanswered,
 * but in the future, we should expand to provide information on specific
 * EAP types where possible.
 *
 ****************************************************/
int backend_sm_timeout_display(struct interface_data *thisint)
{
  switch (thisint->statemachine->lastEapType)
    {
    case EAP_TYPE_IDENTITY:
      // The last EAP message we saw was an Identity request.  We assume that
      // we attempted to send a response to that request.  (If not, it should
      // be reported elsewhere.)  If we get here, then there was a timeout
      // waiting for the authenticator to send us a packet containing the
      // beginning of the actual EAP conversation.

      if (config_get_display_friendly_warnings())
	{
	  debug_printf(DEBUG_NORMAL, "[WARNING]  Timeout waiting for the "
		       "authenticator to begin the EAP conversation.  This "
		       "usually happens when the RADIUS server is "
		       "misconfigured, the authenticator can't talk to the "
		       "RADIUS server, or the username provided is "
		       "invalid.\n");
	}
      break;

    default:
      if (config_get_display_friendly_warnings())
	{
	  debug_printf(DEBUG_NORMAL, "[WARNING] Timeout during the EAP "
		       "conversation!  Please verify that the settings\nin "
		       "your config file are correct and that the "
		       "authenticator and RADIUS servers\nare properly "
		       "configured.  If this error persists, please run "
		       "Xsupplicant in debug\nmode, and e-mail the output, "
		       "along with the config\nfile, and RADIUS config file "
		       "(where possible) to open1x-xsupplicant@"
		       "lists.sourceforge.net.\n");
	}
      break;
    }

  return XENONE;
}

/****************************************************
 *
 * Actually run the backend state machine.
 *
 ****************************************************/
int backend_sm_run(struct interface_data *thisint)
{
  if (thisint->statemachine->initialize || thisint->statemachine->suppAbort)
    {
      thisint->statemachine->beCurState = INITIALIZE;
      debug_printf(DEBUG_STATE, "Defaulting to INITIALIZE state!\n");
    }

  if (thisint->statemachine->eapRestart)
    {
      debug_printf(DEBUG_STATE, "Supplicant PAE has issued a restart.\n");
      eap_reset(thisint);
    }

  switch (thisint->statemachine->beCurState)
    {
    case INITIALIZE:
      abortSupp();
      thisint->statemachine->suppAbort = FALSE;
      thisint->statemachine->beLastState = thisint->statemachine->beCurState;
      thisint->statemachine->beCurState = IDLE;
      debug_printf(DEBUG_STATE, "Backend State : INITIALIZE -> IDLE\n");
      break;

    case IDLE:
      if (thisint->statemachine->beCurState != thisint->statemachine->beLastState)
	{
      	  thisint->statemachine->suppStart = FALSE;
	  thisint->statemachine->beLastState = thisint->statemachine->beCurState;
	}

      if (thisint->statemachine->eapFail && thisint->statemachine->suppStart)
	{
	  thisint->statemachine->beCurState = FAIL;
	  debug_printf(DEBUG_STATE, "Backend State : IDLE -> FAIL\n");
	}

      if (thisint->statemachine->eapolEap && thisint->statemachine->suppStart)
	{
	  thisint->statemachine->beCurState = REQUEST;
	  debug_printf(DEBUG_STATE, "Backend State : IDLE -> REQUEST\n");
	}

      if (thisint->statemachine->eapSuccess && thisint->statemachine->suppStart)
	{
	  thisint->statemachine->beCurState = SUCCESS;
	  debug_printf(DEBUG_STATE, "Backend State : IDLE -> SUCCESS\n");
	}
      break;

    case REQUEST:
      if (thisint->statemachine->beCurState != thisint->statemachine->beLastState)
	{
	  thisint->statemachine->authWhile = 0;
	  thisint->statemachine->eapReq = TRUE;
	  thisint->statemachine->beLastState = REQUEST;
	}

      getSuppRsp(thisint);

      if (thisint->statemachine->eapFail)
	{
	  thisint->statemachine->beCurState = FAIL;
	  debug_printf(DEBUG_STATE, "Backend State : REQUEST -> FAIL\n");
	}

      if (thisint->statemachine->eapNoResp)
	{
	  thisint->statemachine->beCurState = RECEIVE;
	  debug_printf(DEBUG_STATE, "Backend State : REQUEST -> RECEIVE\n");
	}

      if (thisint->statemachine->eapResp)
	{
	  thisint->statemachine->beCurState = RESPONSE;
	  debug_printf(DEBUG_STATE, "Backend State : REQUEST -> RESPONSE\n");
	}

      if (thisint->statemachine->eapSuccess)
	{
	  thisint->statemachine->beCurState = SUCCESS;
	  debug_printf(DEBUG_STATE, "Backend State : REQUEST -> SUCCESS\n");
	}
      break;

    case RESPONSE:
      txSuppRsp(thisint);
      thisint->statemachine->eapResp = FALSE;
      thisint->statemachine->beLastState = thisint->statemachine->beCurState;
      thisint->statemachine->beCurState = RECEIVE;
      debug_printf(DEBUG_STATE, "Backend State : RESPONSE -> RECEIVE\n");
      break;

    case RECEIVE:
      if (thisint->statemachine->beCurState != thisint->statemachine->beLastState)
	{
	  thisint->statemachine->authWhile = thisint->statemachine->authPeriod;
       	  thisint->statemachine->eapolEap = FALSE;
	  thisint->statemachine->eapNoResp = FALSE;
	  thisint->statemachine->beLastState = thisint->statemachine->beCurState;
	}

      if (thisint->statemachine->eapFail)
	{
	  thisint->statemachine->beCurState = FAIL;
	  debug_printf(DEBUG_STATE, "Backend State : RECEIVE -> FAIL\n");
	}

      if (thisint->statemachine->authWhile == 0)
	{
	  thisint->statemachine->beCurState = TIMEOUT;
	  debug_printf(DEBUG_STATE, "Backend State : RECEIVE -> TIMEOUT\n");
	  backend_sm_timeout_display(thisint);
	}

      if (thisint->statemachine->eapSuccess)
	{
	  thisint->statemachine->beCurState = SUCCESS;
	  debug_printf(DEBUG_STATE, "Backend State : RECEIVE -> SUCCESS\n");
	}

      if (thisint->statemachine->eapolEap)
	{
	  thisint->statemachine->beCurState = REQUEST;
	  debug_printf(DEBUG_STATE, "Backend State : RECEIVE -> REQUEST\n");
	}
      break;

    case FAIL:
      thisint->statemachine->suppFail = TRUE;
      thisint->statemachine->beLastState = thisint->statemachine->beCurState;
      thisint->statemachine->beCurState = IDLE;
      debug_printf(DEBUG_STATE, "Backend State : FAIL -> IDLE\n");
      break;

    case TIMEOUT:
      thisint->statemachine->suppTimeout = TRUE;
      thisint->statemachine->beLastState = thisint->statemachine->beCurState;
      thisint->statemachine->beCurState = IDLE;
      debug_printf(DEBUG_STATE, "Backend State : TIMEOUT -> IDLE\n");
      break;

    case SUCCESS:
      thisint->statemachine->keyRun = TRUE;
      thisint->statemachine->suppSuccess = TRUE;
      thisint->statemachine->beLastState = thisint->statemachine->beCurState;
      thisint->statemachine->beCurState = IDLE;
      debug_printf(DEBUG_STATE, "Backend State : SUCCESS -> IDLE\n");
      break;
    }

  return XENONE;
}

/****************************************************
 *
 * Clean up anything that we set up during the use of the state machine.
 *
 ****************************************************/
void backend_sm_deinit(struct interface_data *thisint)
{
  struct config_network *network_data;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network data!  (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  eap_cleanup(&network_data->activemethod);
}

