/*******************************************************************
 * File: cardif_linux_rtnetlink.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * Comments :
 *     Large portions of the rtnetlink code is derived from the 
 *     wpa_supplicant code written by Jouni Malinen.  (In some cases,
 *     it is copied verbatim.)  wpa_supplicant can be found at 
 *     http://hostap.epitest.fi.
 *
 * $Id: cardif_linux_rtnetlink.c,v 1.54 2005/08/20 19:06:54 chessing Exp $
 * $Date: 2005/08/20 19:06:54 $
 * $Log: cardif_linux_rtnetlink.c,v $
 * Revision 1.54  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.53  2005/08/18 03:19:04  chessing
 * Added the ability to define an SSID with static WEP keys.  When we switch to a network that has this type of configuration we will set the keys, and stop the various association timers.
 *
 * Revision 1.52  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#ifdef LINUX_FRAMER

#include <unistd.h>
#include <strings.h>
#include <string.h>
#include <stdlib.h>
#include <asm/types.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <linux/if.h>
#include <sys/ioctl.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/if.h>
#include <linux/wireless.h>
#include <linux/if_packet.h>

#include "../../profile.h"
#include "../../config.h"
#include "../../xsup_debug.h"
#include "../../xsup_err.h"
#include "cardif_linux.h"
#include "cardif_linux_rtnetlink.h"
#include "../../statemachine.h"
#include "../cardif.h"
#include "../../wpa.h"
#include "../../wpa2.h"
#include "../../eap.h"
#include "../../eap_types/mschapv2/mschapv2.h"
#include "../../config_ssid.h"
#include "../../eapol_key_type254.h"
#include "timer.h"
#include "mic.h"

// Define this, so the compiler doesn't complain.
extern char *if_indextoname(unsigned int, char *);

#define INT_DEL    0
#define INT_NEW    1

static int rtnl_sock=-1;
static struct sockaddr_nl rtnl_data;

#ifndef IWEVCUSTOM
#warning IWEVCUSTOM is not defined!  We will define it, and try to continue!
#define IWEVCUSTOM   0x8C02
#endif

/********************************************************
 *
 * Do whatever is needed to establish a netlink socket so that we can
 * catch events, and take action.
 *
 ********************************************************/
void cardif_linux_rtnetlink_init()
{
  rtnl_sock = socket(PF_NETLINK, SOCK_RAW, NETLINK_ROUTE);
  
  if (rtnl_sock < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't establish an rtnetlink socket!  Some functionality won't be available!\n");
      return;
    }

  bzero((char *)&rtnl_data, sizeof(rtnl_data));

  rtnl_data.nl_family = AF_NETLINK;
  rtnl_data.nl_groups = RTMGRP_LINK;

  if (bind(rtnl_sock, (struct sockaddr *)&rtnl_data, sizeof(rtnl_data)) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't bind netlink socket!  Some functionality won't be available!\n");
      close(rtnl_sock);
      return;
    }
}

/********************************************************
 *
 * Do whatever is needed to shutdown the netlink socket that we set up.
 *
 ********************************************************/
void cardif_linux_rtnetlink_cleanup()
{
  debug_printf(DEBUG_INT, "Called cardif_linux_rtnetlink_cleanup()!\n");

  // Close the rtnetlink socket.
  close(rtnl_sock);
}

/********************************************************
 *
 * Decode data, and populate it in our SSID structure.
 *
 ********************************************************/
void cardif_linux_rtnetlink_reap(struct interface_data *intdata,
				 u_char *bptr, u_char *bptrend)
{
  u_char *tptr, essid[IW_ESSID_MAX_SIZE+1], custom[256], *temp=NULL;
  u_char *temp_rsn = NULL, *temp_wpa = NULL;
  unsigned char mac[6];
  unsigned char stale_mac[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  unsigned char zeromac[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  //  unsigned char wpa_match[3] = {0x00,0x50,0xf2};
  struct iw_event *iwe=NULL;
  unsigned char flags=0;
  u_char *wpa_ie=NULL, *rsn_ie=NULL, wpa_ie_len=0, rsn_ie_len=0;
  unsigned int freq;
  u_char temp_rsn_len = 0, temp_wpa_len = 0;

  tptr = bptr;
  bzero(mac, 6);
  bzero(stale_mac, 6);

  debug_printf(DEBUG_INT, "Reaped Data :\n");
  debug_hex_dump(DEBUG_INT, bptr, (bptrend - bptr));

  while (tptr < bptrend)
    {
      flags = 0;

      while ((tptr < bptrend) && (memcmp(stale_mac, mac, 6) == 0))
	{
	  iwe = (struct iw_event *)tptr;
      
	  switch (iwe->cmd)
	    {
	    case SIOCGIWAP:
	      memcpy(mac, iwe->u.ap_addr.sa_data, 6);
	      debug_printf(DEBUG_INT, "AP MAC : ");
	      debug_hex_printf(DEBUG_INT, mac, 6);
	      tptr += iwe->len;
	      break;
	      
	    case SIOCGIWFREQ:
	      memcpy(&freq, &iwe->u.freq, iwe->len);	      
	      freq = freq/100;
	      tptr += iwe->len;
	      break;
	      
	    case SIOCGIWMODE:
	      if (iwe->u.mode == 3) 
		{
		  SET_FLAG(flags, MASTER);
		} else {
		  UNSET_FLAG(flags, MASTER);
		}
	      
	      tptr += iwe->len;
	      break;
	      
	    case SIOCGIWESSID:
	      bzero((char *)&essid, IW_ESSID_MAX_SIZE+1);

	      memcpy((char *)&essid, (char *)&tptr[12], iwe->u.essid.length);
	      essid[iwe->u.essid.length] = '\0';
	      
	      tptr += iwe->len;
	      break;

	    case SIOCGIWNAME:
	      debug_printf(DEBUG_INT, "Protocol : %s\n", iwe->u.name);

	      tptr += iwe->len;
	      break;

#if WIRELESS_EXT > 17
	    case IWEVASSOCREQIE:
	      debug_printf(DEBUG_INT, "IWEVASSOCREQIE\n");
	      break;
	      
	    case IWEVASSOCRESPIE:
	      debug_printf(DEBUG_INT, "IWEVASSOCRESPIE\n");
	      tptr += iwe->len;
	      break;
	      
	    case IWEVGENIE:
	      // XXX Double check this!  We will probably get both RSN and WPA
	      // IEs concatenated together.  We need to check the whole thing
	      // in order to determine which is the best choice.
	      temp = (char *)malloc(iwe->u.data.length);
       	      memcpy(temp, &tptr[12], iwe->u.data.length);

	      if ((temp[0] == WPA_EID)) // && (memcmp(wpa_match, &temp[2], 3) == 0))
		{
		  SET_FLAG(flags, WPA_IE);
		  debug_printf(DEBUG_INT, "AP appears to support WPA!\n");
		  wpa_parse_ie(temp);
		  temp_rsn_len = temp_wpa_len = iwe->u.data.length;
		}

	      if (temp[0] == WPA2_EID)
		{
		  SET_FLAG(flags, RSN_IE);
		  debug_printf(DEBUG_INT, "AP appears to support WPA2/802.11i!\n");
		  debug_printf(DEBUG_NORMAL, "IE : ");
		  debug_hex_printf(DEBUG_NORMAL, temp, iwe->u.data.length);
		  wpa2_parse_ie(temp);
		  temp_rsn_len = temp_wpa_len = iwe->u.data.length;
		}

	      tptr += iwe->len;
	      break;
	      
	    case IWEVMICHAELMICFAILURE:
	      debug_printf(DEBUG_INT, "IWEVMICHAELMICFAILURE\n");
	      tptr += iwe->len;
	      break;
	      
	    case IWEVPMKIDCAND:
	      debug_printf(DEBUG_INT, "IWEVPMKIDCAND\n");
	      tptr += iwe->len;
	      break;
#endif
	      
	    case SIOCGIWENCODE:
	      if (iwe->u.data.flags & IW_ENCODE_DISABLED)
		{
		  UNSET_FLAG(flags, WEP);
		}
	      else
		{
		  SET_FLAG(flags, WEP);
		}
	      
	      tptr += iwe->len;
	      break;
	      
	    case SIOCGIWRATE:
	      tptr += iwe->len;
	      break;
	      
	    case IWEVQUAL:
	      tptr += iwe->len;
	      break;
	      
	    case IWEVCUSTOM:
	      bzero(custom, 256);
	      memcpy(&custom, &tptr[12], (iwe->len - 12));
	      debug_printf(DEBUG_INT, "(Custom) Value : %s\n", custom);

	      if (strncmp("wpa_ie=", custom, 7) == 0)
		{
		  SET_FLAG(flags, WPA_IE);
		  debug_printf(DEBUG_INT, "AP appears to support WPA!\n");
		  temp = (char *)malloc((iwe->len - 12)/2);
		  process_hex(&custom[7], (iwe->len - 12-7), temp);
		  wpa_parse_ie(temp);
		  
		  temp_wpa = temp;
		  temp_wpa_len = ((iwe->len - 12)/2);

		  // Set temp to NULL, so we can know if we are using an IE 
		  // from WE18, or not.
		  temp = NULL;
		}

	      if (strncmp("rsn_ie=", custom, 7) == 0)
		{
		  SET_FLAG(flags, RSN_IE);
		  debug_printf(DEBUG_INT, "AP appears to support WPA2/802.11i!\n");
		  temp = (char *)malloc((iwe->len - 12)/2);
		  process_hex(&custom[7], (iwe->len - 12-7), temp);
		  wpa2_parse_ie(temp);
		  
		  temp_rsn = temp;
		  temp_rsn_len = ((iwe->len - 12)/2);

		  temp = NULL;
		}
	      
	      tptr += iwe->len;
	      break;

	    default:
	      debug_printf(DEBUG_INT, "Unknown event (%04X)\n", iwe->cmd);
	      tptr += iwe->len;
	    }
	}

      // Check and see if we have a new set.
      if (memcmp(stale_mac, mac, 6) != 0)
	{
	  // If the stale MAC is all 0s, then this is our first time, and
	  // we should ignore things.
	  if (memcmp(zeromac, stale_mac, 6) != 0)
	    {
	      if (intdata->cur_essid == NULL)
		{
		  debug_printf(DEBUG_INT, "intdata->cur_essid is NULL!  "
			       "Setting!\n");
		  intdata->cur_essid = (char *)malloc(100);
		  if (intdata->cur_essid == NULL)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory!"
				   "\n");
		      return;
		    }
		  bzero(intdata->cur_essid, 100);

		  cardif_GetSSID(intdata, intdata->cur_essid);
		  
		  if (intdata->cur_essid[0] == 0x00)
		    {
		      debug_printf(DEBUG_NORMAL, "No valid SSID returned from "
				   "wireless card!!!\n");
		    } else {
		      debug_printf(DEBUG_INT, "intdata->cur_essid = %s\n",
				   intdata->cur_essid);
		    }
		}

	      // We have a new set.
	      debug_printf(DEBUG_INT, "ESSID : %s   Flags : %02X (", essid, 
			   flags);

	      if (flags & MASTER) debug_printf_nl(DEBUG_INT, "Master, ");
	      if (flags & WEP) debug_printf_nl(DEBUG_INT, "WEP, ");
	      if (flags & WPA_IE) debug_printf_nl(DEBUG_INT, "WPA IE, ");
	      if (flags & RSN_IE) debug_printf_nl(DEBUG_INT, "RSN IE, ");
	      
	      debug_printf_nl(DEBUG_INT, ")\n");
	      
	      
	      if (flags & WPA_IE)
		{
		  if (temp == NULL)
		    {
		      wpa_ie = temp_wpa;
		    } else {
		      wpa_ie = temp;
		    }
		  wpa_ie_len = temp_wpa_len;
		}

	      if (flags & RSN_IE)
		{
		  if (temp == NULL)
		    {
		      rsn_ie = temp_rsn;
		    } else {
		      rsn_ie = temp;
		    }
		  rsn_ie_len = temp_rsn_len;
		}
	    }

	  if ((temp != NULL) && ((!(flags & WPA_IE))) && (!(flags & RSN_IE)))
	    {
	      debug_printf(DEBUG_NORMAL, "There seems to be a WPA or RSN IE, "
			   "but the WPA_IE or RSN_IE flag(s) aren't set!?!\n");
	    }

	  if (strcmp(essid, intdata->cur_essid) == 0)
	    {
	      debug_printf(DEBUG_INT, "Adding this SSID data!\n");
	      config_ssids_add(essid, flags, wpa_ie, wpa_ie_len, rsn_ie, 
			       rsn_ie_len, freq, stale_mac);
	    }
	  
	  bzero(&custom, 256);
	  if (temp != NULL)
	    {
	      free(temp);
	      temp = NULL;
	    }
	}
      memcpy(stale_mac, mac, 6);
    }

  // If the stale MAC is all 0s, then this is our first time, and
  // we should ignore things.
  if (memcmp(zeromac, stale_mac, 6) != 0)
    {
      // We have a new set.
      debug_printf(DEBUG_INT, "ESSID : %s   Flags : %02X (", essid, flags);

      if (intdata->cur_essid == NULL)
	{
	  debug_printf(DEBUG_INT, "intdata->cur_essid is NULL!  Setting!\n");
	  cardif_GetSSID(intdata, intdata->cur_essid);
	}

      if (flags & MASTER) debug_printf_nl(DEBUG_INT, "Master, ");
      if (flags & WEP) debug_printf_nl(DEBUG_INT, "WEP, ");
      if (flags & WPA_IE) debug_printf_nl(DEBUG_INT, "WPA IE, ");
      if (flags & RSN_IE) debug_printf_nl(DEBUG_INT, "RSN IE, ");
      debug_printf_nl(DEBUG_INT, ")\n");

      if (flags & WPA_IE)
	{
	  if (temp == NULL)
	    {
	      wpa_ie = temp_wpa;
	    } else {
	      wpa_ie = temp;
	    }
	  wpa_ie_len = temp_wpa_len;
	}
      
      if (flags & RSN_IE)
	{
	  if (temp == NULL)
	    {
	      rsn_ie = temp_rsn;
	    } else {
	      rsn_ie = temp;
	    }
	  rsn_ie_len = temp_rsn_len;
	}      
    }

  if (strcmp(essid, intdata->cur_essid) == 0)
    {
      debug_printf(DEBUG_INT, "Adding this SSID data!\n");
      config_ssids_add(essid, flags, wpa_ie, wpa_ie_len, rsn_ie, 
		       rsn_ie_len, freq, mac);
    }
  
  bzero(&custom, 256);
  if (temp != NULL)
    {
      free(temp);
      temp = NULL;
    }
}



/*******************************************************
 *
 * Check to see if we have data in the returned scan buffer, even if 
 * we didn't get a scan complete event.  (Some cards may not send the
 * scan complete event.)
 *
 *******************************************************/
u_char cardif_linux_rtnetlink_check_nets(struct interface_data *idata)
{
  struct lin_sock_data *sockData;
  struct iwreq iwr;
  char buffer[8192];

  if (!(idata->flags & SCANNING)) return XENONE;

  debug_printf(DEBUG_INT, "Checking for returned SSID information....\n");

  sockData = idata->sockData;

  iwr.u.data.pointer = buffer;
  iwr.u.data.flags = 0;
  iwr.u.data.length = sizeof(buffer);

  strcpy(iwr.ifr_name, idata->intName);
  
  if (ioctl(sockData->sockInt, SIOCGIWSCAN, &iwr) < 0)
    {
      debug_printf(DEBUG_INT, "No data available!\n");
      return XENONE;
    }

  // Then harvest the data.
  debug_printf(DEBUG_INT, "Reaping data. (Size : %d)\n", iwr.u.data.length);
  cardif_linux_rtnetlink_reap(idata, (char *)buffer, 
			      ((char *)buffer + iwr.u.data.length));

  UNSET_FLAG(idata->flags, SCANNING);

  return XDATA;
}

/*******************************************************
 *
 * Check to see if we have any events, and act on them if we do.
 * intdata passed in should be the starting pointer in the list!
 *
 *******************************************************/
u_char cardif_linux_rtnetlink_check_event(struct interface_data *idata)
{
  int rtnl_data_size, remain, length;
  char buf[8192];
  struct nlmsghdr *nlhead;

  // Grab the next message off the rtnetlink socket.
  rtnl_data_size = sizeof(rtnl_data);
  if (rtnl_sock < 0)
    {
      debug_printf(DEBUG_NORMAL, "RTnetlink socket not available!\n");
      return XENOSOCK;
    }

  remain = recvfrom(rtnl_sock, buf, sizeof(buf), MSG_DONTWAIT,
			(struct sockaddr *)&rtnl_data, &rtnl_data_size);
  if (remain >= 0)
    {
      // We need a pointer to the buffer to work with.
      nlhead = (struct nlmsghdr *)buf;

      // There may be more than one message in the packet.  So, loop through!
      while (remain >= sizeof(struct nlmsghdr))
	{
	  // Make sure we have enough data for a real message.
	  if ((nlhead->nlmsg_len > remain) || 
	      ((nlhead->nlmsg_len - sizeof(struct nlmsghdr)) < 0))
	    {
	      debug_printf(DEBUG_NORMAL, "Invalid netlink message!\n");
	      break;
	    }
	  
	  // See what kind of message it is.  (New or deleted link?)
	  switch (nlhead->nlmsg_type)
	    {
	    case RTM_NEWLINK:
	      debug_printf(DEBUG_INT, "Got an RTM_NEWLINK!\n");
	      cardif_linux_rtnetlink_do_link(idata, nlhead,
					     nlhead->nlmsg_len, INT_NEW);
	      break;
	      
	    case RTM_DELLINK:
	      debug_printf(DEBUG_INT, "Got an RTM_DELLINK!\n");
	      cardif_linux_rtnetlink_do_link(idata, nlhead, 
					     nlhead->nlmsg_len, INT_DEL);
	      break;
	    }

	  // Find the aligned length of the message, so we can skip
	  // to the next message.
	  length = NLMSG_ALIGN(nlhead->nlmsg_len);

	  remain -= length;
	  
	  nlhead = (struct nlmsghdr *) ((char *)nlhead + length);
	}

      // If we have anything left, then there may be a problem.  So, report
      // the we may have a problem.
      if (remain > 0)
	{
	  debug_printf(DEBUG_NORMAL, "Extra bytes at the end of the netlink message.\n");
	}
    }

  // There may be situations where the wireless card driver can scan,
  // but doesn't provide an event when the scan has completed.  So,
  // we need to check if we have data, even if we don't have an event.
  return cardif_linux_rtnetlink_check_nets(idata);
}

/************************************************************
 *
 *  We got an RTM_NEWLINK or RTM_DELLINK message, so process it, and 
 *  decide how to proceed.
 *
 ************************************************************/
void cardif_linux_rtnetlink_do_link(struct interface_data *intdata,
				    struct nlmsghdr *msg, int len, int type)
{
  struct ifinfomsg *ifinfo;
  int nlmsg_len, rtalen, rtlen;
  struct rtattr *rtattr;

  if (len < sizeof(struct ifinfomsg))
    {
      debug_printf(DEBUG_NORMAL, "Netlink message too short!\n");
      return;
    }

  // Get the actual message from the block.
  ifinfo = NLMSG_DATA(msg);

  // Find out how big the message is.
  nlmsg_len = NLMSG_ALIGN(sizeof(struct ifinfomsg));

  if ((msg->nlmsg_len - nlmsg_len) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Message inside newlink isn't valid!\n");
      return;
    }

  rtattr = (struct rtattr *)(((char *)ifinfo) + nlmsg_len);

  rtalen = RTA_ALIGN(sizeof(struct rtattr));
  
  // Validate the attribute we have, and determine if it is for wireless,
  // or wired.
  while (RTA_OK(rtattr, (msg->nlmsg_len - nlmsg_len)))
    {
      switch (rtattr->rta_type)
	{
	case IFLA_IFNAME:
	  // This is a non-wireless event. (Ignore it.)
	  break;

	case IFLA_WIRELESS:
	  // This is a wireless event.
	  cardif_linux_rtnetlink_ifla_wireless(intdata, ifinfo->ifi_index,
					       ((char *) rtattr)+rtalen,
					       rtattr->rta_len - rtalen);
	  break;
	}

      rtlen = msg->nlmsg_len - nlmsg_len;

      // Get the next attribute
      rtattr = RTA_NEXT(rtattr, rtlen);
    }
}

/***********************************************************
 *
 * Check the string that identifies the custom event that we got. And
 * act on it.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_check_custom(struct interface_data *intdata,
					 char *str)
{
  if (strncmp(str, "MLME-MICHAELMICFAILURE.indication", 33) == 0)
    {
      intdata->statemachine->MICfailures++;
      debug_printf(DEBUG_NORMAL, "MIC failure #%d!\n",
		   intdata->statemachine->MICfailures);
      
      if (strstr(str, " unicast ") != NULL)
	{
	  // The attempted attack was probably against the unicast key.
	  debug_printf(DEBUG_NORMAL, "MIC failure on unicast key!\n");
	  eapol_key_type254_request_new_key(intdata, 1);
	  intdata->send_size = 0;
	} else {
	  // The attempted attack was probably against the group key.
	  debug_printf(DEBUG_NORMAL, "MIC failure on group key!\n");
	  eapol_key_type254_request_new_key(intdata, 0);
	  intdata->send_size = 0;
	}

      if (intdata->statemachine->MICfailures >= 2)
	{
	  // The WPA/802.11i standard requires we assert countermeasures 
	  // for 60 seconds.
	  if (timer_check_existing(COUNTERMEASURE_TIMER))
	    {
	      debug_printf(DEBUG_NORMAL, "For some reason, we already have "
			   "a countermeasure timer in the queue!  Resetting "
			   "the timer!\n");
	      timer_reset_timer_count(COUNTERMEASURE_TIMER, 
				      MIC_COUNTERMEASURE_TIMEOUT);
	    } else {
	      debug_printf(DEBUG_NORMAL, "Enabling MIC countermeasures!\n");
	      timer_add_timer(COUNTERMEASURE_TIMER, MIC_COUNTERMEASURE_TIMEOUT,
			      NULL, &mic_disable_countermeasures);
	    }
	  cardif_countermeasures(intdata, TRUE);
	}
    }
}

/***********************************************************
 *
 * Check to see if we have become disassociated before the rekey_prob_timer
 * (found in profile.h) reaches 0.  If we have, it may indicate that we
 * have a card driver that resets the card on a key set.  This should only be
 * a problem with WEP and older card drivers.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_check_key_prob(struct interface_data *idata)
{
  if (timer_check_existing(REKEY_PROB_TIMER))
    {
      if (config_get_display_friendly_warnings())
	{
	  debug_printf(DEBUG_NORMAL, "** WARNING! ** You were disassocated "
		       "within a short time of setting a key!\nThis usually "
		       "means there is a problem with the card driver.\n"
		       "Please e-mail the developers for more information on "
		       "what this means. Be sure to include the type of card, "
		       "driver in use, and driver version number.\n");
	}
    }
}

/***********************************************************
 *
 * Given that we are now unassociated, we should attempt to associate again,
 * if there isn't already an association timer running.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_do_unassociated(struct interface_data *intdata)
{
  cardif_linux_rtnetlink_check_key_prob(intdata);
  cardif_roam(intdata);
  statemachine_reinit(intdata);

  if (!timer_check_existing(ASSOCIATION_TIMER))
    {
      debug_printf(DEBUG_INT, "No currently active association timer.  "
		   "Attempting to associate again!\n");
      cardif_associate(intdata, intdata->cur_essid);
    }
}

/***********************************************************
 *
 * If the authentication timer has expired, then do whatever is needed to
 * try to get another authentication going.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_authentication_timeout(struct interface_data 
						   *intdata)
{
  // If we get here we need to send a deauth, and see if we can't associate
  // again.

  // Kill our left-over authentication timer bits.
  timer_cancel(AUTHENTICATION_TIMER);

  debug_printf(DEBUG_NORMAL, "Authentication timer expired!  Attempting to "
	       "reassociate!\n");

  // So, send a deauth.  (XXX Double check this!  Should it be something else?)
  cardif_disassociate(intdata, DISASSOC_UNSPECIFIED);

  // Start up our association timer.
  if (timer_check_existing(ASSOCIATION_TIMER))
    {
      timer_reset_timer_count(ASSOCIATION_TIMER, ASSOCIATION_TIMEOUT);
    } else {
      timer_add_timer(ASSOCIATION_TIMER, ASSOCIATION_TIMEOUT, NULL,
		      &cardif_association_timeout_expired);
    }
}

/***********************************************************
 *
 * This is the timer that is set when we have associated, and we are
 * waiting for an authentication to complete.  If it expires, we need to
 * attempt to jump-start the authentication again.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_set_authentication_timer()
{
  if (timer_check_existing(AUTHENTICATION_TIMER))
    {
      timer_reset_timer_count(AUTHENTICATION_TIMER, AUTHENTICATION_TIMEOUT);
    } else {
      timer_add_timer(AUTHENTICATION_TIMER, AUTHENTICATION_TIMEOUT, NULL,
		      &cardif_linux_rtnetlink_authentication_timeout);
    }
}

/***********************************************************
 *
 * Check the MAC that we were given.  If it is all 0s, 4s, or Fs then the
 * event is a disassociation.  If it isn't then it is an association.
 *
 ***********************************************************/
int cardif_linux_rtnetlink_validate(struct interface_data *idata, u_char *mac)
{
  char newmac[6];

  // Is it a disassociation?
  memset(newmac, 0x00, 6);
  if (memcmp(newmac, mac, 6) == 0)
    {
      UNSET_FLAG(idata->flags, ASSOCIATED);
      debug_printf(DEBUG_INT, "Switched to UNASSOCIATED mode!\n");
      cardif_linux_rtnetlink_check_key_prob(idata);
      cardif_roam(idata);
      statemachine_reinit(idata);
      return FALSE;
    }

  memset(newmac, 0x44, 6);
  if (memcmp(newmac, mac, 6) == 0)
    {
      UNSET_FLAG(idata->flags, ASSOCIATED);
      debug_printf(DEBUG_INT, "Switched to UNASSOCIATED mode!\n");
      cardif_linux_rtnetlink_check_key_prob(idata);
      cardif_roam(idata);
      statemachine_reinit(idata);
      return FALSE;
    }

  memset(newmac, 0xff, 6);
  if (memcmp(newmac, mac, 6) == 0)
    {
      UNSET_FLAG(idata->flags, ASSOCIATED);
      debug_printf(DEBUG_INT, "Switched to UNASSOCIATED mode!\n");
      cardif_linux_rtnetlink_check_key_prob(idata);
      cardif_roam(idata);
      statemachine_reinit(idata);
      return FALSE;
    }
  
  // Otherwise, it was an association
  //Now to check to see if it was roaming ww
  if ((idata->flags & ASSOCIATED) & (memcmp(mac, idata->dest_mac, 6) != 0))
    {
      debug_printf(DEBUG_INT, "Roaming to new access point on the same SSID.\n");
      // We don't need to check the keying problem timer here, since if we
      // roamed to a different AP we can assume that the key should change.
      // So, checking for the keying problem would result in a false positive.
      cardif_roam(idata);
      statemachine_reinit(idata);
      cardif_linux_rtnetlink_set_authentication_timer();
      return FALSE;
    }


  SET_FLAG(idata->flags, ASSOCIATED);
  debug_printf(DEBUG_INT, "Switched to associated mode.\n");

  // If we have an association timer ticking, cancel it.
  if (timer_check_existing(ASSOCIATION_TIMER))
    {
      debug_printf(DEBUG_EVERYTHING, "Canceling association timeout!\n");
      timer_cancel(ASSOCIATION_TIMER);
    }

  // DKS -- could be better to call cardif_associate here, but can't -- don't know the ssid. Have
  // to call something, in order to handle clearing keys from cards that can immediately
  // reassociate to a new AP, but cannot do 1x until encryption is turned off. (They try to do
  // 1x, only they encrypt the frames with the key shared with their old AP. This is true
  // at least of prism2 cards, and probably others.)
  cardif_roam(idata);
  statemachine_reinit(idata);
  cardif_linux_rtnetlink_set_authentication_timer();

  return TRUE;
}

/***********************************************************
 *
 * Process wireless events.
 *
 ***********************************************************/
void cardif_linux_rtnetlink_ifla_wireless(struct interface_data *idata, 
					  int ifindex, char *data, int len)
{
  struct iw_event *iwe;
  char *pos, *end, *custom, buf[1500], essid[IW_ESSID_MAX_SIZE+1], mac[6];
  char rescan=0;
  struct config_network *network_data;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  if ((data == NULL) || (len == 0))
    {
      debug_printf(DEBUG_NORMAL, "No data available in event!\n");
      return;
    }

  pos = data;
  end = data+len;

  if (((ifindex != idata->intIndex) &&
       (idata->driver_in_use != DRIVER_HOSTAP)) && 
      (!(idata->flags & ONEDOWN)))
    {
      debug_printf(DEBUG_INT, "Got a wireless event!  Interface index is %d, "
		   "we are using index %d!\n", ifindex, idata->intIndex);
      debug_printf(DEBUG_INT, "Ignoring!\n");
      return;
    }

  if ((idata->driver_in_use == DRIVER_HOSTAP) || (idata->flags & ONEDOWN))
    {
      if ((ifindex < (idata->intIndex - 1)) || (ifindex > (idata->intIndex)))
	{
	  debug_printf(DEBUG_INT, "Got a wireless event! Interface index is "
		       "%d, we are using indexs %d & %d!\n", ifindex,
		       ifindex-1, idata->intIndex);
	  return;
	}
    }

  while (pos + IW_EV_LCP_LEN <= end)
    {
      iwe = (struct iw_event *) pos;
      debug_printf(DEBUG_INT, "Wireless event: cmd=0x%x len=%d\n", iwe->cmd, 
		   iwe->len);

      if (iwe->len <= IW_EV_LCP_LEN)
	{
	  debug_printf(DEBUG_NORMAL, "Not enough data in wireless event!\n");
	  return;
	}

      switch (iwe->cmd)
	{
	case SIOCGIWAP:
	  memcpy(mac, iwe->u.ap_addr.sa_data, 6);
	  debug_printf(DEBUG_INT, "AP MAC : ");
	  debug_hex_printf(DEBUG_INT, mac, 6);
	  if (cardif_linux_rtnetlink_validate(idata, (u_char *)&mac) == 1)
	    {
	      // We have changed to associated mode.  Populate the destination
	      // MAC with the BSSID, as long as we are in auto mode.
	      
	      if (config_get_destination() == DEST_AUTO)
		{
		  memcmp(idata->dest_mac, mac, 6);
		}
	    }
	  break;

	case SIOCGIWSCAN:
	  // We have been told that the scan is complete.
	  debug_printf(DEBUG_INT, "Wireless scan complete!\n");
	  cardif_linux_rtnetlink_check_nets(idata);
	  break;

#ifdef IWEVTXDROP
	  // This is mostly for the gee-whiz factor.
	case IWEVTXDROP:
	  debug_printf(DEBUG_INT, "Wireless TX Drop\n");
	  break;
#endif

#if WIRELESS_EXT > 17
	case IWEVASSOCREQIE:
	  debug_printf(DEBUG_INT, "IWEVASSOCREQIE\n");
	  break;

	case IWEVASSOCRESPIE:
	  debug_printf(DEBUG_INT, "IWEVASSOCRESPIE\n");
	  break;

	case IWEVGENIE:
	  debug_printf(DEBUG_INT, "IWEVGENIE\n");
	  break;

	case IWEVMICHAELMICFAILURE:
	  debug_printf(DEBUG_INT, "IWEVMICHAELMICFAILURE\n");
	  break;

	case IWEVPMKIDCAND:
	  debug_printf(DEBUG_INT, "IWEVPMKIDCAND\n");
	  break;
#endif

	case IWEVCUSTOM:
	  custom = pos + IW_EV_POINT_LEN;
	  bzero(buf, 100);
	  if (custom + iwe->u.data.length > end)
	    {
	      debug_printf(DEBUG_INT, "IWEVCUSTOM called with no data!\n");
	      return;
	    }

	  memcpy(buf, custom, iwe->u.data.length);
	  buf[iwe->u.data.length] = '\0';
	  debug_printf(DEBUG_NORMAL, "Custom Data : \n");
	  debug_hex_dump(DEBUG_NORMAL, buf, iwe->u.data.length);
	  cardif_linux_rtnetlink_check_custom(idata, buf);
	  break;

	case SIOCSIWESSID:
	  bzero((char *)&essid, IW_ESSID_MAX_SIZE+1);

	  custom = pos + IW_EV_POINT_LEN;
	  memcpy((char *)&essid, custom, iwe->u.essid.length);
	  essid[iwe->u.essid.length] = '\0';
	  
	  debug_printf(DEBUG_INT, "ESSID set .. name : %s\n", essid);
	  UNSET_FLAG(idata->flags, DONT_KEY);
	  if (config_build(essid) == FALSE)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't build a valid configuration"
			   " for ESSID %s!\n", essid);
	    }

	  // We changed ssids, so record the new one.
	  if (idata->cur_essid != NULL)
	    {
	      free(idata->cur_essid);

	      idata->cur_essid = (char *)malloc(strlen(essid)+1);
	      if (idata->cur_essid != NULL)
		{
		  strcpy(idata->cur_essid, essid);
		}
	    }

	  eap_clear_active_method(network_data->activemethod);

	  // If we changed ssids, we need to rescan.
	  if (idata->flags & DONTSCAN)
	    {
	      debug_printf(DEBUG_INT, "We were instructed not to scan..\n");
	      UNSET_FLAG(idata->flags, DONTSCAN);
	    } else {
	      rescan = 1;
	    }
	  break;

	case SIOCSIWENCODE:
	  debug_printf(DEBUG_INT, "Encryption key set\n");
	  break;

	}

      pos += iwe->len;
    }
  
  if (rescan == 1)
    {
      cardif_do_wireless_scan(idata);
      cardif_reassociate(idata, 1);
    }
}

#endif
