/*******************************************************************
 * Linux wireless extensions interface.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: cardif_linux_wext.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_linux_wext.c,v 1.37 2005/08/20 19:06:54 chessing Exp $
 * $Date: 2005/08/20 19:06:54 $
 * $Log: cardif_linux_wext.c,v $
 * Revision 1.37  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.36  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#ifdef LINUX_FRAMER

#include <string.h>
#include <strings.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <netinet/in.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/if.h>
#include <linux/if_packet.h>
#include <linux/netlink.h>
#include <linux/wireless.h>
#include <linux/if_arp.h>

#include "config.h"
#include "config_ssid.h"
#include "profile.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "wpa.h"
#include "wpa2.h"
#include "cardif/cardif.h"
#include "cardif/linux/cardif_linux.h"
#include "cardif/linux/cardif_linux_wext.h"
#include "cardif/linux/cardif_linux_rtnetlink.h"

// Old versions of wireless.h may not have this defined.
#ifndef IW_ENCODE_TEMP
#define IW_ENCODE_TEMP            0x0400
#endif

/**************************************************************
 *
 * Tell the wireless card to start scanning for wireless networks.
 *
 **************************************************************/
int cardif_linux_wext_scan(struct interface_data *thisint)
{
  struct lin_sock_data *sockData;
  struct iwreq iwr;

  if (!(thisint->flags & IS_WIRELESS))
    {
      debug_printf(DEBUG_INT, "%s is not a wireless interface!\n", 
		   thisint->intName);
      return XENOWIRELESS;
    }

  sockData = thisint->sockData;

  if (sockData->sockInt <= 0)
    return XENOSOCK;

  debug_printf(DEBUG_INT, "Issuing scan request for interface %s!\n",
	       thisint->intName);

  // XXX Probably want to update this to support extended scanning.
  iwr.u.data.flags = IW_SCAN_DEFAULT;
  iwr.u.data.length = 0;
  iwr.u.data.pointer = NULL;

  strcpy((char *)&iwr.ifr_name, thisint->intName);

  if (ioctl(sockData->sockInt, SIOCSIWSCAN, &iwr) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error with SCAN ioctl!  (Perhaps your card "
		   "doesn't support scanning, or isn't up?)\n");

      return -1;
    }

  while ((cardif_linux_rtnetlink_check_event(thisint) != XDATA) &&
	 (thisint->flags & SCANNING))
    {
      debug_printf(DEBUG_INT, "Waiting for SSID information...\n");
      usleep(250000);
    }

  return XENONE;
}

/*************************************************************
 *
 * Set all of the keys to 0s.
 *
 *************************************************************/
void cardif_linux_wext_set_zero_keys(struct interface_data *thisint)
{
  char zerokey[13];
  char keylen = 13;

  debug_printf(DEBUG_INT, "Setting keys to zeros!\n");

  bzero(zerokey, 13);

  // We set the key index to 0x80, to force key 0 to be set to all 0s,
  // and to have key 0 be set as the default transmit key.
  cardif_set_wep_key(thisint, (char *)&zerokey, keylen, 0x80);
  cardif_set_wep_key(thisint, (char *)&zerokey, keylen, 0x01);
  cardif_set_wep_key(thisint, (char *)&zerokey, keylen, 0x02);
  cardif_set_wep_key(thisint, (char *)&zerokey, keylen, 0x03);
}

/**************************************************************
 *
 * If we have detected, or forced this interface to reset keys, then
 * we need to reset them.  Otherwise, we will just ignore the fact that
 * we changed APs, and return.
 *
 **************************************************************/
void cardif_linux_wext_zero_keys(struct interface_data *thisint)
{
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  if (network_data->wireless_ctrl == CTL_NO) 
    {
      debug_printf(DEBUG_INT, "Config file has instructed us not to reset the key!  Roaming may not work!!!\n");
      return;
    }

  if (thisint->flags & ROAMED)
    {
      return;
    }

  SET_FLAG(thisint->flags, ROAMED);

  cardif_linux_wext_set_zero_keys(thisint);
  cardif_linux_wext_enc_open(thisint);
}

/**************************************************************
 *
 * Disable encryption on the wireless card.  This is used in cases
 * where we roam to a different AP and the card needs to have WEP
 * disabled.
 *
 **************************************************************/
int cardif_linux_wext_enc_disable(struct interface_data *thisint)
{
  int rc = 0;
  int skfd;
  struct iwreq wrq;
  struct config_network *network_data;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  if (network_data->wireless_ctrl == CTL_NO) 
    {
      debug_printf(DEBUG_INT, "Config file has instructed us not to reset the"
		   " key!  Roaming may not work!!!\n");
      return -1;
    }

  bzero((struct iwreq *)&wrq, sizeof(struct iwreq));

  skfd = socket(AF_INET, SOCK_DGRAM, 0);
  if (skfd < 0)
    return -1;

  strncpy(wrq.ifr_name, thisint->intName, IFNAMSIZ);

  // We got some data, so see if we have encryption or not.
  wrq.u.encoding.flags = IW_ENCODE_DISABLED;
  wrq.u.encoding.length = 0;
  wrq.u.encoding.pointer = (caddr_t)NULL;

  rc = ioctl(skfd, SIOCSIWENCODE, &wrq);
  if (rc < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't disable encryption!\n");
    } else {
      debug_printf(DEBUG_INT, "Encryption disabled!\n");
    }
 
  close(skfd);
  return rc;
}

int cardif_linux_wext_get_wpa2_ie(struct interface_data *thisint, 
				  char *iedata, int *ielen)
{
#if WIRELESS_EXT > 17
  // Should we use capabilities here?
  wpa2_gen_ie(thisint, iedata, ielen);

  debug_printf(DEBUG_INT, "Setting WPA2 IE : ");
  debug_hex_printf(DEBUG_INT, iedata, *ielen);
  debug_printf(DEBUG_INT, "\n");
#else
  debug_printf(DEBUG_NORMAL, "WPA2 isn't implemented in this version of the "
	       "wireless extensions!  Please upgrade to the latest version "
	       "of wireless extensions, or specify the driver to use with the"
	       " -D option!\n");

  iedata = NULL;
  *ielen = 0;
#endif
  return XENONE;
}

int cardif_linux_wext_get_wpa_ie(struct interface_data *thisint, 
				 char *iedata, int *ielen)
{
#if WIRELESS_EXT > 17
  // Should we use capabilities here?
  wpa_gen_ie(thisint, iedata);
  *ielen = 24;

  debug_printf(DEBUG_INT, "Setting WPA IE : ");
  debug_hex_printf(DEBUG_INT, iedata, *ielen);
  debug_printf(DEBUG_INT, "\n");
#else
  debug_printf(DEBUG_NORMAL, "WPA isn't implemented in this version of the "
	       "wireless extensions!  Please upgrade to the latest version "
	       "of wireless extensions, or specify the driver to use with the"
	       " -D option!\n");

  iedata = NULL;
  *ielen = 0;
#endif
  return XENONE;
}

/**************************************************************
 *
 * Set encryption on the wireless card to open.
 *
 **************************************************************/
int cardif_linux_wext_enc_open(struct interface_data *thisint)
{
  int rc = 0;
  int skfd;
  struct iwreq wrq;
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  if (network_data->wireless_ctrl == CTL_NO) 
    {
      debug_printf(DEBUG_INT, "Config file has instructed us not to reset the "
		   "key!  Roaming may not work!!!\n");
      return -1;
    }

  bzero((struct iwreq *)&wrq, sizeof(struct iwreq));

  skfd = socket(AF_INET, SOCK_DGRAM, 0);
  if (skfd < 0)
    return -1;

  strncpy(wrq.ifr_name, thisint->intName, IFNAMSIZ);

  // We got some data, so see if we have encryption or not.
  wrq.u.encoding.flags = IW_ENCODE_OPEN;
  wrq.u.encoding.length = 0;
  wrq.u.encoding.pointer = (caddr_t)NULL;

  rc = ioctl(skfd, SIOCSIWENCODE, &wrq);
  if (rc < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't disable encryption!\n");
    } else {
      debug_printf(DEBUG_INT, "Encryption set to Open!\n");
    }
 
  close(skfd);
  return rc;
}


/******************************************
 *
 * Set a WEP key.  Also, based on the index, we may change the transmit
 * key.
 *
 ******************************************/
int cardif_linux_wext_set_WEP_key(struct interface_data *thisint, u_char *key, 
				  int keylen, int index)
{
  int rc = 0;
  int skfd;
  struct iwreq wrq;
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  if (!(thisint->flags & IS_WIRELESS))
    {
      if ((cardif_int_is_wireless(thisint->intName) != TRUE) ||
	  (network_data->type == WIRED) ||
	  (network_data->wireless_ctrl == CTL_NO))
	{
	  debug_printf(DEBUG_NORMAL, "Interface isn't wireless, but an attempt"
		       " to set a key was made!\n");
	  return XENOWIRELESS;
	} else {
	  thisint->flags |= IS_WIRELESS;
	}
    }

  skfd = socket(AF_INET, SOCK_DGRAM, 0);
  if (skfd < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate socket!\n");
      return -1;
    }

  strncpy(wrq.ifr_name, thisint->intName, IFNAMSIZ);

  wrq.u.data.flags = ((index & 0x7f)+1);

  if (thisint->flags & DONT_USE_TEMP)
    wrq.u.data.flags |= IW_ENCODE_OPEN;
  else
    wrq.u.data.flags |= IW_ENCODE_OPEN | IW_ENCODE_TEMP;

  wrq.u.data.length = keylen;
  wrq.u.data.pointer = (caddr_t)key;

  if ((rc = ioctl(skfd, SIOCSIWENCODE, &wrq)) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Failed to set WEP key [%d], error %d : %s\n",
		   (index & 0x7f) + 1, errno, strerror(errno));

      rc = XENOKEYSUPPORT;
    } else {
      debug_printf(DEBUG_INT, "Successfully set WEP key [%d]\n",
		   (index & 0x7f)+1);

      if (index & 0x80)
	{
	  // This is a unicast key, use it for transmissions.
	  strncpy(wrq.ifr_name, thisint->intName, IFNAMSIZ);

	  wrq.u.data.flags = (((index & 0x7f) + 1) & IW_ENCODE_INDEX) | IW_ENCODE_NOKEY;
	  if (thisint->flags & DONT_USE_TEMP)
	    wrq.u.data.flags |= IW_ENCODE_OPEN;
	  else
	    wrq.u.data.flags |= IW_ENCODE_OPEN | IW_ENCODE_TEMP;

	  wrq.u.data.length = 0;
	  wrq.u.data.pointer = (caddr_t)NULL;

	  if (ioctl(skfd, SIOCSIWENCODE, &wrq) < 0)
	    {
	      debug_printf(DEBUG_NORMAL, "Failed to set the WEP transmit key ID [%d]\n", (index & 0x7f)+1);
	      rc = XENOKEYSUPPORT;
	    } else {
	      debug_printf(DEBUG_INT, "Successfully set the WEP transmit key [%d]\n", (index & 0x7f)+1);
	    }
	}  
    }
 
  close(skfd);
  return rc;
}

/**********************************************************
 *
 * Set the SSID of the wireless card.
 *
 **********************************************************/
int cardif_linux_wext_set_ssid(struct interface_data *thisint, char *ssid_name)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;
  char newssid[100];

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed in to %s!\n",
		   __FUNCTION__);
      return XEGENERROR;
    }

  if (ssid_name == NULL)
    {
      debug_printf(DEBUG_NORMAL, "A NULL ssid was passed in to %s! "
		   "Authentication may not work!\n", __FUNCTION__);
      return -1;
    }

  sockData = thisint->sockData;
 
  if (!(thisint->flags & IS_WIRELESS))
    {
      // We want to verify that the interface is in fact, not wireless, and
      // not that we are in a situation where the interface has just been 
      // down.
      if (!(thisint->flags & WAS_DOWN))
	{
	  return XENOWIRELESS;
	}
    } 

  // If we get here, and isWireless == FALSE, then we need to double
  // check that our interface is really not wireless.
  if (!(thisint->flags & IS_WIRELESS))
    {
      if (cardif_int_is_wireless(thisint->intName) == TRUE)
	{
	  thisint->flags |= IS_WIRELESS;
	} else {
	  thisint->flags &= (~IS_WIRELESS);
	}

      if (!(thisint->flags & IS_WIRELESS))
	{
	  thisint->flags &= (~WAS_DOWN);
	}
    }

  // Specify the interface name we are asking about.
  strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  bzero(newssid, 100);
  strcpy(newssid, ssid_name);

  iwr.u.essid.pointer = (caddr_t) newssid;
  iwr.u.essid.length = strlen(newssid)+1;
  iwr.u.essid.flags = 1;

  if (ioctl(sockData->sockInt, SIOCSIWESSID, &iwr) < 0) return XENOWIRELESS;

  thisint->flags &= (~WAS_DOWN);

  return XENONE;
}

/******************************************
 *
 * Set the Broadcast SSID (MAC address) of the AP we are connected to.
 *
 ******************************************/
int cardif_linux_wext_set_bssid(struct interface_data *intdata, u_char *bssid)
{
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  if ((intdata == NULL) || (bssid == NULL))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed to %s!\n", __FUNCTION__);
      return -1;
    }

  sockData = intdata->sockData;

  if (sockData == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid socket data!! (%s:%d)\n", 
		   __FUNCTION__, __LINE__);
      return XENOSOCK;
    }

  bzero(&wrq, sizeof(wrq));

  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);
  memcpy(&wrq.u.ap_addr.sa_data, bssid, 6);

  if (ioctl(sockData->sockInt, SIOCSIWAP, &wrq) < 0)
    {
      // If we couldn't set the BSSID, it isn't the end of the world.  The
      // driver just may not need it.
      //      debug_printf(DEBUG_NORMAL, "Error setting BSSID!  We may not associate/"
      //		   "authenticate correctly!\n");
      return XESOCKOP;
    }

  return XENONE;
}


/******************************************
 *
 * Get the Broadcast SSID (MAC address) of the Access Point we are connected 
 * to.  If this is not a wireless card, or the information is not available,
 * we should return an error.
 *
 ******************************************/
int cardif_linux_wext_get_bssid(struct interface_data *thisint, 
				char *bssid_dest)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;

  if ((thisint == NULL) || (bssid_dest == NULL))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed to %s!\n", __FUNCTION__);
      return -1;
    }

  // If we are a wired interface, don't bother.
  if (!TEST_FLAG(thisint->flags, IS_WIRELESS)) return XENONE;

  sockData = thisint->sockData;

  if (sockData == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid sockData in %s:%d\n", __FUNCTION__,
		   __LINE__);
      debug_printf(DEBUG_NORMAL, "FATAL ERROR!\n");
      exit(1);
    }

  // Specify the interface name we are asking about.
  strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  if (ioctl(sockData->sockInt, SIOCGIWAP, &iwr) < 0) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get MAC address for AP!\n");
      return XENOWIRELESS;
    }

  memcpy(bssid_dest, iwr.u.ap_addr.sa_data, 6);
  return XENONE;
}

/******************************************
 *
 * Ask the wireless card for the ESSID that we are currently connected to.  If
 * this is not a wireless card, or the information is not available, we should
 * return an error.
 *
 ******************************************/
int cardif_linux_wext_get_ssid(struct interface_data *thisint, char *ssid_name)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;
  char newssid[100];

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface structure!  (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return -1;
    }

  if (ssid_name == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid bucket for ssid_name!  (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return -1;
    }

  sockData = thisint->sockData;
 
  if (!(thisint->flags & IS_WIRELESS))
    {
      // We want to verify that the interface is in fact, not wireless, and
      // not that we are in a situation where the interface has just been 
      // down.
      if (!(thisint->flags & WAS_DOWN))
	{
	  debug_printf(DEBUG_NORMAL, "Interface appears to be down!  Cannot "
		       "determine the SSID!\n");
	  return XENOWIRELESS;
	}
    } 

  // If we get here, and isWireless == FALSE, then we need to double
  // check that our interface is really not wireless.
  if (!(thisint->flags & IS_WIRELESS))
    {
      if (cardif_int_is_wireless(thisint->intName) == TRUE)
	{
	  thisint->flags |= IS_WIRELESS;
	} else {
	  thisint->flags &= (~IS_WIRELESS);
	}

      if (!(thisint->flags & IS_WIRELESS))
	{
	  thisint->flags &= (~WAS_DOWN);
	}
    }

  // Specify the interface name we are asking about.
  strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  bzero(newssid, 100);
  iwr.u.essid.pointer = (caddr_t) newssid;
  iwr.u.essid.length = 100;
  iwr.u.essid.flags = 0;

  if (ioctl(sockData->sockInt, SIOCGIWESSID, &iwr) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get ESSID!\n");
      debug_printf(DEBUG_NORMAL, "Error (%d) : %s\n", errno, strerror(errno));
      return XENOWIRELESS;
    }

  thisint->flags &= (~WAS_DOWN);

  strcpy(ssid_name, newssid);

  return XENONE;
}

/***********************************************************************
 *
 *  This function is called when we roam, or disassociate.  It should
 *  reset the card to a state where it can associate with a new AP.
 *
 ***********************************************************************/
int cardif_linux_wext_roam(struct interface_data *intdata, int zero_keys)
{
  if (zero_keys == 0)
    {
      debug_printf(DEBUG_INT, "ROAMING: turning encryption off.\n");
      return cardif_linux_wext_enc_disable(intdata);
    } else if (zero_keys == 1)
      {
      debug_printf(DEBUG_INT, "ROAMING: zero keys.\n");
	cardif_linux_wext_zero_keys(intdata);
	return XENONE;
      }

  return XENOTHING_TO_DO;
}

#if WIRELESS_EXT > 17
int cardif_linux_wext_mlme(struct interface_data *thisint, uint16_t mlme_type,
			   uint16_t mlme_reason)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;
  struct iw_mlme iwm;

  sockData = thisint->sockData;

  // If we get here, and isWireless == FALSE, then we need to double
  // check that our interface is really not wireless.
  if (!(thisint->flags & IS_WIRELESS))
    {
      if (cardif_int_is_wireless(thisint->intName) == TRUE)
	{
	  thisint->flags |= IS_WIRELESS;
	} else {
	  thisint->flags &= (~IS_WIRELESS);
	}

      if (!(thisint->flags & IS_WIRELESS))
	{
	  thisint->flags &= (~WAS_DOWN);
	}
    }  

  memset(&iwr, 0, sizeof(iwr));
  // Specify the interface name we are asking about.
  strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  memset(&iwm, 0, sizeof(iwm));
  
  // Set up our MLME struct.
  iwm.cmd = mlme_type;
  iwm.reason_code = mlme_reason;
  iwm.addr.sa_family = ARPHRD_ETHER;

  // Need to specify the MAC address that we want to do this MLME for.
  memcpy(iwm.addr.sa_data, thisint->source_mac, 6);
  iwr.u.data.pointer = (caddr_t)&iwm;
  iwr.u.data.length = sizeof(iwm);

  if (ioctl(sockData->sockInt, SIOCSIWMLME, &iwr))
    {
      debug_printf(DEBUG_NORMAL, "Couldn't issue MLME request!\n");
    }
  
  return XENONE;
}
#endif

int cardif_linux_wext_disassociate(struct interface_data *intdata, int reason)
{
#if WIRELESS_EXT > 17
  cardif_linux_wext_mlme(intdata, IW_MLME_DISASSOC, reason);
#endif
  return XENONE;
}

int cardif_linux_wext_set_key_ext(struct interface_data *intdata, int alg, 
				   unsigned char *addr, int keyidx, int settx, 
				   char *seq,  int seqlen, char *key, 
				   int keylen)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;
  struct iw_encode_ext *iwee;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  // Allocate enough memory to hold our iw_encode_ext struct, and the
  // key itself.
  iwee = (struct iw_encode_ext *)malloc(sizeof(struct iw_encode_ext) + keylen);

  bzero(iwee, sizeof(struct iw_encode_ext));

  iwee->alg = alg;
  iwee->ext_flags = keyidx+1;

  if ((seq != NULL) && (seqlen > 0))
    {
      iwee->ext_flags |= IW_ENCODE_EXT_RX_SEQ_VALID;
      memcpy(iwee->rx_seq, seq, seqlen);
    }
  
  if (settx) 
    {
      iwee->ext_flags |= IW_ENCODE_EXT_SET_TX_KEY;
      if ((addr != NULL) && (memcmp(addr, "\0xff\0xff\0xff\0xff\0xff\0xff", 6) != 0))
	{
	  memcpy(iwee->addr.sa_data, addr, 6);
	} else {
	  memcpy(iwee->addr.sa_data, intdata->dest_mac, 6);
	}
    } else {
      iwee->ext_flags |= IW_ENCODE_EXT_GROUP_KEY;
      memset(iwee->addr.sa_data, 0xff, 6);
    }
  iwee->addr.sa_family = ARPHRD_ETHER;

  iwee->key_len = keylen;
  memcpy(iwee->key, key, keylen);

  wrq.u.data.pointer = (caddr_t)iwee;
  wrq.u.data.length = sizeof(struct iw_encode_ext) + keylen;

  if (ioctl(sockData->sockInt, SIOCSIWENCODEEXT, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting key!! (IOCTL "
		   "failure.)\n");
      debug_printf(DEBUG_NORMAL, "Error %d : %s\n", errno, strerror(errno));
    }
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}


int cardif_linux_wext_set_tkip_key(struct interface_data *intdata, 
				   unsigned char *addr, int keyidx, int settx, 
				   char *seq,  int seqlen, char *key, 
				   int keylen)
{
#if WIRELESS_EXT > 17

  return cardif_linux_wext_set_key_ext(intdata, IW_ENCODE_ALG_TKIP, addr,
				       keyidx, settx, seq, seqlen, key, 
				       keylen);
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}

int cardif_linux_wext_set_ccmp_key(struct interface_data *intdata,
				   unsigned char *addr, int keyidx, int settx,
				   char *seq, int seqlen, char *key,
				   int keylen)
{
#if WIRELESS_EXT > 17
  return cardif_linux_wext_set_key_ext(intdata, IW_ENCODE_ALG_CCMP, addr,
				       keyidx, settx, seq, seqlen, key,
				       keylen);
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}

int cardif_linux_wext_wpa(struct interface_data *intdata, char state)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.param.flags = IW_AUTH_WPA_ENABLED & IW_AUTH_INDEX;
  wrq.u.param.value = state;

  if (ioctl(sockData->sockInt, SIOCSIWAUTH, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting WPA state enable/disable!\n");
      return -1;
    }
#endif
  return XENONE;
}

int cardif_linux_wext_set_wpa_ie(struct interface_data *intdata, unsigned char *wpaie,
				 unsigned int wpalen)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero (&wrq, sizeof(wrq));

  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.data.pointer = (caddr_t) wpaie;
  wrq.u.data.length = wpalen;
  wrq.u.data.flags = 0;

  if (ioctl(sockData->sockInt, SIOCSIWGENIE, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting WPA IE!\n");
    } 
#endif
  return XENONE;
}


int cardif_linux_wext_wpa_state(struct interface_data *intdata, char state)
{

  // If we have wireless extensions 18 or higher, we can support WPA/WPA2
  // with standard ioctls.

#if WIRELESS_EXT > 17
  char wpaie[24];

  if (state)
    {
      // Enable WPA if the interface doesn't already have it.
      cardif_linux_wext_wpa(intdata, TRUE);


    } else {
      cardif_linux_wext_wpa(intdata, FALSE);

      // Otherwise, make sure we don't have an IE set.
      memset(wpaie, 0, sizeof(wpaie));
      if (cardif_linux_wext_set_wpa_ie(intdata, wpaie, 0) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't clear WPA IE on device %s!\n",
		       intdata->intName);
	}
    }
#endif

  return XENONE;
}

int cardif_linux_wext_unencrypted_eapol(struct interface_data *intdata,
					int state)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.param.flags = IW_AUTH_RX_UNENCRYPTED_EAPOL & IW_AUTH_INDEX;
  wrq.u.param.value = state;

  if (ioctl(sockData->sockInt, SIOCSIWAUTH, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting allow unencrypted EAPOL! "
		   "It is possible that your driver is not yet compatible "
		   "with wireless extensions 17 or higher.\n");
      return -1;
    } else {
      debug_printf(DEBUG_INT, "Set allowing of unencrypted EAPOL!\n");
    }
#endif
  return XENONE;
}

void cardif_linux_wext_associate(struct interface_data *intdata, char *newssid)
{
  char *bssid;
  int len;
  u_char wpaie[255];

  if (config_ssid_get_ssid_abilities() & RSN_IE)
    {
      cardif_linux_wext_get_wpa2_ie(intdata, wpaie, &len);
    } else if (config_ssid_get_ssid_abilities() & WPA_IE)
      {
	cardif_linux_wext_get_wpa_ie(intdata, wpaie, &len);
      }

  if (len > 0)
    {
      if (cardif_linux_wext_set_wpa_ie(intdata, wpaie, len) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set WPA IE on device %s!\n",
		       intdata->intName);
	}
    }

  intdata->flags |= DONTSCAN;
  cardif_linux_wext_set_ssid(intdata, newssid);

  bssid = config_ssid_get_mac();
  if (bssid != NULL)
    {
      debug_printf(DEBUG_INT, "Dest. BSSID : ");
      debug_hex_printf(DEBUG_INT, bssid, 6);
    }

  //  cardif_linux_wext_set_bssid(intdata, bssid);
  
  if ((intdata->wpa_ie == NULL) && (intdata->rsn_ie == NULL))
    {
      // We need to set up the card to allow unencrypted EAPoL frames.
      cardif_linux_wext_unencrypted_eapol(intdata, TRUE);
    }
  
  // We need to force zero keys to cause association. At least this is the default behavior
  // elsewhere.
  /*
    debug_printf(DEBUG_INT, "ASSOCIATING: zeroing keys off.\n");
    cardif_linux_wext_zero_keys(intdata);*/
  return;
}

int cardif_linux_wext_countermeasures(struct interface_data *intdata,
				      char endis)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.param.flags = IW_AUTH_TKIP_COUNTERMEASURES & IW_AUTH_INDEX;
  wrq.u.param.value = endis;

  if (ioctl(sockData->sockInt, SIOCSIWAUTH, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting WPA countermeasures enable/disable!\n");
      return -1;
    }
#endif
  return XENONE;
}

int cardif_linux_wext_drop_unencrypted(struct interface_data *intdata,
				       char endis)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.param.flags = IW_AUTH_DROP_UNENCRYPTED & IW_AUTH_INDEX;
  wrq.u.param.value = endis;

  if (ioctl(sockData->sockInt, SIOCSIWAUTH, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting WPA countermeasures enable/disable!\n");
      return -1;
    }
#endif
  return XENONE;
}

int cardif_linux_wext_delete_key(struct interface_data *intdata, int key_idx,
				 int set_tx)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;
  struct iw_encode_ext iwee;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  bzero(&iwee, sizeof(iwee));

  iwee.alg = IW_ENCODE_ALG_NONE;
  
  if (set_tx) iwee.ext_flags |= IW_ENCODE_EXT_SET_TX_KEY;
  
  iwee.key_len = 0;

  wrq.u.data.pointer = (caddr_t)&iwee;
  wrq.u.data.length = sizeof(iwee);

  if (ioctl(sockData->sockInt, SIOCSIWENCODEEXT, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error with key delete! (IOCTL failure.)\n");
    }
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}


struct cardif_funcs cardif_linux_wext_driver = {
  .scan = cardif_linux_wext_scan,
  .disassociate = cardif_linux_wext_disassociate,
  .set_wep_key = cardif_linux_wext_set_WEP_key,
  .set_tkip_key = cardif_linux_wext_set_tkip_key,
  .set_ccmp_key = cardif_linux_wext_set_ccmp_key,
  .delete_key = cardif_linux_wext_delete_key,
  .set_ssid = cardif_linux_wext_set_ssid,
  .associate = cardif_linux_wext_associate,
  .get_ssid = cardif_linux_wext_get_ssid,
  .get_bssid = cardif_linux_wext_get_bssid,
  .wpa_state = cardif_linux_wext_wpa_state,
  .wpa = cardif_linux_wext_wpa,
  .roam = cardif_linux_wext_roam,
  .countermeasures = cardif_linux_wext_countermeasures,
  .drop_unencrypted = cardif_linux_wext_drop_unencrypted,
  .get_wpa_ie = cardif_linux_wext_get_wpa_ie,
  .get_wpa2_ie = cardif_linux_wext_get_wpa2_ie,
  .enc_disable = cardif_linux_wext_enc_disable
};


#endif
