/*******************************************************************
 * Handle keying for type 1 (RC4, non-TKIP) EAPOL Keys
 * File: eapol_key_type1.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapol_key_type1.c,v 1.32 2005/08/09 01:39:14 chessing Exp $
 * $Date: 2005/08/09 01:39:14 $
 * $Log: eapol_key_type1.c,v $
 * Revision 1.32  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdio.h>
#include <openssl/hmac.h>
#include <openssl/rc4.h>
#include <string.h>
#include <netinet/in.h>
#include "xsup_debug.h"
#include "xsup_err.h"
#include "frame_structs.h"
#include "cardif/cardif.h"
#include "key_statemachine.h"
#include "eapol_key_type1.h"
#include "config.h"
#include "timer.h"

int eapol_dump_keydata(char *inframe, int framesize)
{
  struct key_packet *keydata;
  uint16_t length;

  if (!inframe)
    {
      debug_printf(DEBUG_NORMAL, "Invalid frame passed to eapol_dump_keydata()!\n");
      return XEMALLOC;
    }

  keydata = (struct key_packet *)inframe;

  debug_printf(DEBUG_INT, "Key Descriptor   = %d\n",keydata->key_descr);

  memcpy(&length, keydata->key_length,2);
  debug_printf(DEBUG_INT, "Key Length       = %d\n",ntohs(length));
  debug_printf(DEBUG_INT, "Replay Counter   = ");
  debug_hex_printf(DEBUG_INT, keydata->replay_counter, 8);
  debug_printf(DEBUG_INT, "Key IV           = ");
  debug_hex_printf(DEBUG_INT, keydata->key_iv, 16);
  debug_printf(DEBUG_INT, "Key Index (RAW)  = %02X\n",keydata->key_index);
  debug_printf(DEBUG_INT, "Key Signature    = ");
  debug_hex_printf(DEBUG_INT, keydata->key_signature, 16);

  return XENONE;
}

/************************************
 *
 * Check the HMAC on the key packet we got.  If we can't validate the
 * HMAC, then we return FALSE, indicating an error.
 *
 ************************************/
int eapol_key_type1_check_hmac(struct interface_data *thisint, char *inframe,
			       int framesize)
{
  struct key_packet *keydata;
  char *framecpy, *calchmac;
  int outlen, retVal, length;

  framecpy = NULL;
  calchmac = NULL;
  keydata = NULL;
  outlen = 0;
  retVal = 0;
  length = 0;

  if ((!thisint) || (!inframe))
    {
      debug_printf(DEBUG_NORMAL, "Bad data passed in to eapol_key_type1_check_hmac()!\n");
      return XEMALLOC;
    }

  if (thisint->keyingMaterial == NULL)
    {
      debug_printf(DEBUG_EVERYTHING, "No keying material available!  Ignoring key frame!\n");
      return XEMALLOC;
    }

  // First, make a copy of the frame.
  framecpy = (char *)malloc(framesize);
  if (framecpy == NULL) return XEMALLOC;

  memcpy(framecpy, inframe, framesize);

  // Now, we want to zero out the HMAC.
  keydata = (struct key_packet *)&framecpy[4];

  memcpy(&length, keydata->key_length, 2);

  bzero((char *)&keydata->key_signature, 16);

  // Once we have done that, we need to calculate the HMAC.
  calchmac = (char *)malloc(16);   // The resulting MAC is 16 bytes long.
  if (calchmac == NULL) return XEMALLOC;

  HMAC(EVP_md5(), thisint->keyingMaterial+32, 
       thisint->keyingLength, framecpy, framesize, calchmac, &outlen);

  // Now, we need to compare the calculated HMAC to the one sent to us.
  keydata = (struct key_packet *)&inframe[4];

  eapol_dump_keydata((char *)keydata, framesize);

  if (memcmp(calchmac, keydata->key_signature, 16) == 0)
    {
      // The HMAC is valid.
      retVal = TRUE;
    } else {
      retVal = FALSE;
    }

  // Clean up after ourselves.
  free(framecpy);
  framecpy = NULL;
  free(calchmac);
  calchmac = NULL;

  return retVal;
}

int eapol_key_type1_get_rc4(struct interface_data *thisint, u_char *enckey, 
			    u_char *deckey, int keylen, u_char *iv, int ivlen)
{
  u_char *wholekey = NULL;
  RC4_KEY key;

  if ((thisint == NULL) || (enckey == NULL) || (deckey == NULL) ||
      (iv == NULL))
    {
      debug_printf(DEBUG_NORMAL, "Some value passed in to eapol_key_type1_get_rc4() is NULL!\n");
      return XEMALLOC;
    }

  wholekey = (u_char *)malloc(sizeof(u_char) * (ivlen + thisint->keyingLength));
  if (wholekey == NULL) return XEMALLOC;

  memcpy(wholekey, iv, ivlen);

  if (!thisint->keyingMaterial)
    {
      debug_printf(DEBUG_NORMAL, "Invalid keying material!  Keys will not be handled correctly!\n");
      return XEMALLOC;
    }

  memcpy(wholekey + ivlen, thisint->keyingMaterial, thisint->keyingLength);

  RC4_set_key(&key, ivlen + thisint->keyingLength, wholekey);
  RC4(&key, keylen, enckey, deckey);

  if (wholekey)
    {
      free(wholekey);
      wholekey = NULL;
    }

  return XENONE;
}

/*********************************
 *
 * If our rekey timer expires, we should quietly remove it.  In the case of
 * this timer, we WANT it to expire with no events!  (Otherwise, the card
 * in use may have issues.)
 *
 *********************************/
void eapol_key_type1_clear_timer(struct interface_data *intdata)
{
  debug_printf(DEBUG_EVERYTHING, "Clearing rekey problem timer.  (This is "
	       "harmless!)\n");
  timer_cancel(REKEY_PROB_TIMER);
}

/*********************************
 *
 * Set up our timer to warn if the driver may have the card reset issue.
 *
 *********************************/
void eapol_key_type1_set_rekey_prob_timer()
{
  if (timer_check_existing(REKEY_PROB_TIMER))
    {
      debug_printf(DEBUG_NORMAL, "Only %d seconds have elapsed since the last "
		   "key was set!  Either your AP has the rekey interval set "
		   "dangerously low, or your card may reset everytime new keys"
		   " are set!  Please check the rekey interval time on your "
		   "AP and report your card type, driver, and driver version "
		   "number to the list!\n", REKEY_PROB_TIMEOUT);
      timer_reset_timer_count(REKEY_PROB_TIMER, REKEY_PROB_TIMEOUT);
    } else {
      // There isn't an existing counter, so set up a new one.
      timer_add_timer(REKEY_PROB_TIMER, REKEY_PROB_TIMEOUT, NULL,
		      &eapol_key_type1_clear_timer);
    }
}

/*********************************
 *
 * Display the stale key warning, and disable the timer that was running.
 *
 *********************************/
void eapol_key_type1_stale_key_warn(struct interface_data *intdata)
{
  debug_printf(DEBUG_NORMAL, "Your unicast key has been in use for %d "
	       "minute(s).  Given the time it takes to crack WEP keys, your "
	       "data is now less secure than it was.  You should consider "
	       "asking your network administrator to lower the rekey interval"
	       " for your wireless network.\n\n"
	       " *** PLEASE NOTE ***  This is just a warning message, and does"
	       " not indicate that your WEP key has been broken!\n", 
	       (config_get_stale_key_timeout()/60));
  timer_cancel(STALE_KEY_WARN_TIMER);
}

/*********************************
 *
 * Set a timer that watches how long a key has been in use.  (Should only
 * be used for unicast keys, since broadcast keys aren't very secure to
 * begin with!)  If the timer expires, then we need to warn the user that
 * their security may be weaker than it used to be.
 *
 *********************************/
void eapol_key_type1_set_stale_key_warn_timer()
{
  if (timer_check_existing(STALE_KEY_WARN_TIMER))
    {
      timer_reset_timer_count(STALE_KEY_WARN_TIMER, 
			      config_get_stale_key_timeout());
    } else {
      // Set up a new warning counter.
      timer_add_timer(STALE_KEY_WARN_TIMER, config_get_stale_key_timeout(), 
		      NULL, &eapol_key_type1_stale_key_warn);
    }
}

/*********************************
 *
 * Decrypt the key, and set it on the interface.  If there isn't a key to
 * decrypt, then use the peer key.
 *
 *********************************/
int eapol_key_type1_decrypt(struct interface_data *thisint, char *inframe,
			    int framesize)
{
  struct key_packet *keydata = NULL;
  int keylen, rc=0;
  uint16_t length;
  u_char *newkey = NULL, *enckey = NULL;

  if ((!thisint) || (!inframe))
    {
      debug_printf(DEBUG_NORMAL, "Bad data passed in to eapol_key_type1_decrypt()!\n");
      return XEMALLOC;
    }

  keydata = (struct key_packet *)&inframe[0];

  memcpy(&length, keydata->key_length, 2);
  keylen = ntohs(length);

  debug_printf(DEBUG_INT, "EAPoL Key Processed: %s [%d] %d bytes.\n",
	       keydata->key_index & UNICAST_KEY ? "unicast" : "broadcast",
	       (keydata->key_index & KEY_INDEX)+1, keylen);

  if ((keylen != 0) && ((framesize)-sizeof(*keydata) >= keylen))
    {
      newkey = (u_char *)malloc(sizeof(u_char) * keylen);
      if (newkey == NULL) return XEMALLOC;

      enckey = (u_char *)&inframe[sizeof(struct key_packet)];

      debug_printf(DEBUG_INT, "Key before decryption : ");
      debug_hex_printf(DEBUG_INT, enckey, keylen);

      if (eapol_key_type1_get_rc4(thisint, enckey, newkey, keylen, 
				  keydata->key_iv, 16) != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't decrypt new key!\n");
	  return XEBADKEY;
	}

      debug_printf(DEBUG_INT, "Key after decryption : ");
      debug_hex_printf(DEBUG_INT, newkey, keylen);

      if (cardif_set_wep_key(thisint, newkey, keylen, keydata->key_index)
	  != 0)
	{
	  rc = FALSE;
	} else {

	  //  If the unicast flag is set, start the warning timer.
	  if (keydata->key_index & 0x80)
	    {
	      eapol_key_type1_set_rekey_prob_timer();
	      eapol_key_type1_set_stale_key_warn_timer();
	    }

	  rc = TRUE;
	  UNSET_FLAG(thisint->flags, ROAMED);
	}

      free(newkey);
      newkey = NULL;
    } else {
      debug_printf(DEBUG_INT, "Using peer key!\n");

      if (config_get_display_friendly_warnings())
	{
	  debug_printf(DEBUG_NORMAL, "*WARNING* This AP uses the key "
		       "generated during the authentication\nprocess.  If "
		       "reauthentication doesn't happen frequently enough "
		       "your connection\nmay not be very secure!\n");
	}

      if (cardif_set_wep_key(thisint, thisint->keyingMaterial, keylen, 
			     keydata->key_index) != 0)
	{
	  rc = FALSE;
	} else {
	  rc = TRUE;

	  //  If the unicast flag is set, start the warning timer.
	  if (keydata->key_index & 0x80)
	    {
	      eapol_key_type1_set_rekey_prob_timer();
	      eapol_key_type1_set_stale_key_warn_timer();
	    }
	}
    }

  // If we reach this point, then we should remember the length of the
  // keys for later comparison.
  if (keydata->key_index & UNICAST_KEY)
    {
      thisint->statemachine->unicastKeyLen = keylen;
    } else {
      thisint->statemachine->broadcastKeyLen = keylen;
    }

  return rc;
}

/**********************************
 *
 * We are handed in an EAPoL key frame.  From that frame, we check the frame
 * to make sure it hasn't been changed in transit.  We then determine the 
 * correct key, and make the call to set it.
 *
 **********************************/
void eapol_key_type1_process(struct interface_data *thisint)
{
  struct key_packet *keydata;
  struct eapol_header *eapolheader;
  u_char *inframe;
  int framesize;
  int framelen;

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to eapol_key_type1_process()!\n");
      return;
    }

  inframe = thisint->recvframe;
  framesize = thisint->recv_size;

  eapolheader = (struct eapol_header *)&inframe[OFFSET_PAST_MAC];

  framelen = ntohs(eapolheader->eapol_length);

  keydata = (struct key_packet *)&inframe[OFFSET_TO_EAPOL+4];

  if (keydata->key_descr != RC4_KEY_TYPE)
    {
      debug_printf(DEBUG_NORMAL, "Key type isn't RC4!\n");
      return;
    }

  if (eapol_key_type1_check_hmac(thisint, (char *)&inframe[OFFSET_TO_EAPOL], framelen+4)==FALSE)
    {
      debug_printf(DEBUG_NORMAL, "HMAC failed on key data!  This key will be discarded.\n");
      return;
      }

  if (eapol_key_type1_decrypt(thisint, (char *)&inframe[OFFSET_TO_EAPOL+4],
			      (framelen)) != TRUE)
    {
      debug_printf(DEBUG_NORMAL, "Failed to set wireless key!\n");
      return;
    }

  if ((thisint->statemachine->unicastKeyLen != 0) && 
      (thisint->statemachine->broadcastKeyLen != 0) &&
      (thisint->statemachine->unicastKeyLen != 
       thisint->statemachine->broadcastKeyLen))
    {

      if (config_get_display_friendly_warnings())
	{
	  debug_printf(DEBUG_NORMAL, "[WARNING] Unicast and broadcast keys "
		       "are different lengths!  Some cards/drivers/APs do "
		       "not like this combination!\n");
	}
    }
  thisint->recv_size = 0;
}
