/*******************************************************************
 * The driver function for a Linux application layer EAPOL 
 * implementation
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: xsup_driver.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: xsup_driver.c,v 1.93 2005/08/21 18:12:13 chessing Exp $
 * $Date: 2005/08/21 18:12:13 $
 * $Log: xsup_driver.c,v $
 * Revision 1.93  2005/08/21 18:12:13  chessing
 * Added the ability to reload our configuration using a HUP.  Cleaned up a bunch of memory leaks in the config parse code.
 *
 * Revision 1.92  2005/08/20 19:26:42  chessing
 * Added option -s that will force removal of old unix domain control socket.  This option should only be used on startup, since it could cause some strange problems if it is used when an existing Xsupplicant instance is in operation.
 *
 * Revision 1.91  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/
/***
 *** This code implements 802.1X Authentication on a supplicant
 *** and supports multiple Authentication types.  
 *** See IEEE Draft P802.1X/D11, March 27, 2001 for more details
 ***/

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <strings.h>
#include <string.h>
#include <sys/wait.h>

#include "profile.h"
#include "config.h"
#include "eap.h"
#include "config_ssid.h"
#include "statemachine.h"
#include "xsup_ipc.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "eapol.h"
#include "snmp.h"
#include "cardif/cardif.h"
#include "cardif/core.h"
#include "timer.h"

// This is needed to support the use of IW_ENCODE_TEMP keys for Cisco cards
// based on the airo driver.
#include "cardif/linux/cardif_linux_wext.h"

int wpa_keying = 0;
struct interface_data *intiface;
int dsd = 0;
char block_wpa = 0;
char *config_path = NULL;

/*********************************************
 *
 * When the alarm clock is called, we set the interface tick to true and
 * the statemachine tick to true.
 *
 *********************************************/
void alrmclock()
{
  intiface->statemachine->tick = TRUE;
  intiface->tick = TRUE;
 
  alarm(1);
}

/*********************************************
 *
 * Take a string for a driver name, and return a numeric value that indicates
 * which driver we should use.
 *
 *********************************************/
int xsup_driver_get_driver(char *drname)
{
  if (drname == NULL) return DRIVER_WEXT;

  lowercase(drname);

  if (strcmp(drname, "none") == 0) return DRIVER_NONE;
  if (strcmp(drname, "wext") == 0) return DRIVER_WEXT;
  if (strcmp(drname, "hostap") == 0) 
    {
      printf("Using hostap driver!\n");
      return DRIVER_HOSTAP;
    }

  if (strcmp(drname, "atmel") == 0) 
    {
      printf("Using atmel driver!\n");
      return DRIVER_ATMEL;
    }

  if (strcmp(drname, "ndiswrapper") == 0)
    {
      printf("Using ndiswrapper driver!\n");
      return DRIVER_NDISWRAPPER;
    }

  if (strcmp(drname, "ipw") == 0)
    {
      printf("Using ipw driver!\n");
      return DRIVER_IPW;
    }

  if (strcmp(drname, "madwifi") == 0)
    {
      printf("Using madwifi driver!\n");
      return DRIVER_MADWIFI;
    }

  return DRIVER_WEXT;
}

/*********************************************
 *
 * Initialize all of the pieces that will be needed for our supplicant.
 * We need to initialize in the correct order, in order to make sure
 * that other pieces of the initialization happen with the correct 
 * information available.
 *
 * THIS FUNCTION SHOULD NEVER BE CALLED OUTSIDE OF THIS FILE!
 *
 *********************************************/
int global_init(int xdaemon, char *device, char *config, char *drivernam,
		FDEPTH flags)
{
  struct interface_data *intcur;
  char driveridx = 0;
  char newssid[33];
  char *default_cfg = "/etc/xsupplicant.conf";

  // Set our debug level.
  intiface = NULL;
  intcur = NULL;

  timer_init();

  driveridx = xsup_driver_get_driver(drivernam);
  debug_setdaemon(xdaemon);

  if (config == NULL) 
    {
      printf("Using default config path!\n");
      config = default_cfg;
    }

  config_path = strdup(config);

  // Build up our config information.
  switch(config_setup(config))
    {
    case XECONFIGFILEFAIL:
      debug_printf(DEBUG_NORMAL, "\tPlease ensure that \"%s\" contains a valid"
		   " xsupplicant configuration.\n", config);
      exit(255);
      break;
    case XECONFIGPARSEFAIL:
      debug_printf(DEBUG_NORMAL, "There was a problem with the config file.  "
		   "We cannot continue.\n");
      exit(255);
      break;
    }

  logfile_setup(config_get_logfile());

  // Initalize the event core.
  event_core_setup();

  if (device == NULL)
    {
      device = config_get_default_int();
      if (device == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "No valid interface defined!  Cannot"
		       " continue!\n");
	  return XENOTINT;
	}

      debug_printf(DEBUG_NORMAL, "Interface not specified on the command line"
		   " we will use the default of '%s' instead!\n", device);
    }

  intiface = (struct interface_data *)malloc(sizeof(struct interface_data));
  if (intiface == NULL) return XEMALLOC;
  memset(intiface, 0, sizeof(struct interface_data));
  
  // Start by setting up the structure, and assigning the interface.
  if (init_interface_struct(intiface, device) != XENONE)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't init interface struct for device "
		   "%s!  Cannot continue!\n", device);
      exit(1);
    }

  // Set our flags.
  intiface->flags = flags;
  
  // Establish a handler to the interface. Only need to manually turn on WPA if
  // force is set, otherwise should only turn it on if AP indicates support.
  if (cardif_init(intiface, driveridx, block_wpa) < 0) 
    return XENOTINT;

  debug_printf(DEBUG_INT, "Interface initialized!\n");

  if (cardif_get_if_state(intiface) == FALSE)
    {
      debug_printf(DEBUG_NORMAL, "Interface %s is down!  Bring it up, and try"
		   " again!\n", intiface->intName);
      exit(1);
    }

  // Check to see if this interface is wireless.
  if (cardif_int_is_wireless(intiface->intName) == TRUE)
    {
      SET_FLAG(intiface->flags, IS_WIRELESS);

      bzero(newssid, 33);
      if (cardif_GetSSID(intiface, (char *)&newssid) != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't determine the current SSID!\n");
	} else {
	  debug_printf(DEBUG_NORMAL, "Your card is currently set for "
		       "wireless network \"%s\".  Looking for a config.\n",
		       newssid);

	  intiface->cur_essid = strdup(newssid);

	  if (config_build(intiface->cur_essid) != TRUE)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't build a config for ESSID "
			   "%s!\n", intiface->cur_essid);
	      // XXX  Need to determine what to do when we fail.
	    }
	  config_set_globals(intiface);


	  cardif_do_wireless_scan(intiface);
	  
	  cardif_reassociate(intiface, DISASSOC_UNSPECIFIED);

	  if (intiface->flags & ZEROONROAM)
	    {
	      // DKS -- currently only want to do this if not associated; aren't picking up 
	      // state properly even if we then associatecardif/
	      cardif_roam(intiface);
	    }
	}
    } else {
      UNSET_FLAG(intiface->flags, IS_WIRELESS);
    }
  
  if (config_build(intiface->cur_essid) != TRUE)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't build config for network %s!\n",
		   intiface->cur_essid);
      // XXX Need to determine what to do when we fail.
    }
  config_set_globals(intiface);

  // were doing cardif_roam here if zeros_on_roam is true

  cardif_check_associated(intiface);
  
  // Then, initialize EAPoL (which will initialize EAP methods).

  return XENONE;
}

/***************************************
 *
 * Trap a segfault, and exit cleanly.
 *
 ***************************************/
void global_sigseg()
{
    fprintf(stderr, "[FATAL] SIGSEGV  (Segmentation Fault)!!!\n");
    xsup_ipc_cleanup(intiface);
    fflush(stderr); fflush(stdout);
    exit(-1);
}

/****************************************
 *
 * Clean up any values that we have set up for use by the supplicant.  This
 * includes calling any clean up routines for other modules such as EAPoL
 * or EAP.
 *
 * THIS FUNCTION SHOULD NEVER BE CALLED OUTSIDE THIS FILE!
 *
 ****************************************/
void global_deinit()
{
  // First thing we need to do is kill the alarm clock, so it doesn't try
  // to do something while we are in the middle of quitting! ;)
  alarm(0);

  if (intiface)
    {
      xsup_ipc_cleanup(intiface);
      eapol_cleanup(intiface);

      debug_printf(DEBUG_INT, "Sending Logoff for int %s!\n",
		   intiface->intName);
      txLogoff(intiface);
      cardif_deinit(intiface);
      destroy_interface_struct(intiface);
    }

  if (intiface != NULL)
    {
      free(intiface);
      intiface = NULL;
    }

  config_ssid_clear();

  timer_cleanup();

  event_core_cleanup();

  config_destroy();

  logfile_cleanup();

  free(config_path);
  config_path = NULL;

  exit(0);
}

/****************************************
 *
 * Clear, and reload our config.
 *
 ****************************************/
void global_config_reload()
{
  debug_printf(DEBUG_NORMAL, "Reloading configuration...\n");
  config_destroy();
  config_setup(config_path);
  signal(SIGHUP, global_config_reload);
}


/****************************************
 *
 * Display our usage information.
 *
 ****************************************/
void usage(char *prog)
{
  debug_printf(DEBUG_NORMAL, "Usage: %s "
	       "[-W] "
	       "[-w] "
	       "[-c config file] "
	       "[-i device] "
	       "[-d debug_level] "
	       "[-f] "
	       "[-D driver name] "
	       "[-z] "
	       "[-t] "
	       "[-q] "
	       "[-a] "
	       "[-s] "
	       "\n", prog);

  debug_printf(DEBUG_NORMAL, "\n\n");
  debug_printf(DEBUG_NORMAL, "-W : Use wpa_supplicant for WPA keying\n");
  debug_printf(DEBUG_NORMAL, "-w : Don't enable WPA support.  (On drivers that"
	       "support WPA/WPA2.)\n");
  debug_printf(DEBUG_NORMAL, "-c <path> : Use the config file <path> instead "
	       "of the default.\n");
  debug_printf(DEBUG_NORMAL, "-i <interface> : Authenticate using <interface> "
	       "\n");
  debug_printf(DEBUG_NORMAL, "-d <debug_level/flags> : Set debug verbosity."
	       "\n");
  debug_printf(DEBUG_NORMAL, "-f : Run in forground mode.\n");
  debug_printf(DEBUG_NORMAL, "-D <driver name> : Use <driver name> support. "
	       "(Needed for WPA support.)\n");
  debug_printf(DEBUG_NORMAL, "-z : Reset WEP keys to 0s on roam. (Needed for "
	       "some drivers, such as Orinoco_cs.)\n");
  debug_printf(DEBUG_NORMAL, "-t : Don't use IW_ENCODE_TEMP in key setting "
	       "requests.  (Try this if you are getting errors setting keys.)\n");
  debug_printf(DEBUG_NORMAL, "-q : Terminate when defaulting to authenticated "
	       "state.\n");
  debug_printf(DEBUG_NORMAL, "-a : Watch alternate interface index for "
	       "wireless events.\n");
  debug_printf(DEBUG_NORMAL, "-s : Remove the existing control socket if found.  (Should only be used in system init scripts!)\n");
  debug_printf(DEBUG_NORMAL, "\n\n");

  debug_printf(DEBUG_NORMAL, " <driver name> can be any of : \n");
  debug_printf(DEBUG_NORMAL, "\twext : Interface uses Linux wireless "
	       "extensions. (Default)\n");
  debug_printf(DEBUG_NORMAL, "\thostap : Interface uses hostap driver.\n");
  debug_printf(DEBUG_NORMAL, "\tatmel : Interface uses atmel driver.\n");
  debug_printf(DEBUG_NORMAL, "\tndiswrapper : Interface uses ndiswrapper "
	       "driver.\n");
  debug_printf(DEBUG_NORMAL, "\tipw     : Interface uses IPW2100/2200 "
	       "driver.\n");

#if ENABLE_MADWIFI
  debug_printf(DEBUG_NORMAL, "\tmadwifi : Interface uses madwifi driver.\n");
#endif
  debug_printf(DEBUG_NORMAL, "\n");

  debug_printf(DEBUG_NORMAL, " <debug_level> can be any of : \n");
  debug_printf(DEBUG_NORMAL, "\t0..7 : Old style debug flags.\n");
  debug_printf(DEBUG_NORMAL, "\tA : Enable ALL debug flags.\n");
  debug_printf(DEBUG_NORMAL, "\tc : Enable CONFIG debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\ts : Enable STATE debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\ta : Enable AUTHTYPES debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\ti : Enable INT debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\tn : Enable SNMP debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\te : Enable EVERYTHING debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\tx : Enable EXCESSIVE debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\nValues [0..7] cannot be combined with character attributes!\n");
}

/***************************************
 *
 * The main body of the program.  We should keep this simple!  Process any
 * command line options that were passed in, set any needed variables, and
 * enter the loop to handle sending an receiving frames.
 *
 ***************************************/
int main(int argc, char *argv[])
{
  int op, pid;
  char *theOpts = "c:i:d:D:Wwfztqas";
  char *config = NULL, *device = NULL;
  char *drivername = NULL;
  int xdaemon = 1, new_debug;
  FDEPTH flags = 0;

  new_debug = 0;
  config = NULL;

  // If we don't have at least a -i, then we can't start up!
  if (argc < 2)
    {
      usage(argv[0]);
      exit(0);
    }

  // Process any arguments we were passed in.
  while ((op = getopt(argc, argv, theOpts)) != EOF) 
    {
      switch (op)
	{
	case 'c':
	  // Path to config file.
	  config = optarg;
	  break;

	case 'i':
	  // Interface to use.
	  device = optarg;
	  break;

	case 'd':
	  // Set the debug level.
	  if ((atoi(optarg) == 0) && (optarg[0] != '0'))
	    {
	      debug_alpha_set_flags(optarg);
	    } else {
	      debug_set_flags(atoi(optarg));
	    }
	  break;

	case 'W':
	  // Provide WPA keying material (PMK) to wpa_supplicant.
	  wpa_keying = 1;
	  break;

	case 'f':
	  // Force running in the foreground.
	  xdaemon = 2;
	  break;

	case 'D':
	  // The name of the wireless driver to use.
	  drivername = optarg;
	  break;

	case 'z':
	  // Reset the encryption key to 0s on roam.
	  SET_FLAG(flags, ZEROONROAM);
	  break;

	case 'w':
	  // Force WPA to be enabled.
	  block_wpa = 1;
	  break;

	case 't':
	  // Use IW_ENCODE_TEMP for setting keys.
	  SET_FLAG(flags, DONT_USE_TEMP);
	  break;

	case 'q':
	  // Terminate when we have exhausted the maximum number of starts we
	  // want to send.
	  SET_FLAG(flags, TERM_ON_FAIL);
	  break;

	case 'a':
	  // Enable "off by -1 mode" for wireless cards that provide the
	  // driver events on an interface that is (interface index)-1.
	  SET_FLAG(flags, ONEDOWN);
	  break;
	  
	case 's':
	  // Clear the IPC socket file, if it still exists.
	  SET_FLAG(flags, CLEAR_IPC);
	  break;

	  // added by npetroni, need to do something with bad options.
	  // for now, I say exit.
	default:
	  usage(argv[0]);
	  exit(0);
	  break;
	}
    }

  if (device == NULL)
    {
      // We don't have an interface listed to use!
      debug_printf(DEBUG_NORMAL, "You need to specify an interface to use!"
		   "\n\n");
      usage(argv[0]);
      exit(0);
    }

  // XXX Probably need to move this.
  if (xdaemon == 1)
    {
      printf("Starting XSupplicant v. %s\n",VERSION);
      // We should fork, and let the parent die.
      pid = fork();
      
      if (pid > 0) 
	{
	  // If we are the parent, die.
	  exit(0);
	}
      
      // Otherwise, keep going.
    }

  snmp_init();

  // We have our options set, to initalize, then go in to our event loop.
  if (global_init(xdaemon, device, config, drivername, flags) != 0)
    {
      printf("Couldn't initalize!!!!\n");
      exit(255);
    }

  if (intiface == NULL)
    {
      printf("No valid interface found!\n");
      global_deinit();
      exit(255);
    }

  if (xsup_ipc_init(intiface) != 0)
    {
      printf("Couldn't initalize daemon socket!\n");
      global_deinit();
      exit(255);
    }

  // When we quit, cleanup.
  signal(SIGTERM, global_deinit);
  signal(SIGINT, global_deinit);
  signal(SIGQUIT, global_deinit);
  signal(SIGSEGV, global_sigseg);
  signal(SIGHUP, global_config_reload);

  // Set up a handler, and start our timer.
  signal(SIGALRM, alrmclock);
  alarm(1);

  while (1)
    {
      xsup_ipc_process(intiface);

      event_core(intiface);
    }

  return XENONE;
}
