/*******************************************************************
 * Linux card interface implementation.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: cardif_linux.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_linux.c,v 1.123 2006/03/21 18:22:10 chessing Exp $
 * $Date: 2006/03/21 18:22:10 $
 * $Log: cardif_linux.c,v $
 * Revision 1.123  2006/03/21 18:22:10  chessing
 * Fixes to EAP-AKA code.  Changed a few defaults to disable Xsupplicant based roaming, and passive scanning for this release.  (It would currently cause more problems than it would solve.
 *
 * Revision 1.122  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.121  2006/01/24 04:42:26  chessing
 * A few more fixes to WPA code, along with a fix to the scan reaping code.
 *
 * Revision 1.120  2006/01/23 05:28:37  chessing
 * Fixed a few settings that were causing errors with IOCTLs on some cards.  Updated WPA2 code to properly process group key packets. We now record quality, signal level, and noise level from scan results, so that we can start to make better decisions on which AP to associate to.
 *
 * Revision 1.119  2006/01/19 05:37:04  chessing
 * WPA2 is working correctly.  Added the ability to query the card to gather encryption/authentication capabilities.  1.2.3 is now ready to go.
 *
 * Revision 1.118  2006/01/12 17:54:03  chessing
 * WPA almost works again on cards that use the GENIE setting.  Added a small fix to allow Xsupplicant to build on Open SuSE 10.
 *
 * Revision 1.117  2005/12/24 04:51:51  chessing
 * Removed an unneeded file.  Updated scanning code to parse IWEVGENIE events correctly, and make use of them.
 *
 * Revision 1.116  2005/12/24 02:39:48  chessing
 * Fixes to autoconf script to correctly identify the number of arguments that iw_extract_event_stream() takes, along with checking iwlib.h compiles correctly.
 *
 * Revision 1.115  2005/12/03 22:18:51  chessing
 * Added an include to fix some problems when compiling with WE19.
 *
 * Revision 1.114  2005/11/30 00:42:29  chessing
 * Cleaned up some warnings during the build process.
 *
 * Revision 1.113  2005/11/22 19:49:42  chessing
 * When the desired interface isn't available when Xsupplicant starts, Xsupplicant will wait for it to show up before starting to authenticate.  In this case, the interface isn't 'available' when an ifconfig doesn't show that the interface exists.  It has *NOTHING* to do with the up/down state of the interface.  This is mostly useful for situations where a cardbus wireless card is plugged in before linux is booted.
 *
 * Revision 1.112  2005/11/14 17:27:39  chessing
 * Removed the driver specific code for hostap, ipw, and ndiswrapper, since they all support WE18 now.
 *
 * Revision 1.111  2005/11/10 04:56:54  chessing
 * Added patch from Ben Gardner to add support for setting a specific WEP key prior to attempting to associte.  (With a few slight modifications by me to make it fit in the current CVS code, and get it supported in config-parse.)  Added patch from Pekka Savola to fix some header ordering issues, and a potential buffer overflow.
 *
 * Revision 1.110  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.109  2005/10/14 02:26:18  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.108  2005/09/25 15:03:36  shaftoe
 * Document syslog functionality in example config file.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.107  2005/09/25 00:39:48  shaftoe
 * - More work to pretty up syslog output.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.106  2005/09/19 21:48:05  chessing
 * Added a line that will indicate that the interface is scanning when running in normal (non-debug) mode.
 *
 * Revision 1.105  2005/09/19 21:45:46  chessing
 * Fix for the madwifi driver when it connects to certain APs to do WEP.  (Currently the only known one with this problem is the Trapeze APs when they are running MSS 4.x+.)
 *
 * Revision 1.104  2005/09/14 03:34:54  chessing
 * Small cosmetic changes.  Default association mode is now auto instead of manual. Fixes for bug IDs #1290449 & #1290323.
 *
 * Revision 1.103  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.102  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.101  2005/09/05 01:00:37  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.100  2005/08/25 03:34:05  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.99  2005/08/25 02:20:20  chessing
 * Some cleanup in xsup_debug.c, added the ability to wait for an interface to come up if it is down when Xsupplicant is started.  Roughed in the ability to choose between having Xsupplicant auto associate you, or allow you to set the ssid manually.  The stale key timer can now be set in the config file.  The association timeout can be set in the config file, and will also be used once the functionality is in place to attempt to guess the key settings needed for association, as well as the functionality to auto associate.
 *
 * Revision 1.98  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.97  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#ifdef LINUX_FRAMER

// We want to use kernel headers to build.
#define HEADERS_KERNEL

#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/compiler.h>
#include <linux/wireless.h>
#include <iwlib.h>
#include <linux/if_packet.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <linux/rtnetlink.h>

#include "xsupconfig.h"
#include "profile.h"
#include "config_ssid.h"
#include "cardif/linux/cardif_linux_wext.h"
#include "cardif/cardif.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "snmp.h"
#include "statemachine.h"
#include "wireless_sm.h"
#include "cardif/linux/cardif_linux.h"
#include "cardif/linux/cardif_atmel_driver.h"
#include "cardif/linux/cardif_madwifi_driver.h"
#include "cardif/linux/cardif_linux_rtnetlink.h"
#include "timer.h"

#ifndef ETH_P_EAPOL
#define ETH_P_EAPOL 0x888e
#endif

// Define this, so the compiler doesn't complain.
extern unsigned int if_nametoindex(const char *);

// This contains a pointer to the functions needed for wireless.  
struct cardif_funcs *wireless;

// Store values about what state our interface was in before we start messing
// with it.
struct int_starting_data *startup;

/********************************************************************
 *
 * Clear all keys, and accept unencrypted traffic again.
 *
 ********************************************************************/
void cardif_linux_clear_keys(struct interface_data *ctx)
{
  int i;

  debug_printf(DEBUG_INT, "Clearing keys!\n");
  
  // Clear the PTK.
  cardif_delete_key(ctx, 0, 1);

  for (i=0;i<4;i++)
    {
      cardif_delete_key(ctx, i, 0);
    }

  // Accept unencrypted frames again.
  cardif_drop_unencrypted(ctx, 0);
}

/***********************************************
 *
 * Determine if we are currently associated. 
 *
 ***********************************************/
int cardif_check_associated(struct interface_data *intdata)
{
  char newmac[6], curbssid[6];

  // If we are wired, this function doesn't do anything useful.
  if (!TEST_FLAG(intdata->flags, IS_WIRELESS)) return XENONE;

  cardif_GetBSSID(intdata, curbssid);

  memset(newmac, 0x00, 6);
  if (memcmp(newmac, curbssid, 6) == 0)
    {
      return IS_UNASSOCIATED;
    }

  memset(newmac, 0x44, 6);
  if (memcmp(newmac, curbssid, 6) == 0)
    {
      return IS_UNASSOCIATED;
    }

  memset(newmac, 0xFF, 6);
  if (memcmp(newmac, curbssid, 6) == 0)
    {
      return IS_UNASSOCIATED;
    }

  //Otherwise, we are associated.
  return IS_ASSOCIATED;
}

/***********************************************
 *
 * Set up the wireless cardif_funcs structure to the driver that the user
 * has requested.
 *
 ***********************************************/
void cardif_set_driver(char driver)
{
  switch (driver)
    {
    case DRIVER_NONE:
      wireless = NULL;
      break;

    case DRIVER_WEXT:
      wireless = &cardif_linux_wext_driver;
      break;

#ifdef ENABLE_MADWIFI
    case DRIVER_MADWIFI:
      wireless = &cardif_madwifi_driver;
      break;
#endif
      
    case DRIVER_ATMEL:
      wireless = &cardif_atmel_driver;
      break;
    }
}

/***********************************************
 *
 * Do whatever is needed to get the interface in to a state that we can send
 * and recieve frames on the network.  Any information that we need to later
 * use should be stored in the interface_data structure.
 *
 ***********************************************/
int cardif_init(struct interface_data *thisint, char driver, int block_wpa)
{
  struct ifreq ifr;
  struct lin_sock_data *sockData;
  int sockopts, sockerr, retval;
  struct config_globals *globals;

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid interface data in %s!\n",
		   __FUNCTION__);
      return XEGENERROR;
    }

  // Get the information about the global settings from the config file.
  globals = config_get_globals();

  if (globals == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid global configuration variables!\n");
      return XEGENERROR;
    }

  debug_printf(DEBUG_INT, "Initializing socket for interface %s..\n",
	       thisint->intName);

  // Keep track of which driver we were assigned.
  thisint->driver_in_use = driver;

  // Find out what the interface index is.
  thisint->intIndex = if_nametoindex(thisint->intName);

  // Allocate memory for the things we need.
  thisint->sockData = (void *)malloc(sizeof(struct lin_sock_data));
  if (thisint->sockData == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error allocating memory!\n");
      return XEMALLOC;
    }

  sockData = thisint->sockData;

  // Establish a socket handle.
  sockData->sockInt = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_EAPOL));
  if (sockData->sockInt < 0)
    {
      debug_printf(DEBUG_NORMAL, 
		   "Couldn't initialize raw socket for interface %s!\n",
		   thisint->intName);
      return XENOSOCK;
    }        

  // Tell the ifreq struct which interface we want to use.
  strncpy((char *)&ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));

  retval = ioctl(sockData->sockInt, SIOCGIFINDEX, &ifr);
  if (retval < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error getting interface index value for interface %s\n",
		   thisint->intName);
      return XESOCKOP;
    }

  // Build our link layer socket struct, so we can bind it to a specific
  // interface.
  sockData->sll.sll_family = PF_PACKET;
  sockData->sll.sll_ifindex = ifr.ifr_ifindex;
  sockData->sll.sll_protocol = htons(ETH_P_EAPOL);

  // Bind to the interface.
  retval = bind(sockData->sockInt, (const struct sockaddr *)&sockData->sll, 
		sizeof(struct sockaddr_ll));
  if (retval < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error binding raw socket to interface %s!\n",
		   thisint->intName);
      return XESOCKOP;
    }

  // Get our MAC address.  (Needed for sending frames out correctly.)
  retval = ioctl(sockData->sockInt, SIOCGIFHWADDR, &ifr);
  if (retval < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error getting hardware (MAC) address for interface %s!\n",
		   thisint->intName);
      return XENOTINT;
    }

  // Store a copy of our source MAC for later use.
  memcpy((char *)&thisint->source_mac[0], (char *)&ifr.ifr_hwaddr.sa_data[0], 6);

  // Check if we want ALLMULTI mode, and enable it.
  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_ALLMULTI))
    {
      // Tell the ifreq struct which interface we want to use.
      strncpy((char *)&ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));

      if (ioctl(sockData->sockInt, SIOCGIFFLAGS, &ifr) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't determine if ALLMULTI is enabled!\n");
	} else {
	  if (ifr.ifr_flags & IFF_ALLMULTI)
	    {
	      debug_printf(DEBUG_INT, "Allmulti mode is already enabled on this device!\n");
	      thisint->flags |= ALLMULTI;
	    } else {
	      debug_printf(DEBUG_INT, "Allmulti is currently disabled on this device!\n");
	      thisint->flags &= ~ALLMULTI;
	    }
	}

      ifr.ifr_flags |= IFF_ALLMULTI;
      if (ioctl(sockData->sockInt, SIOCSIFFLAGS, &ifr) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set ALLMULTI mode on this interface!  We will continue anyway!\n");
	}
    }

  // Set our socket to non-blocking.
  sockopts = fcntl(sockData->sockInt, F_GETFL, 0);
  if (sockopts < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error getting socket options for interface %s!\n",
		   thisint->intName);
      return XENOTINT;
    }

  sockerr = fcntl(sockData->sockInt, F_SETFL, sockopts | O_NONBLOCK);
  if (sockerr < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting socket options for interface %s!\n",
		   thisint->intName);
      return XENOTINT;
    }

  // Set up wireless card drivers.
  cardif_set_driver(driver);

  // XXX Do we need this?
  /*
  if (!block_wpa) 
  {
    cardif_enable_wpa(thisint);
  } else {
    debug_printf(DEBUG_NORMAL, "Not turning on WPA support!\n");
  }
  */

  if (cardif_int_is_wireless(thisint->intName) == TRUE)
    {
      // If we have our destination set to AUTO, then preset our destination
      // address.
      if (globals->destination == DEST_AUTO)
	{
	  cardif_GetBSSID(thisint, thisint->dest_mac);
	}
    }
	  
  return XENONE;
}

/**************************************************************
 *
 * Tell the wireless card to start scanning for wireless networks.
 *
 **************************************************************/
int cardif_do_wireless_scan(struct interface_data *thisint, char passive)
{
  if (wireless == NULL) 
    {
      debug_printf(DEBUG_INT, "No valid wireless calls struct! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return -1;
    }

  // If we are already scanning, then we shouldn't get here, but go ahead 
  // and ignore it anyway.
  if (TEST_FLAG(thisint->flags, SCANNING) )
    {
      debug_printf(DEBUG_INT, "Got a request to start a new scan when one is"
		   " already in progress!  Ignoring!\n");
      return XENONE;
    }

  SET_FLAG(thisint->flags, SCANNING);
  config_ssid_clear();

  debug_printf(DEBUG_NORMAL, "Scanning for wireless networks ...\n");
  return wireless->scan(thisint, passive);
}

/**************************************************************
 *
 * Send a disassociate message.
 *
 **************************************************************/
int cardif_disassociate(struct interface_data *thisint, int reason_code)
{
  if (wireless == NULL) return -1;

  debug_printf(DEBUG_INT, "Called %s\n", __FUNCTION__);
  return wireless->disassociate(thisint, reason_code);
}

/**************************************************************
 *
 * Check to see if the BSSID value is valid.  If it is, return TRUE. If
 * it isn't return FALSE.
 *
 **************************************************************/
int cardif_valid_dest(struct interface_data *thisint)
{
  char baddest[6];

  if ((thisint->flags & IS_WIRELESS))
    {
      memset((char *)&baddest, 0x00, 6);
      if (memcmp(thisint->dest_mac, baddest, 6) == 0)
	{
	  debug_printf(DEBUG_INT, "All 0s for dest mac!\n");
	  return FALSE;

	}

      memset((char *)&baddest, 0x44, 6);
      if (memcmp(thisint->dest_mac, baddest, 6) == 0)
	{
	  debug_printf(DEBUG_INT, "All 4s for dest mac!\n");
	  return FALSE;
	}

      memset((char *)&baddest, 0xff, 6);
      if (memcmp(thisint->dest_mac, baddest, 6) == 0)
	{
	  debug_printf(DEBUG_INT, "All Fs for dest mac!\n");
	  return FALSE;
	}
    }  
  return TRUE;
}

/******************************************
 *
 * Return the socket number for functions that need it.
 *
 ******************************************/
int cardif_get_socket(struct interface_data *thisint)
{
  struct lin_sock_data *sockData;

  sockData = thisint->sockData;

  return sockData->sockInt;
}

/******************************************
 *
 * Clean up anything that was created during the initialization and operation
 * of the interface.  This will be called before the program terminates.
 *
 ******************************************/
int cardif_deinit(struct interface_data *thisint)
{
  struct ifreq ifr;
  uint16_t int16;
  struct lin_sock_data *sockData;

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data in %s!\n",
		   __FUNCTION__);
      return XEGENERROR;
    }
  
  sockData = thisint->sockData;

  debug_printf(DEBUG_EVERYTHING, "Cleaning up interface %s...\n",thisint->intName);

  if (TEST_FLAG(thisint->flags, IS_WIRELESS))
  {
    // Remove all of the keys that we have set.
    cardif_linux_clear_keys(thisint);

    debug_printf(DEBUG_INT, "Turning off WPA support/state.\n");
    // DKS -- we should undo all the changes we did to the interface. Currently this
    // does not do things like remove the WPA ie we added. This leaves the card in
    // a potentially inconsistent state.
    cardif_disable_wpa_state(thisint);
  }

  // Check if we want ALLMULTI mode, and enable it.
  if (TEST_FLAG(thisint->flags, ALLMULTI))
    {
      // Tell the ifreq struct which interface we want to use.
      strncpy((char *)&ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));

      if (ioctl(sockData->sockInt, SIOCGIFFLAGS, &ifr) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't get interface flags!\n");
	} else {
	  // Check if allmulti was disabled when we started.  If it was,
	  // then disable it again, so everything is good.
	  if (!(thisint->flags & ALLMULTI))
	    {
	      debug_printf(DEBUG_INT, "Turning off ALLMULTI mode!\n");

	      int16 = ifr.ifr_flags;

	      // ANDing the flags with 0xfdff will turn off the ALLMULTI flag.
	      ifr.ifr_flags = (int16 & 0xfdff);
	      if (ioctl(sockData->sockInt, SIOCSIFFLAGS, &ifr) < 0)
		{
		  debug_printf(DEBUG_NORMAL, "Couldn't set ALLMULTI mode on this interface!  We will continue anyway!\n");
		}
	    }
	}
    }

  close(sockData->sockInt);

  // Now clean up the memory.
  if (thisint->sockData != NULL)
    {
      free(thisint->sockData);
      thisint->sockData = NULL;
    }

  return XENONE;
}

/******************************************
 *
 * Set a WEP key.  Also, based on the index, we may change the transmit
 * key.
 *
 ******************************************/
int cardif_set_wep_key(struct interface_data *thisint, u_char *key, 
		       int keylen, int index)
{
  if (wireless == NULL) return -1;

  return wireless->set_wep_key(thisint, key, keylen, index);
}

/**********************************************************
 *
 * Set a TKIP key. 
 *
 **********************************************************/
int cardif_set_tkip_key(struct interface_data *thisint, char *addr, 
			      int keyidx, int settx, char *seq, int seqlen, 
			      char *key, int keylen)
{
  if (wireless == NULL) return -1;

  return wireless->set_tkip_key(thisint, (u_char *) addr, keyidx, settx, seq, seqlen,
				key, keylen);
}

/**********************************************************
 *
 * Set a CCMP (AES) key
 *
 **********************************************************/
int cardif_set_ccmp_key(struct interface_data *thisint, char *addr, int keyidx,
			int settx, char *seq, int seqlen, char *key,
			int keylen)
{
  if (wireless == NULL) return -1;

  return wireless->set_ccmp_key(thisint, (u_char *) addr, keyidx, settx, seq, seqlen,
				key, keylen);
}

/**********************************************************
 *
 * Delete a key
 *
 **********************************************************/
int cardif_delete_key(struct interface_data *intdata, int key_idx, int set_tx)
{
  if (wireless == NULL) return -1;

  return wireless->delete_key(intdata, key_idx, set_tx);
}

/******************************************
 *
 * Do whatever we need to do in order to associate based on the flags in
 * the ssids_list struct.
 *
 ******************************************/
void cardif_associate(struct interface_data *intdata, char *newssid)
{
  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed to %s!\n",
		   __FUNCTION__);
      return;
    }

  wireless->associate(intdata, newssid);
  
  if (intdata->cur_essid)
    {
      free(intdata->cur_essid);
      intdata->cur_essid = NULL;
    }
  
  intdata->cur_essid = strdup(newssid);
}

/******************************************
 *
 * Ask the wireless card for the ESSID that we are currently connected to.  If
 * this is not a wireless card, or the information is not available, we should
 * return an error.
 *
 ******************************************/
int cardif_GetSSID(struct interface_data *thisint, char *ssid_name)
{
  if (wireless == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "No valid call to get SSID for this driver!"
		   "\n");
      return -1;
    }

  if ((thisint == NULL) || (ssid_name == NULL)) 
  {
    debug_printf(DEBUG_INT, "NULL value passed to %s!\n", __FUNCTION__);
    return -1;
  }

  return wireless->get_ssid(thisint, ssid_name);
}

/******************************************
 *
 * Get the Broadcast SSID (MAC address) of the Access Point we are connected 
 * to.  If this is not a wireless card, or the information is not available,
 * we should return an error.
 *
 ******************************************/
int cardif_GetBSSID(struct interface_data *thisint, char *bssid_dest)
{
  if (wireless == NULL) return -1;

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data structure passed to %s!\n", __FUNCTION__);
      return -1;
    }

  if (bssid_dest == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid bssid_dest in %s!\n", __FUNCTION__);
      return -1;
    }

  return wireless->get_bssid(thisint, bssid_dest);
}

/******************************************
 *
 * Set the flag in the state machine that indicates if this interface is up
 * or down.  If there isn't an interface, we should return an error.
 *
 ******************************************/
int cardif_get_if_state(struct interface_data *thisint)
{
  int retVal;
  struct ifreq ifr;
  struct lin_sock_data *sockData;

  sockData = thisint->sockData;

  strncpy(ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));
  retVal = ioctl(sockData->sockInt, SIOCGIFFLAGS, &ifr);
  if (retVal < 0)
    {
      debug_printf(DEBUG_NORMAL, "Interface %s not found!\n", thisint->intName);
      return FALSE;
    }
  
  if ((ifr.ifr_flags & IFF_UP) == IFF_UP)
    {
      return TRUE;
    } else {
      SET_FLAG(thisint->flags, WAS_DOWN);
      return FALSE;
    }
  return XENONE;
}

/******************************************
 *
 * Send a frame out of the network card interface.  If there isn't an 
 * interface, we should return an error.  We should return a different error
 * if we have a problem sending the frame.
 *
 ******************************************/
int cardif_sendframe(struct interface_data *thisint)
{
  char nomac[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  int retval;
  struct lin_sock_data *sockData;
  struct config_network *network_data;

  sockData = thisint->sockData;

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context in %s!\n",
		   __FUNCTION__);
      return XEMALLOC;
    }

  if (thisint->send_size == 0) 
    {
      debug_printf(DEBUG_INT, "%s:%d -- Nothing to send!\n",
		   __FUNCTION__, __LINE__);
      return XENONE;
    }

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  // The frame we are handed in shouldn't have a src/dest, so put it in.
  memcpy(&thisint->sendframe[0], &thisint->dest_mac[0], 6);
  memcpy(&thisint->sendframe[6], &thisint->source_mac[0], 6);

  if (memcmp(nomac, (char *)&network_data->dest_mac[0], 6) != 0)
    {
      debug_printf(DEBUG_INT, "Static MAC address defined!  Using it!\n");
      memcpy(&thisint->sendframe[0], &network_data->dest_mac[0], 6);
    }

  debug_printf(DEBUG_EVERYTHING, "Frame to be sent (%d) : \n",
	       thisint->send_size);
  debug_hex_dump(DEBUG_EVERYTHING, thisint->sendframe, thisint->send_size);

  snmp_dot1xSuppEapolFramesTx();
  retval = sendto(sockData->sockInt, thisint->sendframe, thisint->send_size, 0,
		  (struct sockaddr *)&sockData->sll, sizeof(sockData->sll));
  if (retval <= 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't send frame! (%s)\n", strerror(errno));
    }

  thisint->send_size = 0;
  
  // Clear out the receive buffer so we don't accidently try to process it
  // again.
  bzero(thisint->recvframe, 1520);
  thisint->recv_size = 0;

  return retval;
}

/******************************************
 * 
 * Get a frame from the network.  Make sure to check the frame, to determine 
 * if it is something we care about, and act accordingly.
 *
 ******************************************/
int cardif_getframe(struct interface_data *thisint)
{
  int newsize=0;
  char dot1x_default_dest[6] = {0x01, 0x80, 0xc2, 0x00, 0x00, 0x03};
  struct lin_sock_data *sockData;
  u_char resultframe[1520];
  int resultsize;
  struct config_globals *globals;

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data in %s!\n",
		   __FUNCTION__);
      return XEGENERROR;
    }

  globals = config_get_globals();

  if (globals == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid configuration globals available "
		   "in %s!\n", __FUNCTION__);
      return XEGENERROR;
    }

  sockData = thisint->sockData;

  errno = 0;
  resultsize = 1520; /* overflows resultframe if too large, or should we increase resultframe instead? */

  newsize = recvfrom(sockData->sockInt, resultframe, resultsize, 0, 0, 0);
  if (newsize <= 0)
    {
      if (errno != EAGAIN)
	{
	  debug_printf(DEBUG_NORMAL, "Error (%d) : %s  (%s:%d)\n", errno,
		       strerror(errno), __FUNCTION__, __LINE__);
	}
      return XENOFRAMES;
    } else {
      debug_printf(DEBUG_EVERYTHING, "Got Frame (%d) : \n", newsize);
      debug_hex_dump(DEBUG_EVERYTHING, resultframe, newsize);
    }

  snmp_dot1xSuppEapolFramesRx();

  // Make sure that the frame we got is for us..
  if ((memcmp(&thisint->source_mac[0], &resultframe[0], 6) == 0) ||
      ((memcmp(&resultframe[0], &dot1x_default_dest[0], 6) == 0) &&
       (memcmp(&resultframe[6], &thisint->source_mac[0], 6) != 0)))
    {
      // Since we now know this frame is for us, record the address it
      // came from.
      snmp_dot1xSuppLastEapolFrameSource((u_char *)&resultframe[6]);

      resultsize = newsize;

      switch (globals->destination)
	{
	case DEST_AUTO:
	  // If it is a wired interface, only change the destination if
	  // the recieved frame destination isn't the multicast address.
	  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
	    {
	      if (memcmp(&resultframe[0], dot1x_default_dest, 6) == 0)
		{
		  break;
		}
	      // Otherwise, fall through.
	    }
	case DEST_SOURCE:
	  if (memcmp(thisint->dest_mac, &resultframe[6], 6) != 0)
	    {
	      debug_printf(DEBUG_INT, "Changing destination mac to source.\n");
	    }
	  memcpy(thisint->dest_mac, &resultframe[6], 6);
	  break;

	case DEST_MULTICAST:
	  memcpy(thisint->dest_mac, dot1x_default_dest, 6);
	  break;

	case DEST_BSSID:
	  cardif_GetBSSID(thisint, thisint->dest_mac);
	  break;

	default:
	  debug_printf(DEBUG_NORMAL, "Unknown destination mode!\n");
	  break;
	}

      thisint->recv_size = newsize;

      memcpy(thisint->recvframe, resultframe, newsize);
      return newsize;
    }

  // Otherwise it isn't for us. 
  debug_printf(DEBUG_INT, "Got a frame, not for us.\n");
  return XENOFRAMES;
}

/**************************************************************
 *
 * Set the state needed to associate to a WPA enabled AP, and actually
 * do a WPA authentication.
 *
 **************************************************************/
int cardif_enable_wpa_state(struct interface_data *thisint)
{
  if (wireless == NULL) return -1;

  debug_printf(DEBUG_INT, "WPA: Enabling WPA state on interface %s.\n",thisint->intName);

  return wireless->wpa_state(thisint, TRUE);
}

/**************************************************************
 *
 * Clear the state needed to associate to a WPA enabled AP, and actually
 * do a WPA authentication.
 *
 **************************************************************/
int cardif_disable_wpa_state(struct interface_data *thisint)
{
  if (wireless == NULL) return -1;

  return wireless->wpa_state(thisint, FALSE);
}

/**************************************************************
 *
 * Enable WPA (if it is supported.)
 *
 **************************************************************/
int cardif_enable_wpa(struct interface_data *thisint)
{
  if (wireless == NULL) return -1;

  debug_printf(DEBUG_INT, "WPA: Enabling WPA on interface %s.\n",thisint->intName);

  return wireless->wpa(thisint, TRUE);
}

/**************************************************************
 *
 * Call this when we roam to a different AP, or disassociate from an AP.
 *
 **************************************************************/
int cardif_wep_associate(struct interface_data *thisint, int zeros)
{
  if (wireless == NULL) return -1;

  if (!config_ssid_using_wep())
    {
      debug_printf(DEBUG_INT, "Doing WPA/WPA2 mode! Not "
		   "setting/unsetting keys.\n");
      return 0;
    }

  return wireless->wep_associate(thisint, zeros); 
}

/******************************************
 * 
 * Return true if there is a frame in the queue to be processed.
 *
 ******************************************/
int cardif_frameavail(struct interface_data *thisint)
{
  int newsize=0;
  char resultframe[1520];
  struct lin_sock_data *sockData;

  sockData = thisint->sockData;

  newsize = recvfrom(sockData->sockInt, &resultframe, 1520, MSG_PEEK, 0, 0);
  if (newsize > 0) return TRUE;

  return FALSE;
}

/******************************************
 *
 * Validate an interface, based on if it has a MAC address.
 *
 ******************************************/
int cardif_validate(char *interface)
{
  int sd, res;
  struct ifreq ifr;

  strncpy(ifr.ifr_name, interface, sizeof(interface)+1);

  sd = socket(PF_PACKET, SOCK_RAW, 0);
  if (sd < 0)
    return FALSE;
  res = ioctl(sd, SIOCGIFHWADDR, &ifr);
  close(sd);
  if (res < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get information for interface %s!\n",interface);
    } else {
      switch (ifr.ifr_hwaddr.sa_family)
	{
	case ARPHRD_ETHER:
	case ARPHRD_IEEE80211:
	  return TRUE;
	}
    }
  return FALSE;
}

/******************************************
 *
 * (en)/(dis)able countermeasures on this interface.
 *
 ******************************************/
int cardif_countermeasures(struct interface_data *intdata, char endis)
{
  if (wireless == NULL) return -1;

  return wireless->countermeasures(intdata, endis);
}

/******************************************
 *
 * (en)/(dis)able receiving of unencrypted frames on this interface.
 *
 ******************************************/
int cardif_drop_unencrypted(struct interface_data *intdata, char endis)
{
  if (wireless == NULL) return -1;

  if (config_ssid_using_wep()) return XENONE;
  
  return wireless->drop_unencrypted(intdata, endis);
}

/******************************************
 *
 * Get the name of an interface, based on an index value.
 *
 ******************************************/
#define PROC_DEV_FILE  "/proc/net/dev"

int cardif_get_int(int index, char *retInterface)
{
  FILE *fp;
  int hits;
  char line[1000], *lineptr;

  hits = 0;

  fp = fopen(PROC_DEV_FILE, "r");
  if (fp == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't access /proc/net/dev!\n");
      exit(250);
    }

  bzero(line, 1000);

  while ((hits <= index) && (fgets(line, 999, fp) != NULL))
    { 
      lineptr = strchr(line, ':');
      
      if (lineptr == NULL) continue;

      *lineptr = '\0';
      lineptr = &line[0];

      while (*lineptr == ' ') lineptr++;  // Strip out blanks.
      
      strcpy(retInterface, lineptr);
      hits++;
    }

  if (hits <= index)
    {
      debug_printf(DEBUG_INT, "No more interfaces to look at!\n");
      return XNOMOREINTS;
    }

  debug_printf(DEBUG_INT, "Found interface : %s\n",retInterface);

  fclose(fp);

  return XENONE;   // No errors.
}


/*******************************************************
 *
 * Check to see if an interface is wireless.  On linux, we look in
 * /proc/net/wireless to see if the interface is registered with the
 * wireless extensions.
 *
 *******************************************************/
#define PROC_WIRELESS_FILE  "/proc/net/wireless"

int cardif_int_is_wireless(char *interface)
{
  FILE *fp;
  char line[1000], *lineptr=NULL;
  int done;

  done = FALSE;

  fp = fopen(PROC_WIRELESS_FILE, "r");
  if (fp == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't access /proc/net/wireless!  (You probably don't have wireless extensions enabled!)\n");
      return -1;
    }

  bzero(line, 1000);

  while ((!done) && (fgets(line, 999, fp) != NULL))
    { 
      lineptr = strchr(line, ':');
      
      if (lineptr != NULL)
	{
	  
	  *lineptr = '\0';
	  lineptr = &line[0];
	  
	  while (*lineptr == ' ') lineptr++;  // Strip out blanks.
	  if (lineptr != NULL)
	    {
	      if (strcmp(lineptr, interface) == 0) done=TRUE;
	    }
	}
    }
  fclose(fp);
  
  if ((lineptr != NULL) && (strcmp(lineptr, interface) == 0))
    {
      debug_printf(DEBUG_INT, "Interface %s is wireless!\n",interface);
      return TRUE;
    } else {
      debug_printf(DEBUG_INT, "Interface %s is NOT wireless!\n",interface);
      return FALSE;
    }
  return XENONE;   // No errors.
}

int cardif_get_wpa_ie(struct interface_data *intdata, char *iedata, int *ielen)
{
  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error!  Invalid interface data structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (iedata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid bucket for IE data! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  return wireless->get_wpa_ie(intdata, iedata, ielen);
}

int cardif_get_wpa2_ie(struct interface_data *intdata, char *iedata, int *ielen)
{
  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error!  Invalid interface data structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (iedata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid bucket for IE data! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  return wireless->get_wpa2_ie(intdata, iedata, ielen);
}

/**************************************************************
 *
 * This function should clear out all keys that have been applied to the card.
 * It should be indepentant of the type (WEP/TKIP/CCMP) of key that was
 * applied.
 *
 **************************************************************/
int cardif_clear_keys(struct interface_data *intdata)
{
  int retVal = 0, i;

  debug_printf(DEBUG_INT, "Called %s!\n", __FUNCTION__);
  // Clear the TX key.
  retVal = cardif_delete_key(intdata, 0, 1);
  if (retVal != XENONE) 
    {
      debug_printf(DEBUG_NORMAL, "Error clearing default TX key!\n");
      return retVal;
    }

  // Now, clear out the broadcast/multicast/group key(s).
  for (i=0;i<4;i++)
    {
      retVal = cardif_delete_key(intdata, i, 0);
      if (retVal != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "Error clearing key %d!\n", i);
	  return retVal;
	}
    }

  return XENONE;
}

/*******************************************************************
 *
 * Attempt to reassociate to the network we were previously connected to.
 *
 *******************************************************************/
void cardif_reassociate(struct interface_data *intiface, u_char reason)
{
  if (!config_ssid_using_wep())
    {
      debug_printf(DEBUG_NORMAL, "%s is WPA/WPA2 capable. WPA/WPA2 is enabled "
		      "on this connection.\n", intiface->cur_essid);
      
      // Since we are doing WPA/WPA2, we need to disassociate from 
      // the network, and reassociate with WPA/WPA2 set up.
      cardif_enable_wpa(intiface);
      cardif_enable_wpa_state(intiface);

      // In general, we need to set requirements for reassociation & force 
      // reassociation.
      cardif_disassociate(intiface, reason);  
      cardif_clear_keys(intiface);
    }
  cardif_associate(intiface, intiface->cur_essid);  
}

/*****************************************************************
 *
 * Disable encryption on the card.  (Set the interface to open mode.)
 *
 *****************************************************************/
int cardif_enc_disable(struct interface_data *intdata)
{
  if (!intdata)
    {
      debug_printf(DEBUG_NORMAL, "Error!  Invalid interface data structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  return wireless->enc_disable(intdata);
}

/******************************************************************
 *
 * Determine what abilities this card has.  (WPA, WPA2, TKIP, CCMP, WEP40, 
 * etc.)
 *
 ******************************************************************/
void cardif_get_abilities(struct interface_data *intdata)
{
  if (!intdata)
    {
      debug_printf(DEBUG_NORMAL, "Error!  Invalid interface data structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  if (!wireless->enc_capabilities)
    {
      intdata->enc_capa = 0;
      return;
    }

  wireless->enc_capabilities(intdata);
}

/******************************************************************
 *
 * Change the BSSID that we are currently connected to.
 *
 ******************************************************************/
void cardif_setBSSID(struct interface_data *ctx, char *new_bssid)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Error!  Invalid interface data structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  if (!wireless->setbssid)
    {
      return;
    }

  wireless->setbssid(ctx, new_bssid);
}

/******************************************************************
 *
 * Wait for an interface to be "attached" to the system before starting to
 * attempt authentication with it.  This is a blocking call that should
 * *ONLY* return when the interface is available.  (Note : 'available'
 * does not mean that the interface is UP.  The 802.1X state machine will
 * deal with the interface if it is down.  We just need to wait for an
 * interface to exist so that we can use it.
 *
 ******************************************************************/
void cardif_wait_for_int(char *intname)
{
  int idx = -1;

  idx = if_nametoindex(intname);
  if (idx < 1)
    {
      debug_printf(DEBUG_NORMAL, "Waiting for interface to be inserted, or "
		   "driver to be loaded.\n");
      while (if_nametoindex(intname) < 1)
	{
	  sleep(1);
	}
    }
}

/********************************************************************
 *
 * The passive scan timer expried.  So, we need to issue a scan request,
 * and reset our timer to recheck the scan results periodically.
 *
 ********************************************************************/
void cardif_passive_scan_timeout(struct interface_data *ctx)
{
  struct config_globals *globals;
  char *ssid, *mac;

#warning FINISH!  We get scan data results, but we still need to do something with them.

  if (!TEST_FLAG(ctx->flags, PASV_SCANNING))
    {
     if (!TEST_FLAG(ctx->flags, SCANNING))
	{
	  timer_reset_timer_count(PASSIVE_SCAN_TIMER, 5);
	  cardif_do_wireless_scan(ctx, 1);
	  SET_FLAG(ctx->flags, PASV_SCANNING);
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "Got a request to start a new passive scan "
		       "when a previous one has not completed!\n");
	}
    }
  else
    {
      // If the scanning flag is no longer set, then we need to make a decision
      // about how to associate.
      debug_printf(DEBUG_NORMAL, "Looking for the best network to connect to.\n");
      // Clear the passive scanning flag.
      UNSET_FLAG(ctx->flags, PASV_SCANNING);

      // Reset the timer so that we scan again.
      
      globals = config_get_globals();
      
      if (!globals)
	{
	  debug_printf(DEBUG_NORMAL, "No global data!  Passive scanning will"
		       " be broken until the next time an authentication "
		       "completes.\n");
	}
      else
	{
	  debug_printf(DEBUG_EVERYTHING, "Resetting passive scan timer.\n");
	  timer_reset_timer_count(PASSIVE_SCAN_TIMER, globals->passive_timeout);
	  
	}

      ssid = config_ssid_get_desired_ssid(ctx);

      if (strcmp(ssid, ctx->cur_essid) != 0)
	{
	  debug_printf(DEBUG_NORMAL, "The best AP to connect to appears to be"
		       " in a different ESSID!  It is likely that your card"
		       " doesn't support the needed passive scanning flags."
		       "\n");
	  // Don't do anything with the result.
	} 
      else
	{
	  // We got a valid result.  So, see if it is a different AP.  If it
	  // is, then jump to it.
	  mac = config_ssid_get_mac();
	  if (memcmp(ctx->dest_mac, mac, 6) != 0)
	    {
	      debug_printf(DEBUG_INT, "Jumpping to a BSSID with a better "
			   "signal.  (BSSID : ");
	      debug_hex_printf(DEBUG_INT, mac, 6);
	      debug_printf_nl(DEBUG_INT, ")\n");

	      // Then change our BSSIDs.
	      cardif_setBSSID(ctx, mac);
	    }
	  else
	    {
	      debug_printf(DEBUG_EVERYTHING, "We are connected to the best "
			   "BSSID already.\n");
	    }
	}
    }
}

#endif
