/*******************************************************************
 * File: eapol.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapol.c,v 1.94 2005/10/17 03:56:53 chessing Exp $
 * $Date: 2005/10/17 03:56:53 $
 * $Log: eapol.c,v $
 * Revision 1.94  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.93  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.92  2005/09/26 02:04:07  shaftoe
 * Yet another round of log output cleanup.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.91  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.90  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.89  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <netinet/in.h>
#include <strings.h>

#include "profile.h"
#include "xsupconfig.h"
#include "snmp.h"
#include "frame_structs.h"
#include "statemachine.h"
#include "eapol.h"
#include "eap.h"
#include "cardif/cardif.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "key_statemachine.h"


/********************************************
 *
 * Set up anything that we will need for EAPoL.  This includes setting the
 * default values for the state machine.
 *
 ********************************************/
int eapol_init(struct interface_data *newint)
{
  if (!newint)
    {
      debug_printf(DEBUG_NORMAL, "NULL data passed in to eapol_init()!\n");
      return XEMALLOC;
    }

  statemachine_init(newint);

  return XENONE;
}

/*****************************************
 *
 * Do anything that is needed to clean up, and exit cleanly.
 *
 *****************************************/
int eapol_cleanup(struct interface_data *thisint)
{
  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "NULL data passed in to eapol_cleanup()!\n");
      return XEMALLOC;
    }

  statemachine_cleanup(thisint);

  return XENONE;
}

/*****************************************
 *
 * Get the EAPOL version that we should be using.
 *
 *****************************************/
u_char eapol_get_eapol_ver()
{
  struct config_network *network_data;
  u_char eapolver;

  // Get our current network configuration.
  network_data = config_get_network_config();

  eapolver = network_data->force_eapol_ver;

  // If we have not hard set the eapol version, then use the last
  // version that the authenticator gave us.  This isn't in line with
  // the 802.1X-REV-2004 standard, but many authenticators are checking
  // version #s, and breaking when it isn't 1. :-/
  if ((eapolver < 1) || (eapolver > MAX_EAPOL_VER))
    {
      eapolver = snmp_get_dot1xSuppLastEapolFrameVersion();
      if (eapolver == 0)
	{
	  eapolver = MAX_EAPOL_VER;
	} 
    }

  return eapolver;
}

/*****************************************
 *
 * Add an EAPOL header to a frame that is in the "frame" buffer.
 *
 *****************************************/
void eapol_build_header(int eapolType, int length, int eapolVer, char *frame)
{
  struct eapol_header *eapolheader;

  eapolheader = (struct eapol_header *)&frame[OFFSET_PAST_MAC];
  
  eapolheader->eapol_length = htons(length);
  eapolheader->frame_type = htons(0x888e);
  if ((eapolVer < 1) || (eapolVer > MAX_EAPOL_VER))
    {
      eapolheader->eapol_version = MAX_EAPOL_VER;
    } else {
      eapolheader->eapol_version = eapolVer;
    }
  eapolheader->eapol_type = eapolType;
}

/*****************************************
 *
 * Actually check to see if we have a frame, and process it if we do.
 *
 *****************************************/
int eapol_execute(struct interface_data *workint)
{
  int retval=0;
  struct eapol_header *temp;
  char *inframe;    // A pointer to our frame data.  (Normally will point
                    // to the newframe[] array.)

  if (!workint)
    {
      debug_printf(DEBUG_NORMAL, "NULL data passed in to eapol_execute()!\n");
      return XEMALLOC;
    }

  if (cardif_getframe(workint) < 0)
    {
      retval = XENOFRAMES;
      inframe = NULL;         // Have the EAP types process, if they are in
                              // a state to request information from the GUI.
    } else {

      // We want to let getframe be called, even if we don't have any
      // config information.  That will keep the frame queue empty so that
      // when we do have enough config information we can start by processing
      // an EAP request that is valid.  If we don't have any config informtion,
      // we should just bail here, and not return an error.

      inframe = (char *)&workint->recvframe;

      temp = (struct eapol_header *)&inframe[OFFSET_PAST_MAC];

      if (ntohs(temp->frame_type) == 0x888e)
	{
	  if (temp->eapol_version > MAX_EAPOL_VER)
	    {
	      debug_printf(DEBUG_EVERYTHING, "Got invalid EAPOL frame!\n");
	      snmp_inc_dot1xSuppInvalidEapolFramesRx();
	      workint->recv_size = 0;
	    } else {
	      snmp_dot1xSuppLastEapolFrameVersion(temp->eapol_version);

	      switch (temp->eapol_type)
		{
		case EAP_PACKET:
		  eap_process_header(workint);
		  break;

		case EAPOL_START:
		  debug_printf(DEBUG_NORMAL, "Got EAPoL-Start! Ignoring!\n");
		  return XEIGNOREDFRAME;

		case EAPOL_LOGOFF:
		  debug_printf(DEBUG_NORMAL, "Got EAPoL-Logoff! Ignoring!\n");
		  return XEIGNOREDFRAME;

		case EAPOL_KEY:
		  debug_printf(DEBUG_EVERYTHING, "Processing EAPoL-Key!\n");
		  workint->statemachine->rxKey = TRUE;
		  key_statemachine_run(workint);
		  return XGOODKEYFRAME;

		case EAPOL_ASF_ALERT:
		  debug_printf(DEBUG_NORMAL, "Got EAPoL-ASF-Alert!\n");
		  return XEIGNOREDFRAME;

		default:
		  debug_printf(DEBUG_NORMAL, "Unknown EAPoL type! (%02X)\n",
			       temp->eapol_type);
		  return XEIGNOREDFRAME;
		}
	    }
	} else {
	  debug_printf(DEBUG_EVERYTHING, "Got a frame, but it isn't an EAPoL "
		       "frame, ignoring.\n");
	}
    }

  // Process our state machine.
  statemachine_run(workint);

  return XENONE;
}
