
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/types.h>

#ifdef linux
#include <endian.h>
#elif defined (FreeBSD)
#include <machine/endian.h>
#elif defined (sgi)
#include <sys/endian.h>
#elif defined (sun)
#include <sys/byteorder.h>
#endif


#include "types.h"
#include "audio_file.h"
#include "ieee/ieee.h"
#include "aifc.h"
#include "endian.h"

static int aifc_new(Audio_File *af);

bool is_aifc(int fd)
{
   Aif_Form form;

    if ((read(fd,&form,sizeof(Aif_Form)))==-1) return(False);
   if ((lseek(fd,0,SEEK_SET))==-1) return(False);

#ifdef little_endian
    M_32_SWAP(form.form);
    M_32_SWAP(form.length);
    M_32_SWAP(form.type);
#endif
   if (form.form!=FORM) return(False);
   if ((form.type!=AIFF)&&(form.type!=AIFC)) return(False);
   return(True);
}


int aifc_open(Audio_File *af,int mode)
{
   int i,count;
   Aif_Form form;
   Aif_Chunk chunk;
   ulong frames,stamp,block_size,offset;
   short channels,bps;
   ID comp_typ;
   unsigned char freq[10],plength,*pstring;
   bool com_found=False;
   bool snd_found=False;
   bool ver_found=False;
   
   if (mode==AF_NEW) return(aifc_new(af));
 
   if (read(af->fd,&form,sizeof(Aif_Form))!=sizeof(Aif_Form)) 
     return(AF_ERROR);

#ifdef little_endianN
   M_32_SWAP(form.form);
   M_32_SWAP(form.length);
   M_32_SWAP(form.type);
#endif
   
   if (form.form!=FORM) return(AF_ERROR);
   
   if ((form.type!=AIFF)&&(form.type!=AIFC)) return(AF_ERROR);
   
   while(1) {
      if ((i=read(af->fd,&chunk,sizeof(Aif_Chunk)))!=sizeof(Aif_Chunk))
	return(AF_ERROR);
      
#ifdef little_endianN
      M_32_SWAP(chunk.type);
      M_32_SWAP(chunk.length);
#endif

      if (chunk.type==FVER) {
	 ver_found=True;
	 if ((i=read(af->fd,&stamp,sizeof(stamp)))!=sizeof(stamp))
	   return(AF_ERROR);
	 if ((lseek(af->fd,-i,SEEK_CUR))==-1) return(AF_ERROR);
      }
      
      if (chunk.type==COMM) {
	 com_found=True;
	 count=0;
	 i=read(af->fd,&channels,sizeof(channels));count+=i;
	 i=read(af->fd,&frames,sizeof(frames));count+=i;
	 i=read(af->fd,&bps,sizeof(bps));count+=i;
	 i=read(af->fd,&freq,sizeof(freq));count+=i;
	 if (form.type==AIFC) {
	    i=read(af->fd,&comp_typ,sizeof(comp_typ));count+=i;
	    
#ifdef little_endianN
	    M_32_SWAP(comp_typ);
#endif

	    i=read(af->fd,&plength,sizeof(plength));count+=i;
	    pstring=malloc(plength);
	    if ((i!=-1)&&(pstring!=NULL))
	      i=read(af->fd,pstring,plength);count+=i;
	 } else comp_typ=AIFF;
	 
	 if (i<=0) return(AF_ERROR);
	 
	 /* SSND Chunk was above, read COMM Chunk and jump to begin */
	 if (snd_found) { 
	    if ((lseek(af->fd,sizeof(Aif_Form),SEEK_SET))==-1)
	      return(AF_ERROR);
	 } else {
	    /* seek to begin of comm chunk (below we seek to chunk.length) */  
	    if ((lseek(af->fd,-count,SEEK_CUR))==-1) return(AF_ERROR);
	 }

#ifdef little_endianN
	 M_16_SWAP(channels);
	 M_32_SWAP(frames);
	 M_16_SWAP(bps);
#endif
      }

      /* pad byte */
      if (chunk.length % 2) {
	 chunk.length++;
      }

      if (chunk.type==SSND) {
	 snd_found=True;
	 count=chunk.length;
	
	 /* COMM Chunk was above -> break, else search for */
	 if (com_found) {
	    /* offset and block_size following after chunk length */
	    if ((i=read(af->fd,&offset,sizeof(offset)))!=sizeof(offset))
	      return(AF_ERROR);
	    if ((i=read(af->fd,&block_size,sizeof(block_size)))
		!=sizeof(block_size)) return(AF_ERROR);
	    if ((i=lseek(af->fd,0,SEEK_CUR))==-1) return(AF_ERROR);
	    af->headoffs=i;
	    break;
	 }
      }
      if ((lseek(af->fd,chunk.length,SEEK_CUR))==-1) return(AF_ERROR);
   }

   af->freq=(int) ConvertFromIeeeExtended(freq);
   af->bps=bps;
   af->channels=channels;
   af->comp=comp_typ;
   af->type=AF_AIFC;
   /* offset+block_size=8 */
   af->length=count-8;

   switch (comp_typ) {
    case AIFF:
    case CNONE: 
      af->comp=AF_PCM;
      break;
    default:
      return(AF_NOTSUPPORTED);
   }
   
   return(AF_AIFC);
}

int aifc_new(Audio_File *af)
{
   Aif_Form form;
   Aif_Chunk chunk;
   Com_Chunk com;
   int headoffs=0;
   char freq[10];
   ulong stamp,offset,block_size;
   int i;

   com.channels=af->channels;
   com.s_frames=af->length/(af->bps/8)/af->channels;
   com.bps=af->bps;
   com.freq=af->freq;
   com.comp_typ=af->comp;
   com.pstring=NULL;
   
   form.form=FORM;
   form.type=AIFC;
   /* FORM and length are not in chunk length, but AIFC is */
   form.length=sizeof(form.type);

   if ((i=write(af->fd,&form,sizeof(form)))!=sizeof(form))
     return(AF_ERROR);
   headoffs+=i;
   
   /* FVER chunk */
   chunk.type=FVER;
   stamp=AIFCVERSION_1;
   chunk.length=sizeof(stamp);
#ifdef little_endianN
   M_32_SWAP(chunk.type);
   M_32_SWAP(chunk.length);
   M_32_SWAP(stamp);
#endif
   if ((i=write(af->fd,&chunk,sizeof(chunk)))!=sizeof(chunk))
     return(AF_ERROR);
   headoffs+=i;
   if ((i=write(af->fd,&stamp,sizeof(stamp)))!=sizeof(stamp))
     return(AF_ERROR);
   headoffs+=i;
   
   /* COMM chunk */
   chunk.type=COMM;
   chunk.length=COMMLENGTH+sizeof(com.comp_typ);
   switch (com.comp_typ) {
    case AF_PCM:
      if ((com.pstring=malloc(strlen(CNONESTR)))==NULL) return(AF_ERROR);
      strcpy(com.pstring,CNONESTR);
      com.plength=strlen(com.pstring);
      com.comp_typ=CNONE;
      chunk.length+=sizeof(com.plength)+com.plength;
      chunk.length+=(chunk.length % 2);
      break;
    default:
      return(AF_ERROR);
   }

#ifdef little_endianN
   M_32_SWAP(chunk.type);
   M_32_SWAP(chunk.length);
   M_16_SWAP(com.channels);
   M_32_SWAP(com.s_frames);
   M_16_SWAP(com.bps);
   M_32_SWAP(com.comp_typ);
#endif
   ConvertToIeeeExtended(com.freq,freq);
   
   i=write(af->fd,&chunk,sizeof(chunk));headoffs+=i;
   i=write(af->fd,&com.channels,sizeof(com.channels));headoffs+=i;
   i=write(af->fd,&com.s_frames,sizeof(com.s_frames));headoffs+=i;
   i=write(af->fd,&com.bps,sizeof(com.bps));headoffs+=i;
   i=write(af->fd,&freq,sizeof(freq));headoffs+=i;
   i=write(af->fd,&com.comp_typ,sizeof(com.comp_typ));headoffs+=i;
   i=write(af->fd,&com.plength,sizeof(com.plength));headoffs+=i;
   i=write(af->fd,com.pstring,com.plength);headoffs+=i;
   
   /* pad byte, odd because com.plength is 1 byte */
   if (!(com.plength % 2)) {
      unsigned char pad=0;
      i=write(af->fd,&pad,sizeof(pad));headoffs+=i;
   }
   
   if (i==-1) {
      if (com.pstring!=NULL) free(com.pstring);
      return(AF_ERROR);
   }
   
   /* Pad byte if write chunk again, after some sound data is written */
   if (af->length>0) if (af->length % 2) af->length++;
   
   /* SSND chunk */
    
   chunk.type=SSND;
   chunk.length=af->length+sizeof(offset)+sizeof(block_size);
   offset=0;
   block_size=0;
   
#ifdef little_endianN
   M_32_SWAP(chunk.type);
   M_32_SWAP(chunk.length);
   M_32_SWAP(offset);
   M_32_SWAP(block_size);
#endif
   
   i=write(af->fd,&chunk,sizeof(chunk));headoffs+=i;
   i=write(af->fd,&offset,sizeof(offset));headoffs+=i;
   i=write(af->fd,&block_size,sizeof(block_size));headoffs+=i;
   if (i==-1) {
      if (com.pstring!=NULL) free(com.pstring);
      return(AF_ERROR);
   }
   
   form.length=headoffs+af->length-sizeof(form.form)-sizeof(form.length);
   
#ifdef little_endianN
   M_32_SWAP(form.form);
   M_32_SWAP(form.length);
   M_32_SWAP(form.type);
#endif
   
   if (aifc_seek(*af,0,SEEK_SET)==AF_ERROR) {
      if (com.pstring!=NULL) free(com.pstring);
      return(AF_ERROR);
   }
   
   if ((write(af->fd,&form,sizeof(form)))!=sizeof(form)) {
      if (com.pstring!=NULL) free(com.pstring);
      return(AF_ERROR);
   }
   
   if (aifc_seek(*af,headoffs,SEEK_SET)==AF_ERROR) {
      if (com.pstring!=NULL) free(com.pstring);
      return(AF_ERROR);
   }
   
   af->headoffs=headoffs;
   return(AF_AIFC);
}

int aifc_read(Audio_File af,char *buffer,int size)
{
   if ((size=read(af.fd,buffer,size))==-1) return (AF_ERROR);
#ifdef little_endianN
   /* swap byte order in 16bit sample */
   if (af.bps==16) {
      register int _tmp,i;
      for (i=0;i<size;i+=2) {
	 _tmp=buffer[i+1];
	 buffer[i+1]=buffer[i];
	 buffer[i]=_tmp;
      }
   }
#endif
   return(size);
}

int aifc_write(Audio_File af,char *buffer,int size)
{
#ifdef little_endianN
   /* swap byte order in 16bit sample */
   if (af.bps==16) {
      register int _tmp,i;
      for (i=0;i<size;i+=2) {
	 _tmp=buffer[i+1];
	 buffer[i+1]=buffer[i];
	 buffer[i]=_tmp;
      }
   }
#endif
   if ((size=write(af.fd,buffer,size))==-1) return (AF_ERROR);
   return(size);
}

int aifc_seek(Audio_File af,int pos,int mode)
{
   switch (af.comp) {
    case AF_PCM:
      return(lseek(af.fd,pos,mode));
   }
   return(AF_ERROR);
}

int aifc_close(Audio_File af)
{
   if (aifc_seek(af,0,SEEK_SET)==AF_ERROR) return(AF_ERROR);
   if (aifc_new (&af)==AF_ERROR) return(AF_ERROR);
   return(close(af.fd));
}



