#!perl -w
#
# KConfig -- encapsulate kernel configuration file and builtin modules
#   Copyright (C) 2005  Erik van Konijnenburg
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
#
#
# KConfig is a mapping from kernel define symbol to value.
# Common values are 'y' (compiled in), 'm' modular.
# The "no" choice in kernel configuration is represented by absence of the
# symbol.  Other values (for example decimal or hex numbers) also occur.
#
# We also maintain a list of known modules and corresponding kernel define.
# If the define is 'y', we need not load the corresponding module.
# This is not relevant to hardware (if a PCI controller is builtin,
# it will not occur in modules.pcimap, so no loading is attempted),
# but it does help with filesystems, where an 'ext3' line in /etc/fstab
# means the ext3 module needs to be loaded unless its compiled in.
#
# All this is complicated by the fact that modules may be aliases for each
# other.  Point in case: aes.  There is both CRYPTO_AES, CRYPTO_AES_586
# and CRYPTO_AES_X86_64.  All are three-state, the first is generic,
# and module aes_i586 contains a line MODULE_ALIAS ("aes") that makes modprobe
# decide that aes_i586 is just as good as aes.  Thus for the code that loads
# an encryption algorithm just before cryptsetup, the difference between
# aes and aes_i865 is transparant.  However, the code in yaird that decides
# aes is built into the kernel so no modprobe is required to understand
# that CRYPTO_AES_586 is equivalent to CRYPTO_AES.
#

use strict;
use warnings;
use Base;
use Conf;
package KConfig;

my $kConfMap = undef;
my $kConfList = undef;

sub init () {
	if (defined ($kConfMap)) {
		return;
	}
	$kConfMap = {};
	my $name = Conf::get('kernConf');
	if (! open (IN, "<", "$name")) {
		Base::fatal ("can't open kernel config file $name");
	}
	while (defined (my $line = <IN>)) {
		chomp $line;
		$line =~ s/^\s+//;
		$line =~ s/\s+$//;
		$line =~ s/#.*//;
		next if ($line eq "");

		if ($line !~ /^([A-Z][A-Za-z0-9_]+)=(.+)$/) {
			# lowercase is uncommon, but: CONFIG_SCx200=m
			Base::fatal "bad line in $name: $line";
		}
		my $key = $1;
		my $value = $2;
		if ($value eq 'y'
			|| $value eq 'm'
			|| $value =~ /^-?\d+$/
			|| $value =~ /^0x[[:xdigit:]]+$/
			|| $value =~ /^"[-a-zA-Z0-9@\$,.:_\/= ]*"$/
		) {
			$kConfMap->{$key} = $value;
		}
		else {
			Base::fatal "bad value in $name: $line";
		}
	}
	if (! close (IN)) {
		Base::fatal "could not read kernel config file $name";
	}
	$kConfList = [ sort keys %{$kConfMap} ];

	Base::debug ("KConfig::init completed - $name");
}

#
# Map module name to kernel define.  Module names here
# must use hyphen, not underscore.
#
my $moduleMap = {
	# user interface devices
	fbcon => [ 'FRAMEBUFFER_CONSOLE' ],
	vesafb => [ 'FB_VESA' ],
	serio => [ 'SERIO' ],
	i8042 => [ 'SERIO_I8042' ],
	usbhid => [ 'USB_HID' ],
	atkbd => [ 'KEYBOARD_ATKBD' ],
	sunkbd => [ 'KEYBOARD_SUNKBD' ],
	mousedev => [ 'INPUT_MOUSEDEV' ],
	evdev => [ 'INPUT_EVDEV' ],
	psmouse => [ 'MOUSE_PS2' ],
	sermouse => [ 'MOUSE_SERIAL' ],

	# file systems
	ext2 => [ 'EXT2_FS' ],
	ext3 => [ 'EXT3_FS' ],
	jbd => [ 'JBD' ],
	reiserfs => [ 'REISERFS_FS' ],
	jfs => [ 'JFS_FS' ],
	xfs => [ 'XFS_FS' ],
	minix => [ 'MINIX_FS' ],
	romfs => [ 'ROMFS_FS' ],
	isofs => [ 'ISO9660_FS' ],
	udf => [ 'UDF_FS' ],
	fat => [ 'FAT_FS' ],
	msdos => [ 'MSDOS_FS' ],
	vfat => [ 'VFAT_FS' ],
	# broken, and nonmodular: umsdos => [ 'UMSDOS_FS' ],
	ntfs => [ 'NTFS_FS' ],
	adfs => [ 'ADFS_FS' ],
	affs => [ 'AFFS_FS' ],
	hfs => [ 'HFS_FS' ],
	hfsplus => [ 'HFSPLUS_FS' ],
	befs => [ 'BEFS_FS' ],
	bfs => [ 'BFS_FS' ],
	efs => [ 'EFS_FS' ],
	jffs => [ 'JFFS_FS' ],
	jffs2 => [ 'JFFS2_FS' ],
	cramfs => [ 'CRAMFS' ],
	freevxfs => [ 'VXFS_FS' ],
	hpfs => [ 'HPFS_FS' ],
	qnx4 => [ 'QNX4FS_FS' ],
	sysv => [ 'SYSV_FS' ],
	ufs => [ 'UFS_FS' ],
	nfs => [ 'NFS_FS' ],
	smbfs => [ 'SMB_FS' ],
	cifs => [ 'CIFS' ],
	ncpfs => [ 'NCP_FS' ],
	coda => [ 'CODA_FS' ],
	kafs => [ 'AFS_FS' ],

	# native language support
	'nls-cp437' => [ 'NLS_CODEPAGE_437' ],
	'nls-cp737' => [ 'NLS_CODEPAGE_737' ],
	'nls-cp775' => [ 'NLS_CODEPAGE_775' ],
	'nls-cp850' => [ 'NLS_CODEPAGE_850' ],
	'nls-cp852' => [ 'NLS_CODEPAGE_852' ],
	'nls-cp855' => [ 'NLS_CODEPAGE_855' ],
	'nls-cp857' => [ 'NLS_CODEPAGE_857' ],
	'nls-cp860' => [ 'NLS_CODEPAGE_860' ],
	'nls-cp861' => [ 'NLS_CODEPAGE_861' ],
	'nls-cp862' => [ 'NLS_CODEPAGE_862' ],
	'nls-cp863' => [ 'NLS_CODEPAGE_863' ],
	'nls-cp864' => [ 'NLS_CODEPAGE_864' ],
	'nls-cp865' => [ 'NLS_CODEPAGE_865' ],
	'nls-cp866' => [ 'NLS_CODEPAGE_866' ],
	'nls-cp869' => [ 'NLS_CODEPAGE_869' ],
	'nls-cp936' => [ 'NLS_CODEPAGE_936' ],
	'nls-cp950' => [ 'NLS_CODEPAGE_950' ],
	'nls-cp932' => [ 'NLS_CODEPAGE_932' ],
	'nls-cp949' => [ 'NLS_CODEPAGE_949' ],
	'nls-cp874' => [ 'NLS_CODEPAGE_874' ],
	'nls-iso8859-8' => [ 'NLS_ISO8859_8' ],
	'nls-cp1250' => [ 'NLS_CODEPAGE_1250' ],
	'nls-cp1251' => [ 'NLS_CODEPAGE_1251' ],
	'nls-ascii' => [ 'NLS_ASCII' ],
	'nls-iso8859-1' => [ 'NLS_ISO8859_1' ],
	'nls-iso8859-2' => [ 'NLS_ISO8859_2' ],
	'nls-iso8859-3' => [ 'NLS_ISO8859_3' ],
	'nls-iso8859-4' => [ 'NLS_ISO8859_4' ],
	'nls-iso8859-5' => [ 'NLS_ISO8859_5' ],
	'nls-iso8859-6' => [ 'NLS_ISO8859_6' ],
	'nls-iso8859-7' => [ 'NLS_ISO8859_7' ],
	'nls-iso8859-9' => [ 'NLS_ISO8859_9' ],
	'nls-iso8859-13' => [ 'NLS_ISO8859_13' ],
	'nls-iso8859-14' => [ 'NLS_ISO8859_14' ],
	'nls-iso8859-15' => [ 'NLS_ISO8859_15' ],
	'nls-koi8-r' => [ 'NLS_KOI8_R' ],
	'nls-koi8-u' => [ 'NLS_KOI8_U' ],
	'nls-utf8' => [ 'NLS_UTF8' ],

	# network
	'af-packet' => [ 'PACKET' ],

	# device mapper: raid and lvm.
	linear => [ 'MD_LINEAR' ],
	raid0 => [ 'MD_RAID0' ],
	raid1 => [ 'MD_RAID1' ],
	raid10 => [ 'MD_RAID10' ],
	raid5 => [ 'MD_RAID5' ],
	raid6 => [ 'MD_RAID6' ],
	multipath => [ 'MD_MULTIPATH' ],
	faulty => [ 'MD_FAULTY' ],
	md => [ 'BLK_DEV_MD' ],
	'dm-mod' => [ 'BLK_DEV_DM' ],
	'dm-crypt' => [ 'DM_CRYPT' ],
	'dm-snapshot' => [ 'DM_SNAPSHOT' ],
	'dm-mirror' => [ 'DM_MIRROR' ],
	'dm-zero' => [ 'DM_ZERO' ],

	# crypto
	hmac => [ 'CRYPTO_HMAC' ],
	'crypto-null' => [ 'CRYPTO_NULL' ],
	md4 => [ 'CRYPTO_MD4' ],
	md5 => [ 'CRYPTO_MD5' ],
	sha1 => [ 'CRYPTO_SHA1' ],
	sha256 => [ 'CRYPTO_SHA256' ],
	sha512 => [ 'CRYPTO_SHA512' ],
	wp512 => [ 'CRYPTO_WP512' ],
	des => [ 'CRYPTO_DES' ],
	blowfish => [ 'CRYPTO_BLOWFISH' ],
	twofish => [ 'CRYPTO_TWOFISH' ],
	serpent => [ 'CRYPTO_SERPENT' ],
	aes => [ 'CRYPTO_AES', 'CRYPTO_AES_586', 'CRYPTO_AES_X86_64' ],
	cast5 => [ 'CRYPTO_CAST5' ],
	cast6 => [ 'CRYPTO_CAST6' ],
	arc4 => [ 'CRYPTO_ARC4' ],
	tea => [ 'CRYPTO_TEA' ],
	khazad => [ 'CRYPTO_KHAZAD' ],
	anubis => [ 'CRYPTO_ANUBIS' ],
	deflate => [ 'CRYPTO_DEFLATE' ],
	'michael-mic' => [ 'CRYPTO_MICHAEL_MIC' ],
	crc32c => [ 'CRYPTO_CRC32C' ],
	tcrypt => [ 'CRYPTO_TEST' ],

	# IDE
	'ide-generic' => [ 'IDE_GENERIC' ],
	'ide-disk' => [ 'BLK_DEV_IDEDISK' ],
	'ide-cd' => [ 'BLK_DEV_IDECD' ],
	'ide-tape' => [ 'BLK_DEV_IDETAPE' ],
	'ide-floppy' => [ 'BLK_DEV_IDEFLOPPY' ],

	# SCSI
	'sd-mod' => [ 'BLK_DEV_SD' ],
	'st' => [ 'CHR_DEV_ST' ],
	'sr-mod' => [ 'BLK_DEV_SR' ],
	'sg' => [ 'CHR_DEV_SG' ],

	# MMC block device driver
	'mmc-block' => [ 'MMC_BLOCK' ],

	# Compaq Smart Array controllers
	'cpqarray' => [ 'BLK_CPQ_DA' ],
	'cciss' => [ 'BLK_CPQ_CISS_DA' ],
};


#
# all -- return a list of all known configuration defines
#
sub all	() {
	init;
	return $kConfList;
}

#
# getValue -- return config value for a given config key
#
sub getValue ($) {
	my ($confKey) = @_;
	init;
	my $confVal = $kConfMap->{"CONFIG_$confKey"};
	if (defined ($confVal)) {
		# config type of "string"
		$confVal =~ s/^"(.*)"$/$1/;
	}
	return $confVal;
}

#
# allKnownModules -- return list of all module names for
# which a corresponding kernel define is known.
#
sub allKnownModules () {
	init;
	return [ sort keys %{$moduleMap} ];
}

#
# isBuiltIn -- true if the module is known to be compiled
# into the kernel.  Or if something is compiled in that 
# has the module as an alias.
#
sub isBuiltIn ($) {
	my ($module) = @_;
	init;
	$module =~ s!_!-!g;
	my $confKeys = $moduleMap->{$module};
	if (! defined ($confKeys)) {
		Base::debug ("KConfig::isBuiltIn $module - unknown");
		return 0;
	}
	for my $confKey (@{$confKeys}) {
		my $confVal = $kConfMap->{"CONFIG_$confKey"};
		if (defined ($confVal) && $confVal eq 'y') {
			Base::debug ("KConfig::isBuiltIn $module, $confKey - yes");
			return 1;
		}
	}

	Base::debug ("KConfig::isBuiltIn $module - no");
	return 0;
}


1;

