/*****************************************************************************
 *                                                                           *
 *   Copyright (C) 2005 by Chazal Francois             <neptune3k@free.fr>   *
 *   website : http://workspace.free.fr                                      *
 *                                                                           *
 *                     =========  GPL Licence  =========                     *
 *    This program is free software; you can redistribute it and/or modify   *
 *   it under the terms of the  GNU General Public License as published by   *
 *   the  Free  Software  Foundation ; either version 2 of the License, or   *
 *   (at your option) any later version.                                     *
 *                                                                           *
 *****************************************************************************/

//== INCLUDE REQUIREMENTS =====================================================

/*
** Local libraries */
#include "tabbed_widget.h"
#include "tabbed_widget.moc"



//== CONSTRUCTORS AND DESTRUCTORS =============================================


TabbedWidget::TabbedWidget(QWidget * parent, const char * name) : QWidget(parent, name)
{
    // Initializes the background pixmap ------------------

    root_pixmap = new KRootPixmap(this, "Transparent background");
    root_pixmap->setCustomPainting(true);
    root_pixmap->start();

    connect(root_pixmap, SIGNAL(backgroundUpdated(const QPixmap &)), this, SLOT(slotUpdateBuffer(const QPixmap &)));

    selected_font = font();
    unselected_font = font();
    selected_font.setBold(true);

    refreshBuffer();
}

TabbedWidget::~TabbedWidget()
{
    delete root_pixmap;
}



//== PUBLIC METHODS ===========================================================


/******************************************************************************
** Adds an item to the tabs
*****************************/

void    TabbedWidget::addItem(int id)
{
    items.append(id);
    areas.append(0);
    captions.append("console "+ QString::number(id + 1));

    refreshBuffer();
}


/******************************************************************************
** Selects an item in the tabs
********************************/

void    TabbedWidget::selectItem(int id)
{
    selected_id = items.findIndex(id);
    refreshBuffer();
}


/******************************************************************************
** Removes an item from the tabs
**********************************/

int     TabbedWidget::removeItem(int id)
{
    int index = items.findIndex(id);

    items.remove(items.at(index));
    areas.remove(areas.at(index));
    captions.remove(captions.at(index));

    if (index != items.size())
        selected_id = index;
    else if (index != 0)
        selected_id = index - 1;
    else
        selected_id = -1;

    refreshBuffer();

    if (selected_id != -1)
        emit itemSelected(items[selected_id]);

    return selected_id;
}


/******************************************************************************
** Selects the new item in the list
*************************************/

void    TabbedWidget::selectNextItem()
{
    if (selected_id != items.size() - 1)
        selected_id++;

    refreshBuffer();

    emit itemSelected(items[selected_id]);
}


/******************************************************************************
** Selects the previous item in the list
******************************************/

void    TabbedWidget::selectPreviousItem()
{
    if (selected_id != 0)
        selected_id--;

    refreshBuffer();

    emit itemSelected(items[selected_id]);
}


/******************************************************************************
** Sets the font color
************************/

void    TabbedWidget::setFontColor(const QColor & color)
{
    font_color = color;
}


/******************************************************************************
** Loads the background pixmap
********************************/

void    TabbedWidget::setBackgroundPixmap(const QString & path)
{
    background_image.load(path);
    resize(width(), background_image.height());

    repaint();
}


/******************************************************************************
** Loads the separator pixmap
*******************************/

void    TabbedWidget::setSeparatorPixmap(const QString & path)
{
    separator_image.load(path);
}


/******************************************************************************
** Loads the unselected background pixmap
*******************************************/

void    TabbedWidget::setUnselectedPixmap(const QString & path)
{
    unselected_image.load(path);
}


/******************************************************************************
** Loads the selected background pixmap
*****************************************/

void    TabbedWidget::setSelectedPixmap(const QString & path)
{
    selected_image.load(path);
}


/******************************************************************************
** Loads the selected left corner pixmap
******************************************/

void    TabbedWidget::setSelectedLeftPixmap(const QString & path)
{
    selected_left_image.load(path);
}


/******************************************************************************
** Loads the selected right corner pixmap
*******************************************/

void    TabbedWidget::setSelectedRightPixmap(const QString & path)
{
    selected_right_image.load(path);
}



//== PROTECTED METHODS ========================================================


/******************************************************************************
** Repaints the widget when asked
***********************************/

void    TabbedWidget::paintEvent(QPaintEvent *)
{
    bitBlt(this, 0, 0, &buffer_image);
}


/******************************************************************************
** Modifies button's state (mouse up)
***************************************/

void    TabbedWidget::mouseReleaseEvent(QMouseEvent *e)
{
    uint    id;
    int     width;

    if (e->x() < 0)
        return ;

    for (id = 0, width = 0; (id < areas.size()) && (e->x() >= width); id++)
        width += areas[id];

    if (e->x() <= width)
    {
        selected_id = id - 1;
        refreshBuffer();

        emit itemSelected(items[selected_id]);
    }

    if (e->button() == Qt::RightButton)
    {
        bool ok;
        QString tab_name = KInputDialog::getText(i18n("Tab Name"),
                           i18n("Enter new tab name"),
                           captions[selected_id], &ok);

        if (ok && (tab_name != NULL))
        {
            captions[selected_id] = tab_name;
            refreshBuffer();
        }
    }
}



//== PRIVATE METHODS ==========================================================


/******************************************************************************
** Refreshes the back buffer
******************************/

void   TabbedWidget::refreshBuffer()
{
    int         x = 0;
    QPainter    painter;

    buffer_image.resize(size());
    painter.begin(&buffer_image);

    painter.drawTiledPixmap(0, 0, width(), height(), desktop_image);

    for (uint i = 0; i < items.size(); i++)
        x = drawButton(i, painter);

    painter.drawTiledPixmap(x, 0, width() - x, height(), background_image);

    painter.end();

    repaint();
}


/******************************************************************************
** Draws the tabs button
**************************/

const int   TabbedWidget::drawButton(int id, QPainter & painter)
{
    static int  x = 0;
    QPixmap     tmp_pixmap;

    areas[id] = 0;
    x = (!id) ? 0 : x;

    // Initializes the painter ----------------------------

    painter.setPen(font_color);
    painter.setFont((id == selected_id) ? selected_font : unselected_font);

    // draws the left border ------------------------------

    if (id == selected_id)
        tmp_pixmap = selected_left_image;
    else if (id != selected_id + 1)
        tmp_pixmap = separator_image;

    painter.drawPixmap(x, 0, tmp_pixmap);
    areas[id] += tmp_pixmap.width();
    x += tmp_pixmap.width();

    // draws the main contents ----------------------------

    int             width;
    QFontMetrics    metrics(painter.font());

    width = metrics.width(captions[id]) + 10;

    tmp_pixmap = (id == selected_id) ? selected_image : unselected_image;
    painter.drawTiledPixmap(x, 0, width, height(), tmp_pixmap);

    painter.drawText(x, 0, width + 1, height() + 1,
                     Qt::AlignHCenter | Qt::AlignVCenter, captions[id]);

    areas[id] += width;
    x += width;

    // draws the right border if needed -------------------

    if (id == selected_id)
    {
        painter.drawPixmap(x, 0, selected_right_image);
        areas[id] += selected_right_image.width();
        x += selected_right_image.width();
    }
    else if (id == (int) items.size() - 1)
    {
        painter.drawPixmap(x, 0, separator_image);
        areas[id] += separator_image.width();
        x += separator_image.width();
    }

    return x;
}



//== PRIVATE SLOTS ============================================================


void    TabbedWidget::slotUpdateBuffer(const QPixmap & pixmap)
{
    desktop_image = pixmap;

    refreshBuffer();
}
