;;; yasgml.el --- Yet Another linuxdocSGML mode

;; Copyright (C) 1999 by Free Software Foundation, Inc.

;; Author: Takashi Nishimoto <g96p0935@mse.waseda.ac.jp>
;; Keywords: sgml

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; YaTeX 桼Τ sgml mode Ǥ
;;
;; SGML ȤΤϡ HTML ˻ʸˡΥ** plain text 
;; LaTeXHTMLinfo ʤɤΤΥեޥåȤʸͥʪ
;; ǤĤޤꡢĤΥ饪饤ʸ񡢤줤ʰʪۡ
;; ڡκ뤳ȤǤʸݼΩޤ
;;
;; LaTeX 񤯤Τ YaTeX ȤȤ䤹ͭ̾ major-mode 
;; ޤ HTML 񤯤Τˤ yahtml ޤȤˡ 
;; yahtml  YaTeX Ʊ褦ˡˤʤäƤ뤿ᡢ YaTeX 桼
;; ϤʤޤYaTeX 桼 SGML ʸ񤯤Ȥ YaTeX 
;; Ӥ major-mode ̵ΤȤȤˤʤޤǰ
;; ¸ߤʤ褦ʤΤǡ˻ޤ
;;
;;  yasgml ϡ linuxdoc DTD ˹碌ƺäƤޤ linuxdoc DTD 
;; ϡȤ Linux  HOWTO ʸ񤯤Τ˺줿ΤǤǤ
;; ̤ʸ񤭤ˤȤޤ
;;
;; Хոʤɤޤ顢 g96p0935@mse.waseda.ac.jp ޤǥ᡼
;; ꤤޤǤ¤˾ˤϱ褦ȻפäƤޤ
;;
;; ɥȽ񤭤ˤ뤵 memo-mode.el ѤƤޤ
;; ޡʸϤ¤ΤˤΤǤƱ̤
;; ǥȤʤ顢󤢤뤤ϽطɽƤޤ
;; memo-mode.el  
;; http://mibai.tec.u-ryukyu.ac.jp/%7Eoshiro/Programs/elisp/memo-mode.el
;; ˤޤ
;;

;;; Installation;

;;-->YaTeX 򥤥󥹥ȡ뤷ƤΥץ YaTeX Ȥ
;;   ° yahtml Ȥޤ
;;   
;;-->sgml-tools 򥤥󥹥ȡ뤷ƤꥸʥΤޤޤǤܸ
;;   бƤޤΤǡܸѥåƤ뤳ȡ
;;   ftp://jf.linux.or.jp/pub/JF/misc/sgml-tools/ ˤޤ
;;   
;;-->Υե (yasgml.el)  load-path ̤äǥ쥯ȥ˥ԡޤ
;;   
;;-->.emacs ꤷޤ
;;  -->(setq auto-mode-alist
;;       (nconc
;;         '(("\\.sgml$" . yasgml-mode)) auto-mode-alist))
;;     (autoload 'yasgml-mode "yasgml" "Major mode for editing files of SGML." t)
;;     2Ԥ .emacs ˲äޤ
;;  -->¾ΥޥˤĤƤϡ֥ޥǽѿˤĤơ
;;     򻲾ȤΤȡ
;;     
;;-->Хȥѥ뤷ޤ
;;  -->M-x byte-compile (Υե) ¹Ԥ뤳ȡ

;;-->ޥǽѿˤĤ  ()ϥǥե
;;  -->yasgml-author ("Takashi Nishimoto (g96p0935@mse.waseda.ac.jp)")
;;    -->SGMLʸκ(ʤʬ)̾
;;  -->yasgml-default-article ("article")
;;    -->SGMLʸΥǥեȥ
;;  -->yasgml-default-dtd ("linuxdoc")
;;    -->ǥեȤ DTDޤ linuxdoc ˤбƤޤ
;;  -->yasgml-prefix ("\C-c")
;;    -->yasgml ΥץեåǤ
;;  -->yasgml-fill-column (72)
;;    -->yasgml ˤ fill-column Ǥ
;;  -->yasgml-fill-prefix (nil)
;;    -->yasgml ˤ fill-prefix Ǥ
;;  -->yasgml-environment-indent (1)
;;    -->ĶΥǥȤοǤ
;;  -->yasgml-special-characters-input-method ('auto)
;;    -->üʸˡǤ 'auto  'bell Τɤ餫Ǥ
;;       üʸˡפ򻲾ȤΤ
;;  -->yasgml-do-special-characters-check (t)
;;    -->ץåüʸˤĤƤδñʥåԤɤ
;;  -->yasgml-typeset-program ("sgml2txt -f -l ja -c nippon")
;;    -->ץåȤԤޥɤǤ
;;       "[prefix] tp" Ǥˤä줿 .txt СΥץӥ塼
;;       Ԥޤ
;;  -->yasgml-format-program-alist (ĹΤǾά)
;;    -->եޥåȤȤ뤿Υޥɤ alist Ǥ
;;  -->yasgml-preview-function ('yasgml-show-txt-other-window)
;;    -->ץӥ塼η̤ɽХåեɤΤ褦ɽ뤫
;;       ޤ
;;      -->'yasgml-show-txt-other-window ʤ顢̤2ʬ䤷ɽޤ
;;      -->'yasgml-show-txt-other-frame ʤ顢̤ʥե졼ɽޤ
;;      -->'yasgml-show-txt-bully ʤ顢̤ǥץӥ塼Хåեɽ
;;         ޤ

;;-->üʸˡ
;;  -->SGML ˤơ &<>$#%"~</ üʸȤư 
;;     ޤ餽ΤޤϤǤޤ verbcodequote ĶǤ
;;      󤬴ˤʤޤϤ &  </ ϤΤޤޤǤϤǤ
;;      줾 &amp;&etago; (ʲ &foo; त&󥹡
;;     ȸƤӤޤ)ϤʤФʤޤüʸˤĤƲͤ
;;     ʸ񤤤ƤȡĤδ֤ˤüʸ򤽤ΤޤϤƤޤ
;;     ǽ⤤Ǥ⡢顼򸫤ĤΤϤȤƤ⺤Ǥ
;;     ϤǵդСä顼ǽ
;;     㤯ʤޤ
;;  -->ǥեȤǤüʸΥáȤʸʸɤ
;;     ˡ׽ФʸϤޤ
;;    -->yasgml-special-characters-input-method  'auto ˤʤäƤޤ
;;    -->& 򲡤ȤϤĤǤ &amp; Ϥޤ
;;      -->Ͻ鿴ԤΤθǤΤǡʤС
;;         (setq yasgml-special-characters-input-method 'bell)
;;          .emacs ˲äƤξ֤Ǥ & 򲡤Ȥ٥
;;         Ĥꡢ & üʸǤ뤳Ȥٹ𤹤褦ˤʤޤ
;;    -->< θ / ϤȤưŪ &etago; Ѥޤ
;;    -->ʳʸˤĤ
;;      -->⤷ verbcodequote ĶˤȤϤʸ򤽤Τ
;;         ޽Ϥޤ
;;      -->ǤʤȤϡʸб & Ϥ
;;         


;;; Usage:

;; ʲˤơä˥ޥ򤷤Ƥʤ¤ꡢ[prefix] 
;; C-c ̣ޤ
;;  * [prefix] b X	`</enum>' ȤäλɬפȤʤ`enum'Τ褦
;;			ʴĶ䴰Ϥޤ
;;  * [prefix] s	ʲΤ褦䴰Ԥޤ
;;                        <url url="http://aaaa.com" name="test">
;;                        <sect>̾<p>
;;  * [prefix] l	`<em> ... </em>'  `<tag> ... </tag>' Τ褦
;;			ʥƥȥΥ䴰ޤ
;;			䴰ǽ̾ [prefix] b 䴰ǤΤ
;;			ǽ񤭤ȤˤѤ뤳Ȥޤ
;;  * [prefix] m	`<toc>'  `<item> '`<p>' ñΥ䴰
;;			Ԥޤ
;;  * [prefix] p	֤<p>ޤ
;;  * [prefix] g	б륿<verb> <-> </verb>  url="xxx" 
;;			褦 TAG ˥פޤ
;;                        url=hoge.html ξhoge.htmlӤ
;;			
;;  * [prefix] k	ݥȾ SGML õޤ
;;			⤷ universal-argument դ(C-u˲
;;			)SGMLǰϤޤ줿ƤƱ˾õޤ
;;  * [prefix] c	ݥȾΥѹޤ
;;  * [prefix] t j	ȥեФ sgml2txt ƤӤޤ
;;  * [prefix] t p	SGML 줿 txt ǥץӥ塼ޤ
;;  * [prefix] a	YaTeX Υȵ䴰ƱǤ
;;			&lt; &gt; ϤǤޤ
;;
;;  * [prefix] b 䴰Ǥ륿()ϥ硼ȥåȥǤ
;;                        abstract ([prefix] b a)
;;                        code     ([prefix] b c)
;;                        descrip  ([prefix] b d)
;;                        enum     ([prefix] b e)
;;                        itemize  ([prefix] b i)
;;                        quote    ([prefix] b q)
;;                        tscreen  ([prefix] b t)
;;                        verb     ([prefix] b v)
;;  * [prefix] l 䴰Ǥ륿
;;                        tag
;;                        bf
;;                        cparam
;;                        em
;;                        it
;;                        sf
;;                        sl
;;                        tt
;;                        bf
;;                        [prefix] b 䴰Ǥ륿٤
;;  * [prefix] m 䴰Ǥ륿
;;                        item
;;                        p
;;                        toc
;;                        idx
;;                        cdx
;;                        nidx
;;                        ncdx
;;                        file
;;  * [prefix] s 䴰Ǥ륿
;;                        sect
;;                        sect1
;;                        sect2
;;                        sect3
;;                        sect4
;;                        url
;;                        htmlurl
;;                        label
;;                        ref

;;; *****Attention*****

;; ߤΤȤ sgml-tools ܸбƤեޥåȤϡ 
;; Plain text  HTML ǤߤΤȤ Linux 
;; ʸ SGML 塹ȿʤǤꡢ sgml2info  sgml2latex ܸ
;; Сܸǽ񤫤줿 SGML ʸ⤢եޥåȤѴ
;; 뤳ȤǤ

;;; Code:

(require 'yahtml)
(require 'yatex)

;;; --- customizable variable starts here ---
(defvar yasgml-author "Takashi Nishimoto (g96p0935@mse.waseda.ac.jp)"
  "*Author of SGML document.")
(defvar yasgml-default-style "article"
  "*Default style of SGML document.")
(defvar yasgml-default-dtd "linuxdoc"
  "*Default DTD of SGML document. ")

(defvar yasgml-prefix "\C-c"
  "*Prefix key stroke of yasgml functions.")
(defvar yasgml-fill-column 72 "*fill culumn used for yasgml-mode")
(defvar yasgml-fill-prefix nil "*fill prefix for yasgml-mode")
(defvar yasgml-environment-indent 1
  "*Indentation depth of SGML's listing environment")
(defvar yasgml-special-characters-input-method 'auto
  "*How to input special characters, e.g. &amp; &lt; &gt; ...
auto     Auto conversion.
bell     Warn when you press '&' key.
")
(defvar yasgml-do-special-characters-check t
  "*If non-nil, check special characters before typesetting.
It is useful to find errors about special characters.")

;(defvar YaTeX-japan (or (boundp 'NEMACS) (boundp 'MULE) YaTeX-emacs-20)
;  "Whether yatex mode is running on Japanese environment or not.")

(defvar yasgml-typeset-program "sgml2txt -f -l ja -c nippon"
  "*Typesetting program for SGML.")
(defvar yasgml-format-program-alist
  '((text . "sgml2txt -l ja -c nippon")
    (man . "sgml2txt --man")
    (latex . "sgml2latex")
    (ps . "sgml2latex --output=ps")
    (dvi . "sgml2latex --output=dvi")
    (html . "sgml2html -l ja -c nippon")
    (info . "sgml2info")
    (lyx . "sgml2lyx")
    (rtf . "sgml2rtf")
    (xml . "sgml2xml")
    )
  "*Program alist to convert SGML to various formats.")
(defvar yasgml-preview-function 'yasgml-show-txt-other-window
  "*Preview function.")

;;; --- customizable variable ends here ---
(defvar yasgml-kanji-code 3
  "Kanji coding system number of sgml file; 1=sjis, 2=jis, 3=euc")
(defvar yasgml-coding-system
  (cdr (assq yasgml-kanji-code YaTeX-kanji-code-alist))
  "Kanji coding system")
(defvar yasgml-shell-command-option  (or (and (boundp 'shell-command-option) shell-command-option)
      (if (eq system-type 'ms-dos) "/c" "-c")))

;;; -------- tags ----------

(defvar yasgml-env-table
  '(("abstract") ("code") ("descrip") ("enum") ("itemize") ("quote") ("tscreen") ("verb")))
(defvar yasgml-typeface-table
  (append
   '(("tag") ("bf") ("cparam") ("em") ("it") ("sf") ("sl") ("tt") )
   yasgml-env-table)
  "Default completion table of typeface designator")
(defvar yasgml-single-cmd-table
  '(("item") ("p") ("toc") 
    ("idx") ("cdx") ("nidx") ("ncdx") ("file"))
  "Default completion table of SGML single command.")
(defvar yasgml-sect-table
  '(("sect") ("sect1") ("sect2") ("sect3") ("sect4"))
  "Default completion table of sect commands.")
(defvar yasgml-form-table
  (append
   '(("url") ("htmlurl") ("label") ("ref"))
   yasgml-sect-table)
  "Default completion table of SGML form.")
(defvar yasgml-special-characters-alist
  '(("<" . "&lt;") (">" . "&gt;") ("$" . "&dollar;")
    ("#" . "&num;") ("%" . "&percnt;") ("\"" . "&dquot;") ("~" . "&tilde;"))
  "Default special characters.")
(defvar yasgml-user-env-table nil)
(defvar yasgml-tmp-env-table nil)

(defvar yasgml-user-typeface-table nil)
(defvar yasgml-tmp-typeface-table nil)
(defvar yasgml-last-typeface-cmd "em")

(defvar yasgml-user-single-cmd-table nil)
(defvar yasgml-tmp-single-cmd-table nil)

(defvar yahtml-user-form-table nil)
(defvar yahtml-tmp-form-table nil)
(defvar yahtml-last-form "sect")


(defvar yasgml-struct-name-regexp
  (concat
   "\\<\\("
   (mapconcat (function (lambda (x) (car x))) yasgml-typeface-table "\\|")
   "\\)\\b")
  "Regexp of structure beginning.")

;;; --------- keymaps -----------

(defvar yasgml-prefix-map nil)
(defvar yasgml-mode-map nil "Keymap used in yasgml-mode.")
(defvar yasgml-lint-buffer-map nil "Keymap used in lint buffer.")

(defun yasgml-define-begend-key-normal (key env &optional map)
  "Define short cut yasgml-insert-begend key."
  (YaTeX-define-key
   key
   (list 'lambda '(arg) '(interactive "P")
	 (list 'yasgml-insert-begend 'arg env))
   map))

(defun yasgml-define-begend-region-key (key env &optional map)
  "Define short cut yasgml-insert-begend-region key."
  (YaTeX-define-key key (list 'lambda nil '(interactive)
			      (list 'yasgml-insert-begend t env)) map))

(defun yasgml-define-begend-key (key env &optional map)
  "Define short cut key for begin type completion both for
normal and region mode.  To customize yasgml, user should use this function."
  (yasgml-define-begend-key-normal key env map)
  (if YaTeX-inhibit-prefix-letter nil
    (yasgml-define-begend-region-key
     (concat (upcase (substring key 0 1)) (substring key 1)) env map)))


(if yasgml-mode-map nil
  (setq yasgml-mode-map (make-sparse-keymap)
	yasgml-prefix-map (make-sparse-keymap))
  (define-key yasgml-mode-map yasgml-prefix yasgml-prefix-map)
  (define-key yasgml-mode-map "\M-f" 'yasgml-forward-word)
  (define-key yasgml-mode-map "\M-\C-@" 'yahtml-mark-begend)
  (if (and (boundp 'window-system) (eq window-system 'x) YaTeX-emacs-19)
      (define-key yasgml-mode-map [?\M-\C- ] 'yahtml-mark-begend))
  (define-key yasgml-mode-map "\M-\C-a" 'yasgml-beginning-of-environment)
  (define-key yasgml-mode-map "\M-\C-e" 'yasgml-end-of-environment)
  (define-key yasgml-mode-map "\M-\C-m" 'newline-and-indent)
  ;(define-key yasgml-mode-map "\C-i" 'yahtml-indent-line)
  ;; special characters replace
  (mapcar (lambda (x)
            (define-key yasgml-mode-map (car x)
              `(lambda ()
                 (interactive)
                 (insert (if (yasgml-in-verb-p) ,(car x) ,(cdr x))))))
          yasgml-special-characters-alist)
  (define-key yasgml-mode-map "&" 'yasgml-ampersand)
  (define-key yasgml-mode-map "/" 'yasgml-insert-etago)
  (let ((map yasgml-prefix-map))
    (YaTeX-define-key "^" 'yahtml-visit-main map)
    (YaTeX-define-key "4^" 'yahtml-visit-main-other-window map)
    (YaTeX-define-key "4g" 'yahtml-goto-corresponding-*-other-window map)
    (YaTeX-define-key "44" 'YaTeX-switch-to-window map)
    (and YaTeX-emacs-19 window-system
	 (progn
	   (YaTeX-define-key "5^" 'yahtml-visit-main-other-frame map)
	   (YaTeX-define-key "5g" 'yahtml-goto-corresponding-*-other-frame map)
	   (YaTeX-define-key "55" 'YaTeX-switch-to-window map)))
    (YaTeX-define-key "v" 'YaTeX-version map)
    (YaTeX-define-key "}" 'YaTeX-insert-braces-region map)
    (YaTeX-define-key "]" 'YaTeX-insert-brackets-region map)
    (YaTeX-define-key ")" 'YaTeX-insert-parens-region map)
    (YaTeX-define-key "s" 'yasgml-insert-form map)
    (YaTeX-define-key "l" 'yahtml-insert-tag map)
    (YaTeX-define-key "L" 'yahtml-insert-tag-region map)
    (YaTeX-define-key "m" 'yahtml-insert-single map)
    (YaTeX-define-key "p" 'yasgml-insert-p map)
    (if YaTeX-no-begend-shortcut
	(progn
	  (YaTeX-define-key "B" 'yasgml-insert-begend-region map)
	  (YaTeX-define-key "b" 'yasgml-insert-begend map))
      (yasgml-define-begend-key "ba" "abstract" map)
      (yasgml-define-begend-key "bc" "code" map)
      (yasgml-define-begend-key "bd" "descrip" map)
      (yasgml-define-begend-key "be" "enum" map)
      (yasgml-define-begend-key "bi" "itemize" map)
      (yasgml-define-begend-key "bq" "quote" map)
      (yasgml-define-begend-key "bt" "tscreen" map)
      (yasgml-define-begend-key "bv" "verb" map)
      (YaTeX-define-key "b " 'yasgml-insert-begend map)
      (YaTeX-define-key "B " 'yasgml-insert-begend-region map)
      )
    (YaTeX-define-key "e" 'YaTeX-end-environment map)
    (YaTeX-define-key ">" 'yahtml-comment-region map)
    (YaTeX-define-key "<" 'yahtml-uncomment-region map)
    (YaTeX-define-key "g" 'yasgml-goto-corresponding-* map)
    (YaTeX-define-key "k" 'yahtml-kill-* map)
    (YaTeX-define-key "c" 'yahtml-change-* map)
    (YaTeX-define-key "t" 'yasgml-typeset-menu map)
    (YaTeX-define-key "a" 'yasgml-complete-mark map)
    (YaTeX-define-key "'" 'yahtml-prev-error map)
    ;;;;;(YaTeX-define-key "i" 'yahtml-fill-item map)
    ))

;;; ------- paragraph -----------


(defvar yasgml-paragraph-start
  (concat
   "^$\\|<!--\\|^[ \t]*</?\\(sect\\|sect[1-4]\\|article\\|book\\|report\\|manpage\\|abstract\\|code\\|descrip\\|enum\\|itemize\\|quote\\|tscreen\\|verb\\|p\\)\\b")
  "*Regexp of sgml paragraph separater")
(defvar yasgml-paragraph-separate
  (concat
   "^$\\|<!--\\|^[ \t]*</?\\(sect\\|sect[1-4]\\|article\\|book\\|report\\|manpage\\|abstract\\|code\\|descrip\\|enum\\|itemize\\|quote\\|tscreen\\|verb\\|p\\|!--\\)\\b")
  "*Regexp of sgml paragraph separater")

;;; ------- yasgml-mode ---------

(defun yasgml-mode ()
  (interactive)
  (cond
   ((and YaTeX-emacs-20 (fboundp 'coding-system-equal))
    (if t (or (coding-system-equal
	       yasgml-coding-system buffer-file-coding-system)
	      (set-buffer-file-coding-system yasgml-coding-system))
      ;;^v which is better?
      (let ((bmp (buffer-modified-p)))
	(set-buffer-file-coding-system yasgml-coding-system)
	(set-buffer-modified-p bmp))))
   ((featurep 'mule)
    (set-file-coding-system yasgml-coding-system))
   ((boundp 'NEMACS)
    (make-local-variable 'kanji-fileio-code)
    (setq kanji-fileio-code yasgml-kanji-code)))
  (setq major-mode 'yasgml-mode
	mode-name "yasgml")
  (mapcar
   (function (lambda (x)
	       (make-local-variable (car x))
	       (set (car x) (if (and (symbolp (cdr x))
				     (boundp (cdr x)))
				(symbol-value (cdr x))
			      (cdr x)))))
   '((YaTeX-ec . "")
     (YaTeX-struct-begin . "<%1%2")
     (YaTeX-struct-end . "</%1>")
     (YaTeX-struct-name-regexp . yasgml-struct-name-regexp)
     (YaTeX-comment-prefix . "<!--")
     (YaTeX-coding-system . yasgml-coding-system)
     (YaTeX-typesetting-mode-map . yahtml-lint-buffer-map)
     (fill-prefix . yasgml-fill-prefix) (fill-column . yasgml-fill-column)
     (paragraph-start . yasgml-paragraph-start)
     (paragraph-separate . yasgml-paragraph-separate)
     (comment-start . "<!-- ") (comment-end . " -->")
     (comment-start-skip . comment-start)
     (indent-line-function . indent-relative-maybe)
     ;;
     (yahtml-env-table . yasgml-env-table)
     (yahtml-user-env-table . nil)
     (yahtml-tmp-env-table . nil)
     ;;
     (yahtml-typeface-table . yasgml-typeface-table)
     (yahtml-user-typeface-table . nil)
     (yahtml-tmp-typeface-table . nil)
     (yahtml-last-typeface-cmd . "em")
     ;;
     (yahtml-single-cmd-table . yasgml-single-cmd-table)
     (yahtml-user-single-cmd-table . nil)
     (yahtml-tmp-single-cmd-table . nil)
     (yahtml-last-single-cmd . "toc")
     ;;
     (yahtml-form-table . yasgml-form-table)
     (yahtml-user-form-table . nil)
     (yahtml-tmp-form-table . nil)
     (yahtml-last-form . "sect")
     ;;
     (yahtml-struct-name-regexp . yasgml-struct-name-regexp)
     (yasgml-preview-point . 1)
     (yahtml-prefer-upcases . nil)
      ))

  (set-syntax-table yahtml-syntax-table)
  (use-local-map yasgml-mode-map)
  (YaTeX-read-user-completion-table)
  (turn-on-auto-fill)			;Sorry, this is prerequisite
  (and (= 0 (buffer-size)) (call-interactively 'yasgml-make-document))
  (run-hooks 'text-mode-hook 'yasgml-mode-hook))

;;; ------ menu-bar ------
(defvar yasgml-menu-map nil "Menu map of yasgml")
(defvar yasgml-menu-map-sectioning nil "Menu map of yasgml(sectioning)")
(defvar yasgml-menu-map-environment nil "Menu map of yasgml(environment)")
(defvar yasgml-menu-map-listing nil "Menu map of yasgml(listing)")
(defvar yasgml-menu-map-typeface nil "Menu map of yasgml(typeface tags)")
(defvar yasgml-menu-map-form nil "Menu map of yasgml(URL, label, ref)")
(defvar yasgml-menu-map-item nil "Menu map of yasgml(single tags)")

(cond
 ((and YaTeX-emacs-19 (null yasgml-menu-map))
  (setq yasgml-menu-map (make-sparse-keymap "yasgml"))
  (setq yasgml-menu-map-sectioning (make-sparse-keymap "sectioning menu"))
  (YaTeX-define-menu
   'yasgml-menu-map-sectioning
   (nreverse
    '((1 "sect"  . (lambda () (interactive) (yasgml-insert-sect "sect")))
      (2 "sect1" . (lambda () (interactive) (yasgml-insert-sect "sect1")))
      (3 "sect2" . (lambda () (interactive) (yasgml-insert-sect "sect2")))
      (4 "sect3" . (lambda () (interactive) (yasgml-insert-sect "sect3")))
      (5 "sect4" . (lambda () (interactive) (yasgml-insert-sect "sect4")))
      )))

  (setq yasgml-menu-map-typeface (make-sparse-keymap "typeface tags"))
  (YaTeX-define-menu
   'yasgml-menu-map-typeface
   (nreverse
    '((bf	"Bold" .
	  (lambda () (interactive) (yasgml-insert-tag nil "bf")))
      (tt	"Typewriter" .
	  (lambda () (interactive) (yasgml-insert-tag nil "tt"))) 
      (it	"Italic" .
	  (lambda () (interactive) (yasgml-insert-tag nil "it")))
      ;; Other fonts should not be used.
      )))
  (setq yasgml-menu-map-environment (make-sparse-keymap "environment"))
  (YaTeX-define-menu
   'yasgml-menu-map-environment
   (nreverse
    '((abstract	"Abstract" .
		(lambda () (interactive) (yasgml-insert-begend nil "abstract")))
      (code	"Code" .
		(lambda () (interactive) (yasgml-insert-begend nil "code")))
      (quote	"Quote" .
		(lambda () (interactive) (yasgml-insert-begend nil "quote")))
      (tscreen	"Typewriter screen" .
		(lambda () (interactive) (yasgml-insert-begend nil "tscreen")))
      (verb	"Verbatim" .
		(lambda () (interactive) (yasgml-insert-begend nil "verb")))
      )))
  (setq yasgml-menu-map-listing (make-sparse-keymap "listing"))
  (YaTeX-define-menu
   'yasgml-menu-map-listing
   (nreverse
    '((descrip	"Description" .
		(lambda () (interactive) (yasgml-insert-begend nil "descrip")))
      (enum	"Enumerate" .
		(lambda () (interactive) (yasgml-insert-begend nil "enum")))
      (itemize	"Itemize" .
		(lambda () (interactive) (yasgml-insert-begend nil "itemize")))
      )))
  (setq yasgml-menu-map-form (make-sparse-keymap "label & ref"))
  (YaTeX-define-menu
   'yasgml-menu-map-form
   (nreverse
    '((url	"Url" .
		(lambda () (interactive) (yasgml-insert-form "url")))
      (htmlurl	"HTMLurl" .
		(lambda () (interactive) (yasgml-insert-form "htmlurl")))
      (label	"Label" .
		(lambda () (interactive) (yasgml-insert-form "label")))
      (ref	"Ref" .
		(lambda () (interactive) (yasgml-insert-form "ref")))
      )))
  (setq yasgml-menu-map-item (make-sparse-keymap "item"))
  (YaTeX-define-menu
   'yasgml-menu-map-item
   (nreverse
    '((item	"item" .
		(lambda () (interactive) (yasgml-insert-single "item")))
      (p	"P)aragraph" .
		(lambda () (interactive) (yasgml-insert-p)))
      (toc	"Table of Contents" .
		(lambda () (interactive) (yasgml-insert-single "toc")))
      ;;These tags are ignored by all backends except LaTeX.
      ;;But LaTeX-backend is not support Japanese yet.
;      (idx	"index" .
;		(lambda () (interactive) (yasgml-insert-single "idx")))
;      (cdx	"index(constant-width font)" .
;		(lambda () (interactive) (yasgml-insert-single "cdx")))
;      (nidx	"index(no appearance)" .
;		(lambda () (interactive) (yasgml-insert-single "nidx")))
;      (ncdx	"index(constant-width font, no appearance)" .
;		(lambda () (interactive) (yasgml-insert-single "ncdx")))
;      (file	"filename" .
;		(lambda () (interactive) (yasgml-insert-single "file")))
      )))
  (define-key yasgml-mode-map [menu-bar yasgml]
    (cons "yasgml" yasgml-menu-map))
  (let ((keys (where-is-internal 'fill-paragraph global-map)))
    (while keys
      (define-key yasgml-mode-map (car keys) 'yahtml-fill-paragraph)
      (setq keys (cdr keys))))
  (YaTeX-define-menu
   'yasgml-menu-map
   (nreverse
    (list
     (cons (list 'sect "Sectioning")
	   (cons "sectioning" yasgml-menu-map-sectioning))
     (cons (list 'env  "Environment")
           (cons "environment" yasgml-menu-map-environment))
     (cons (list 'list "Listing")
	   (cons "Listing" yasgml-menu-map-listing))
     (cons (list 'item "Item")
	   (cons "Itemizing" yasgml-menu-map-item));;; 
     (cons (list 'type "Typeface tags")
	   (cons "typeface" yasgml-menu-map-typeface))
     (cons (list 'form "URL, label, ref")
           (cons "url, label, ref" yasgml-menu-map-form))
     )))
  (if (featurep 'xemacs)
      (add-hook 'yasgml-mode-hook
		'(lambda ()
		   (or (assoc "yasgml" current-menubar)
		       (progn
			 (set-buffer-menubar (copy-sequence current-menubar))
			 (add-submenu nil yasgml-menu-map))))))
  ))


;;; ------ insertion -------
(defalias 'yasgml-insert-single 'yahtml-insert-single)
(defalias 'yasgml-insert-tag 'yahtml-insert-tag)

(defun yasgml-make-document (style title author date)
  "Set up a SGML document."
  (interactive (let* ((result (progn (message "A)rticle B)ook R)eport M)anpage (default: %s)" yasgml-default-style)
				     (read-char)))
                      (title  (read-string "Title: "))
                      (author (read-string "Author: " yasgml-author))
                      (date   (read-string "Date: " (current-time-string)))) 
		 (list (downcase result) title author date)))
  (let ((style-string
	 (cond
	  ((eq style ?a)"article")
	  ((eq style ?b)"book")
	  ((eq style ?r)"report")
	  ((eq style ?m)"manpage")
	  (t yasgml-default-style))))
    (insert "<!doctype " yasgml-default-dtd " system>\n\n")
    (insert "<" style-string ">\n\n<title>" title
            "\n<author>" author "\n<date>" date "\n\n")
    (save-excursion (insert "\n</" style-string ">\n"))))

(defun yasgml-in-verb-p ()
  "Return t if the (point) is between <verb> and </verb>."
  (save-excursion
    (if (re-search-backward "<\\(/?\\)\\(verb\\|code\\|quote\\)>" nil t)
        (cond ((string= (YaTeX-match-string 1) "") t)
              ((string= (YaTeX-match-string 1) "/") nil))
      nil)))

(defun yasgml-ampersand ()
  "When you press '&' key, execute this function."
  (interactive)
  (cond ((eq yasgml-special-characters-input-method 'auto)
         (insert "&amp;"))
        ((eq yasgml-special-characters-input-method 'bell)
         (insert "&")
         (ding)
         (message (substitute-command-keys
                   "Warning: Bare '&' is an error in SGML. Use '\\[yasgml-complete-mark]' to input special-char.")))
        ))

(defun yasgml-complete-mark ()
  "Complete &amp, &lt, &gt, &dollar, &num, &percnt, &dquot, &tilde, &etago."
  (interactive)
  (message "%s" "a:& s:</ d:< f:> g:$ h:\# j:% k:\" l:~")
  (let ((c (downcase(read-char))) s del)
    (if (eq (char-before (point)) ?&) (progn
                                        (delete-backward-char 1)
                                        (setq del t)))
    (setq c (cond ((eq c ?a) ?&)
                  ((eq c ?s) ?/)
                  ((eq c ?d) ?<)
                  ((eq c ?f) ?>)
                  ((eq c ?g) ?$)
                  ((eq c ?h) ?#)
                  ((eq c ?j) ?%)
                  ((eq c ?k) ?\")
                  ((eq c ?l) ?~)
                  (t c)))
    (cond ((eq c ?/) (insert "&etago;"))
          ((eq c ?&) (insert "&amp;"))
          (t (if (setq s (cdr (assoc (char-to-string c)
                                     yasgml-special-characters-alist)))
                 (insert s)
               (if del (insert "&")))))))

        


(defun yasgml-insert-etago ()
  "Intelligent insertion of &etago;. The binding of this command is \"/\".
If previous char is \"<\", replace it with \"&etago;\".
Otherwise insert \"/\"."
  (interactive)
  (cond ((eq (char-before) ?<)
         (delete-backward-char 1) (insert "&etago;"))
        (t
         (insert "/"))))

(defun yasgml-insert-p ()
  "Insert <p>."
  (interactive)
  (yahtml-insert-single "p") (newline))

(defvar yasgml-last-begend "verb")
(defun yasgml-insert-begend (&optional region env)
  "Insert <cmd> ... </cmd>."
  (interactive "P")
  (let*((completion-ignore-case t)
	(cmd
	 (or env
	     (YaTeX-cplread-with-learning
	      (format "Environment(default %s): " yasgml-last-begend)
	      'yahtml-env-table 'yahtml-user-env-table 'yahtml-tmp-env-table)))
	(bolp (save-excursion
		(skip-chars-backward " \t" (point-beginning-of-line)) (bolp)))
	(cc (current-column)))
    (if (string< "" cmd) (setq yasgml-last-begend cmd))
    (setq yahtml-last-begend
	  (or (cdr (assoc yasgml-last-begend yahtml-env-table))
	      yasgml-last-begend))
    (setq cmd yasgml-last-begend)
    (if yahtml-prefer-upcases (setq cmd (upcase cmd)))
    (if region
	(let ((beg (region-beginning))
	      (end (region-end))
	      (addin (yasgml-addin cmd)))
	  (goto-char end)
	  (insert (format "</%s>%s" cmd (if bolp "\n" "")))
	  (goto-char beg)
	  (insert (format "<%s%s>%s" cmd addin (if bolp "\n" ""))))
      (insert (format "<%s%s>" cmd (yasgml-addin cmd)))
      (save-excursion
	(insert "\n")
	(indent-to-column cc)
	(insert (format "</%s>" cmd)))
      (yasgml-intelligent-newline nil)
      )))


(defun yasgml-insert-form (&optional form)
  "Insert <FORM option=\"argument\">."
   (interactive)
   (or form
       (let ((completion-ignore-case t))
	 (setq form
	       (YaTeX-cplread-with-learning
		(format "Form(default %s): " yahtml-last-form)
		'yahtml-form-table 'yahtml-user-form-table
		'yahtml-tmp-form-table))))
   (let ((p (point)) q)
     (if (string= form "") (setq form yahtml-last-form))
     (setq yahtml-last-form form)
     (if (assoc form yasgml-sect-table)
         (yasgml-insert-sect form)
       (insert (format "<%s%s>" form (yasgml-addin form))))
     ;;(indent-relative-maybe)
     (if (cdr (assoc form yahtml-form-table))
	 (save-excursion (insert (format "</%s>" form))))
     (if (search-backward "\"\"" p t) (forward-char 1))))

(defun yasgml-insert-sect (cmd)
  "Insert <sect> .. <sect4>."
  (interactive
   (list (completing-read "<sect>, <sect1> .. <sect4>: " yasgml-sect-table nil t "sect"))) 
  (insert "<" cmd ">" (read-string "title: ") "\n<p>\n"))
    

;;; ---------- Add-in ----------
(defun yasgml-addin (form)
  "Check add-in function's existence and call it if exists."
   (let ((addin (concat "yasgml:" (downcase form))) s)
     (if (and (intern-soft addin) (fboundp (intern-soft addin))
	      (stringp (setq s (funcall (intern addin))))
	      (string< "" s))
	 (if (eq (aref s 0) ? ) s (concat " " s))
       "")))

(defun yasgml:url ()
  "Add-in function for url."
  (let ((href ""))
    (setq yahtml-completing-buffer (current-buffer)
	  href (read-from-minibuffer "url: " "" yahtml-url-completion-map)
	  ;; yahtml-urls-local is buffer-local, so we must put
	  ;; that into yahtml-urls here
	  yahtml-urls (append yahtml-urls-private yahtml-urls-local))
    (prog1
	(concat (yahtml-make-optional-argument
		 "url" href)
		(yahtml-make-optional-argument
		 "name" (read-string "name: ")))
      (if (and (string-match "^http://" href)
	       (null (assoc href yahtml-urls)))
	  (YaTeX-update-table
	   (list href)
	   'yahtml-urls-private 'yahtml-urls-private 'yahtml-urls-local))
      )))

(fset 'yasgml:htmlurl 'yasgml:url)

(defun yasgml:label ()
  "Add-in function for label."
  (let ()
    (concat "id=\"" (read-string "Label-id: " nil) "\"")))

(defun yasgml:ref ()
  "Add-in function for ref."
  (let ()
    (concat "id=\"" (read-string "Ref-id: " nil) "\" "
            "name=\"" (read-string "Name: " nil) "\"")))


;;; ---------- Jump ----------

(defun yasgml-forward-word (count)
  "Forward-word for yasgml."
  (interactive "p")
  (if (zerop count) t
    (if (eq (char-after (point)) ?<)
        (search-forward ">" nil t)
      (forward-word 1))
    (yasgml-forward-word (1- count))))

(defun yasgml-on-url-p ()
  ""
  (let ((p (point)) e cmd)
    (save-excursion
      (and (goto-char (1+ (search-backward "<" nil t)))
           (looking-at "\\(htmlurl\\|url\\)")
           (save-excursion
             (search-forward ">" nil t)
             (setq e (point)))
           (search-forward "url" e t)
           (search-forward "=" e t)
	   (progn
	     (skip-chars-forward " \t\n")
	     (looking-at "\"?\\([^\"> \t\n]+\\)\"?"))
	   (< p (match-end 0))
	   (YaTeX-match-string 1)
	   ))))

(defun yasgml-goto-corresponding-url (&optional other)
  "Go to corresponding name."
  (let ((href (yasgml-on-url-p)) file name (parent buffer-file-name))
    (if href
	(cond
	 ((string-match "^\\(ht\\|f\\)tp:" href)
	  (yahtml-browse-html href))
	 (t (setq file (substring href 0 (string-match "#" href)))
	    (if (string-match "#" href)
		(setq name (substring href (1+ (string-match "#" href)))))
	    (if (string< "" file)
		(progn
		  (if (string-match "/$" file)
		      (setq file (concat file yahtml-directory-index)))
		  (if (string-match "^/" file)
		      (setq file (yahtml-url-to-path file)))
		  (if other (YaTeX-switch-to-buffer-other-window file)
		    (YaTeX-switch-to-buffer file))
		  (or YaTeX-parent-file (setq YaTeX-parent-file parent))))
	    (if name
		(progn (set-mark-command nil) (yasgml-jump-to-name name)))
	    t)))))

(defun yasgml-jump-to-name (name)
  "Jump to html's named tag."
  (setq name (format "<label[^>]+name\\s *=\\s *\"?%s\"?" name))
  (or (and (re-search-forward name nil t) (goto-char (match-beginning 0)))
      (and (re-search-backward name nil t) (goto-char (match-beginning 0)))
      (message "Named tag `%s' not found" (substring href 1))))
           

(defun yasgml-goto-corresponding-* (&optional other)
  "Go to corresponding object."
  (interactive)
  (let ((major-mode 'yahtml-mode))
  (cond
   ((yasgml-goto-corresponding-url other))
   ;((yasgml-goto-corresponding-label))
   ((yahtml-goto-corresponding-begend))
   (t (message "I don't know where to go."))
   )))

(defun yasgml-beginning-of-environment (&optional limit-search-bound end)
  "YaTeX-beginning-of-environment for yasgml-mode."
  (interactive)
  (let ((major-mode 'yahtml-mode))
    (YaTeX-beginning-of-environment limit-search-bound end)))

(defun yasgml-end-of-environment (&optional limit-search-bound)
  "YaTeX-end-of-environment for yasgml-mode."
  (interactive)
  (yasgml-beginning-of-environment limit-search-bound t))

;;; 
;;; ---------- Typesetting and Previewing ----------
;;; 
(defun yasgml-typeset-menu ()
  "Browsing menu"
  (interactive)
  (message "J)typeset P)review T)ext H)tml I)nfo L)atex D)vi y)lYx R)tf X)ml...")
  (let ((c (char-to-string (read-char))))
    (cond
     ((string-match "j" c)
      (yasgml-typeset-this-buffer))
     ((string-match "p" c)
      (yasgml-preview-this-buffer))
     ((string-match "t" c)
      (yasgml-format-this-buffer 'text))
     ((string-match "h" c)
      (yasgml-format-this-buffer 'html))
     ((string-match "i" c)
      (yasgml-format-this-buffer 'info))
     ((string-match "l" c)
      (yasgml-format-this-buffer 'latex))
     ((string-match "d" c)
      (yasgml-format-this-buffer 'dvi))
     ((string-match "y" c)
      (yasgml-format-this-buffer 'lyx))
     ((string-match "r" c)
      (yasgml-format-this-buffer 'rtf))
     ((string-match "x" c)
      (yasgml-format-this-buffer 'xml))
     )))

(defvar yasgml-typeset-buffer "*yasgml-typesetting*")

(defun yasgml-typeset-this-buffer ()
  "Validate this buffer"
  (interactive)
  (if yasgml-do-special-characters-check
      (yasgml-special-characters-check))
  (require 'yatexprc)
  (YaTeX-save-buffers)
  (YaTeX-typeset
   (concat yasgml-typeset-program " "
	   (file-name-nondirectory (buffer-file-name (current-buffer))))
   yasgml-typeset-buffer  "typeset" "typeset")
  )

(defun yasgml-special-characters-check ()
  "Check for special characters. "
  (interactive)
  (let ((pt (point))
        amp colon)
    (goto-char (point-min))
    (while (prog1
             (setq amp (search-forward "&" nil t))
             (setq colon (save-excursion
                           (re-search-forward "[&;]" nil t)
                           (if (eq (char-before (point)) ?&) nil (point)))))
      (if (or (null colon) 
              (> (-  colon amp) 7 ))
          (if (progn
                (momentary-string-display "=>" (1- (point)) 0
                                          "This '&' should be '&amp;', FIX? ")
                (y-or-n-p "This '&' should be '&amp;', FIX? "))
              (insert "amp;")
            (ding)
            (message "WARNING: THIS FILE MAY BE ERROR.")
            (sit-for 2))))
    (goto-char pt)
    (message "")
    (sit-for 0)))

  

(defun yasgml-format-this-buffer (to)
  "Make various formats from SGML file."
  (let ((cmd (cdr (assq to yasgml-format-program-alist)))
        )
    (if cmd
        (YaTeX-typeset
         (concat cmd " "
                 (file-name-nondirectory (buffer-file-name (current-buffer))))
         "*yasgml-formatting*"  "format" "format")
      (error "Command for %s is not specified" (symbol-name to)))))

(defun yasgml-preview-this-buffer ()
  "Preview. Executing function specified in yasgml-preview-function."
  (interactive)
  (funcall yasgml-preview-function))

(defun yasgml-get-filename (&optional ext)
  "Get current-file's basename if EXT is nil, otherwise (concat basename EXT)."
  (let* ((fname (file-name-nondirectory (buffer-file-name)))
         (period (rindex fname ?.)))
    (concat (substring fname 0 period) (if ext ext ""))
    ))

(defvar yasgml-preview-point 1)

(defun yasgml-show-txt (display-function)
  "Show txt version (for preview)."
  (let* ((dir default-directory)
         (txtfile (yasgml-get-filename ".txt"))
         (buf (get-buffer "*yasgml-preview*")))
    (if buf
        (save-excursion
          (set-buffer buf)
          (setq yasgml-preview-point (point)))
      (setq buf (get-buffer-create "*yasgml-preview*")
            yasgml-preview-point 1))
    (save-excursion
      (set-buffer buf)
      (setq buffer-read-only nil)
      (erase-buffer)
      (cd dir)
      (insert-file txtfile)
      (set-buffer-modified-p nil)
      (setq buffer-read-only t)
      )
    (funcall display-function buf)
    (goto-char yasgml-preview-point)
    ))

(defun yasgml-show-txt-other-window ()
  "Show txt version other window (for preview)."
  (interactive)
  (yasgml-show-txt 'switch-to-buffer-other-window))

(defun yasgml-show-txt-other-frame ()
  "Show txt version other frame (for preview)."
  (interactive)
  (yasgml-show-txt 'switch-to-buffer-other-frame))

(defun yasgml-show-txt-bully ()
  "Show txt version in one window (for preview)."
  (interactive)
  (delete-other-windows)
  (yasgml-show-txt 'switch-to-buffer))

;;; ------- Intelligent newline -------
(defun yasgml-inner-environment ()
  (let ((major-mode 'yahtml-mode))
    (YaTeX-inner-environment)))

(defun yasgml-intelligent-newline (&optional arg)
  "Intelligent newline for SGML"
  (interactive "P")
  (let (env func)
    (end-of-line)
    (setq env (downcase (yasgml-inner-environment)))
    (setq func (intern-soft (concat "yasgml-intelligent-newline-" env)))
    (newline)
    (if (and env func (fboundp func))
	;; if intelligent line function is defined, call that
	(funcall func)
      ;; else do the default action
      )))
(defun yasgml-intelligent-newline-descrip ()
  (interactive)
  (yasgml-insert-tag nil "tag")
  (setq yahtml-last-typeface-cmd "tag"))

(defun yasgml-intelligent-newline-enum ()
  (interactive)
  (yasgml-insert-single "item")
  (setq yahtml-last-single-cmd "item"))
(defalias 'yasgml-intelligent-newline-itemize 'yasgml-intelligent-newline-enum)

;;; ------- for debug ---------
(defun yasgml-reload ()
  (interactive)
  (setq yasgml-mode-map nil)
  (load "yasgml")
  (yasgml-mode))
;;; yasgml.el ends here