/**
 * qtclientform.cpp
 * This file is part of the YATE Project http://YATE.null.ro
 *
 * A Qt based universal telephony client
 * Author: Dorin Lazar <lazar@deuromedia.ro>
 *
 * Yet Another Telephony Engine - a fully featured software PBX and IVR
 * Copyright (C) 2004 Null Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "qtclientform.hpp"
#include "qtclientform.moc"
#include "yatedialbutton.hpp"
#include "yatenumberdisplay.hpp"
#include "yateaddressbook.hpp"
#include "yateaddressinput.hpp"
#include "yateaddressbookview.hpp"
#include "yatecallhistory.hpp"

#include <qpushbutton.h>
#include <qapplication.h>
#include <qlabel.h>

#include <telengine.h>

QtClientForm::QtClientForm () :
	QWidget (NULL, "Qt Yate Client")
{
	setFixedSize (300, 300);

	numberDisplay = new YateNumberDisplay(this);
	numberDisplay->move (10, 50);
	numberDisplay->show();

	correctDigit = new QPushButton (this);
	correctDigit->setGeometry (90, 90, 30, 20);
	correctDigit->setText ("<=");
	QKeySequence seqCorrectDigit (UNICODE_ACCEL + '-');//Key_Backspace
	correctDigit->setAccel (seqCorrectDigit);
	connect (correctDigit, SIGNAL(clicked()), numberDisplay, SLOT(removeLastDigit()));
	correctDigit->setFocusPolicy (NoFocus);

	rejectCall = new QPushButton (this);
	rejectCall->setGeometry (67, 120, 53, 20);
	rejectCall->setText ("Reject");
	QKeySequence seqRejectCall (UNICODE_ACCEL + '/');
	rejectCall->setAccel (seqRejectCall);
	rejectCall->setFocusPolicy (NoFocus);
	connect (rejectCall, SIGNAL (clicked()),
		 this, SLOT(rejectPressed()));

	dialButton = new QPushButton (this);
	dialButton->setGeometry (10, 120, 53, 20);
	dialButton->setText ("Dial");
	QKeySequence seqDialButton (Key_Enter);
	dialButton->setAccel (seqDialButton);
	dialButton->setFocusPolicy (NoFocus);
	connect (dialButton, SIGNAL (clicked()),
		 this, SLOT(dialPressed()));

	quitButton = new QPushButton (this);
	quitButton->setGeometry (10, 90, 53, 20);
	quitButton->setText ("Quit");
	QKeySequence seqQuitButton (UNICODE_ACCEL + 'q');
	quitButton->setAccel (seqQuitButton);
	quitButton->setFocusPolicy (NoFocus);
	connect (quitButton, SIGNAL (clicked()),
		 this, SLOT(quitPressed()));

	statusLabel = new QLabel (this);
	statusLabel->setGeometry (10, 10, 180, 40);
	setStatus (YCS_IDLE);

	errorMessage = new QLabel (this);
	errorMessage->setGeometry (10, 270, 280, 30);
	setErrorMessage ("Welcome to YATE Qt Client");

	char dials[] = "123456789*0#";
	for (unsigned int i=0; i<YATE_NDIALBUTTONS; i++) {
		dialButtons[i] = new YateDialButton(this, numberDisplay, dials[i]);
		dialButtons[i]->move (10 + (i%3) * 40, 150 + (i/3) * 30);
		dialButtons[i]->show();
	}

	addressbook = new YateAddressBook;

	addToAddressBookButton = new QPushButton (this);
	addToAddressBookButton->setGeometry (215, 50, 75, 25);
	addToAddressBookButton->setText ("Add");
	QKeySequence seqAddButton (UNICODE_ACCEL + 'a');
	addToAddressBookButton->setAccel (seqAddButton);
	addToAddressBookButton->setFocusPolicy (NoFocus);
	connect (addToAddressBookButton, SIGNAL (clicked()),
		 this, SLOT(addToAddressBookPressed()));

	yabv = new YateAddressBookView (this, addressbook);
	yabv->move (130, 85);
	connect (yabv, SIGNAL(dialRequest(const char *)), 
		 this, SLOT(dialRequest(const char *)));

	ych = new YateCallHistory ();
	numberDisplay->setHistoryItems (ych->items());
}

void __ye_cancel_pending_call (void)
{
	TelEngine::Message msg("call.drop");
	msg.addParam("id","oss/");
	TelEngine::Engine::dispatch(msg);
}

QtClientForm::~QtClientForm()
{
	//if (telEngineStatus!=YCS_IDLE)
//		__ye_cancel_pending_call();
	delete numberDisplay; numberDisplay = NULL;
	delete correctDigit; correctDigit = NULL;
	delete rejectCall; rejectCall = NULL;
	delete dialButton; dialButton = NULL;
	for (unsigned int i=0; i<YATE_NDIALBUTTONS; i++) {
		delete dialButtons[i];
		dialButtons[i] = NULL;
	}
	delete addressbook; addressbook = NULL;
	delete ych; ych = NULL;
	delete yabv; yabv = NULL;
}

void QtClientForm::dialPressed (void)
{
	qDebug ("DIAL request");
	if (telEngineStatus==YCS_RINGIN) {
		setStatus(YCS_INCALL);
		qDebug ("Call accepted");
		/* add it to history */
		ych->addNumber (dialer);
		numberDisplay->setHistoryItems(ych->items());
		return;
	}
	if (telEngineStatus==YCS_RINGOUT)
		__ye_cancel_pending_call();

	if (numberDisplay->getNumber().length()<=0) {
		qDebug ("Invalid number dialed");
		return;
	}

	TelEngine::Message msg("call.execute");
	QString address (numberDisplay->getNumber());
	ych->addNumber (address.latin1());
	numberDisplay->setHistoryItems (ych->items());

	if (::strchr(address,'/'))
		msg.addParam("direct",address.latin1());
	else
		msg.addParam("target",address.latin1());
	msg.addParam("callto","oss///dev/dsp");
	msg.addParam("caller","oss///dev/dsp");
	msg.addParam("id","oss///dev/dsp");

	if (TelEngine::Engine::dispatch(msg)) {
		setErrorMessage ("");
		setStatus (YCS_INCALL);
	}
	else {
		setErrorMessage ("Call failed!");
		setStatus (YCS_IDLE);
	}
}

void QtClientForm::rejectPressed (void)
{
	qDebug ("REJECT request");
	switch (telEngineStatus) {
	case YCS_RINGIN:
		break;
	case YCS_RINGOUT:
	case YCS_INCALL:
		__ye_cancel_pending_call();
		break;
	case YCS_IDLE:
	default:
		numberDisplay->setNumber("");
	}
	setStatus (YCS_IDLE);
}

void QtClientForm::quitPressed (void)
{
	qApp->exit(0);
}

static char *__ycs_names[1 + YCS_MAXSTATUS] = {
	"Idle",
	"RING IN",
	"RING OUT",
	"IN CALL",
	NULL
};

bool QtClientForm::setStatus (YateClientStatus status)
{
	if (telEngineStatus != YCS_IDLE && status==YCS_RINGIN) {
		return false;
	}
	telEngineStatus = status;
	statusLabel->setText (QString ("Engine Status: %1%2").arg(__ycs_names[status])
			      .arg(status==YCS_RINGIN ? (QString("\nfrom %1").arg(dialer)) : QString("")));
	switch (telEngineStatus) {
	case YCS_RINGIN:
		dialButton->setText ("Answer");
		dialButton->setEnabled (true);
		numberDisplay->setReadOnly (false);
		break;
	case YCS_RINGOUT:
		dialButton->setEnabled (false);
		numberDisplay->setReadOnly (true);
		break;
	case YCS_INCALL:
		dialButton->setText ("Dial");
		dialButton->setEnabled (false);
		numberDisplay->setReadOnly (true);
		break;
	default:
	case YCS_IDLE: 
		{
		dialButton->setText ("Dial");
		dialButton->setEnabled (true);
		QString str(numberDisplay->getNumber());
		//if (str.contains('p')>=0) {
		//	str.remove (str.find('p'), str.length());
		//	numberDisplay->setNumber (str);
		//}
		numberDisplay->setReadOnly (false);
		}
		break;
	}
	if (status==YCS_RINGIN) {
	}
	else
		dialButton->setText ("Dial");
	return true;
}

void QtClientForm::setErrorMessage (QString str)
{
	errorMessage->setText (str);
}

void QtClientForm::addToAddressBookPressed (void)
{
	qDebug ("Adding to address book");
	YateAddressInput *yai = new YateAddressInput (this, numberDisplay->getNumber().latin1());
	if (yai->exec()==QDialog::Accepted) {
		addressbook->addAddress(yai->getAlias(), yai->getAddress());
		yabv->rescanAddressBook();
		addressbook->store();
	}
	delete yai;
}

void QtClientForm::dialRequest (const char *number)
{
	numberDisplay->setNumber(QString (number));
	dialPressed();
}

void QtClientForm::emitDTMF (char value)
{
	char c[2];
	c[1] = '\0';
	c[0] = value;
	// masquerade the message with support from the real channel
	TelEngine::Message *m = new TelEngine::Message ("chan.masquerade");
	m->addParam ("id", "oss/");
	m->addParam ("text", c);
	m->addParam ("message", "chan.dtmf");
	TelEngine::Engine::enqueue (m);
}

