/*
 * yeti.i --
 *
 *	Main startup file for Yeti (an extension of Yorick).
 *
 *-----------------------------------------------------------------------------
 *
 *	Copyright (C) 1999-2007 Eric Thibaut.
 *
 *	This file is part of Yeti.
 *
 *	Yeti is  free software;  you can redistribute  it and/or  modify it
 *	under  the terms of  the GNU  General Public  License version  2 as
 *	published by the Free Software Foundation.
 *
 *	Yeti is distributed in the hope that it will be useful, but WITHOUT
 *	ANY WARRANTY; without even  the implied warranty of MERCHANTABILITY
 *	or FITNESS  FOR A PARTICULAR  PURPOSE.  See the GNU  General Public
 *	License for more details.
 *
 *	You should have  received a copy of the  GNU General Public License
 *	along with  Yeti (file "COPYING"  in the top source  directory); if
 *	not, write to  the Free Software Foundation, Inc.,  51 Franklin St,
 *	Fifth Floor, Boston, MA 02110-1301 USA
 *
 *-----------------------------------------------------------------------------
 *
 * History:
 *	$Id: yeti.i,v 1.8 2007/05/11 10:29:46 eric Exp $
 *	$Log: yeti.i,v $
 *	Revision 1.8  2007/05/11 10:29:46  eric
 *	 - Implementation of symbolic links.
 *
 *	Revision 1.7  2007/04/24 07:21:58  eric
 *	 - New setup_package function.
 *	 - Documentation of make_dimlist updated.
 *
 *	Revision 1.6  2007/04/19 17:08:19  eric
 *	 - The `symbol_names` function can now specifically select
 *	   lists, hash tables and/or auto-loaded functions.
 *	 - The `about` routine now account for auto-loaded functions.
 *
 *	Revision 1.5  2007/03/23 16:31:59  eric
 *	 - Updated documentation for mvmult, since possible values
 *	   for JOB/FLAGS in evaluation of matrix products are now
 *	   restricted: must be 0 (default) for direct product and 1
 *	   for transpose product.
 *
 *	Revision 1.4  2007/03/23 11:58:09  eric
 *	 - Hash table objects can now have their own evaluator, which can be
 *	   queried/set by the `h_evaluator` function.
 *	 - New function `h_number` to query number of entries in a hash table.
 *	 - Function `is_hash` returns 2 for a hash table object implementing
 *	   its own evaluator.
 *	 - `h_clone`, `h_copy`, `h_info` and `h_show` fixed to account for the
 *	   evaluator of an hash table object.
 *
 *	Revision 1.3  2006/12/05 07:18:49  eric
 *	 - Renamed `typeIDof` as `identof` and added defintions of
 *	   constants `T_CHAR`, `T_SHORT`, ...
 *
 *	Revision 1.2  2006/07/19 17:32:34  eric
 *	 - New built-in function insure_temporary.
 *
 *	Revision 1.1  2005/05/24 13:06:25  eric
 *	Initial revision
 *-----------------------------------------------------------------------------
 */

if (is_func(plug_in) && is_func(yeti_init) != 2) plug_in, "yeti";

local YETI_VERSION;
local YETI_HOME;
extern yeti_init;
/* DOCUMENT YETI_HOME       the directory where Yeti is installed
 *     -or- YETI_VERSION    version of current Yeti interperter
 *     -or- yeti_init;
 *     -or- yeti_init();
 *
 *   YETI_VERSION and YETI_HOME are global variables predefined by Yeti to
 *   store its version number (as "MAJOR.MINOR.MICRO") and installation
 *   directory (e.g. "/usr/local/lib/yeti-VERSION").
 *
 *   The function yeti_init can be used to restore the values of
 *   YETI_VERSION and YETI_HOME.  When called as a function, yeti_init()
 *   returns Yeti version as a string in the form "MAJOR.MINOR.MICRO".
 *
 *   If Yeti is loaded as a plugin, YETI_HOME is left undefined and no path
 *   initialization is performed.  Otherwise, the first time yeti_init is
 *   called (this is automatically done at Yeti startup), it set the
 *   default path list for Yeti applications.
 *
 *   A convenient way to check if your script is parsed by Yeti is to do:
 *
 *     if (is_func(yeti_init) == 2) {
 *       // we are in Yeti
 *       ...
 *     } else {
 *       // not in Yeti
 *       ...
 *     }
 *
 * SEE ALSO: Y_LAUNCH, Y_HOME, Y_SITE, Y_VERSION,
 *           get_path, set_path.
 */
if (batch()) {
  yeti_init;
} else {
  write, format=" Yeti %s ready.  Copyright (c) 1996-2007, Eric Thibaut.\n",
    yeti_init();
}

func setup_package(plugname)
/* DOCUMENT PACKAGE_HOME = setup_package();
 *     -or- PACKAGE_HOME = setup_package(plugname);
 *
 *   The setup_package function must be directly called in a Yorick source
 *   file, the so-called Yorick package source file.  This function
 *   determines the package directory which is the absolute directory name
 *   of the package source file and setup Yorick search paths to include
 *   this directory.  The returned value is the package directory
 *   (guaranteed to be terminated by a slash "/").
 *
 *   If PLUGNAME is specified, the corresponding plugin is loaded
 *   (preferentially from the package directory).
 *
 *
 * SEE ALSO: plug_in, plug_dir, current_include, get_path, set_path.
 */
{
  /* Quick check. */
  path = current_include();
  if (is_void(path)) {
    error, "setup_package must be called from a Yorick source file";
  }

  /* Figure out the absolute directory from where we are called. */
  cwd = cd(".");
  j = where(strchar(path) == '/');
  if (is_array(j)) {
    pkgdir = cd(strpart(path, 1:j(0)));
    cd, cwd;
  } else {
    pkgdir = cwd;
  }
  if (is_void(pkgdir)) {
    error, "bad path for include file: \"" + path + "\"";
  }
  if (strpart(pkgdir, 0:0) != "/") {
    pkgdir += "/";
  }

  /* Setup Yorick search path. */
  list = get_path();
  if (! strlen(list)) {
    list = [];
    flag = 1n;
  } else {
    c = strchar(list);
    j = where(c == ':');
    if (is_array(j)) {
      c(j) = 0;
      list = strchar(c);
    }
    found = (list == pkgdir);
    if (noneof(found)) {
      flag = 1n;
    } else if (! found(1) || sum(found) > 1) {
      flag = 1n;
      list = list(where(! found));
    } else {
      flag = 0n; /* no need to add PKGDIR */
    }
  }
  if (flag) {
    set_path, (numberof(list) ? pkgdir + sum(":" + list) : pkgdir);
  }

  /* Setup list of directories for plugins so that the package directory is
     searched first and load package plugin. */
  if (! is_void(plugname) && is_func(plug_in)) {
    list = plug_dir();
    if (is_void(list)) {
      plug_dir, pkgdir;
    } else {
      /* move directory in first position */
      plug_dir, grow(pkgdir, list(where(list != pkgdir)));
    }
    plug_in, plugname;
  }

  return pkgdir;
}

/*---------------------------------------------------------------------------*/
/* SORTING */

extern heapsort;
/* DOCUMENT heapsort(a)
       -or- heapsort, a;
     When called as a function, returns a vector of numberof(A) longs
     containing index values such that A(heapsort(A)) is a monotonically
     increasing vector.  When called as a subroutine, performs in-place
     sorting of elements of array A.  This function uses the heap-sort
     algorithm which may be superior to the quicksort algorithm (for
     instance for integer valued arrays).  Beware that headpsort(A) and
     sort(A) differ for multidimensional arrays.

   SEE ALSO: quick_select, sort. */

extern quick_select;
/* DOCUMENT quick_select(a, k [, first, last])
 *     -or- quick_select, a, k [, first, last];
 *
 *   Find the K-th smallest element in array A.  When called as a function,
 *   the value of the K-th smallest element in array A is returned.  When
 *   called as a subroutine, the elements of A are re-ordered (in-place
 *   operation) so that A(K) is the K-th smallest element in array A and
 *   A(J) <= A(K) for J <= K and A(J) >= A(K) for J >= K.
 *
 *   Optional arguments FIRST and LAST can be used to specify the indices
 *   of the first and/or last element of A to consider: elements before
 *   FIRST and after LAST are ignored and left unchanged when called as a
 *   subroutine; index K however always refers to the full range of A.  By
 *   default, FIRST=1 and LAST=numberof(A).
 *
 *   Yorick indexing rules are supported for arguments K, FIRST and LAST
 *   (i.e. 0 means last element, etc).
 *
 *
 * EXAMPLES
 *
 *   The index K which splits a sample of N=numberof(A) elements into
 *   fractions ALPHA (before K, that is K - 1 elements) and 1 - ALPHA
 *   (after K, that is N - K elements) is such that:
 *
 *       (1 - ALPHA)*(K - 1) = ALPHA*(N - K)
 *
 *   hence:
 *
 *       K = 1 + ALPHA*(N - 1)
 *
 *   Accounting for rounding to nearest integer, this leads to:
 *
 *       quick_select(A, long(1.5 + ALPHA*(numberof(A) - 1)))
 *
 *   Therefore the first inter-quartile split is at (1-based and rounded to
 *   nearest integer) index:
 *
 *       K1 = (N + 5)/4     (with integer division)
 *
 *   the second inter-quartile (median) is at:
 *
 *       K2 = N/2 + 1       (with integer division)
 *
 *   the third inter-quartile is at:
 *
 *       K3 = (3*N + 3)/4   (with integer division)
 *
 *
 * SEE ALSO: quick_median, quick_interquartile_range, sort, heapsort.
 */

func quick_interquartile_range(a)
/* DOCUMENT quick_interquartile_range(a)
 *   Returns the interquartile range of values in array A.
 *
 * SEE ALSO
 *   quick_median, quick_select, insure_temporary.
 */
{
  n = numberof(a);
  k1 = (n + 5)/4; /* first inter-quartile */
  k3 = (3*n + 3)/4; /* third inter-quartile */
  insure_temporary, a;
  quick_select, a, k1;
  quick_select, a, k3, k1 + 1;
  return (a(k3) - a(k1));
}

func quick_median(a)
/* DOCUMENT quick_median(a)
 *   Returns the median of values in array A.
 *
 * SEE ALSO
 *   median, quick_interquartile_range,
 *   quick_select, insure_temporary.
 */
{
  n = numberof(a);
  k = (n + 1)/2;
  if (n % 2) {
    /* odd number of elements */
    return quick_select(a, k);
  } else {
    /* even number of elements */
    insure_temporary, a;
    quick_select, a, k;
    return (a(k) + a(min:k+1:n))/2.0;
  }
}

/*---------------------------------------------------------------------------*/
/* SYMBOLIC LINKS */

extern symlink_to_variable;
extern symlink_to_name;
extern is_symlink;
extern name_of_symlink;
extern value_of_symlink;
/* DOCUMENT lnk = symlink_to_variable(var)
 *     -or- lnk = symlink_to_name(varname)
 *     -or- is_symlink(lnk)
 *     -or- name_of_symlink(lnk)
 *     -or- value_of_symlink(lnk)
 *
 *   The call symlink_to_variable(var) creates a symbolic link to variable
 *   VAR.  The call symlink_to_name(varname) creates a symbolic link to
 *   variable whose name is VARNAME.  When the link object LNK is used in
 *   an 'eval' context or a 'get member' context (see examples below), LNK
 *   gets replaced 'on the fly' by the symbol which is actually stored into
 *   the corresponding Yorick's variable.  Therefore LNK adds no additional
 *   reference to the variable which only has to exist when LNK is later
 *   used.  This functionality can be used to implement 'virtual' methods
 *   for pseudo-object in Yorick (using hash tables).
 *
 *   For instance:
 *
 *      > lnk = symlink_to_variable(foo);  // variable foo does not yet exists
 *      > lnk = symlink_to_name("foo");    // same link, using a name
 *      > func foo(x) { return 2*x; }
 *      > lnk(9)
 *       18
 *      > func foo(x) { return 3*x; }
 *      > lnk(9)
 *       27
 *
 *      > z = array(complex, 10, 4);
 *      > lnk = symlink_to_variable(z);
 *      > info, lnk.re;
 *       array(double,10,4)
 *
 *   The function is_symlink(LNK) check whether LNK is a symbolic link.
 *
 *   The function name_of_symlink(LNK) returns the name of the variable
 *   linked by LNK.
 *
 *   The function value_of_symlink(LNK) returns the actual value of the
 *   variable corresponding to the symbolic link LNK.  This function can be
 *   used to force the substitution in a context where it is not
 *   automatically done.  For instance:
 *
 *     > lnk = symlink_to_variable(a);
 *     > a = random(10);
 *     > avg(lnk)
 *     ERROR (*main*) avg requires numeric argument
 *     > avg(value_of_symlink(lnk))
 *     0.383679
 *     > avg(a)
 *     0.383679
 *
 *
 * SEE ALSO: h_new.
 */

/*---------------------------------------------------------------------------*/
/* HASH TABLE OBJECTS */

extern h_debug;
/* DOCUMENT h_debug, object, ...
     Print out some debug information on OBJECT.

     ****************************
     *** WILL BE REMOVED SOON ***
     ****************************/

extern h_new;
/* DOCUMENT h_new();
       -or- h_new(key=value, ...);
       -or- h_new("key", value, ...);
     Returns  a new  hash table  object with  member(s) KEY  set  to VALUE.
     There may be  any number of KEY-VALUE pairs. A  particular member of a
     hash table  OBJ can be  specified as a  scalar string, i.e.  "KEY", or
     using keyword syntax,  i.e. KEY=.  The keyword syntax  is however only
     possible  if  KEY is  a  valid Yorick's  symbol  name.   VALUE can  be
     anything (even a non-array object).

     A  hash  table   object  can  be  used  to   implement  some  kind  of
     object-oriented  abstraction in  Yorick.  However,  in Yorick,  a hash
     table  must have  a simple  tree structure  -- no  loops or  rings are
     allowed (loops break Yorick's memory  manager -- beware).  You need to
     be careful not to do this as the error will not be detected.

     The difference between  a hash table object and a  list object is that
     items are retrieved by key  identifier rather than by order (by h_get,
     get_member or  dot dereferenciation).   It is possible  to dereference
     the contents of OBJ using the dot operator (as for a structure) or the
     get_member function.  For instance, it is legal to do:
       obj = h_new(x=span(-7,7,100), name="my name", op=sin, scale=33);
       plg, obj.op(obj.x), obj.x;
     but  the member  must already  exists  and there  are restrictions  to
     assignation, i.e. only contents of array members can be assigned:
       obj.name() = "some other string"; // ok
       obj.name   = "some other string"; // error
       obj.x(RANGE_OR_INDEX) = EXPR;     // ok if conformable AND member X
                                         // is not a 'fast' scalar (int,
                                         // long or double scalar)
       obj.x                 = EXPR;     // error
     and  assignation cannot therefore  change the  dimension list  or data
     type of  a hash table  member.  Redefinition/creation of a  member can
     always be performed  with the h_set function which  is the recommended
     method to set the value of a hash table member.

     Hash table objects behave differently depending how they are used:

        obj.key      - de-reference hash member
        obj("key")   - returns member named "key" in hash table object OBJ,
                       this is exactly the same as: h_get(obj, "key")
        obj()        - returns number of elements in hash table OBJ
        obj(i)       - returns i-th member in hash table OBJ; i is a scalar
                       integer and can be less or equal zero to start from
                       the last one; if the hash table is unmodified,
                       obj(i) is the same as obj(keys(i)) where
                       keys=h_keys(obj).

     However, beware that the behaviour of calls such that OBJ(...) may be
     changed if the has table object implements its own "evaluator" (see
     h_evaluator).

     For instance, to explorate the whole hash table, there are two
     possibilities:
       keys = h_keys(obj);
       n = numberof(keys);   // -or- n = obj()
       for (i=1 ; i<=n ; ++i) {
         a = obj(key(i));
         ...;
       }
     or:
       n = obj();
       for (i=1 ; i<=n ; ++i) {
         a = obj(i);
         ...;
       }
     the second form is simpler but is slower for large tables.

     An  important point to  remember when  using hash  table is  that hash
     members are references to their contents, i.e.
       h_set, hash, member=x;
     makes an additional  reference to array X and does  not copy the array
     although you can force that, e.g.:
       tmp = x;                   // make a copy of array X
       h_set, hash, member=tmp;   // reference copy in hash table
       tmp = [];                  // delete one reference to the copy
     Because assignation  result is  its rhs (right-hand-side),  you cannot
     do:
       h_set, hash, member=(tmp = x);   // assignation result is X
     Similarly,  unlike Yorick array  data types,  a statement  like x=hash
     does not make a copy of  the hash table, it merely makes an additional
     reference to the list.


   CAVEATS:
     In  Yorick (or  Yeti), many  objects can  be used  to  reference other
     objects: pointers, lists and hash  tables.  Since Yorick uses a simple
     reference counter to delete unused object, cyclic references (i.e.  an
     object  referencing itself  either directly  or indirectly)  result in
     objects  that  will   not  be  properly  deleted.   It   is  the  user
     reponsibility to create no cyclic  references in order to avoid memory
     leaks.   Checking a  potential (or  effective) cyclic  reference would
     require recursive  investigation of all  members of the  parent object
     and could be very time consuming.


   SEE ALSO: h_copy, h_get, h_has, h_keys, h_pop, h_set, h_stat, h_first,
             h_next, _lst, get_member. */

extern h_get;
/* DOCUMENT h_get(obj, key=);
       -or- h_get(obj, "key");
     Returns  the value  of member  KEY of  hash table  object OBJ.   If no
     member  KEY exists  in  OBJ,  nil is  returned.  h_get(OBJ, "KEY")  is
     identical to get_member(OBJ, "KEY") and also to OBJ("KEY").

   SEE ALSO h_new, get_member. */

extern h_set;
/* DOCUMENT h_set, obj, key=value, ...;
       -or- h_set, obj, "key", value, ...;
     Stores VALUE in member KEY of hash table object OBJ.  There may be any
     number of KEY-VALUE pairs.  If  called as a function, the return value
     is OBJ.

   SEE ALSO h_new, h_set_copy. */

func h_set_copy(obj, ..)
/* DOCUMENT h_set_copy, obj, key, value, ...;
     Set member KEY (a scalar string) of hash table OBJ with VALUE.  Unlike
     h_set, VALUE is duplicated if it is an array.  There may be any number
     of KEY-VALUE pairs.

   SEE ALSO h_copy, h_new, h_set. */
{
  while (more_args()) {
    key = next_arg();
    value = next_arg();
    h_set, obj, key, value;
  }
  return obj;
}

func h_copy(obj, recursively)
/* DOCUMENT h_copy(obj);
       -or- h_copy(obj, recursively);
     Effectively copy  contents of  hash table object  OBJ into a  new hash
     table that is  returned.  If argument RECURSIVELY is  true, every hash
     table object contained into OBJ  get also duplicated.  This routine is
     needed because doing CPY=OBJ, where  OBJ is a hash table object, would
     only  make a  new reference  to OBJ:  CPY and  OBJ would  be  the same
     object.

   SEE ALSO h_new, h_set, h_clone. */
{
  key_list = h_keys(obj);
  n = h_number(obj); /* number of members */
  new = h_new();
  h_evaluator, new, h_evaluator(obj);
  if (recursively) {
    for (i=1 ; i<=n ; ++i) {
      key = key_list(i);
      member = h_get(obj, key);
      h_set, new, key, (is_hash(member) ? h_copy(member, 1) : member);
    }
  } else {
    for (i=1 ; i<=n ; ++i) {
      key = key_list(i);
      member = h_get(obj, key);
      h_set, new, key, member;
    }
  }
  return new;
}

/*
 * NOTE: h_clone(obj, copy=1)           is the same as h_copy(obj)
 *       h_clone(obj, copy=1, depth=-1) is the same as h_copy(obj, 1)
 */
func h_clone(obj, copy=, depth=)
/* DOCUMENT h_clone(obj, copy=, depth=);
     Make a new  hash table with same contents as OBJ.   If keyword COPY is
     true,  a fresh  copy  is  made for  array  members.  Otherwise,  array
     members are just  referenced one more time by the  new hash table.  If
     keyword DEPTH is  non-zero, every hash table object  referenced by OBJ
     get also cloned (this is  done recursively) until level DEPTH has been
     reached  (infinite recursion  if  DEPTH is  negative).   The value  of
     keyword COPY is kept the same across the recursions.

   SEE ALSO h_new, h_set, h_copy. */
{
  local member;
  key_list = h_keys(obj); /* list of hash keys */
  n = h_number(obj); /* number of members */
  new = h_new();
  h_evaluator, new, h_evaluator(obj);
  if (depth) {
    --depth;
    for (i=1 ; i<=n ; ++i) {
      key = key_list(i);
      if (copy) member = h_get(obj, key);
      else eq_nocopy, member, h_get(obj, key);
      h_set, new, key,
        (is_hash(member) ? h_clone(member, copy=copy, depth=depth) : member);
    }
  } else if (copy) {
    for (i=1 ; i<=n ; ++i) {
      key = key_list(i);
      member = h_get(obj, key);
      h_set, new, key, member;
    }
  } else {
    for (i=1 ; i<=n ; ++i) {
      key = key_list(i);
      h_set, new, key, h_get(obj, key);
    }
  }
  return new;
}

extern h_number;
/* DOCUMENT h_number(obj)
     Returns number of entries in hash table object OB.

   SEE ALSO h_new, h_keys. */

extern h_keys;
/* DOCUMENT h_keys(obj)
     Returns list of members of hash table object OBJ as a string vector of
     key names.  The order in which keys are returned is arbitrary.

   SEE ALSO h_new, h_first, h_next, h_number. */

extern h_has;
/* DOCUMENT h_has(obj, "key")
       -or- h_has(obj, key=)
     Returns 1 if member KEY is defined in hash table object OBJ, else 0.

   SEE ALSO h_new. */

extern h_first;
extern h_next;
/* DOCUMENT h_first(obj)
       -or- h_next(obj, key)
     Get first or next key in hash table object OBJ.  Useful to run through
     all entries in a hash table (however beware that the hash table should
     be left unchanged during the scan).  For instance:

       for (key = h_first(obj) ; key ; key = h_next(obj, key)) {
         value = obj(key);
         ...;
       }

   SEE ALSO h_new, h_keys. */

extern h_evaluator;
/* DOCUMENT h_evaluator(obj)
 *     -or- h_evaluator(obj, evl);
 *     -or- h_evaluator, obj, evl;
 *
 *   Set/query evaluator function of hash table object OBJ.  When called as
 *   a function, the evaluator of OBJ prior to any change is returned as a
 *   scalar string.  If EVL is specified, it becomes the new evaluator of
 *   OBJ.  EVL must be a scalar string (the name of the evaluator
 *   function), or a function, or nil.  If EVL is explicitely nil (for
 *   instance []) or a NULL-string (for instance string(0)), the default
 *   behaviour is restored.
 *
 *   When hash table object OBJ is used as:
 *
 *     OBJ(...)
 *
 *   where "..." represents any list of arguments (including none) then its
 *   evaluator get called as:
 *
 *     EVL(OBJ, ...)
 *
 *   that is with OBJ prepended to the same argument list.
 *
 *
 * EXAMPLES:
 *   // create a hash table object:
 *   obj = h_new(data=random(200), count=0);
 *
 *   // define a fucntion:
 *   func eval_me(self, incr)
 *   {
 *      if (incr) h_set, self, count = (self.count + incr);
 *      return self.data(1 + abs(self.count)%200);
 *   }
 *
 *   // set evaluator (which must be already defined as a function):
 *   h_evaluator, obj, eval_me;
 *
 *   obj(49);   // return 49-th value
 *   obj();     // return same value
 *   obj(3);    // return 51-th value
 *   h_evaluator, obj, []; // restore standard behaviour
 *
 *   // set evaluator (not necessarily already defined as a function):
 *   h_evaluator, obj, "some_name";
 *
 *   // then define the function code prior to use:
 *   func some_name(self, a, b) { return self.count; }
 *
 *
 * SEE ALSO: h_new, h_get.
 */

func h_info(obj, align)
/* DOCUMENT h_info, obj;
       -or- h_info, obj, align;
     List contents of hash table OBJ in alphabetical order of keys.
     If second argument is true, the key names are right aligned.

   SEE ALSO: h_new, h_keys, h_first, h_next, h_show, sort. */
{
  key_list = h_keys(obj);
  if (is_void(key_list)) return;
  key_list = key_list(sort(key_list));
  n = numberof(key_list);
  width = max(strlen(key_list));
  format = swrite(format=(align?"%%%ds":"%%-%ds"), width + 1);
  for (i=1 ; i<=n ; ++i) {
    key = key_list(i);
    write, format=format, key+":";
    info, h_get(obj, key);
  }
}
local _h_show_worker;
func h_show(obj, prefix=, maxcnt=, depth=)
/* DOCUMENT h_show, obj;
 *   Display contents of hash table OBJ in a tree-like
 *   representation.  Keyword PREFIX can be used to prepend a
 *   prefix to the printed lines.  Keyword MAXCNT (default 5)
 *   can be used to specify the maximum number of elements for
 *   printing array values.
 * SEE ALSO: h_info, h_keys.
 */
{
  _h_show_maxcnt = (is_void(maxcnt) ? 5 : maxcnt);
  _h_show_worker, obj, "TOP", (is_void(prefix) ? "" : prefix), 0;
}
func _h_show_worker(obj, name, prefix, stage)
{
  if (stage == 1) {
    prefix1 = prefix + " |-";
    prefix2 = prefix + " | ";
  } else if (stage == 2) {
    prefix1 = prefix + " `-";
    prefix2 = prefix + "   ";
  } else {
    prefix1 = prefix;
    prefix2 = prefix;
  }
  if (is_hash(obj)) {
    key_list = h_keys(obj);
    if (is_array(key_list)) {
      key_list = key_list(sort(key_list));
      //width = max(strlen(key_list));
      //format = swrite(format=(align?"%%%ds":"%%-%ds"), width + 1);
    }
    n = numberof(key_list);
    e = h_evaluator(obj);
    write, format="%s %s (hash_table, %s%d %s)\n",
      prefix1, name, (e ? "evaluator=\""+e+"\", " : ""),
      n, (n <= 1 ? "entry" : "entries");
    for (k = 1; k <= n; ++k) {
      key = key_list(k);
      _h_show_worker, h_get(obj,key), key, prefix2, 1 + (k == n);
    }
  } else if (is_array(obj)) {
    descr = typeof(obj);
    dims = dimsof(obj);
    n = numberof(dims);
    k = 1;
    while (++k <= n) {
      descr += swrite(format=",%d", dims(k));
    }
    if (numberof(obj) <= _h_show_maxcnt) {
      write, format="%s %s (%s) %s\n", prefix1, name, descr, sum(print(obj));
    } else {
      write, format="%s %s (%s)\n", prefix1, name, descr;
    }
  } else if (is_void(obj)) {
    write, format="%s %s (void) []\n", prefix1, name;
  } else {
    write, format="%s %s (%s)\n", prefix1, name, typeof(obj);
  }
}

extern h_pop;
/* DOCUMENT h_pop(obj, "key")
       -or- h_pop(obj, key=)
     Pop  member KEY  out of  hash table  object OBJ  and return  it.  When
     called  as a subroutine,  the net  result is  therefore to  delete the
     member from the hash table.

   SEE ALSO h_new, h_delete. */

func h_delete(h, ..)
/* DOCUMENT h_delete(obj, "key", ...)
     Delete members KEY, ... from hash table object OBJ and return it.  Any
     KEY arguments may be present and must be array of strings or nil.

   SEE ALSO h_new, h_pop. */
{
  local key;
  while (more_args()) {
    eq_nocopy, key, next_arg();
    n = numberof(key);
    for (i=1 ; i<=n ; ++i) h_pop, h, key(i);
  }
  return h;
}

extern h_stat;
/* DOCUMENT h_stat(obj)
     Returns an histogram of the  slot occupation in hash table object OBJ.
     The  result is  a long  integer vector  with i-th  value equal  to the
     number of slots  with (i-1) items.  Note: efficient  hash table should
     keep the number of items per slot as low as possible.

   SEE ALSO h_new. */

func h_list(obj, sorted)
/* DOCUMENT h_list(obj)
       -or- h_list(obj, sorted)
     Convert hash table object OBJ  into a list: _lst("KEY1", VALUE1, ...).
     The order  of key-value pairs  is arbitrary unless argument  SORTED is
     true in which case keys get sorted in alphabetical order.

   SEE ALSO h_new, _lst, sort. */
{
  keylist = h_keys(obj);
  n = numberof(keylist);
  if (sorted && n>1) keylist = keylist(sort(keylist)(::-1));
  list = _lst();
  for (i=1 ; i<=n ; ++i) {
    /* grow the list the fast way, adding new values to its head (adding to
       the tail would make growth an N^2 proposition, as would using the
       grow function) */
    key = keylist(i);
    list = _cat(key, h_get(obj, key), list);
  }
  return list;
}

func h_cleanup(obj, recursively)
/* DOCUMENT h_cleanup, obj, 0/1;
       -or- h_cleanup(obj, 0/1);
     Delete all void  members of hash table object OBJ  and return OBJ.  If
     the second argument is a true (non nil and non-zero) empty members get
     deleted recursively.

   SEE ALSO h_new. */
{
  local member;
  keylist = h_keys(obj);
  n = numberof(keylist);
  for (i=1 ; i<=n ; ++i) {
    key = keylist(i);
    eq_nocopy, member, h_get(obj, key);
    if (is_void(member)) h_pop, obj, key;
    else if (recursively && is_hash(member)) h_cleanup, member, recursively;
  }
  return obj;
}

func h_save_symbols(____l____, ..)
/* DOCUMENT h_save_symbols(namelist, ...);
       -or- h_save_symbols(flag);
     Return  hash  table which  references  symbols  given  in NAMELIST  or
     selected by FLAG (see symbol_names).  Of course, the symbol names will
     be used as member names in the result.

   SEE ALSO h_new, h_restore_builtin, symbol_names. */
{
  /* Attempt to use dummy symbol names in this routine to avoid clash with
     the symbols ddefined in caller's context. */
  while (more_args()) grow, ____l____, next_arg();
  if ((____s____ = structof(____l____)) != string) {
    if ((____s____!=long && ____s____!=int && ____s____!=short &&
         ____s____!=char) || dimsof(____l____)(1))
      error, "expected a list of names, or nil, or a scalar integer";
    ____l____ = symbol_names(____l____);
  }
  ____s____ = h_new();
  ____n____ = numberof(____l____);
  for (____i____=1 ; ____i____<=____n____ ; ++____i____) {
    ____k____ = ____l____(____i____);
    h_set, ____s____, ____k____, symbol_def(____k____);
  }
  return ____s____;
}

local SAVE_BUILTINS;
local __h_saved_builtins;
func h_restore_builtin(name) { return h_get(__h_saved_builtins, name); }
/* DOCUMENT h_restore_builtin(name);
     Get the original definition of  builtin function NAME.  This is useful
     if you deleted by accident a  builtin function and want to recover it;
     for instance:
       sin = 1;
       ...
       sin = h_restore_builtin("sin");
     would restore the  definition of the sine function  that was redefined
     by the assignation.

     To  enable   this  feature,  you  must  define   the  global  variable
     SAVE_BUILTINS  to  be  true  before  loading the  Yeti  package.   For
     instance:
       SAVE_BUILTINS = 1;
       include, "yeti.i";
     then  all  all  current  definitions  of  builtin  functions  will  be
     referenced  in  global  hash  table __h_saved_builtins  and  could  be
     retrieved by calling h_restore_builtin.

     Note that this feature is disabled in batch mode.

   SEE ALSO h_new, h_save_symbols, batch. */
if (! batch() && SAVE_BUILTINS && ! is_hash(__h_saved_builtins)) {
  __h_saved_builtins = h_save_symbols(32);
}

/*---------------------------------------------------------------------------*/
/* MORPHO-MATH OPERATORS */

extern morph_dilation;
extern morph_erosion;
/* DOCUMENT morph_dilation(a, r);
 *     -or- morph_erosion(a, r);
 *   These functions perform a dilation/erosion morpho-math operation onto
 *   input array A which must have at most 3 dimensions.  A dilation
 *   (erosion) operation replaces every voxel of A by the maximum (minimum)
 *   value found in the voxel neighborhood as defined by the structuring
 *   element. Argument R defines the structuring element as follows:
 *
 *     - If R is a scalar integer, then it is taken as the radius (in
 *       voxels) of the structuring element.
 *
 *     - Otherwise, R gives the offsets of the structuring element relative
 *       to the coordinates of the voxel of interest.  In that case, R must
 *       an array of integers with last dimension equals to the number of
 *       dimensions of A.  In other words, if A is a 3-D array, then the
 *       offsets are:
 *
 *         DX = R(1,..)
 *         DY = R(2,..)
 *         DZ = R(3,..)
 *
 *       and the neighborhood of a voxel at (X,Y,Z) is defined as: (X +
 *       DX(I), Y + DY(I), Z + DZ(i)) for i=1,...,numberof(DX).
 *       Conversely, R = [DX, DY, DZ].  Thanks to that definition,
 *       structuring element with arbitrary shape and relative position can
 *       be used in morpho-math operations.
 *
 *       For instance, the dilation of an image (a 2-D array) IMG by a
 *       3-by-5 rectangular structuring element centered at the pixel of
 *       interest is obtained by:
 *
 *         dx = indgen(-1:1);
 *         dy = indgen(-2:2);
 *         result =  morph_dilation(img, [dx, dy(-,)])
 *
 *
 * SEE ALSO: morph_closing, morph_opening, morph_white_top_hat,
 *           morph_black_top_hat.
 */

func morph_closing(a, r)
{ return morph_erosion(morph_dilation(a, r), r); }
func morph_opening(a, r)
{ return morph_dilation(morph_erosion(a, r), r); }
/* DOCUMENT morph_closing(a, r);
       -or- morph_opening(a, r);
     Perform an image closing/closing of A by a structuring element R.  A
     closing is a dilation followed by an erosion, whereas an opening is an
     erosion followed by a dilation.  See morph_dilation for the meaning of
     the arguments.

   SEE ALSO: morph_dilation, morph_white_top_hat,
             morph_black_top_hat. */

func morph_white_top_hat(a, r, s) {
  if (! is_void(s)) a = morph_closing(a, s);
  return a - morph_opening(a, r); }
func morph_black_top_hat(a, r, s) {
  if (! is_void(s)) a = morph_opening(a, s);
  return morph_closing(a, r) - a; }
/* DOCUMENT morph_white_top_hat(a, r);
       -or- morph_white_top_hat(a, r, s);
       -or- morph_black_top_hat(a, r);
       -or- morph_black_top_hat(a, r, s);
     Perform a summit/valley detection by applying a top-hat filter to
     array A.  Argument R defines the structuring element for the feature
     detection.  Optional argument gives the structuring element used to
     apply a smoothing to A prior to the top-hat filter.  If R and S are
     specified as the radii of the structuring elements, then S should be
     smaller than R.  For instance:

       morph_white_top_hat(bitmap, 3, 1)

     may be used to detect text or lines in a bimap image.


   SEE ALSO: morph_dilation, morph_closing. */

/*---------------------------------------------------------------------------*/
/* COST FRUNCTIONS */

extern cost_l2;
extern cost_l2l1;
extern cost_l2l0;
/* DOCUMENT cost_l2(hyper, res [, grd])
 *     -or- cost_l2l1(hyper, res [, grd])
 *     -or- cost_l2l0(hyper, res [, grd])
 *
 *   These functions compute the cost for an array of residuals RES and
 *   hyper-parameters HYPER (which can have 1, 2 or 3 elements).  If
 *   optional third argument GRD is provided, it must be a simple variable
 *   reference used to store the gradient of the cost function with respect
 *   to the residuals.
 *
 *   The cost_l2() function returns the sum of squared residuals times
 *   HYPER(1):
 *
 *      COST_L2 = MU*sum(RES^2)
 *
 *   where MU = HYPER(1).
 *
 *   The cost_l2l1() and cost_l2l0() functions are quadratic (L2) for small
 *   residuals and non-quadratic (L1 and L0 respectively) for larger
 *   residuals.  The thresholds for L2 / non-L2 transition are given by
 *   the second and third value of HYPER.
 *
 *   If HYPER = [MU, TINF, TSUP] with TINF < 0 and TSUP > 0, an asymmetric
 *   cost function is computed as:
 *
 *      COST_L2L0 = MU*(TINF^2*sum(atan(RES(INEG)/TINF)^2) +
 *                      TSUP^2*sum(atan(RES(IPOS)/TPOS)^2))
 *
 *      COST_L2L1 = 2*MU*(TINF^2*sum(RES(INEG)/TINF -
 *                                   log(1 + RES(INEG)/TINF)) +
 *                        TSUP^2*sum(RES(IPOS)/TSUP -
 *                                   log(1 + RES(IPOS)/TSUP)))
 *
 *   with INEG = where(RES < 0) and IPOS = where(RES >= 0).  If any or the
 *   thresholds is negative or zero, the L2 norm is used for residuals with
 *   the corresponding sign (same as having an infinite threshold level).
 *   The different cases are:
 *
 *      TINF < 0    ==> L2-L1/L0 norm for negative residuals
 *      TINF = 0    ==> L2 norm for negative residuals
 *      TSUP = 0    ==> L2 norm for positive residuals
 *      TSUP > 0    ==> L2-L1/L0 norm for positive residuals
 *
 *   For residuals much smaller (in magnitude) than the thresholds, the
 *   non-L2 cost function behave as the L2 one.  For residuals much larger
 *   (in magnitude), than the thresholds, the L2-L1 cost function is L1
 *   (i.e. scales as abs(RES)) and the L2-L0 cost function is L0 (tends to
 *   saturate).
 *
 *   If HYPER = [MU, T], with T>0, a symmetric non-L2 cost function is
 *   computed with TINF = -T and TSUP = +T; in other words:
 *
 *      COST_L2L0 = MU*T^2*sum(atan(RES/T)^2)
 *
 *      COST_L2L1 = 2*MU*T^2*sum(abs(RES/T) - log(1 + abs(RES/T)))
 *
 *   If HYPER has only one element (MU) the L2 cost function is used.  Note
 *   that HYPER = [MU, 0] or HYPER = [MU, 0, 0] is the same as HYPER = MU
 *   (i.e. L2 cost function).
 *
 *
 * SEE ALSO:
 */

/*---------------------------------------------------------------------------*/
/* 1D CONVOLUTION AND "A TROUS" WAVELET TRANSFORM */

extern __yeti_convolve_f;
/* PROTOTYPE
     void yeti_convolve_f(float array dst, float array src, int stride,
                          int n, int nafter, float array ker, int w,
		          int scale, int border, float array ws); */

extern __yeti_convolve_d;
/* PROTOTYPE
     void yeti_convolve_d(double array dst, double array src, int stride,
		          int n, int nafter, double array ker, int w,
                          int scale, int border, double array ws); */

func yeti_convolve(a, which=, kernel=, scale=, border=, count=)
/* DOCUMENT ap = yeti_convolve(a)
     Convolve  array A along  its dimensions  (all by  default) by  a given
     kernel.  By default, the convolution kernel is [1,4,6,4,1]/16.0.  This
     can be  changed by using keyword  KERNEL (but the kernel  must have an
     odd number  of elements).  The following operation  is performed (with
     special  handling for the  boundaries, see  keyword BORDER)  along the
     direction(s) of interest:
     |         ____
     |         \
     |   AP(i)= \ KERNEL(j+W) * A(i + j*SCALE)
     |          /
     |         /___
     |     -W <= j <= +W
     |
     where  numberof(KERNEL)=2*W+1.  Except  for  the SCALE  factor, AP  is
     mostly  a convolution  of A  by array  KERNEL along  the  direction of
     interest.

     Keyword WHICH can be used  to specify the dimension(s) of interest; by
     default, all  dimensions get convolved.   As for indices,  elements in
     WHICH less than  1 is taken as relative to the  final dimension of the
     array.  You may specify  repeated convolution along some dimensions by
     using them several times in array WHICH (see keyword COUNT).

     Keyword BORDER can be used to set the handling of boundary conditions:
       BORDER=0  Extrapolate  missing  values  by  the left/rightmost  ones
                 (this is the default behaviour).
       BORDER=1  Extrapolate missing left  values by zero and missing right
                 values by the rightmost one.
       BORDER=2  Extrapolate  missing left  values by the  leftmost one and
                 missing right values by zero.
       BORDER=3  Extrapolate missing left/right values by zero.
       BORDER=4  Use periodic conditions.
       BORDER>4 or BORDER<0
                 Do   not   extrapolate   missing  values   but   normalize
                 convolution product  by sum  of kernel weights  taken into
                 account (assuming they are all positive).

     By  default, SCALE=1 which  corresponds to  a simple  convolution.  An
     other value can be used thanks  to keyword SCALE (e.g. for the wavelet
     "a trou" method).  The value of SCALE must be a positive integer.

     Keyword COUNT  can be used to  augment the amount  of smoothing: COUNT
     (default COUNT=1) is  the number of convolution passes.   It is better
     (i.e. faster) to use only one pass with appropriate convolution kernel
     (see keyword KERNEL).

  SEE ALSO yeti_wavelet.

  RESTRICTIONS
    1. Should use the in-place ability of the operation to limit the number
       of array copies.
    2. Complex convolution not yet  implemented (although it exists in the
       C-code). */
{
  /* Check data type of A. */
  type = structof(a);
  if (type == complex) {
    return (yeti_convolve(double(a), which=which, kernel=kernel, scale=scale,
                          border=border, count=count)
            + 1i*yeti_convolve(a.im, which=which, kernel=kernel, scale=scale,
                               border=border, count=count));
  } else if (type == double) {
    op = __yeti_convolve_d;
  } else if (type == float || type == long || type == int || type == short ||
             type == char) {
    op = __yeti_convolve_f;
    type = float;
  } else {
    error, "bad data type";
  }
  a = type(a);  /* force a private copy of A */

  /* Check dimensions of A and keyword WHICH. */
  dims = dimsof(a);
  rank = dims(1);
  if (is_void(which)) {
    which = indgen(rank);
  } else {
    which += (which <= 0)*rank;
    if (min(which) < 1 || max(which) > rank)
      error, "dimension index out of range in WHICH";
  }

  /* Check KERNEL and other keywords. */
  if (is_void(kernel)) {
    k0= type(0.375);  /* 6.0/16.0 */
    k1= type(0.25);   /* 4.0/16.0 */
    k2= type(0.0625); /* 1.0/16.0 */
    kernel= [k2, k1, k0, k1, k2];
  }
  if ((w = numberof(kernel))%2 != 1)
    error, "KERNEL must have an odd number of elements";
  if (is_void(scale)) scale = 1;
  else if (structof(scale+0)!=long || scale<=0)
    error, "bad value for keyword SCALE";
  if (is_void(border)) border = 0;
  if (is_void(count)) count = 1;

  /* Compute strides. */
  stride = array(1, rank);
  for (s=1,i=2 ; i<=rank ; ++i) stride(i) = stride(i-1)*dims(i);
  stride = stride(which);
  dims = dims(which + 1);
  nafter = numberof(a)/(dims*stride);

  /* Apply the operator along every dimensions of interest. */
  for (i=1 ; i<=numberof(which) ; ++i) {
    len = dims(i);
    for (j=1 ; j<=count ; ++j) {
      op, a, a, stride(i), len, nafter(i), kernel, (w-1)/2, scale, border,
        array(type, 2*len);
    }
  }
  return a;
}

func yeti_wavelet(a, order, which=, kernel=, border=)
/* DOCUMENT cube = yeti_wavelet(a, order)
     Compute the "a trou" wavelet transform of A.  The result is such
     that:
       CUBE(.., i) = S_i - S_(i+1)
     where:
       S_1 = A
       S_(i+1) = yeti_convolve(S_i, SCALE=2^(i-1))
     As a consequence:
       CUBE(..,sum) = A;

  SEE ALSO yeti_convolve. */
{
  if (((s=structof(order)) != long && s!=int && s!=short && s!=char) ||
      dimsof(order)(1) || order<0) {
    error, "ORDER must be a non-negative integer";
  }
  dims = dimsof(a);
  grow, dims, order+1;
  ++dims(1);
  cube = array(structof(a(1)+0.0f), dims);
  for (scale=1, i=1 ; i<=order ; ++i, scale*=2) {
    ap = a;
    a = yeti_convolve(a, which=which, kernel=kernel, scale=scale,
                      border=border);
    cube(..,i) = ap-a;
  }
  cube(..,0) = a;
  return cube;
}

extern smooth3;
/* DOCUMENT smooth3(a)
     Returns array A smoothed by a simple 3-element convolution (but for
     the edges).  In one dimension, the smoothing operation reads:
        smooth3(A)(i) = C*A(i) + D*(A(i-1) + A(i+1))
     but for the first and last element for which:
        smooth3(A)(1) = E*A(1) + D*A(2)
        smooth3(A)(n) = E*A(n) + D*A(n-1)
     where N is the length of the dimension and the coefficients are:
        C = 0.5
        D = 0.25
        E = 0.75
     With the default value of C (see keyword C below), the smoothing
     operation is identical to:

        smooth3(A) = A(pcen)(zcen)             for a 1D array
        smooth3(A) = A(pcen,pcen)(zcen,zcen)   for a 2D array
        ...                                    and so on

     Keyword C can be used to specify another value for the coefficient
     C (default: C=0.5); coefficients D and E are computed as follows:
        D = 0.5*(1 - C)
        E = 0.5*(1 + C)

     The default is to smooth A along all its dimensions, but keyword WHICH
     can be used to specify the only dimension to smooth.  If WHICH is less
     or equal zero, then the smoothed dimension is the last one + WHICH.

     The smoothing operator implemented by smooth3 has the following
     properties:

     1. The smoothing operator is linear and symmetric (for any number of
        dimensions in A).  The symmetry of the smoothing operator is
        important for the computation of gradients in regularization.  For
        instance, let Y = smooth3(X) and Q be a scalar function of Y, then
        then the gradient of Q with respect to X is simply:
           DQ_DX = smooth3(DQ_DY)
        where DQ_DY is the gradient of Q with respect to Y.

     2. For a vector, A, smooth3(A)=S(,+)*A(+) where the matrix S is
        tridiagonal:

           [E D         ]
           [D C D       ]
           [  D C D     ]
           [   \ \ \    ]    where, to improve readability,
           [    \ \ \   ]    missing values are all zero.
           [     D C D  ]
           [       D C D]
           [         D E]

        You can, in principle, reverse the smoothing operation with TDsolve
        along each dimensions of smooth3(A).  Note: for a vector A, the
        operator S-I applied to A (where I is the identity matrix) is the
        finite difference 2nd derivatives of A (but for the edges).

     3. The definition of coefficients C, D and E insure that the smoothing
        operator does not change the sum of the element values of its
        argument, i.e.: sum(smooth3(A)) = sum(A).

     4. Only an array with all elements having the same value is invariant
        by the smoothing operator.  In fact "slopes" along dimensions of A
        are almost invariant, only the values along the edges are changed.


   KEYWORDS: c, which.

   SEE ALSO: TDsolve. */

/*---------------------------------------------------------------------------*/
/* GIST WINDOW AND ALARM CALLBACK */

extern window_geometry;
/* DOCUMENT window_geometry()
       -or- window_geometry(win)
     Get geometry settings of the  visible region of display window WIN (or
     current window  if WIN is nil  or not specified).   These settings are
     subject to change  each time the window get resized.   The result is a
     vector of 6 doubles:
       [DPI, ONE_PIXEL, XBIAS, YBIAS, WIDTH, HEIGHT]
     where:
       DPI = dot-per-inch of WIN
       ONE_PIXEL = pixel size in NDC units
       XBIAS = abscissa offset in NDC units
       YBIAS = ordinate offset in NDC units
       WIDTH = width of visible region in pixels
       HEIGHT = height of visible region in pixels
     Pixel coordinates (XPIX,YPIX) run  from top-left (0,0) to bottom-right
     (WIDTH-1,HEIGHT-1).  The conversion to NDC coordinates is:
       XNDC = XBIAS + XPIX*ONE_PIXEL;
       YNDC = YBIAS - YPIX*ONE_PIXEL;

     If window WIN does not exists, all output values are zero.

     Notes:
       (1) The  top/left  margin(s) used  by  Gist window  to display  some
           message are not considered as part of the "visible" region.
       (2) An  extra 0.5  pixel offset has  been added to  (XBIAS,YBIAS) to
           avoid rounding errors.

   SEE ALSO: window, current_window, viewport, limits. */

extern window_select;
extern window_exists;
extern window_list;
/* DOCUMENT window_select(n)
       -or- window_exists(n)
       -or- window_list()

     The function window_select makes window number N the current one and
     return 1 (true); unless window number N does not exists, in which case
     the current window is left unchanged and 0 (false) is returned.

     The function window_exists returns 1 or 0 whether or not window number
     N exists.

     The function window_list returns the list of existing windows as a
     vector of longs or nil if no window currently exists.

   SEE ALSO: window, current_window, redraw, fma, limits, window_geometry.
 */

extern set_alarm;
/* DOCUMENT set_alarm, secs, callback;
     Arrange for function CALLBACK to get called with no argument in SECS
     seconds.  CALLBACK can be specified either as a Yorick function or as
     a function name.  If CALLBACK is given as a name, the symbol is
     resolved at alarm time (after SECS seconds) this permits to (re)define
     the effective CALLBACK function before alarm expires.  When called as
     a function, the returned value is the alarm time in WALL seconds.

   SEE ALSO: set_idler. */

/*---------------------------------------------------------------------------*/
/* STRING ROUTINES */

func strtrimleft(s)  {return strtrim(s, 1);}
func strtrimright(s) {return strtrim(s, 2);}
/* DOCUMENT strtrimleft(s);
       -or- strtrimrigth(s);
     Returns input (array of) string(s) S without leading or trailing
     blanks.

   SEE ALSO strlower, strupper, string, strtrim. */

func strlower(s) { return strcase(0, s); }
func strupper(s) { return strcase(1, s); }
/* DOCUMENT strlower(s);
       -or- strupper(s);
     Returns input (array of) string(s) S converted to lower/upper case
     letters.

   SEE ALSO string, strcase, strtrimleft. */

/*---------------------------------------------------------------------------*/
/* FILE ROUTINES */

extern filepath;
/* DOCUMENT filepath(file);
     Return full path name of file(s).  Argument FILE can be either an open
     binary/text file or  an array of file names (in  the latter case tilde
     expansion  is performed and  the result  will have  the same  shape as
     input).

   SEE ALSO open. */

//extern tmpfile; // FIXME:
//extern named_reference; // FIXME:

/*---------------------------------------------------------------------------*/
/* MATH ROUTINES */

extern sinc;
/* DOCUMENT sinc(x);
     Returns the "sampling function" of X as defined by Woodward (1953) and
     Bracewell (1999):

       sinc(x) = 1                 for x=0
                 sin(PI*x)/(PI*x)  otherwise

     Note: This definition correspond to the "normalized sinc function";
     some other authors may define the sampling function without the PI
     factors in the above expression.


   REFERENCES
     Bracewell, R. "The Filtering  or Interpolating Function, sinc(x)."  In
     "The  Fourier Transform  and  Its Applications",  3rd  ed.  New  York:
     McGraw-Hill, pp. 62-64, 1999.

     Woodward, P.  M. "Probability and Information Theory with Applications
     to Radar". New York: McGraw-Hill, 1953.

  SEE ALSO: sin. */

extern round;
/* DOCUMENT round(x);
     Returns X rounded to the nearest integer (as a floating point value). */

extern arc;
/* DOCUMENT arc(x);
     Returns angle X wrapped in range (-PI, +PI]. */

/*---------------------------------------------------------------------------*/
/* SPARSE MATRICES AND MATRIX-VECTOR MULTIPLICATION */

extern sparse_matrix;
/* DOCUMENT s = sparse_matrix(coefs, row_dimlist, row_indices,
 *                                   col_dimlist, col_indices);
 *
 *   Returns a sparse matrix object.  COEFS is an array with the non-zero
 *   coefficients of the full matrix.  ROW_DIMLIST and COL_DIMLIST are the
 *   dimension lists of the matrix 'rows' and 'columns'.  ROW_INDICES and
 *   COL_INDICES are the 'row' and 'column' indices of the non-zero
 *   coefficients of the full matrix.
 *
 *   The sparse matrix object S can be used to perform sparse matrix
 *   multiplication as follows:
 *
 *     S(x) or S(x, 0) yields the result of matrix multiplication of
 *             'vector' X by S; X must be an array with dimension list
 *             COL_DIMLIST (or a vector with as many elements as an array
 *             with such a dimension list); the result is an array with
 *             dimension list ROW_DIMLIST.
 *
 *     S(y, 1) yields the result of matrix multiplication of 'vector' Y by
 *             the transpose of S; Y must be an array with dimension list
 *             ROW_DIMLIST (or a vector with as many elements as an array
 *             with such a dimension list); the result is an array with
 *             dimension list COL_DIMLIST.
 *
 *    The contents of the sparse matrix object S can be queried as with a
 *    regular Yorick structure: S.coefs, S.row_dimlist, S.row_indices,
 *    S.col_dimlist or S.col_indices are valid expressions if S is a sparse
 *    matrix.
 *
 *
 *  SEE ALSO: is_sparse_matrix, mvmult,
 *            sparse_expand, sparse_squeeze, sparse_grow.
 */

extern is_sparse_matrix;
/* DOCUMENT is_sparse_matrix(obj)
 *   Returns true if OBJ is a sparse matrix object; false otherwise.
 *
 *  SEE ALSO: sparse_matrix.
 */


func sparse_grow(s, coefs, row_indices, col_indices)
/* DOCUMENT sparse_grow(s, coefs, row_indices, col_indices);
 *
 *   Returns a sparse matrix object obtained by growing the non-zero
 *   coefficients of S by COEFS with the corresponding row/column indices
 *   given by ROW_INDICES and COL_INDICES which must have the same number
 *   of elements as COEFS.
 *
 *  SEE ALSO: sparse_matrix.
 */
{
  return sparse_matrix(grow(s.coefs, coefs),
                       s.row_dimlist, grow(s.row_indices, row_indices),
                       s.col_dimlist, grow(s.col_indices, col_indices));
}

func sparse_squeeze(a, n)
/* DOCUMENT s = sparse_squeeze(a);
 *     -or- s = sparse_squeeze(a, n);
 *   Convert array A into its sparse matrix representation.  Optional
 *   argument N (default, N=1) is the number of dimensions of the input
 *   space.  The dimension list of the input space are the N trailing
 *   dimensions of A and, assuming that A has NDIMS dimensions, the
 *   dimension list of the output space are the NDIMS - N leading
 *   dimensions of A.
 *
 * SEE ALSO: sparse_matrix, sparse_expand.
 */
{
  if (! is_array(a)) error, "unexpected non-array";
  dimlist = dimsof(a);
  ndims = dimlist(1);
  if (is_void(n)) n = 1; /* one trailing dimension for the input space */
  if ((m = ndims - n) < 0) error, "input space has too many dimensions";
  if (! is_array((i = where(a)))) error, "input array is zero everywhere!";
  (row_dimlist = array(long, m + 1))(1) = m;
  stride = 1;
  if (m >= 1) {
    row_dimlist(2:) = dimlist(2:m+1);
    for (j=m+1;j>=2;--j) stride *= dimlist(j);
  }
  (col_dimlist = array(long, n + 1))(1) = n;
  if (n >= 1) col_dimlist(2:) = dimlist(m+2:0);
  j = i - 1;
  return sparse_matrix(a(i),
                       row_dimlist, 1 + j%stride,
                       col_dimlist, 1 + j/stride);
}

func sparse_expand(s)
/* DOCUMENT a = sparse_expand(s);
 *   Convert sparse matrix S into standard Yorick's array A.
 *
 * SEE ALSO: sparse_squeeze, histogram.
 */
{
  row_dimlist = s.row_dimlist;
  stride = 1;
  j = row_dimlist(1) + 2;
  while (--j >= 2) stride *= row_dimlist(j);
  a = array(structof(s.coefs), row_dimlist, s.col_dimlist);
#if 0
  /* We cannot do that because, coefficients may not be unique. */
  a(s.row_indices + (s.col_indices - 1)*stride) = s.coefs;
#endif
  a(*) = histogram(s.row_indices + (s.col_indices - 1)*stride,
                   s.coefs, top=numberof(a));
  return a;
}

extern mvmult;
/* DOCUMENT y = mvmult(a, x);
 *     -or- y = mvmult(a, x, 0/1);
 *
 *   Returns the result of (generalized) matrix-vector multiplication of
 *   vector X (a regular Yorick array) by matrix A (a regular Yorick array
 *   or a sparse matrix).  The matrix-vector multiplication is performed as
 *   if there is only one index running over the elements of X and the
 *   trailing/leading dimensions of A.
 *
 *   If optional last argument is omitted or false, the summation index
 *   runs across the trailing dimensions of A which must be the same as
 *   those of X and the dimensions of the result are the remaining leading
 *   dimensions of A.
 *
 *   If optional last argument is 1, the matrix operator is transposed: the
 *   summation index runs across the leading dimensions of A which must be
 *   the same as those of X and the dimensions of the result are the
 *   remaining trailing dimensions of A.
 *
 * SEE ALSO: sparse_matrix, sparse_squeeze.
 */


/*---------------------------------------------------------------------------*/
/* ACCESSING YORICK'S INTERNALS */

extern is_scalar;
extern is_vector;
extern is_matrix;
/* DOCUMENT is_scalar(x)
 *     -or- is_vector(x)
 *     -or- is_matrix(x)
 *   These functions return true if X is (respectively) a scalar, a vector
 *   (i.e. a 1-D array), or a matrix (i.e. a 2-D array).
 *
 *  SEE ALSO: dimsof,
 *            is_array, is_func, is_hash, is_integer, is_list, is_range,
 *            is_stream, is_struct, is_void.
 */

extern is_integer;
extern is_real;
extern is_complex;
extern is_numerical;
extern is_string;
extern is_pointer;
/* DOCUMENT is_integer(x)
 *     -or- is_real(x)
 *     -or- is_complex(x)
 *     -or- is_numerical(x)
 *     -or- is_string(x)
 *     -or- is_pointer(x)
 *   These functions  return true if  X is an  array of type:  integer, real
 *   (i.e.  double or  float), complex,  numerical (i.e.  integer,  real or
 *   complex), string, or pointer.
 *
 * SEE ALSO: structof, dimsof,
 *           is_array, is_func, is_hash, is_list, is_range, is_scalar,
 *           is_stream, is_struct, is_void.
 */

extern is_hash;
/* DOCUMENT is_hash(object)
 *   Returns 1, if OBJECT is a regular hash table; returns 2, if OBJECT is
 *   a hash table with a specialized evaluator; returns 0, if OBJECT is not
 *   a hash table.
 *
 * SEE ALSO: h_new, h_evaluator,
 *           is_array, is_func, is_integer, is_list, is_range, is_scalar,
 *           is_stream, is_struct, is_void.
 */

// extern is_list;
// /* DOCUMENT is_list(object)
//  *   Returns 1 if OBJECT is a list, else 0.
//  *
//  * SEE ALSO: _lst,
//  *          is_array, is_func, is_hash, is_integer, is_range, is_scalar,
//  *          is_stream, is_struct, is_void.
//  */

local T_CHAR, T_SHORT, T_INT, T_LONG;
local T_FLOAT, T_DOUBLE, T_COMPLEX;
local T_STRING, T_POINTER, T_STRUCT;
local T_RANGE, T_LVALUE, T_VOID;
local T_FUNCTION, T_BUILTIN;
local T_STRUCTDEF, T_STREAM, T_OPAQUE;
local typeIDof; /* FIXME: obsolete */
extern identof;
/* DOCUMENT identof(object)
 *   Returns type identifier of OBJECT as a long integer:
 *      0 (T_CHAR)      for an array of char('s)
 *      1 (T_SHORT)     for an array of short('s)
 *      2 (T_INT)       for an array of int('s)
 *      3 (T_LONG)      for an array of long('s)
 *      4 (T_FLOAT)     for an array of float('s)
 *      5 (T_DOUBLE)    for an array of double('s)
 *      6 (T_COMPLEX)   for an array of complex('s)
 *      7 (T_STRING)    for an array of string('s)
 *      8 (T_POINTER)   for an array of pointer('s)
 *      9 (T_STRUCT)    for a structure object
 *     10 (T_RANGE)     for a range object
 *     11 (T_LVALUE)    for a lvalue
 *     12 (T_VOID)      for a void (undefined) object
 *     13 (T_FUNCTION)  for a function array
 *     14 (T_BUILTIN)   for a builtin array
 *     15 (T_STRUCTDEF) for a data type or structure definition
 *     16 (T_STREAM)    for a file stream
 *     17 (T_OPAQUE)    for an opaque object
 *
 * SEE ALSO typeof, structof.
 */
if (! is_func(typeIDof)) typeIDof = identof; /* FIXME: obsolete */
T_CHAR = 0;
T_SHORT = 1;
T_INT = 2;
T_LONG = 3;
T_FLOAT = 4;
T_DOUBLE = 5;
T_COMPLEX = 6;
T_STRING = 7;
T_POINTER = 8;
T_STRUCT = 9;
T_RANGE = 10;
T_LVALUE = 11;
T_VOID = 12;
T_FUNCTION = 13;
T_BUILTIN = 14;
T_STRUCTDEF = 15;
T_STREAM = 16;
T_OPAQUE = 17;

extern nrefsof;
/* DOCUMENT nrefsof(object)
     Returns number of references on OBJECT.

   SEE ALSO: unref. */

extern get_encoding;
/* DOCUMENT get_encoding(name);
     Return the data layout for machine NAME, one of:
       "native"   the current machine
          (little-endians)
       "i86"      Intel x86 Linux
       "ibmpc"    IBM PC (2 byte int)
       "alpha"    Compaq alpha
       "dec"      DEC workstation (MIPS), Intel x86 Windows
       "vax"      DEC VAX (H-double)
       "vaxg"     DEC VAX (G-double)
          (big-endians)
       "xdr"      External Data Representation
       "sun"      Sun, HP, SGI, IBM-RS6000, MIPS 32 bit
       "sun3"     Sun-2 or Sun-3 (old)
       "sgi64"    SGI, Sun, HP, IBM-RS6000 64 bit
       "mac"      MacIntosh 68000 (power Mac, Gx are __sun)
       "macl"     MacIntosh 68000 (12 byte double)
       "cray"     Cray XMP, YMP

     The result is a vector of 32 long's as follow:
       [size, align, order] repeated 6  times for  char,  short, int, long,
                            float,  and double, except  that char  align is
                            always  1,   so  result(2)  is   the  structure
                            alignment (see struct_align).
       [sign_address,  exponent_address, exponent_bits,
        mantissa_address, mantissa_bits,
        mantissa_normalization, exponent_bias]  repeated  twice  for  float
                            and double.  See the comment at the top of file
                            prmtyp.i for an explanation of these fields.

     The total number of items is therefore 3*6 + 7*2 = 32.

   SEE ALSO get_primitives, set_primitives, install_encoding, machine_constant. */

func install_encoding(file, encoding)
/* DOCUMENT install_encoding, file, encoding;
     Set layout of  primitive data types for binary  stream FILE.  ENCODING
     may be  one of the  names accepted by  get_encoding or an array  of 32
     integers as explained in get_encoding documentation.

    SEE ALSO: get_encoding, install_struct. */
{
  /* Get encoding parameters with minimal check. */
  if (structof(encoding) == string) {
    p = get_encoding(encoding);
  } else {
    if ((s = structof(encoding)) == long) p = encoding;
    else if (/*s==char || s==short || */s==int) p = long(encoding);
    else error, "bad data type for ENCODING";
    if (numberof(p) != 32) error, "bad number of elements for encoding";
  }

  /* Install primitive definitions. */
  install_struct, file, "char",    1,     1,     p( 3);
  install_struct, file, "short",   p( 4), p( 5), p( 6);
  install_struct, file, "int",     p( 7), p( 8), p( 9);
  install_struct, file, "long",    p(10), p(11), p(12);
  install_struct, file, "float",   p(13), p(14), p(15), p(19:25);
  install_struct, file, "double",  p(16), p(17), p(18), p(26:32);
  struct_align, file, p(2);
}

func same_encoding(a, b)
/* DOCUMENT same_encoding(a, b)
     Compare primitives  A and B  which must be conformable  integer arrays
     with first dimension equals to 32 (see set_primitives).  The result is
     an array  of int's with one  less dimension than A-B  (the first one).
     Some checking is  performed for the operands.  The  byte order for the
     char data type is ignored in the comparison.

   SEE ALSO install_encoding, get_encoding.*/
{
  if (! is_array((d = dimsof(a, b))) || d(1) < 1 || d(2) != 32)
    error, "bad dimensions";
  diff = abs(a - b);
  if ((s = structof(diff)) != long && s != int) error, "bad data type";
  if (anyof(a(1,..) != 1) || anyof(b(1,..) != 1))
    error, "unexpected sizeof(char) != 1";
  diff(3, ..) = 0; /* ignore byte order for type char */
  return ! diff(max,);
}

local DBL_EPSILON, DBL_MIN, DBL_MAX;
local FLT_EPSILON, FLT_MIN, FLT_MAX;
extern machine_constant;
/* DOCUMENT machine_constant(str)
 *   Returns the value of the machine dependent constant given its name
 *   STR.  STR is a scalar string which can be one of (prefixes "FLT_" and
 *   "DBL_" are for single/double precision respectively):
 *
 *     "FLT_MIN",
 *     "DBL_MIN" - minimum normalized positive floating-point number;
 *
 *     "FLT_MAX",
 *     "DBL_MAX" - maximum representable finite floating-point number;
 *
 *     "FLT_EPSILON",
 *     "DBL_EPSILON" - the difference between 1 and the least value greater
 *             than 1 that is representable in the given floating point
 *             type: B^(1 - P);
 *
 *     "FLT_MIN_EXP",
 *     "DBL_MIN_EXP" - minimum integer EMIN such that FLT_RADIX^(EMIN - 1)
 *             is a normalized floating-point value;
 *
 *     "FLT_MIN_10_EXP"
 *     "DBL_MIN_10_EXP" - minimum negative integer such that 10 raised to
 *             that power is in the range of normalized floating-point
 *             numbers: ceil(log10(B)*(EMIN - 1));
 *
 *     "FLT_MAX_EXP",
 *     "DBL_MAX_EXP" - maximum integer EMAX such that FLT_RADIX^(EMAX - 1)
 *             is a normalized floating-point value;
 *
 *     "FLT_MAX_10_EXP"
 *     "DBL_MAX_10_EXP" - maximum integer such that 10 raised to that power
 *             is in the range of normalized floating-point numbers:
 *             floor(log10((1 - B^(-P))*(B^EMAX)))
 *
 *     "FLT_RADIX" - radix of exponent representation, B;
 *
 *     "FLT_MANT_DIG",
 *     "DBL_MANT_DIG" - number of base-FLT_RADIX significant digits P in the
 *             mantissa;
 *
 *     "FLT_DIG",
 *     "DBL_DIG" - number of decimal digits, Q, such that any floating-point
 *             number with Q decimal digits can be rounded into a
 *             floating-point number with P (FLT/DBL_MANT_DIG) radix B
 *             (FLT_RADIX) digits and back again without change to the Q
 *             decimal digits:
 *                 Q = P*log10(B)                if B is a power of 10
 *                 Q = floor((P - 1)*log10(B))   otherwise
 *
 *
 *  SEE ALSO: get_encoding.
 */
DBL_EPSILON = machine_constant("DBL_EPSILON");
DBL_MIN = machine_constant("DBL_MIN");
DBL_MAX = machine_constant("DBL_MAX");
FLT_EPSILON = machine_constant("FLT_EPSILON");
FLT_MIN = machine_constant("FLT_MIN");
FLT_MAX = machine_constant("FLT_MAX");

extern symbol_exists;
/* DOCUMENT symbol_exists(name)
     Check  if variable/function named  NAME exists.   This routine  can be
     used  prior  to  symbol_def  to  check existence  of  a  symbol  since
     symbol_def raise an error for non-existing symbol.

   SEE ALSO symbol_def, symbol_names, symbol_set.*/

extern symbol_names;
/* DOCUMENT symbol_names()
       -or- symbol_names(flags)
     Return an  array of  strings with  the names of  all symbols  of given
     type(s) found in  global symbol table.  To select  the type of symbol,
     FLAGS is be the bitwise-or of one or more of the following bits:
         1 - basic array symbols
         2 - structure instance symbols
         4 - range symbols
         8 - nil symbols (i.e. symbols undefined at current scope level)
        16 - interpreted function symbols
        32 - builtin function symbols
        64 - structure definition symbols
       128 - file stream symbols
       256 - opaque symbols
       512 - list objects
      1024 - hash-table objects
      2048 - auto-loaded functions

     The special value FLAGS=-1 can be used to get all names found in
     global symbol table.  The default (if FLAGS is nil or omitted) is to
     return the names of all symbols but the nil ones.  Beware that lists,
     hash tables and auto-loaded functions are also opaque symbols.

   SEE ALSO symbol_def, symbol_exists, symbol_info, symbol_set.*/

func symbol_info(____n____)
/* DOCUMENT symbol_info, flags;
       -or- symbol_info, names;
       -or- symbol_info;
     Print out some information about  Yorick's symbols.  FLAGS is a scalar
     integer used to select symbol types (as in symbol_names).  NAMES is an
     array  of symbol  names.  If  argument  is omitted  or undefined,  all
     defined array symbols get selected (as with FLAGS=3).

   SEE ALSO: mem_info, symbol_def, symbol_names.*/
{
  /* attempt to use _very_ odd names to avoid clash with caller */
  if (is_void(____n____)) ____n____ = symbol_names(3);
  else if (structof(____n____) != string) ____n____ = symbol_names(____n____);
  for (____i____=1 ; ____i____<=numberof(____n____) ; ++____i____) {
    write, format="%s:", ____n____(____i____);
    info, symbol_def(____n____(____i____));
  }
}

func mem_info(____a____)
/* DOCUMENT mem_info;
       -or- mem_info, count;
     Print out some information about memory occupation. If COUNT is
     specified, the COUNT biggest (in bytes) symbols are listed (use
     COUNT<0 to list all symbols sorted by size).

   BUGS:
     Only the memory used by Yorick's array symbols is considered,
     e.g. arrays only referenced by a pointer are not considered.  Symbols
     which are aliases (e.g. by using eq_nocopy) may be considered several
     times.

   SEE ALSO: symbol_def, symbol_info, symbol_names. */
{
  ____n____ = symbol_names(3);
  ____i____ = numberof(____n____);
  ____s____ = array(long, ____i____);
  while (____i____ > 0) {
    ____s____(____i____) = sizeof(symbol_def(____n____(____i____)));
    --____i____;
  }
  ____i____ = sum(____s____);
  write, format="Total memory used by array symbols: %d bytes (%.3f Mb)\n",
    ____i____, ____i____/1024.0^2;
  if (____a____) {
    ____i____ = sort(____s____);
    if (____a____ > 0 && ____a____ < numberof(____i____))
      ____i____ = ____i____(1-____a____:0);
    ____n____ = ____n____(____i____);
    ____s____ = ____s____(____i____);
    ____i____ = numberof(____i____);
    if (____i____ > 1) {
      write, format="The %d biggest symbols are:\n", ____i____;
    } else {
      write, format="%s", "The biggest symbol is:\n";
    }
    ____a____ = swrite(format="  %%%ds: %%%.0fd bytes,",
                       max(strlen(____n____)), ceil(log10(max(____s____))));
    while (____i____ > 0) {
      write, format=____a____, ____n____(____i____), ____s____(____i____);
      info, symbol_def(____n____(____i____));
      --____i____;
    }
  }
}

func about(____n____, ____a____)
/* DOCUMENT about, pattern;
       -or- about, pattern, 1;
     Search and display documentation about functions (or all symbols if
     second argument is true) matching regular expression PATTERN.  If
     multiple matches are found, the user is prompted to select a subject.
     PATTERN may be a string, or a function or structure definition.  If
     PATTERN is a string with a trailing "/i", the other part of the
     regular expression is interpreted so as to ignore case.

   SEE ALSO help, info, symbol_def, symbol_names,
            strgrep, strcase,
            select_item_in_string_list. */
{
  /* attempt to use _very_ odd names to avoid clash with caller */
  ____a____ = symbol_names((____a____ ? -1 : 2096)); /* get names of functions */
  if (structof(____n____) != string) {
    ____n____ = nameof(____n____);
    if (structof(____n____) != string)
      error, "expecting a string, a function, or a structure definition";
  }
  if (strpart(____n____, -1:0) == "/i") {
    /* There is no 'ignore case' flag in strgrep,
       so we have to emulate this feature... */
    ____n____ = strgrep(strcase(0, strpart(____n____, 1:-2)),
                        strcase(0, ____a____));
  } else {
    ____n____ = strgrep(____n____, ____a____);
  }
  ____n____ = where(____n____(1,..) <= ____n____(2,..));
  if (! is_array(____n____)) {
    write, " Sorry no match found.";
    return;
  }
  ____a____ = ____a____(____n____);
  ____a____ = ____a____(sort(____a____));
  ____a____ = select_item_in_string_list(____a____, bol=" ",
                                         prompt=" Choose one subject: ");
  if (! is_void(____a____)) {
    /* Explicitely filter range operators which have a built-in
       function counterpart to avoid a deadly bug in Yorick. */
    if (____a____ == "sum") {
      write, format="%s\n%s\n%s\n%s\n",
        "/* DOCUMENT sum(x)",
        "     Returns sum of values in array X.",
        "     Can also be used as a range operator.",
        "   SEE ALSO avg, max, min. */";
    } else if (____a____ == "avg") {
      write, format="%s\n%s\n%s\n%s\n",
        "/* DOCUMENT avg(x)",
        "     Returns average of values in array X.",
        "     Can also be used as a range operator.",
        "   SEE ALSO max, min, sum. */";
    } else if (____a____ == "min") {
      write, format="%s\n%s\n%s\n%s\n",
        "/* DOCUMENT min(x)",
        "     Returns minimum value in array X.",
        "     Can also be used as a range operator.",
        "   SEE ALSO avg, max, sum. */";
    } else if (____a____ == "max") {
      write, format="%s\n%s\n%s\n%s\n",
        "/* DOCUMENT max(x)",
        "     Returns maximum value in array X.",
        "     Can also be used as a range operator.",
        "   SEE ALSO avg, min, sum. */";
    } else {
      help, symbol_def(____a____);
    }
  }
}

extern insure_temporary;
/* DOCUMENT insure_temporary, var1 [, var2, ...];
     Insure that symbols VAR1 (VAR2 ...) are temporary variables referring to
     arrays.  Useful prior to in-place operations to avoid side-effects for
     caller.

   SEE ALSO: eq_nocopy, nrefsof, swap, unref. */

extern swap;
/* DOCUMENT swap, a, b;
     Exchanges  the contents  of variables  A and  B without  requiring any
     temporary copy.  The result of the call is identical to:
       tmp = a; a = b; b = tmp;
     which makes  a copy of  A and then  a copy of B.   Another possibility
     which avoids any copy of A nor B is:
       local tmp;
       eq_nocopy, tmp, a; eq_nocopy, a, b; eq_nocopy, b, tmp;

   SEE ALSO: eq_nocopy, unref. */

extern unref;
/* DOCUMENT unref(x)
     Returns X,  destroying X in the process  if it is an  array (useful to
     deal with temporary big arrays).  Written after Yorick's FAQ.

   SEE ALSO: eq_nocopy, swap, nrefsof. */

extern current_include;
/* DOCUMENT current_include()
     If Yorick is parsing a file, this function returns the absolute path
     of this file; otherwise, this function returns nil.

   SEE ALSO include, require. */

extern get_includes;
/* DOCUMENT get_includes()
     Returns an array of strings with the names of all included files so
     far.

   SEE ALSO: set_path, current_include. */

extern mem_base;
extern mem_copy;
extern mem_peek;
/* DOCUMENT mem_base(array);
       -or- mem_copy, address, expression;
       -or- mem_peek(address, type, dimlist);
     Hacker routines  to read/write data  at given memory  location.  These
     routines allow the user to de _very_ nasty but sometimes needed things
     and do  not provide  the safety level  of ususal Yorick  routines, and
     must therefore be used with  extreme care (you've bee warned).  In all
     these routines,  ADDRESS is either a  long integer scalar  or a scalar
     pointer (e.g. &OBJECT).

     mem_base returns the address  (as a long scalar)  of the first element
              of array object ARRAY.  You can use this function if you need
              to add some offset to the address of an object, e.g. to reach
              some particular element of an array or a structure.

     mem_copy copy the contents of EXPRESSION at memory location ADDRESS.

     mem_peek returns  a new  array of data  type TYPE  and dimension  list
              DIMLIST  filled  with  memory  contents starting  at  address
              ADDRESS.

   EXAMPLE
     The following statement converts the contents of complex array Z as an
     array of doubles:
       X = mem_peek(mem_base(Z), double, 2, dimsof(Z));
     then:
       X(1,..) is Z.re
       X(2,..) is Z.im

   SEE ALSO reshape, native_byte_order. */

func native_byte_order(type)
/* DOCUMENT native_byte_order()
       -or- native_byte_order(type)
     Returns the native byte  order, one of: "LITTLE_ENDIAN", "BIG_ENDIAN",
     or  "PDP_ENDIAN".  Optional  argument  TYPE is  an  integer data  type
     (default is long).

   SEE ALSO mem_peek. */
{
  if (is_void(type)) type = long;
  size = sizeof(type);
  (carr = array(char, size))(*) = indgen(size:1:-1);
  value = mem_peek(mem_base(carr), type);
  if (size == 4) {
    if (value == 0x01020304) {
      return "LITTLE_ENDIAN";
    } else if (value == 0x04030201) {
      return "BIG_ENDIAN";
    } else if (value == 0x03040102) {
      return "PDP_ENDIAN";
    }
  } else if (size == 2) {
    if (value == 0x0102) {
      return "LITTLE_ENDIAN";
    } else if (value == 0x0201) {
      return "BIG_ENDIAN";
    }
  }
  error, "unknown byte order";
}

extern make_dimlist;
/* DOCUMENT make_dimlist(arg1, arg2, ...)
 *     -or- make_dimlist, arg1, arg2, ...;
 *
 *   Concatenate all arguments as a single dimension list.  The function
 *   form returns the resulting dimension list whereas the subroutine form
 *   redefines the contents of its first argument which must be a simple
 *   variable reference.  The resulting dimension list is always of the
 *   form [NDIMS, DIM1, DIM2, ...].
 *
 *
 * EXAMPLES
 *
 *   In the following example, a first call to make_dimlist is needed to
 *   make sure that input argument DIMS is a valid dimension list if there
 *   are no other input arguments:
 *
 *       func foo(a, b, dims, ..)
 *       {
 *         // build up dimension list:
 *         make_dimlist, dims;
 *         while (more_args()) make_dimlist, dims, next_arg();
 *         ...;
 *       }
 *
 *   Here is an other example:
 *
 *       func foo(a, b, ..)
 *       {
 *         // build up dimension list:
 *         dims = [0];
 *         while (more_args()) make_dimlist, dims, next_arg();
 *         ...;
 *       }
 *
 *
 * SEE ALSO: array, build_dimlist.
 */

/*---------------------------------------------------------------------------*/
/* SIMPLE INTERACTIVE ROUTINES */

local __select_file_dir;
func select_file(dir, prompt=, width=, forever=, all=, suffix=, pattern=)
/* DOCUMENT select_file()
       -or- select_file(dir)
     Interactively  select name  of an  existing file  starting  at current
     working  directory or at  last selected  directory or  at DIR  if this
     argument  is specified.  The  function returns  full path  of selected
     file or nil  [] if no valid selection is made.   If keyword FOREVER is
     true, a file must be selected for the function to return.

     If keyword ALL  is true, then all files  and directories get displayed
     -- even the "hidden" ones which  name start with a dot.  In any cases,
     the current  and parent  directories ("." and  "..") get  displayed to
     allow the  user to  re-scan the  current directory or  to go  into the
     parent directory.

     Keyword SUFFIX  can be set  to a scalar  string to only  display files
     that match SUFFIX.

     Keyword  PATTERN can be  set to  a regular  expression to  select only
     files that match PATTERN.  For instance,  PATTERN="\\.(tgz|tar\\.gz)$"
     would match any files with suffix ".tgz" or ".tar.gz".

     Keyword WIDTH can  be used to specify a different  text width than the
     default of 79 characters.

     Keyword PROMPT can be set to change the default prompt:
       " Select file/directory: "

   SEE ALSO lsdir, regmatch, pretty_print_string_list. */
{
  /* fool codger */ extern __select_file_dir;
  local dir_list;
  if (is_void(width)) width = 79;
  if (is_void(prompt)) prompt=" Select file/directory: ";
  if (structof(pattern) == string) pattern = regcomp(pattern, nosub=1);
  cwd = get_cwd();
  if (! is_void(dir)) __select_file_dir = dir;
  if (structof(__select_file_dir) != string) {
    __select_file_dir = cwd;
  } else {
    __select_file_dir = cd(__select_file_dir);
  }

  suffix_length = is_void(suffix) ? 0 : strlen(suffix);

  hline = "-------------------------------------";
  for (;;) {
    file_list = lsdir(__select_file_dir, dir_list);
    if (! all) {
      if ((n = numberof(file_list))) {
        i = where(strpart(file_list, 1:1) != ".");
        if (numberof(i) != n) file_list = file_list(i);
      }
      if ((n = numberof(dir_list))) {
        i = where(strpart(dir_list, 1:1) != ".");
        if (numberof(i) != n) dir_list = dir_list(i);
      }
    }
    if (suffix_length > 0 && (n = numberof(file_list))) {
      i = where(strpart(file_list, 1-suffix_length:0) == suffix);
      if (numberof(i) != n) file_list = file_list(i);
    }
    if (pattern && (n = numberof(file_list))) {
      i = where(regmatch(pattern, file_list));
      if (numberof(i) != n) file_list = file_list(i);
    }
    grow, dir_list, ".", ".."; /* use . to allow reading directory again */
    dir_list = dir_list(sort(dir_list));
    list = dir_list + "/";
    if (is_array(file_list)) {
      grow, list, file_list(sort(file_list));
      file_list = [];
    }
    ndirs = numberof(dir_list);
    number = numberof(list);

    /* Print out directory list. */
    text = pretty_print_string_list(list, numbered=": ", width=width,
                                    sep=, eol=" ", bol="| ", maxcols=);
    text_len = strlen(text(1))+1;
    len = max(width, text_len);
    while (strlen(hline) < len) hline += hline;
    head_line = "[" + __select_file_dir + "]";
    n = (len - strlen(head_line) - 2)/2;
    if (n > 0) head_line = strpart(hline, 1:n)+head_line;
    n = len - strlen(head_line) - 2;
    if (n > 0) head_line += strpart(hline, 1:n);
    write, format=",%s.\n", head_line;
    write, format=swrite(format="%%-%ds|\n", len-1), text;
    write, format="`%s'\n", strpart(hline, 1:len-2);
    for (;;) {
      t = string(0);
      k = 0;
      s = rdline(prompt=prompt);
      if (sread(s, format="%d %s", k, t) == 1 && k >= 1 && k <= number) break;
      if (numberof((k = where(list == s))) == 1) {
        k = k(1);
        break;
      }
      if (numberof((k = where(dir_list == s))) == 1) {
        k = k(1);
        break;
      }
      if (! forever) {
        cd, cwd;
        return;
      }
    }
    if (k > ndirs) {
      cd, cwd;
      return __select_file_dir + list(k);
    }
    __select_file_dir = cd(__select_file_dir + dir_list(k));
  }
}

func select_item_in_string_list(list, index=,  prompt=, forever=,
                                numfmt=, width=, sep=, eol=, bol=, maxcols=)
/* DOCUMENT select_item_in_string_list(list)
     Print out  array of strings LIST  (using pretty_print_string_list) and
     interactively ask  the user a number/item  in the list  and return the
     selected item.  If keyword INDEX  is true, the item number is returned
     rather  than its value.   The prompt  string can  be set  with keyword
     PROMPT (default is " Select one  item: ").  If keyword FOREVER is true
     the user is prompted until a valid choice is mage.

     Other  keywords  are passed  to  pretty_print_string_list: NUMFMT  (as
     NUMBERED), WIDTH, SEP, EOL, BOL and MAXCOLS.

   SEE ALSO pretty_print_string_list. */
{
  number = numberof(list);
  pretty_print_string_list, list, numbered=(is_void(numfmt) ? " - " : numfmt),
    width=width, sep=sep, eol=eol, bol=bol, maxcols=maxcols;
  if (is_void(prompt)) prompt=" Select one item: ";
  for (;;) {
    t = string(0);
    k = 0;
    s = rdline(prompt=prompt);
    if (sread(s, format="%d %s", k, t) == 1 && k >= 1 && k <= number) break;
    if (numberof((k = where(list == s))) == 1) {
      k = k(1);
      break;
    }
    if (! forever) return;
  }
  return (index ? k : list(k));
}

func pretty_print_string_list(list, numbered=, width=, start=,
                              sep=, eol=, bol=, maxcols=)
/* DOCUMENT pretty_print_string_list, list;
       -or- pretty_print_string_list(list);
     Write  array of  strings LIST  in  columns.  In  subroutine form,  the
     result is  printed to standard  output; otherwise, return an  array of
     strings (one per row).

     The  maximum  width (in  number  of characters)  of  each  row can  be
     precised with  keyword WIDTH  (default 79).  But  actual width  may be
     larger, since at least one column is produced.

     The maximum number of columns  may be limited by using keyword MAXCOLS
     (by default, there is no limit).

     Keywords BOL,  SEP and  EOL, can be  set to  scalar strings to  use at
     begin-of-line, between each  column, at end-of-line respectively.  SEP
     can  also be  the number  of spaces  to insert  between  columns.  The
     default are: BOL="", SEP=5 (five spaces) and EOL=string(0).

     Keyword NUMBERED can be used to  number items. If NUMBERED is a scalar
     string  which  contains  a "%d",  it  is  used  to format  the  index.
     Otherwise, NUMBERED  is the string or  the number of spaces  to use as
     separator between indices and items.  For instance:
       numbered="[%d] "  yields: "[1] first_item    [2] second_item  ..."
       numbered=" - "    yields: "1 - first_item    2 - second_item  ..."
       numbered=3        yields: "1 first_item      2 second_item ..."
     Keyword START can be used to specify the starting  index for numbering
     items (default START=1).

   SEE ALSO swrite. */
{
  number = numberof(list);
  if (numbered) {
    /* Numbered list. */
    if (is_void(start)) start = 1;
    index = indgen(start:start+number-1);
    if (start >= 0) ndigits = 1 + long(floor(log10(index(0))));
    else ndigits = 2 + long(floor(log10(-start)));
    if (structof(numbered) != string) {
      /* Convert integer into spaces. */
      index = swrite(format=swrite(format="%%%dd%%%ds", ndigits, numbered),
                     index, "");
    } else if (dimsof(numbered)(1)==0 && strmatch(numbered, "%d")) {
      index = swrite(format=numbered, index);
      len = strlen(index);
      if (max(len) != min(len)) {
        /* Justify index list. */
        index = swrite(format=swrite(format="%%%ds", max(len)), index);
      }
    } else {
      /* Use NUMBERED as separator. */
      index = swrite(format=swrite(format="%%%dd%s", ndigits),
                     index) + numbered;
    }
    list = index + list(*);
  } else {
    list = list(*);
  }

  if (is_void(bol)) bol = "";
  if (is_void(eol)) eol = string(0);
  if (is_void(width)) width = 79;
  if (structof(sep) != string) {
    /* Convert margin separator into spaces. */
    if (is_void(sep)) sep = 5;
    sep = swrite(format=swrite(format="%%%ds", sep), "");
  }
  len = max(strlen(list));
  slen = strlen(sep);
  ncols = (width + slen - strlen(bol) - strlen(eol))/(len + slen);
  if (! is_void(maxcols) && ncols > maxcols) ncols = maxcols;
  if (ncols < 1) ncols = 1;
  nrows = (number+ncols-1)/ncols;

  (tmp = array(string, nrows, ncols))(1:number) = unref(list);
  if (ncols > 1) {
    fmt = swrite(format="%%-%ds%s", len, sep);
    for (j=1 ; j<ncols ; ++j) bol += swrite(format=fmt, tmp(,j));
  }
  fmt = (eol ? swrite(format="%%-%ds%s", len, eol) : "%s");
  bol += swrite(format=fmt, tmp(,ncols));
  if (! am_subroutine()) return bol;
  write, format="%s\n", bol;
}

/*---------------------------------------------------------------------------*
 * Local Variables:                                                          *
 * mode: Yorick                                                              *
 * tab-width: 8                                                              *
 * fill-column: 75                                                           *
 * coding: latin-1                                                           *
 * End:                                                                      *
 *---------------------------------------------------------------------------*/
