/*
 *  Zapping (TV viewer for the Gnome Desktop)
 *
 * Copyright (C) 2000 Iaki Garca Etxebarria
 * Copyright (C) 2003 Michael H. Schimek
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* $Id: v4l.c,v 1.11 2005/02/25 18:10:32 mschimek Exp $ */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "zapping_setup_fb.h"

#ifdef ENABLE_V4L

#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <assert.h>

#include "common/videodev.h"
#include "common/_videodev.h"

#define v4l_ioctl(fd, cmd, arg)						\
  (IOCTL_ARG_TYPE_CHECK_ ## cmd (arg),					\
   device_ioctl (log_fp, fprint_v4l_ioctl_arg, fd, cmd, arg))

int
setup_v4l			(const char *		device_name,
				 const tv_overlay_buffer *buffer)
{
  int fd;
  struct video_capability caps;
  struct video_buffer fb;
  const tv_pixel_format *pf;

  message (2, "Opening video device.\n");

  if (-1 == (fd = device_open_safer (device_name, 81, O_RDWR)))
    return -1;

  message (2, "Querying device capabilities.\n");

  if (-1 == v4l_ioctl (fd, VIDIOCGCAP, &caps))
    {
      errmsg ("VIDIOCGCAP ioctl failed,\n  probably not a V4L device");
      close (fd);
      return -1;
    }

  message (1, "Using V4L interface.\n");

  message (2, "Checking overlay capability.\n");

  if (!(caps.type & VID_TYPE_OVERLAY))
    {
      message (1, "Device '%s' does not support video overlay.\n",
	       device_name);
      goto failure;
    }

  message (2, "Getting current FB parameters.\n");

  if (-1 == v4l_ioctl (fd, VIDIOCGFBUF, &fb))
    {
      errmsg ("VIDIOCGFBUF ioctl failed");
      goto failure;
    }

  fb.base		= (void *) buffer->base;
  fb.width		= buffer->format.width;
  fb.height		= buffer->format.height;

  pf = buffer->format.pixel_format;

  if (32 == pf->bits_per_pixel)
    fb.depth		= 32; /* depth 24 bpp 32 */
  else
    fb.depth		= pf->color_depth; /* 15, 16, 24 */

  fb.bytesperline	= buffer->format.bytes_per_line[0];

  message (2, "Setting new FB parameters.\n");

  /* This ioctl is privileged because it sets up
     DMA to a random (video memory) address. */
  {
    int success;
    int saved_errno;

    if (!restore_root_privileges ())
      goto failure;

    success = ioctl (fd, VIDIOCSFBUF, &fb);
    saved_errno = errno;

    drop_root_privileges ();

    if (-1 == success)
      {
	errno = saved_errno;

        errmsg ("VIDIOCSFBUF ioctl failed");

        if (EPERM == saved_errno && ROOT_UID != euid)
	  privilege_hint ();

      failure:
	close (fd);
        return 0;
      }
  }

  close (fd);

  return 1;
}

#else /* !ENABLE_V4L */

int
setup_v4l			(const char *		device_name,
				 const tv_overlay_buffer *buffer)
{
  return -1;
}

#endif /* !ENABLE_V4L */
