/*
    Copyright (c) 2007-2010 iMatix Corporation

    This file is part of 0MQ.

    0MQ is free software; you can redistribute it and/or modify it under
    the terms of the Lesser GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    0MQ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    Lesser GNU General Public License for more details.

    You should have received a copy of the Lesser GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "zmq_encoder.hpp"
#include "i_inout.hpp"
#include "wire.hpp"

zmq::zmq_encoder_t::zmq_encoder_t (size_t bufsize_) :
    encoder_t <zmq_encoder_t> (bufsize_),
    source (NULL),
    trim (false)
{
    zmq_msg_init (&in_progress);

    //  Write 0 bytes to the batch and go to message_ready state.
    next_step (NULL, 0, &zmq_encoder_t::message_ready, true);
}

zmq::zmq_encoder_t::~zmq_encoder_t ()
{
    zmq_msg_close (&in_progress);
}

void zmq::zmq_encoder_t::set_inout (i_inout *source_)
{
    source = source_;
}

void zmq::zmq_encoder_t::trim_prefix ()
{
    trim = true;
}

bool zmq::zmq_encoder_t::size_ready ()
{
    //  Write message body into the buffer.
    if (!trim) {
        next_step (zmq_msg_data (&in_progress), zmq_msg_size (&in_progress),
            &zmq_encoder_t::message_ready, false);
    }
    else {
        size_t prefix_size = *(unsigned char*) zmq_msg_data (&in_progress);
        next_step (
            (unsigned char*) zmq_msg_data (&in_progress) + prefix_size + 1,
            zmq_msg_size (&in_progress) - prefix_size - 1,
            &zmq_encoder_t::message_ready, false);
    }
    return true;
}

bool zmq::zmq_encoder_t::message_ready ()
{
    //  Destroy content of the old message.
    zmq_msg_close(&in_progress);

    //  Read new message from the dispatcher. If there is none, return false.
    //  Note that new state is set only if write is successful. That way
    //  unsuccessful write will cause retry on the next state machine
    //  invocation.
    if (!source || !source->read (&in_progress)) {
        zmq_msg_init (&in_progress);
        return false;
    }

    //  Get the message size. If the prefix is not to be sent, adjust the
    //  size accordingly.
    size_t size = zmq_msg_size (&in_progress);
    if (trim) {
        zmq_assert (size);
        size_t prefix_size =
            (*(unsigned char*) zmq_msg_data (&in_progress)) + 1;
        zmq_assert (prefix_size <= size);
        size -= prefix_size;
    }

    //  Account for the 'flags' byte.
    size++;

    //  For messages less than 255 bytes long, write one byte of message size.
    //  For longer messages write 0xff escape character followed by 8-byte
    //  message size. In both cases empty 'flags' field follows.
    if (size < 255) {
        tmpbuf [0] = (unsigned char) size;
        tmpbuf [1] = (in_progress.flags & ~ZMQ_MSG_SHARED);
        next_step (tmpbuf, 2, &zmq_encoder_t::size_ready,
            !(in_progress.flags & ZMQ_MSG_TBC));
    }
    else {
        tmpbuf [0] = 0xff;
        put_uint64 (tmpbuf + 1, size);
        tmpbuf [9] = (in_progress.flags & ~ZMQ_MSG_SHARED);
        next_step (tmpbuf, 10, &zmq_encoder_t::size_ready,
            !(in_progress.flags & ZMQ_MSG_TBC));
    }
    return true;
}
