package Zim::Formats::Html;

use strict;

our $VERSION = '0.15';

=head1 NAME

Zim::Formats::Html - Html dumper for zim

=head1 DESCRIPTION

This is a dumper that can be used to export zim wiki pages to html.

=head1 METHODS

=over 4

=item C<save_tree(IO, TREE, PAGE)>

Converts a parse tree into plain text.

=cut

sub save_tree {
	my ($class, $io, $tree, $page) = @_;

	my $template = $page->{repository}{_template};
	die "Html format needs a template\n" unless $template;
	my ($head, $foot) = split /\[\%\s*BODY\s*\%\]/, _read_file($template), 2;
	my $title = $page->basename;
	if (ref $$tree[2] and $$tree[2][0] =~ /^head/) {
		# FIXME better logic for this ?
		$title = $$tree[2];
		$$tree[2] = '';
		$title = join '', @$title[2 .. $#$title];
	}
	my $prev = $page->{repository}{_prev}->name if $page->{repository}{_prev};
	my $next =  $page->{repository}{_next}->name if $page->{repository}{_next};
	my @path = $page->namespaces;
	my $root = '../' x scalar @path || './';
	$page->{_root_dir} = $root;
	for my $p ($head, $foot) {
		$p =~ s/\[\%\s*$$_[0]\s*\%\]/$$_[1]/ig for
			[TITLE       => $title              ],
			[PREV        => $prev               ],
			[PREV_FILE   => _href($page, $prev) ],
			[NEXT        => $next               ],
			[NEXT_FILE   => _href($page, $next) ],
			[ZIM_VERSION => $Zim::VERSION       ],
			[ROOT_DIR    => $root               ] ;
	}
	
	my $old_fh = select $io;
	eval {
		print $head;
		_dump_node($tree, $page);
		print $foot;
	};
	select $old_fh;
	die $@ if $@;
}

sub _dump_node {
	no warnings;
	
	my ($node, $page) = @_;
	my ($name, $opt) = splice @$node, 0, 2;
	
	my @tags;
	if ($name =~ /^head(\d+)/) { @tags = ("<h$1>", "</h$1>\n") }
	elsif ($name eq 'Para') {
		@tags = ("<p>\n", "</p>\n");
		if (! ref $$node[0] and $$node[0] =~ /^\s*\*\s+/m) {
			# ugly hack ... should not be necessary
			parse_list($page, @$node);
			@$node = ();
		}
	}
	elsif ($name eq 'Verbatim') {
		@tags = ("<blockquote>\n<pre>\n", "</pre>\n</blockquote>\n")
	}
	elsif ($name eq 'bold'     ) { @tags = ('<b>', '</b>')   }
	elsif ($name eq 'italic'   ) { @tags = ('<i>', '</i>')   }
	elsif ($name eq 'underline') { @tags = ('<u>', '</u>')   }
	elsif ($name eq 'strike')    { @tags = ('<strike>', '</strike>') }
	elsif ($name eq 'verbatim' ) { @tags = ('<tt>', '</tt>') }
	elsif ($name eq 'link') {
		my ($type, $href) = $page->parse_link($$opt{to});
		$href = _href($page, $page->resolve_link($$opt{to})->name)
			if $type eq 'page';
		print STDERR "  link: $$opt{to} => $href\n";
		@tags = ("<a href='$href'>", "</a>");
	}
	elsif ($name eq 'image') {
		@tags = ("<div><img src='$$opt{src}' /></div>", '');
	}
	# else recurs silently

	print $tags[0];
	for (@$node) {
		if (ref $_) { _dump_node($_, $page) } # recurse
		else {
			$_ =~ s/([^-\n\t !\#\$\%\(\)\*\+,\.\~\/\:\;=\?\@\[\\\]\^_\`\{\|\}abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789])/'&#'.(ord($1)).';'/eg; # RegEx copy pasted from Pod::Simple::Html
			print $_;
		}
	}
	print $tags[1];
}

sub _href {
	my ($page, $href) = @_;
	my $namespace = $page->namespace;
	#print STDERR "page: $href, relative: $namespace => ";
	$href =~ s#^:*#:#;
	unless ($href =~ s#^($namespace):*##) {
		$href =~ s/^:*//;
		$href = $page->{_root_dir}.$href;
	}
	$href =~ s#:+$##;
	$href =~ s#[/:]+#/#g;
	$href .= '.html';
	#print STDERR $href."\n";
	return $href;
}

sub parse_list { # FIXME totally wrong level for this kind of parsing :(
	no warnings;

	my ($page, @nodes) = @_;
	
	print "<ul>\n";
	my ($level, $close);
	for (@nodes) {
		if (ref $_) { _dump_node($_, $page) }
		else {
			$_ =~ s/([^-\n\t !\#\$\%\(\)\*\+,\.\~\/\:\;=\?\@\[\\\]\^_\`\{\|\}abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789])/'&#'.(ord($1)).';'/eg; # RegEx copy pasted from Pod::Simple::Html
			my @lines = split /[\n\r]+/, $_;
			for (@lines) {
				if ($_ =~ s/^(\s*)\*\s+//) {
					print $close;
					my $lvl = length $1;
					if ($lvl > $level) {
						print "<ul>\n" for 1 .. $lvl - $level;
					}
					elsif ($lvl < $level) {
						print "</ul>\n" for 1 .. $level - $lvl;
					}
					$level = $lvl;
					print "<li>".$_;
					$close = "</li>\n";
				}
				else { print $_ }
			}
		}
	}
	print $close;
	print "</ul>\n" for 1 .. $level + 1;
}

sub _read_file {
	my $file = shift;
	open FILE, "<$file" or die "Could not read file: $file\n";
	my $text = join '', <FILE>;
	close FILE;
	return $text;
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

