package Zim::GUI::PreferencesDialog;

use strict;
use File::BaseDir qw/xdg_data_files xdg_data_dirs xdg_data_home/;
use Gtk2;

our $VERSION = '0.17';

=head1 NAME

Zim::GUI::PreferencesDialog - dialog for zim

=head1 DESCRIPTION

This module contains the preferences dialog for zim.
It is autoloaded when needed.

=head1 METHODS

=over 4

=item C<new()>

Simple constructor

=cut

sub new {
	my ($class, %self) = @_;
	bless \%self, $class;
}

=item C<show()>

Present the dialog.

=cut

sub show {
	my $self = shift;
	my $app = $self->{app};
	my $settings = $self->{app}{settings};
	
	my $dialog = Gtk2::Dialog->new(
		"Preferences - Zim", $app->{window},
	       	[qw/destroy-with-parent no-separator/],
		'gtk-help'   => 'help',
		'gtk-cancel' => 'cancel',
		'gtk-ok'     => 'ok',
	);
	$dialog->set_resizable(0);
	
	my $tabs = Gtk2::Notebook->new;
	$dialog->vbox->add($tabs);
	
	## General tab
	my $vbox1 = Gtk2::VBox->new(0,5);
	$vbox1->set_border_width(5);
	$tabs->append_page($vbox1, 'General');
	
	my @settings = qw/date_string browser file_browser email_client/;
	my @labels = (
		'Date format',
		'Browser',
		'File browser',
		'Email client',
		'Font'	);
	my @entries;
	my $table = Gtk2::Table->new(scalar(@labels)+1, 2);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$vbox1->pack_start($table,0,1,0);
	my $i = 0;
	for (@labels) {
		my $label = Gtk2::Label->new($_.':');
		my $align = Gtk2::Alignment->new(0,0.5, 0,0);
		$align->add($label);
		$table->attach_defaults($align, 0,1, $i,$i+1);
		if (/Font/) {
			my $font = $$settings{textfont} ||
				$app->PageView->get_style->font_desc->to_string;
			#warn "font: $font\n";
			my $button = Gtk2::FontButton->new_with_font($font);
			$button->signal_connect(font_set => sub {
					my $string = $button->get_font_name;
					$$settings{textfont} = $string;
					$app->PageView->set_font($string);
			} );
			$table->attach_defaults($button, 1,2, $i,$i+1);
		}
		else {
			my $entry = Gtk2::Entry->new;
			$entry->set_text( $$settings{$settings[$i]} );
			push @entries, $entry;
			$table->attach_defaults($entry, 1,2, $i,$i+1);
		}
		$i++;
	}
	my $tearoff =  Gtk2::CheckButton->new_with_label('Tearoff menus');
	$table->attach_defaults($tearoff, 0,2, $i,$i+1);
	$tearoff->set_active($$settings{tearoff_menus});

	## Editing tab
	my $vbox3 = Gtk2::VBox->new(0,5);
	$vbox3->set_border_width(5);
	$tabs->append_page($vbox3, 'Editing');
	my @edit_settings = qw/
		follow_new_link
		use_camelcase
		use_linkfiles
		use_utf8_ent
		backsp_unindent
		use_autoselect
		follow_on_enter
		use_ctrl_space
	/;
	my @edit_labels = (
		'Follow new link',
		'Use CamelCase',
		'Auto-link files',
		'Use utf8 entities',
		'Backspace un-indents',
       		'Use auto-select',
		'Follow link on <enter>',
		'Use "Ctrl-Space" to switch focus' );
	my @edit_tooltips = (
		"Creating a new link directly\nopens the page",
		"CamelCase words will be\nturned into links automaticly",
		"Files will be turned\ninto links automaticly",
		"Use autoformatting to type\nutf8 entities",
		"Use the backspace key to\nun-indent bullet lists",
		"Automaticly select the\ncurrent word when you\ntoggle the format",
		"Follow links using the\n<enter> key. If disabled\nyou still can use <alt><enter>",
		"Use the \"Ctrl-Space\" key combo\nto switch focus between text\nand side pane. If disabled\nyou can still use \"Alt-Space\".",
	);
	my @toggles;
	my $tooltips = Gtk2::Tooltips->new;
	$table = Gtk2::Table->new(scalar(@edit_labels), 1);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$vbox3->pack_start($table, 0,1,0);
	$i = 0;
	for (@edit_labels) {
		my $box = Gtk2::CheckButton->new_with_label($_);
		$box->set_active($$settings{$edit_settings[$i]});
		$tooltips->set_tip($box, $edit_tooltips[$i]);
		$table->attach_defaults($box, 0,1, $i,$i+1);
		$i++;
		push @toggles, $box;
	}

	## Plugins tab
	my $vbox2 = Gtk2::VBox->new(0,5);
	$vbox2->set_border_width(5);
	$tabs->append_page($vbox2, 'Plugins');

	$vbox2->pack_start(Gtk2::Label->new(
		"You need to restart the application\nfor plugin changes to take effect."), 0,1,0 );
	my @plugins;
	my @plugged = split /,/, $$settings{plugins};
	for (xdg_data_home(), xdg_data_dirs()) {
		my $dir = File::Spec->catdir($_, 'zim', 'plugins');
		next unless -d $dir;
		for my $f (Zim::File->list_dir($dir)) {
			next unless $f =~ s/\.pl$//;
			next if grep {$$_[0] eq $f} @plugins;
			my $bit = grep {$_ eq $f} @plugged;
			push @plugins, [$f, $bit];
		}
	}
	@plugins = sort {$$a[0] cmp $$b[0]} @plugins;
	
	$table = Gtk2::Table->new(scalar(@plugins), 1);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$vbox2->pack_start($table, 0,1,0);
	$i = 0;
	for (@plugins) { # TODO add descriptions
		my $box = Gtk2::CheckButton->new_with_label($$_[0]);
		$box->set_active($$_[1]);
		$table->attach_defaults($box, 0,1, $i,$i+1);
		$i++;
		push @$_, $box;
	}
	
	## Show it all
	$dialog->show_all;
	while ($_ = $dialog->run) {
		if ($_ eq 'help') {
			$app->ShowHelp(':zim:usage:preferences');
			next;
		}
		last unless $_ eq 'ok';
		
		# set setting
		$$settings{$settings[$_]} = $entries[$_]->get_text
			for 0 .. $#settings;
			
		unless (length $$settings{date_string}) {
			$$settings{date_string} = '%A %d/%m/%Y';
			$app->{repository}{date_string} = '%A %d/%m/%Y';
		}
		else { $app->{repository}{date_string} = $$settings{date_string} } # Tmp hack till more fully template support gets implemented
		
		$$settings{tearoff_menus} = $tearoff->get_active || '0';
		$app->{ui}->set_add_tearoffs($$settings{tearoff_menus});

		# set editing settings
		$$settings{$edit_settings[$_]} = $toggles[$_]->get_active || '0'
			for 0 .. $#edit_settings;
		$app->PageView->{htext}{follow_on_enter} = $$settings{follow_on_enter}; # FIXME signal for this ?
		
		# set plugins
		@plugged = map $$_[0], grep $$_[2]->get_active, @plugins;
		$$settings{plugins} = join ',', @plugged;

		last;
	}
	$dialog->destroy;
	$app->save_config;
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2006 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

