package Zim::Formats::Html;

use strict;
use Encode;
use Zim::Template;
use Zim::Formats;

our $VERSION = '0.18';
our @ISA = qw/Zim::Formats/;

=head1 NAME

Zim::Formats::Html - Html dumper for zim

=head1 DESCRIPTION

This is a dumper that can be used to export zim wiki pages to html.

=head1 METHODS

=over 4

=item C<save_tree(IO, TREE, PAGE)>

Converts a parse tree into plain text.

=cut

our %entities = (
	'&' => 'apos',
	'<' => 'lt',
	'>' => 'gt',
);
	
sub save_tree { # HACK: template and temp_data are unsupported args
	my ($class, $io, $tree, $page, $template, $temp_data) = @_;
	#use Data::Dumper; warn Dumper $tree;

	$template ||= $page->{repository}{_template} || '';
	$template = length($template)
		? Zim::Template->new($template)
		: Zim::Template->new(\'[% page.body %]')
		unless ref $template ;

	my ($title, $heading) = ($page->basename, '');
	if (ref $$tree[2] and $$tree[2][0] =~ /^head/) {
		# FIXME This logic belongs somewhere central
		$title = $$tree[2];
		$$tree[2] = '';
		$title = join '', @$title[2 .. $#$title];
		$heading = $title;
	}
	# else above defaults

	my @path = $page->namespaces;
	my $root = '../' x scalar @path || './';
	$page->{_root_dir} = $root;

	my $data = {
		page => {
			name => '',
			namespace => '',
			title => $title,
			heading => $heading,
			links => sub {
				[ map { {name => $_, file => href($page, $_)} }
					sort $page->list_links ]
			},
			backlinks => sub {
				return [ map { {name => $_, file => href($page, $_)} }
					sort $page->list_backlinks ] ;
			},
		},
		prev => {},
		next => {},
		repository => { root => $root },
		zim => { version => $VERSION },
	};
	$data->{page}{body} = sub {
		my (undef, $fh) = @_;
		return unless $fh;
		my $old_fh = select $fh;
		eval {
			$page->{_mode} = lc $data->{mode};
			print qq#<div class="slide">\n# if $page->{_mode} eq 's5';
			$class->_dump_node($tree, $page);
			print qq#</div>\n# if $page->{_mode} eq 's5';
		};
		select $old_fh;
		die $@ if $@;
		return undef; # no caching here
	};
	if ($page->{repository}{_prev}) {
		my $prev = $page->{repository}{_prev}->name;
		$data->{prev} = {
			name => $prev,
			file => href($page, $prev),
		};
	}
	if ($page->{repository}{_next}) {
		my $next = $page->{repository}{_next};
		$data->{next} = {
			name => $next,
			file => href($page, $next),
		};
	}
	$data = {%$data, %$temp_data} if $temp_data;

	$template->process($data => $io);
}

sub _dump_node {
	my $class = shift;
	no warnings;
	
	my ($node, $page) = @_;
	my ($name, $opt) = splice @$node, 0, 2;
	
	my @tags;
	if ($name =~ /^head(\d+)/) { @tags = ("<h$1>", "</h$1>\n") }
	elsif ($name eq 'Para') {
		@tags = ("<p>\n", "\n</p>\n");
		if (grep {!ref($_) and /^\s*\x{2022}\s+/m} @$node) {
			# ugly hack ... should not be necessary
			$class->parse_list($page, @$node);
			@$node = ();
		}
		elsif ( $page->{_mode} eq 's5'
				and ! ref $$node[0]
				and $$node[0] =~ s/^----+\s+//
		) {
			print qq#</div>\n\n\n<div class="slide">\n#;
			shift @$node unless $$node[0] =~ /\S/;
		}
	}
	elsif ($name eq 'Verbatim' ) {  @tags = ("<pre>\n", "</pre>\n") }
	elsif ($name eq 'bold'     ) { @tags = ('<b>', '</b>')   }
	elsif ($name eq 'italic'   ) { @tags = ('<i>', '</i>')   }
	elsif ($name eq 'underline') { @tags = ('<u>', '</u>')   }
	elsif ($name eq 'strike'   ) { @tags = ('<strike>', '</strike>') }
	elsif ($name eq 'verbatim' ) { @tags = ('<tt>', '</tt>') }
	elsif ($name eq 'link') {
		my ($type, $href) = $page->parse_link($$opt{to});
		if ($type eq 'page') {
			$href = href($page, $page->resolve_name($href));
		}
		elsif ($type eq 'file') {
			$href = href_file($href);
		}
		warn "#  link: $$opt{to} => $href\n";
		@tags = ("<a href='$href'>", "</a>");
	}
	elsif ($name eq 'image') {
		my $src = href_file($$opt{src});
		my $alt = $src;
		$alt =~ s#.*[/\\]##g;
		@tags = ("<img src='$src' alt='$name' />", '');
	}
	# else recurs silently

	print $tags[0];
	for (@$node) {
		if (ref $_) { $class->_dump_node($_, $page) } # recurse
		else {
			s#([<>&])#\&$entities{$1}\;#g; # encode
			s#\n#<br />\n#g unless $tags[0] =~ m#<pre>#;
			print $_;
		}
	}
	print $tags[1];
}

=item C<href(PAGE, LINK)>

Function that returns an url for a link.

=item C<href_file(FILE)>

Function that returns an url for a file link.

=cut

sub href_file {
	my $href = shift;
	$href = Encode::encode_utf8($href); # turn utf8 flag off
	$href =~ s{ ([^A-Za-z0-9\-\_\.\!\~\*\'\(\)\/\:]) }
	          { sprintf("%%%02X",ord($1))            }egx;
	# url encoding - char set from man uri(7), see relevant rfc
	# added '/' and ':' to char set for readability of uris
	return $href;
}

sub href {
	my ($page, $href) = @_;
	return '' unless length $href;
	my $namespace = $page->namespace;
	#print STDERR "page: $href, relative: $namespace => ";
	$href =~ s#^:*#:#;
	unless ($href =~ s#^(\Q$namespace\E):*##) {
		$href =~ s/^:*//;
		$href = $page->{_root_dir}.$href;
	}
	$href =~ s#:+$##;
	$href =~ s#[/:]+#/#g;
	$href .= '.html';
	#print STDERR $href."\n";
	$href = Encode::encode_utf8($href); # turn utf8 flag off
	$href =~ s{ ([^A-Za-z0-9\-\_\.\!\~\*\'\(\)\/\:]) }
	          { sprintf("%%%02X",ord($1))            }egx;
	# url encoding - char set from man uri(7), see relevant rfc
	# added '/' and ':' to char set for readability of uris
	return $href;
}

sub parse_list { # FIXME wrong level for this kind of parsing :(
	my $class = shift;
	no warnings;

	my ($page, @nodes) = @_;
	
	my ($level, $close) = (-1, '');
	for (@nodes) {
		if (ref $_) { $class->_dump_node($_, $page) }
		else {
			s#([<>&])#\&$entities{$1}\;#g; # encode
			my @lines = split /[\n\r]+/, $_;
			for (@lines) {
				if ($_ =~ s/^(\s*)\x{2022}\s+//) {
					print $close;
					my $lvl = length $1;
					if ($lvl > $level) {
						print "<ul>\n" for 1 .. $lvl - $level;
					}
					elsif ($lvl < $level) {
						print "</ul>\n" for 1 .. $level - $lvl;
					}
					$level = $lvl;
					print "<li>".$_;
					$close = "</li>\n";
				}
				else { print $_ }
			}
		}
	}
	print $close;
	print "</ul>\n" for 1 .. $level + 1;
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

